﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <cstring>
#include <functional>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_Account.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_AccountCommand.h"
#include "accounts/DevMenuCommand_AccountUtil.h"

// system wide
#define NN_DEVMENU_COMMAND_ADD_USAGE        "account add [--register_nsa] [--count <count, 1..8>]"
#define NN_DEVMENU_COMMAND_DELETE_USAGE     "account delete --index <index, 0..8> [--keep_savedata]"
#define NN_DEVMENU_COMMAND_UNREGISTER_NSA_USAGE "account unregister_nsa --index <index, 0..8> [--keep_savedata]"
#define NN_DEVMENU_COMMAND_CLEARALL_USAGE   "account clear_all [--keep_savedata]"
#define NN_DEVMENU_COMMAND_LIST_USAGE       "account list"
#define NN_DEVMENU_COMMAND_ENABLE_USAGE     "account enable <key>"
#define NN_DEVMENU_COMMAND_DISABLE_USAGE    "account disable <key>"
// system wide / Per user
#define NN_DEVMENU_COMMAND_SET_USAGE        "account set [--index <index, 0..8>] <key>=<value>"
// Per user
#define NN_DEVMENU_COMMAND_REFRESH_USAGE    "account refresh --index <index, 0..8> <key>"
#define NN_DEVMENU_COMMAND_LINK_USAGE       "account link --index <index, 0..8> --id <email or login ID> --password <password>"

namespace {
nn::account::Uid GetUidFromIndex(const char* indexString) NN_NOEXCEPT
{
    if (indexString && std::strlen(indexString) == 1 && std::isdigit(indexString[0]))
    {
        int count;
        nn::account::Uid users[nn::account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, std::extent<decltype(users)>::value));
        NN_LOG("[INFO] index counld be in [0,%d]\n", count);

        auto index = indexString[0] - '0';
        if (index >= 0 && index < count)
        {
            return users[index];
        }
    }
    return nn::account::InvalidUid;
}
bool EnsureNpnsJid() NN_NOEXCEPT
{
    char jid[nn::npns::JidLength];
    if (!nn::npns::GetJid(jid, sizeof(jid)).IsSuccess())
    {
        ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(nn::npns::CreateJid());
        NN_LOG("[INFO] JID is newly created\n");
    }
    return true;
}

void UsageForAccountLinkCommand() NN_NOEXCEPT
{
    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_LINK_USAGE "\n");
}

bool AccountLinkCommand(const Option& option) NN_NOEXCEPT
{
    if (!(true
        && option.HasKey("--index")
        && option.HasKey("--id")
        && option.HasKey("--password")))
    {
        UsageForAccountLinkCommand();
        return false;
    }

    nn::account::Uid uid = GetUidFromIndex(option.GetValue("--index"));
    if (!uid)
    {
        UsageForAccountLinkCommand();
        return false;
    }

    accounts::NetworkConnection connection;
    ACCOUNTS_FAILURE_UNLESS(connection.IsAvailable());
    ACCOUNTS_FAILURE_UNLESS(EnsureNpnsJid());

    return accounts::LinkNa(uid, option.GetValue("--id"), option.GetValue("--password"));
}

void UsageForAccountAddCommand() NN_NOEXCEPT
{
    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ADD_USAGE "\n");
}
bool AccountAddCommand(const Option& option) NN_NOEXCEPT
{
    // 作成数の取得
    int count = 1;
    if (option.HasKey("--count"))
    {
        bool success = false;
        auto countString = option.GetValue("--count");
        if (countString && std::strlen(countString) == 1 && std::isdigit(countString[0]))
        {
            int current;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&current));
            NN_LOG("[INFO] count counld be in [1,%d]\n", nn::account::UserCountMax - current);

            count = countString[0] - '0';
            if (count >= 1 && count <= nn::account::UserCountMax - current)
            {
                success = true;
            }
        }
        if (!success)
        {
            UsageForAccountAddCommand();
            return false;
        }
    }

    // NSA 作成の事前条件を整理
    bool toRegisterNsa = option.HasKey("--register_nsa");
    nn::util::optional<accounts::NetworkConnection> connection;
    if (toRegisterNsa)
    {
        connection.emplace();
        ACCOUNTS_FAILURE_UNLESS(connection->IsAvailable());
        ACCOUNTS_FAILURE_UNLESS(EnsureNpnsJid());
    }

    // UA 作成処理
    for (int i = 0; i < count; ++i)
    {
        NN_LOG("%d of %d\n", i + 1, count);
        nn::account::Uid u;
        if (!(true
            && accounts::CreateUser(&u)
            && (!toRegisterNsa || accounts::RegisterNsa(u))
            && accounts::SynchronizeProfileIfNsaAvailable(u)))
        {
            NN_LOG("[ERROR] Failed to add UA\n");
            return false;
        }
    }
    return true;
}

bool AccountDeleteCommand(const Option& option) NN_NOEXCEPT
{
    // 引数解析
    nn::account::Uid uid = nn::account::InvalidUid;;
    bool keepSaveData = false;
    if (option.HasKey("--index"))
    {
        uid = GetUidFromIndex(option.GetValue("--index"));
    }
    if (option.HasKey("--keep_savedata"))
    {
        keepSaveData = true;
    }

    // 引数チェック
    if (!uid)
    {
        NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_DELETE_USAGE "\n");
        return false;
    }

    accounts::NetworkConnection connection;
    if (!connection.IsAvailable())
    {
        NN_LOG("[WARN] Internet connection unavailable\n");
    }
    return true
        && accounts::UnregisterNsa(uid, keepSaveData)
        && accounts::DeleteUser(uid, keepSaveData);
}

bool UnregisterNsaCommand(const Option& option) NN_NOEXCEPT
{
    // 引数解析
    nn::account::Uid uid = nn::account::InvalidUid;;
    bool keepSaveData = false;
    if (option.HasKey("--index"))
    {
        uid = GetUidFromIndex(option.GetValue("--index"));
    }
    if (option.HasKey("--keep_savedata"))
    {
        keepSaveData = true;
    }

    // 引数チェック
    if (!uid)
    {
        NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_UNREGISTER_NSA_USAGE "\n");
        return false;
    }

    accounts::NetworkConnection connection;
    ACCOUNTS_FAILURE_UNLESS(connection.IsAvailable());

    return accounts::UnregisterNsa(uid, keepSaveData);
}

bool AccountClearAllCommand(const Option& option) NN_NOEXCEPT
{
    bool toKeepSavedata = option.HasKey("--keep_savedata");

    accounts::NetworkConnection connection;
    if (!connection.IsAvailable())
    {
        NN_LOG("[WARN] Internet connection unavailable\n");
    }

    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_RESULT_DO(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));

    for (int i = 0; i < count; ++i)
    {
        NN_LOG("%d of %d\n", i + 1, count);
        auto& u = users[i];
        if (!(true
            && accounts::UnregisterNsa(u, toKeepSavedata)
            && accounts::DeleteUser(u, toKeepSavedata)))
        {
            return false;
        }
    }
    return true;
}

bool AccountListCommand(const Option& option) NN_NOEXCEPT
{
    NN_UNUSED(option);

    int count;
    nn::account::Uid users[nn::account::UserCountMax];
    NN_RESULT_DO(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));

    for (int i = 0; i < count; ++ i)
    {
        auto& u = users[i];
        char uidString[64];
        accounts::UidToString(uidString, sizeof(uidString), u);
        NN_LOG("%s\n", uidString);
    }
    return true;
}

const struct
{
    const char* name;
    void(*accessor)(bool enable);
} Configs[] = {
    {
        "fwdbg:require_na_for_nwservice",
        [] (bool enable) {
            nn::settings::fwdbg::SetSettingsItemValue("account", "na_required_for_network_service", &enable, sizeof(enable));
        }
    },
    {
        "fwdbg:na_license_verification_enabled",
        [] (bool enable) {
            nn::settings::fwdbg::SetSettingsItemValue("account", "na_license_verification_enabled", &enable, sizeof(enable));
        }
    },
    {
        "cfg:skip_selection_if_possible",
        [] (bool enable) {
            nn::settings::system::AccountSettings cfg;
            nn::settings::system::GetAccountSettings(&cfg);
            if (enable)
            {
                cfg.userSelectorSettings.flags.Set<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>();
            }
            else
            {
                cfg.userSelectorSettings.flags.Reset<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>();
            }
            nn::settings::system::SetAccountSettings(cfg);
        }
    },
};

bool Configure(const Option& option, bool value) NN_NOEXCEPT
{
    auto target = option.GetTarget();
    for (auto c : Configs)
    {
        if (strcmp(c.name, target) == 0)
        {
            c.accessor(value);
            NN_LOG("NOTE: Required to reset development kit to take effect.\n");
            return true;
        }
    }
    NN_LOG("ERROR: Specified config not found.\n");
    NN_LOG(" - Unknown config specified: %s\n", target);
    NN_LOG(" - Available commands are:\n");
    for (auto c : Configs)
    {
        NN_LOG("    - %s\n", c.name);
    }
    return false;
}
bool AccountEnableCommand(const Option& option) NN_NOEXCEPT
{
    return Configure(option, true);
}
bool AccountDisableCommand(const Option& option) NN_NOEXCEPT
{
    return Configure(option, false);
}

bool AccountSetCommand(const Option& option) NN_NOEXCEPT
{
    // 引数解析
    nn::account::Uid uid = nn::account::InvalidUid;;
    char key[128] = {};
    char value[128] = {};
    if (option.HasKey("--index"))
    {
        uid = GetUidFromIndex(option.GetValue("--index"));
        if (!uid)
        {
            return false;
        }
    }
    if (option.HasTarget())
    {
        auto target = option.GetTarget();
        auto split = std::strchr(target, '=');
        if (split
            && (split - target) > 0 && (split - target) < sizeof(key)
            && strlen(split + 1) > 0 && strlen(split + 1) < sizeof(value))
        {
            std::strncpy(key, target, split - target);
            key[split - target] = '\0';
            std::strcpy(value, split + 1);
            NN_LOG("key=%s, value=%s\n", key, value);
        }
    }

    // 引数チェック
    if (!(key[0] != '\0' && value[0] != '\0'))
    {
        NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_USAGE "\n");
        return false;
    }

    struct Handler
    {
        const char* key;
        std::function<bool(char* value)> handler;
    } handlers[] = {
        {
            "fwdbg:daemon.background_awaking_periodicity",
            [&](const char* value) -> bool
            {
                auto v = std::strtoul(value, nullptr, 10);
                nn::settings::fwdbg::SetSettingsItemValue("account.daemon", "background_awaking_periodicity", &v, sizeof(v));
                return true;
            },
        },
        {
            "fwdbg:daemon.schedule_periodicity",
            [&](const char* value) -> bool
            {
                auto v = std::strtoul(value, nullptr, 10);
                nn::settings::fwdbg::SetSettingsItemValue("account.daemon", "schedule_periodicity", &v, sizeof(v));
                return true;
            },
        },
        {
            "fwdbg:daemon.profile_sync_interval",
            [&](const char* value) -> bool
            {
                auto v = std::strtoul(value, nullptr, 10);
                nn::settings::fwdbg::SetSettingsItemValue("account.daemon", "profile_sync_interval", &v, sizeof(v));
                return true;
            },
        },
        {
            "fwdbg:daemon.na_info_refresh_interval",
            [&](const char* value) -> bool
            {
                auto v = std::strtoul(value, nullptr, 10);
                nn::settings::fwdbg::SetSettingsItemValue("account.daemon", "na_info_refresh_interval", &v, sizeof(v));
                return true;
            },
        },
        {
            "dbg:user_state",
            [&](const char* value) -> bool
            {
                if (!uid)
                {
                    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_USAGE "\n");
                    return false;
                }

                nn::account::NetworkServiceAccountAdministrator admin;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));

#define SET_USER_STATE_INVALID_IF_MATCH(category, state) \
                else if (std::strcmp(value, #category ":" #state) == 0) \
                    ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(admin.DebugSetNetworkServiceAccountAvailabilityError(nn::account::Result ## category ## state ()))

                if (NN_STATIC_CONDITION(false))
                {
                    // Unreachable
                }
                SET_USER_STATE_INVALID_IF_MATCH(NetworkServiceAccount, CredentialBroken);
                SET_USER_STATE_INVALID_IF_MATCH(NetworkServiceAccount, Unmanaged);
                SET_USER_STATE_INVALID_IF_MATCH(NetworkServiceAccount, Banned);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, LinkageBroken);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateOtherButInteractionRequired);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateDeleted);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateBanned);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateSuspended);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateWithdrawn);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateTermsAgreementRequired);
                SET_USER_STATE_INVALID_IF_MATCH(NintendoAccount, StateReauthorizationRequired);
                else
                {
                    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_USAGE "\n");
                    NN_LOG(" - Unknown value specified for key='%s'\n", key);
                    return false;
                }
#undef SET_USER_STATE_INVALID_IF_MATCH
                return true;
            },
        },
    };

    for (const auto& h : handlers)
    {
        if (std::strncmp(key, h.key, sizeof(key)) == 0)
        {
            return h.handler(value);
        }
    }

    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_USAGE "\n");
    NN_LOG(" - Unknown key specified: %s\n", key);
    NN_LOG(" - Available keys are:\n");
    for (auto c : handlers)
    {
        NN_LOG("    - %s\n", c.key);
    }
    return false;
} // NOLINT(readability/fn_size)

bool AccountRefreshCommand(const Option& option) NN_NOEXCEPT
{
    // 引数解析
    nn::account::Uid uid = nn::account::InvalidUid;;
    char key[128] = {};
    if (option.HasKey("--index"))
    {
        uid = GetUidFromIndex(option.GetValue("--index"));
    }
    if (option.HasTarget())
    {
        auto target = option.GetTarget();
        if (strlen(target) > 0 && strlen(target) < sizeof(key))
        {
            std::strncpy(key, target, sizeof(key));
        }
    }

    // 引数チェック
    if (!(uid && key[0] != '\0'))
    {
        NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_REFRESH_USAGE "\n");
        return false;
    }

    struct Handler
    {
        const char* key;
        std::function<bool()> handler;
    } handlers[] = {
        {
            "profile",
            [&]() -> bool
            {
                nn::account::NetworkServiceAccountAdministrator admin;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));
                nn::account::AsyncContext ctx;
                ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(admin.SynchronizeProfileAsync(&ctx));
                nn::os::SystemEvent e;
                NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
                e.Wait();
                ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
                return true;
            },
        },
        {
            "na_info_cache",
            [&]() -> bool
            {
                nn::account::NetworkServiceAccountManager manager;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, uid));
                nn::account::AsyncContext ctx;
                ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNintendoAccountInfoAsync(&ctx));
                nn::os::SystemEvent e;
                NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
                e.Wait();
                ACCOUNTS_FAILURE_UNLESS_RESULT_SUCCESS(ctx.GetResult());
                return true;
            },
        },
    };

    for (const auto& h : handlers)
    {
        if (std::strncmp(key, h.key, sizeof(key)) == 0)
        {
            return h.handler();
        }
    }

    NN_LOG("usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_REFRESH_USAGE "\n");
    NN_LOG(" - Unknown key specified: %s\n", key);
    return false;
}
} // ~namespace <anonymous>

//------------------------------------------------------------------------------------------------

nn::Result AccountCommand(bool* outValue, const Option& option) NN_NOEXCEPT
{
    static const struct
    {
        std::string name;
        bool (*function)(const Option&);
    } SubCommands[] = {
        {"add", AccountAddCommand},
        {"delete", AccountDeleteCommand},
        {"unregister_nsa", UnregisterNsaCommand},
        {"clear_all", AccountClearAllCommand},
        {"list", AccountListCommand},
        {"enable", AccountEnableCommand},
        {"disable", AccountDisableCommand},
        {"set", AccountSetCommand},
        {"refresh", AccountRefreshCommand},
        {"link", AccountLinkCommand},
    };

    static const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ADD_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_DELETE_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_UNREGISTER_NSA_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_CLEARALL_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_LIST_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ENABLE_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_DISABLE_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_REFRESH_USAGE "\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_LINK_USAGE "\n"
#endif
        ""; // 終端

    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::account::InitializeForAdministrator();
    for (const auto& subCommand : SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            *outValue = subCommand.function(option);
            NN_RESULT_SUCCESS;
        }
    }

    NN_LOG("'%s' is not a DevMenu account command. See '" DEVMENUCOMMAND_NAME " account --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
