﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>  // std::memset, std::memcpy
#include <mutex>   // std::lock_guard

#include <nn/nn_Macro.h>
#include "wlan_MezcalType.h"
#include "wlan_MsgBuffer.h"
#include "wlan_StateMachine.h"
#include "wlan_WpaSupplicant.h"
#include "wlan_Settings.h"
#include "wlan_MemoryInit.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/settings/factory/settings_WirelessLan.h>
#endif

namespace nn { namespace wlan {

/* イベントウェイトしているコマンドをリスト化する為のバッファ作成 ---------- */
/* ひどい手抜き実装なんで、要調整 */

//WlanCommand* g_pWaitingEventList[StateMachine::CommandMessageCountMax];
WlanCommand** g_pWaitingEventList;
void CreateWiatingEventList(int cmd_max)
{
    g_pWaitingEventList = new WlanCommand*[StateMachine::CommandMessageCountMax];

    /* 超手抜き実装(リスト構造にするのが良い) */
    for(int i = 0; i < StateMachine::CommandMessageCountMax; i++)
    {
        g_pWaitingEventList[i] = NULL;
    }
}

void DestroyWiatingEventList()
{
    //BlackFree( g_pWaitingEventList );
    delete[] g_pWaitingEventList;
}

void StateMachine::AppendWaitEventList(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    int i = 0;

//    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    nn::os::LockMutex( &m_WaitingEventMutex );

    for(; i < m_CommandMessageMax; i++)
    {
        if(nn::wlan::g_pWaitingEventList[i] == NULL)
        {
            g_pWaitingEventList[i] = pcmdbuff;
//            WLAN_LOG_INFO("WDMS: Append wait list[%d] <= %lp\n", i, pcmdbuff);
            break;
        }
    }

    nn::os::UnlockMutex( &m_WaitingEventMutex );

    if(i == m_CommandMessageMax)
    {
        WLAN_LOG_INFO("WDMS: Fail Appnd wait list\n");
    }

//    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::PostWaitingEventList(uint32_t id, bool result) NN_NOEXCEPT
{
    int i = 0;

    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    nn::os::LockMutex( &m_WaitingEventMutex );

    for(; i < m_CommandMessageMax; i++)
    {
        if(g_pWaitingEventList[i] != NULL)
        {
            if(g_pWaitingEventList[i]->id == id)
            {
                g_pWaitingEventList[i]->Result = result;
                nn::os::SignalEvent(&g_pWaitingEventList[i]->_finishEvent);
                g_pWaitingEventList[i] = NULL;
                WLAN_LOG_INFO("WDMS: Signal waiting event\n");
            }
        }
    }

    nn::os::UnlockMutex( &m_WaitingEventMutex );

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

bool StateMachine::m_IsDrvInitFail;
bool StateMachine::m_IsSleepState;
nn::os::MutexType StateMachine::m_TxOperateMutex;
/* コンストラクタ ---------------------------------------------------------- */
StateMachine::StateMachine() NN_NOEXCEPT
{
    m_IsInit = false;
    m_Authorized = false;
    m_IsMatchingDataRegistered = false;
    m_IsSleepState = false;
    m_WdrvState.SetMain( WdrvMainState_Stop );
    m_WdrvState.SetSub( WdrvSubState_None );
    m_WdrvStateBeforeSleep.SetMain( WdrvMainState_Stop );
    m_WdrvStateBeforeSleep.SetSub( WdrvSubState_None );

    m_pWlanFuncs = NULL;
    m_pSuppFuncs = NULL;
    m_StateMachineEventCallback = NULL;

    std::memset(&m_ConnectionStatus, 0, sizeof(ConnectionStatus));
    std::memset(&m_StaticAesKey[0], 0x00, 16);
    m_StaticAesMode = false;

    m_IsBeaconActionFrameSet = false;
    m_IsBeaconActionFrameTransferring = false;

    m_ownMacAddr.SetDirect(0,0,0,0,0,0);

    nn::os::InitializeEvent(&m_StateMachineInitEvent, false, nn::os::EventClearMode_AutoClear);

    if( nn::wlan::FirmwareDebugSettings::IsSkipBoot() == true )
    {
        m_WdrvState.SetMain( WdrvMainState_OutOfService );
    }

    std::memset(&m_AllowedChannel, 0, sizeof(WlanAllowedChannels));

    m_IsEnabledMc2 = false;

    std::memset(&m_BeaconActionFrame, 0, sizeof(BeaconActionFrame));

    // スキャン結果管理情報を初期化
    m_ScanResultInfo.minRssiValue = InvalidRssiValue;  // 0は意味を持ってしまうので、あり得ない値で初期化しておく
    m_ScanResultInfo.pMinRssiInfo = NULL;

    // 無線ドライバーの初期化に終了したことがあるかどうか
    m_IsDrvInitFail = false;

    // WoWLで起床要因となるTcpパケットに関する情報がセットされているかどうか
    m_IsSetTcpInfoWoWL = false;
    // WoWLに関するパラメータ
    std::memset(&m_WowlParams, 0, sizeof(WlanWowlSetupParams));
    // WoWLの起床要因
    m_WowlWakeReason = 0;
    // WoWLの起床要因別発生回数
    std::memset(&m_WowlWakeCount, 0, sizeof(WowlWakeCount));
    // Wowlスリープスタッツ
    std::memset(&m_WowlSleepStats, 0, sizeof(WowlSleepStats));
    // GTKがセットされているか
    m_IsGtkSet = false;
    // ローカル通信LCSモード中か否か
    m_IsLcsMode = false;
    // 4way-handshake失敗時にLinkDownイベントとWpaサプリカントからのイベント処理順を判断するためのフラグ(SIGLO-64396)
    m_IsLinkDownBeforeWpaFailureEv = false;
    // アンテナ設定が変更されているか
    m_IsAntennaChangedByUser = false;
    m_EventMessageSleepCnt = 0;
    // すれちがいスリープが予約されているかどうか
    m_IsReservedDetectSleep = false;
    // 何のモードのスリープに入ったか
    m_sleepMode = SleepMode_Normal;
    // すれちがいスリープ用データの初期化
    m_DetectSetupParams.hashList.pHashList = NULL;
    InitializeDetectData();
}

StateMachine::~StateMachine() NN_NOEXCEPT
{
    m_IsInit = false;
    m_WdrvState.SetMain( WdrvMainState_Stop );
    m_WdrvState.SetSub( WdrvSubState_None );
    m_WdrvStateBeforeSleep.SetMain( WdrvMainState_Stop );
    m_WdrvStateBeforeSleep.SetSub( WdrvSubState_None );
    m_pWlanFuncs = NULL;
    m_pSuppFuncs = NULL;
    nn::os::FinalizeEvent(&m_StateMachineInitEvent);
}

/* State Machine から抜けて欲しい場合にコール ------------------------------ */
void StateMachine::ExitStateMachineRequest() NN_NOEXCEPT
{
    m_exitStateMachine = true;
    nn::os::ReleaseSemaphore( &this->m_MessageSemaphore );
}

/* State Machineにメッセージを突っ込むための、メッセージバッファの初期化 --- */
bool StateMachine::InitMessageSystem() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    uint8_t* evtBuffer = new uint8_t[sizeof(uintptr_t) * m_EventMessageMax];
    uint8_t* cmdBuffer = new uint8_t[sizeof(uintptr_t) * m_CommandMessageMax];
    uint8_t* evtSlpBuffer = new uint8_t[sizeof(uintptr_t) * EventSleepMessageCountMax];
    m_EventBuffer = reinterpret_cast<uintptr_t*>(evtBuffer);
    m_CommandBuffer = reinterpret_cast<uintptr_t*>(cmdBuffer);
    m_EventSleepBuffer = reinterpret_cast<uintptr_t*>(evtSlpBuffer);

    CreateWiatingEventList( m_CommandMessageMax );
    nn::os::InitializeMutex(&m_WaitingEventMutex, false, 0);

    /* メッセージ */
    nn::os::InitializeSemaphore(&m_MessageSemaphore,
                                0, m_MessageSemaphoreMax);
    nn::os::InitializeSemaphore(&m_MessageSleepSemaphore,
                                0, EventSleepMessageCountMax);

    /* メッセージキューを初期化する */
    nn::os::InitializeMessageQueue(&m_CommandQueue,
                                   m_CommandBuffer,
                                   m_CommandMessageMax);
    nn::os::InitializeMessageQueue(&m_EventQueue,
                                   m_EventBuffer,
                                   m_EventMessageMax);
    nn::os::InitializeMessageQueue(&m_EventSleepQueue,
                                   m_EventSleepBuffer,
                                   EventSleepMessageCountMax);

    nn::wlan::InitalWlanMessageStruct(m_CommandMessageMax, m_EventMessageMax);

    m_IsInit = 1;

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* State Machineにメッセージを突っ込むための、メッセージバッファの初期化 --- */
bool StateMachine::FinalizeMessageSystem() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    nn::os::FinalizeMessageQueue( &m_CommandQueue );
    nn::os::FinalizeMessageQueue( &m_EventQueue );
    nn::os::FinalizeMessageQueue( &m_EventSleepQueue );
    nn::os::FinalizeSemaphore( &m_MessageSemaphore );
    nn::os::FinalizeSemaphore( &m_MessageSleepSemaphore );

    DestroyWiatingEventList();
    nn::os::FinalizeMutex( &m_WaitingEventMutex );

    if(m_EventBuffer != NULL)
    {
        //BlackFree( m_EventBuffer );
        delete m_EventBuffer;
    }
    if(m_CommandBuffer != NULL)
    {
        //BlackFree( m_CommandBuffer );
        delete m_CommandBuffer;
    }
    if(m_EventSleepBuffer != NULL)
    {
        delete m_EventSleepBuffer;
    }

    nn::wlan::FinalizeWlanMessageStruct();

    m_IsInit = 0;

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* ConnectionEvent と CancelEvent の初期化 */
void StateMachine::InitializeSystemEvent() NN_NOEXCEPT
{
    nn::os::CreateSystemEvent(&m_ConnectionStatusEvent, nn::os::EventClearMode_AutoClear, true);
    nn::os::CreateSystemEvent(&m_ScanCompletionEvent, nn::os::EventClearMode_AutoClear, true);
    nn::os::CreateSystemEvent(&m_ConnectCompletionEvent, nn::os::EventClearMode_AutoClear, true);
    nn::os::InitializeMutex(&m_ConnectionStatusMutex, false, 0);
    nn::os::InitializeMutex(&m_TxOperateMutex, false, 0);
}

/* ConnectionEvent と CancelEvent の終了処理 */
void StateMachine::FinalizeSystemEvent() NN_NOEXCEPT
{
    nn::os::DestroySystemEvent(&m_ConnectionStatusEvent);
    nn::os::DestroySystemEvent(&m_ScanCompletionEvent);
    nn::os::DestroySystemEvent(&m_ConnectCompletionEvent);
    nn::os::FinalizeMutex(&m_ConnectionStatusMutex);
    nn::os::FinalizeMutex(&m_TxOperateMutex);
}


/*
 * Message用のバッファ処理。MsgBufferの仕組みは、
 * ステートマシーンの中に隠蔽する方針
 */
/* Commandバッファを取得する。正常に取得できたかは、呼び出し元が確認する --- */
WlanCommand* StateMachine::WlanGetCommandBuff(size_t arg_size) NN_NOEXCEPT
{
    WlanCommand* pcmdbuff;

    pcmdbuff = nn::wlan::GetCommandBuffer( arg_size );

    return pcmdbuff;
}

void StateMachine::WlanReleaseCommandBuff(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    nn::wlan::ReleaseCommandBuffer( pcmdbuff );
}

/* Eventバッファを取得する。正常に取得できたかは、呼び出し元が確認する ----- */
WlanEvent* StateMachine::WlanGetEventBuff(size_t arg_size) NN_NOEXCEPT
{
    WlanEvent* pevnbuff;

    pevnbuff = nn::wlan::GetEventBuffer( arg_size );

    return pevnbuff;
}

void StateMachine::WlanReleaseEventBuff(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    nn::wlan::ReleaseEventBuffer( pevnbuff );
}

/* Massage をキューイング -------------------------------------------------- */
bool StateMachine::PostMassage(nn::os::MessageQueueType* msgq, uintptr_t msg) NN_NOEXCEPT
{
    bool ret;

    ret = nn::os::TrySendMessageQueue( msgq, msg);

    return ret;
}

/* Message post 待ち ------------------------------------------------------- */
int StateMachine::WaitMessage() NN_NOEXCEPT
{
    int signal = nn::os::WaitAny( &m_MessageSemaphore, &m_MessageSleepSemaphore );
    if( signal == 0 )
    {
        nn::os::AcquireSemaphore( &m_MessageSemaphore );
    }
    else if( signal == 1 )
    {
        nn::os::AcquireSemaphore( &m_MessageSleepSemaphore );
    }
    else
    {
        NN_ABORT("[WLAN]%s: Unexpected return value from WaitAny()\n", __FUNCTION__);
    }

    return signal;
}

/* Event Callback 関数の登録 ----------------------------------------------- */
void StateMachine::RegisterEventCallback( void (*pfunc)(WlanEventId id, void* ptr) ) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    m_StateMachineEventCallback = pfunc;
}

/* Event Callback 関数の呼び出し ------------------------------------------- */
void StateMachine::CallEventCallbackFunction(uint32_t id, void* ptr) NN_NOEXCEPT
{
    if(m_StateMachineEventCallback != NULL)
    {
        m_StateMachineEventCallback(static_cast<WlanEventId>(id), ptr);
    }
}

/* Scan Complete イベントの処理 -------------------------------------------- */
void StateMachine::ScanComplete(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            // ローカル通信の既接続スキャン中でMC2モードの無効化が出来なかった場合があるので、ここで無効化しておく(SIGLO-61129)
            if( m_WdrvState.GetMain() == WdrvMainState_LocalClientIdle ||
                    m_WdrvState.GetMain() == WdrvMainState_LocalSpectatorIdle )
            {
                if( m_IsEnabledMc2 == true )
                {
                    WLAN_LOG_INFO("Disable Coex Mc2 @ %s\n", __FUNCTION__);
                    m_pWlanFuncs->DisableCoexMc2();
                    m_IsEnabledMc2 = false;
                }
                if( m_IsLcsMode == true )
                {
                    WLAN_LOG_INFO("Revert CW params @ %s\n", __FUNCTION__);
                    m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
                    m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                }
            }
            m_WdrvState.SetSub( WdrvSubState_Idle );
            nn::os::SignalSystemEvent(&m_ScanCompletionEvent);
            break;
        default:
            if( m_WdrvState.GetMain() == WdrvMainState_InfraIdle &&
                    m_WdrvState.GetSub() == WdrvSubState_Connecting )
            {
                WLAN_LOG_INFO("WDMS: Received scan complete meant for supplicant%s\n", __FUNCTION__);
            }
            else
            {
                DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            }
            break;
        }
        break;

    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            m_WdrvState.SetSub( WdrvSubState_Connected );
            nn::os::SignalSystemEvent(&m_ScanCompletionEvent);
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

}

/* Scan Indication --------------------------------------------------------- */
void StateMachine::ScanIndication(WlanEvent *pevnbuff) NN_NOEXCEPT
{
    //    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    WlanBssInfo* pbssinfo = static_cast<WlanBssInfo*>(pevnbuff->Args);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Scanning:
            BssInfoListAdd( pbssinfo );
            break;
        case WdrvSubState_Connecting:
            if( m_WdrvState.GetMain() == WdrvMainState_InfraIdle )
            {
                // WPA supplicant calls scan request command internally.
                // So, scan result event can come in this state.
                // Wlan process does not need to store such BSS information.
                // Ignores this event.
                WLAN_LOG_DEBUG("Scan result event comes in [Infra, connecting] state.\n");
            }
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }
}

/* Connect indicate -------------------------------------------------------- */
void StateMachine::ConnectionIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    WlanConnectIndicate* pind;
    ConnectionStatus status;  // Masterステートの時は使用されない。

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
#if defined(USE_WPA_SUPPLICANT)
        // Need to wait notification event from WPA supplicant.
        // Do nothing here.
        break;
#endif
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
            // Driverが抱えているConnectionStatusをもらう
            m_pWlanFuncs->GetConnectionStatus(&status);

            pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);

            if( pind->Result == false )
            {
                // LocalClientの場合接続成功ケースはLinkChangeIndicate()で処理しているのでここは失敗ケースのみ処理する。
                WLAN_LOG_INFO("Connection event comes.[FAILURE]\n");

                // MC2モードを無効化しておく
                if( m_IsEnabledMc2 == true )
                {
                    m_pWlanFuncs->DisableCoexMc2();
                    m_IsEnabledMc2 = false;
                }
                if( m_IsLcsMode == true )
                {
                    WLAN_LOG_INFO("Revert CW params @ %s\n", __FUNCTION__);
                    m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
                    m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                }

                m_WdrvState.SetSub( WdrvSubState_Idle );
                status.state = ConnectionState_Idle;
                if(m_WdrvState.GetMain() == WdrvMainState_LocalSpectatorIdle)
                {
                    status.cause = CauseOfInfo_JoinFailure;
                }
                else
                {
                    status.cause = CauseOfInfo_ConnectFailure;
                }

                // ConnectionStatusの更新
                UpdateConnectionStatus(status);

                // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

                // 接続完了したので、通知用イベントをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            }
            else
            {
                if( m_WdrvState.GetMain() == WdrvMainState_LocalSpectatorIdle )
                {
                    // LocalSpectatorの場合接続成功ケースもここで扱う。
                    WLAN_LOG_INFO("Join event comes.[SUCCESS]\n");
                    m_WdrvState.SetSub( WdrvSubState_Connected );
                    status.state = ConnectionState_Joined;
                    status.cause = CauseOfInfo_JoinRequest;

                    // ConnectionStatusの更新
                    UpdateConnectionStatus(status);

                    // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                    nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

                    // 接続完了したので、通知用イベントをシグナルする
                    nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
                }
            }

            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
        // Can be ignored because already in connected state.
        WLAN_LOG_DEBUG("Connection indication occurs in LocalSta connected state.\n");
        break;
    case WdrvMainState_LocalMasterIdle:
        // DestroyAP()とほぼ同タイミングでClientからの接続があると、WDMでIdleステートに移行してから
        // このイベントを受け取る可能性がある。どのみちDestroyが行われるのでケア不要。
        // CreateAp()時にWDMで抱えている接続済みClientリストも初期化するので接続リストにズレは発生しない。
        break;

    case WdrvMainState_LocalMasterBss:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_CancelingScan:
        case WdrvSubState_Scanning:
            pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);
            if(pind->Result == true)
            {
                CallEventCallbackFunction(pevnbuff->id, pind);
            }

            // ClientStatusはイベントコールバック時に更新されている

            if( m_StaticAesMode == true )
            {
                WlanStaticAesInfo keyInfo;
                // StaticAESが選択されている場合、ここでUnicast用のStaticAES鍵を登録しておく
                keyInfo.keyMode = 0;  // Unicast用は0
                std::memcpy(&keyInfo.peerMacAddr[0], &pind->PeerAddr[0], MacAddress::MacAddressSize);
                std::memcpy(&keyInfo.key[0], &m_StaticAesKey[0], 16);
                m_pWlanFuncs->SetStaticAesKey(&keyInfo);
            }

            // CLIENTの接続状態に変化が起きたのでConnectionStatusEventをシグナルする
            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}  //NOLINT(impl/function_size)

/* Disassoc フレーム受信 --------------------------------------------------- */
void StateMachine::DisassocIndicateEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    WlanDisconnectInfo* pInfo;
    pInfo = reinterpret_cast<WlanDisconnectInfo*>(pevnbuff->Args);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
        case WdrvSubState_Disconnecting:
            // LINK DOWNイベントが発生するのでそこでステート遷移を行う。
            // ここではConnectionStatusの更新のみ行う。
            {
                // 接続状態管理オブジェクトの更新。
                ConnectionStatus status;
                status.aid = 0;
                status.beaconInterval = 0;
                status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
                status.capabilityInfo = 0;
                status.cause = CauseOfInfo_RecieveDisconnect;
                status.ssid.Set("");
                status.state = ConnectionState_Idle;
                status.statusReasonCode = pInfo->reason;
                status.channel = 0;
                UpdateConnectionStatus(status);
            }
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            // 基本無視する
            WLAN_LOG_INFO("[WLAN]Unknown disassoc indication\n");
            // 解析のためにアサートを仕掛けておく
            NN_SDK_ASSERT(false, "Unexpected state @ %s\n", __FUNCTION__); // 条件なしでアサートさせるマクロが無いので、TORIAEZU
            break;
        }
        break;

    case WdrvMainState_LocalMasterBss:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
        case WdrvSubState_Disconnecting:
            if( pInfo->reason == Dot11ReasonCode_DisassocStaLeaveLEV )
            {
                if( m_pWlanFuncs->IsMatchAssocList(pInfo->PeerAddr) == true )
                {
                    WLAN_LOG_INFO("The client[%02X:%02X:%02X:%02X:%02X:%02X] is still associated.\n",
                            pInfo->PeerAddr[0], pInfo->PeerAddr[1], pInfo->PeerAddr[2],
                            pInfo->PeerAddr[3], pInfo->PeerAddr[4], pInfo->PeerAddr[5]);
                    break;
                }
            }
            {
                // 接続状態管理オブジェクトの更新
                // 本来はEventCallBackFromDriverの中で更新しておきたいが、Infraモードで発生したのかMasterモードで発生したのか判断出来ないため、
                // StateMachineでやらざるを得ない。
                ClientStatus status;
                if( m_pWlanFuncs->GetClientStatusByMac(&status, MacAddress(pInfo->PeerAddr)) == true )
                {
                    // 接続状態に変更があるかチェック
                    if( status.state == ConnectionState_Connected )
                    {
                        status.capabilityInfo = 0;
                        status.cause = CauseOfInfo_DisconnectReq;  // CLIENT視点では自ら切断要求を出したことになるので、この理由となる
                        status.clientMacAddress.Set(&pInfo->PeerAddr[0]);
                        status.state = ConnectionState_Disconnected;
                        status.statusReasonCode = pInfo->reason;
                        status.rssi = -128;
                        m_pWlanFuncs->UpdateClientStatus(status);
                        // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                        nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
                    }
                }
            }
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            NN_SDK_ASSERT(false, "[WLAN]Disassoc indication is received in invalid state\n");
            break;
        }
        break;

    default:
        // do nothing
        // It can happen that WDM receives this event in unexpected state.
        // Maybe received delayed disassoc frame.
        // WDM can ignore it.
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }
}

/* Deauth フレーム受信 --------------------------------------------------- */
void StateMachine::DeauthIndicateEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    // DeauthIndイベントはLocalMaster以外が受信の可能性がある

    WlanDisconnectInfo* pInfo;
    pInfo = reinterpret_cast<WlanDisconnectInfo*>(pevnbuff->Args);
    ConnectionStatus status;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Disconnecting:
            // 自分でDisassociateを出し、Master側で受信されたが、MasterからのAckが届いていないので、Disassociateを再送した。
            // Masterからすれば、既に切断したはずの相手から再度Disassociateが来たので、Deauthを投げた。
            // 上記の一連の流れによるDeauthIndication。
            // 切断行為自体は成功なので、Disconnect関数を成功としておく。
        case WdrvSubState_Connected:
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            // Deauthフレームを受信しても、無線ドライバステートはまだ接続ステートなので、明示的に切断する必要がある。
            m_pWlanFuncs->Disassociate();
            {
                // 接続状態管理オブジェクトの更新。
                status.aid = 0;
                status.beaconInterval = 0;
                status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
                status.capabilityInfo = 0;
                status.cause = CauseOfInfo_RecieveDisconnect; // このイベントはDisassocフレーム受信時のみ発生する
                status.ssid.Set("");
                status.state = ConnectionState_Idle;
                status.statusReasonCode = pInfo->reason;
                status.channel = 0;
                UpdateConnectionStatus(status);
            }
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            // 基本無視する
            WLAN_LOG_INFO("[WLAN]Unknown disassoc indication\n");
            // 解析のためにアサートを仕掛けておく
            NN_SDK_ASSERT(false, "Unexpected state @ %s\n", __FUNCTION__); // 条件なしでアサートさせるマクロが無いので、TORIAEZU
            break;
        }
        break;

    case WdrvMainState_LocalClientIdle:
        /*
         * 接続試行中に接続先のMASTERが破棄され、さらにその直後にを同チャンネルに別SSIDでMASTERが作成された場合、
         * Clientの接続要求フレームを受け取ったMASTERはDeauthフレームを投げてくるので、そのケースに対応する。
         * 接続試行中に接続先からDeauthフレームを受け取ると、接続失敗イベントも発行されない仕様なので、
         * ここで接続失敗と判断する。
         */

        // 接続先相手からかチェック
        // Authフレーム送信時に相手先BSSIDを保存している
        m_pWlanFuncs->GetConnectionStatus(&status);
        if( std::memcmp(&pInfo->PeerAddr[0], status.bssid.GetMacAddressData(),
                MacAddress::MacAddressSize ) == 0 )
        {
            WLAN_LOG_INFO("Connection event comes.[FAILURE] Reason:%04X.\n", pInfo->reason);

            // 接続状態管理オブジェクトの更新。
            status.aid = 0;
            status.beaconInterval = 0;
            status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
            status.capabilityInfo = 0;
            status.cause = CauseOfInfo_ConnectFailure;
            status.ssid.Set("");
            status.state = ConnectionState_Idle;
            status.statusReasonCode = pInfo->reason;
            status.channel = 0;
            UpdateConnectionStatus(status);

            // MC2モードを無効化しておく
            if( !(m_WdrvState.GetSub() == WdrvSubState_Scanning ||
                    m_WdrvState.GetSub() == WdrvSubState_CancelingScan) )
            {
                // スキャン中に行うと失敗するので、スキャン中でない時に無効化する。
                // スキャン中であった場合はスキャン完了後に無効化する。->スキャン完了イベント処理内で行う。
                if( m_IsEnabledMc2 == true )
                {
                    m_pWlanFuncs->DisableCoexMc2();
                    m_IsEnabledMc2 = false;
                }
                if( m_IsLcsMode == true )
                {
                    WLAN_LOG_INFO("Revert CW params @ %s\n", __FUNCTION__);
                    m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
                    m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                }
            }

            m_WdrvState.SetSub( WdrvSubState_Idle );

            // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

            // 接続完了したので、通知用イベントをシグナルする
            nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);

            if( pInfo->reason != Dot11ReasonCode_NonauthClass2Frame )
            {
                // 通常はDot11ReasonCode_NonauthClass2FrameのReasonを投げてくる
                WLAN_LOG_ERROR("Received unexpected deauth frame!. Reason %04X\n", pInfo->reason);
            }
        }
        else
        {
            // 無視する
            WLAN_LOG_INFO("Received unexpected deauth frame from [%02X:%02X:%02X:%02X:%02X:%02X]\n",
                    pInfo->PeerAddr[0], pInfo->PeerAddr[1], pInfo->PeerAddr[2],
                    pInfo->PeerAddr[3], pInfo->PeerAddr[4], pInfo->PeerAddr[5]);
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        }
        break;
    default:
        // do nothing
        // It can happen that WDM receives this event in unexpected state.
        // Maybe received delayed deauth frame.
        // WDM can ignore it.
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }
}

/* Disassoc フレーム送信完了 --------------------------------------------------- */
void StateMachine::DisassocCompleteEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Disconnecting:
        // KeepAliveによる自動切断で勝手に切断フレームが送信されている場合があるので以下のサブステートも対象に加えておく
        case WdrvSubState_Connected:
        case WdrvSubState_Scanning:
        case WdrvSubState_CancelingScan:
            // LINK DOWNイベントが発生するので、そこでステートの更新を行う。
            // ここではConnectionStatusの更新のみ行う。
            {
                ConnectionStatus status;
                // (SIGLO-66079) wlanライブラリからの自発的切断の場合はConnectionStatusの更新は行わない
                GetConnectionStatus(&status);
                if( status.cause != CauseOfInfo_ConnectFailureAfterAssoc )
                {
                    status.aid = 0;
                    status.state = ConnectionState_Idle;
                    status.channel = 0;
                    status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
                    status.cause = CauseOfInfo_DisconnectReq;
                    status.ssid.Set("");
                    status.statusReasonCode = 0;
                    status.beaconInterval = 0;
                    status.capabilityInfo = 0;
                    UpdateConnectionStatus(status);
                }
                else
                {
                    WLAN_LOG_INFO("%s happens when cause is CauseOfInfo_ConnectFailureAfterAssoc.\n", __FUNCTION__);
                }
            }
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            NN_SDK_ASSERT(false, "[WLAN]Unexpected event\n");
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);

        // Disconnect完了イベントを待つ前に、相手からの切断フレームを受信してDisconnect_Indイベントを受信していたら
        // 既にIDLE状態に遷移しているので、ここで受け取る可能性はあるので、基本的には無視しておく。
        // また、WPAサプリカントでは4way-handshakeの失敗で自分から切断する場合もあり、その場合にも本ケースが発生する。
        WLAN_LOG_INFO("%s called in state[%d, %d]\n", __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
}

/* Deauth フレーム送信完了 --------------------------------------------------- */
// TODO DeauthRequestを実行した時点で切断処理は完了とするので、下記関数は不要となる(タイミングを見て削除しておきます)
void StateMachine::DeauthCompleteEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterBss:
    {
        uint8_t* pFlag = reinterpret_cast<uint8_t*>(pevnbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pFlag);
        if( *pFlag == 1 )
        {
            // 自動切断によるDeauth送信だった。
            // 子機の接続状態に変化が起きたのでConnectionStatusEventをシグナルする
            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
        }
        break;
    }
    default:
        // ドライバの仕様により、想定外のタイミングで本イベントが発生することもあるので、基本的にはスルーしておく。
        WLAN_LOG_INFO("%s called in an unexpected state[%d, %d]\n", __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
}


/* Beacon miss indication -------------------------------------------------- */
void StateMachine::BmissIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    WlanConnectIndicate* pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
        case WdrvSubState_Disconnecting:
        case WdrvSubState_CancelingScan:
        case WdrvSubState_Scanning:
            if(pind->Result == true)
            {
                CallEventCallbackFunction(pevnbuff->id, pind);
            }

            // BMISS発生したので、接続情報の更新
            m_ConnectionStatus.cause = CauseOfInfo_BeaconLost;

            // 上位層に通知する
            // 切断するかどうかは上位層の判断に任せる
            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            NN_SDK_ASSERT(false, "Unexpected state @ %s\n", __FUNCTION__);
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        NN_SDK_ASSERT(false, "Unexpected state @ %s\n", __FUNCTION__);
        break;
    }
}

/* Link change event indicate ---------------------------------------------- */
void StateMachine::LinkChangeIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    if( m_WdrvState.GetMain() == WdrvMainState_OutOfService )
    {
        return;
    }

    // Link changeイベントで明確にリンク状態が変化するので、WLANステートもこれに合わせて変化させておく
    WlanLinkInfo* linkInfo = static_cast<WlanLinkInfo*>(pevnbuff->Args);

    switch( linkInfo->reason )
    {
    case LinkChangeReason_Up:
        switch( m_WdrvState.GetMain() )
        {
        // ローカル通信のSTAのみ接続完了をここで判断する。
        // ConnectionEventは、リンクアップ後に一発目のビーコンを受け取るまで発生しない仕様となっており、
        // その間に切断が起きるとConnectionEventが発生しなくなってしまう。こうなるとステートがいつまでもConnectingのままになり遷移が不可能となる。
        // ただ、接続失敗はConnectionEventを見ないと判断出来ないので、その場合はConnectionIndicate()に任せる。
        // インフラ通信ではWPAサプリカントが接続成功/失敗を管理しているので、このようなことをする必要はない。
        case WdrvMainState_LocalClientIdle:
        case WdrvMainState_LocalSpectatorIdle:
            switch( m_WdrvState.GetSub() )
            {
            case WdrvSubState_Connecting:
            {
                ConnectionStatus status;
                // Driverが抱えているConnectionStatusをもらう
                m_pWlanFuncs->GetConnectionStatus(&status);
                WLAN_LOG_INFO("Connection event comes.[SUCCESS]\n");
                m_WdrvState.SetSub( WdrvSubState_Connected );
                if(m_WdrvState.GetMain() ==WdrvMainState_LocalClientIdle)
                {
                    m_WdrvState.SetMain( WdrvMainState_LocalClient );
                    status.state = ConnectionState_Connected;
                    status.cause = CauseOfInfo_ConnectRequest;

                    if( m_IsMatchingDataRegistered == true )
                    {
                        // マッチング用データが登録済みの場合のみ、認証済みと判断する
                        m_Authorized = true;
                    }
                }
                else if(m_WdrvState.GetMain() ==WdrvMainState_LocalSpectatorIdle)
                {
                    m_WdrvState.SetMain( WdrvMainState_LocalSpectator );
                    status.state = ConnectionState_Joined;
                    status.cause = CauseOfInfo_JoinRequest;

                    // Spectatorの場合はマッチング用データの登録は不要
                    m_Authorized = true;
                }

                if( m_StaticAesMode == true )
                {
                    // StaticAESが選択されている場合、ここでBroadcast/Unicast両方のStaticAES鍵を登録しておく
                    WlanStaticAesInfo keyInfo;
                    keyInfo.keyMode = 0;  // Unicast用は0
                    std::memcpy(&keyInfo.peerMacAddr[0], &linkInfo->PeerAddr[0], MacAddress::MacAddressSize);
                    std::memcpy(&keyInfo.key[0], &m_StaticAesKey[0], 16);
                    m_pWlanFuncs->SetStaticAesKey(&keyInfo);

                    keyInfo.keyMode = 1;  // Broadcast用は1
                    std::memset(&keyInfo.peerMacAddr[0], 0x00, MacAddress::MacAddressSize);
                    std::memcpy(&keyInfo.key[0], &m_StaticAesKey[0], 16);
                    m_pWlanFuncs->SetStaticAesKey(&keyInfo);
                }

                // Get bss channel
                int16_t channel;
                m_pWlanFuncs->GetChannel(&channel);
                status.channel = channel;

                // ConnectionStatusの更新
                UpdateConnectionStatus(status);

                // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

                // 接続完了したので、通知用イベントをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);

                if(m_WdrvState.GetMain() == WdrvMainState_LocalSpectatorIdle)
                {
                    PostWaitingEventList(WLAN_JOIN_NETWORK_SPECTATOR, true);
                }
            }
                break;
            default:
                DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
                // リンクアップが想定外のステートで行われると、その後の動作を保証出来ないのでABORTさせておく
                NN_ABORT_UNLESS_RESULT_SUCCESS(ResultUnexpectedLinkup());
                break;
            }
            break;
        default:
            break;
        }
        break;
    case LinkChangeReason_Disassoc:
    case LinkChangeReason_BcnLoss:  // Link down due to beacon lost.
        WLAN_LOG_INFO("Link down event occurs due to %s.\n",
                (linkInfo->reason == LinkChangeReason_BcnLoss) ? "beacon lost" : "disassoc request" );
        switch( m_WdrvState.GetMain() )
        {
        case WdrvMainState_LocalClient:
        case WdrvMainState_LocalSpectator:
            // MC2モードを無効化しておく
            if( !(m_WdrvState.GetSub() == WdrvSubState_Scanning ||
                    m_WdrvState.GetSub() == WdrvSubState_CancelingScan) )
            {
                // スキャン中に行うと失敗するので、スキャン中でない時に無効化する。
                // スキャン中であった場合はスキャン完了後に無効化する。->スキャン完了イベント処理内で行う。
                if( m_IsEnabledMc2 == true )
                {
                    m_pWlanFuncs->DisableCoexMc2();
                    m_IsEnabledMc2 = false;
                }
                if( m_IsLcsMode == true )
                {
                    WLAN_LOG_INFO("Revert CW params @ %s\n", __FUNCTION__);
                    m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
                    m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                }
            }
        case WdrvMainState_InfraSta:
            // update main state
            if(m_WdrvState.GetMain() == WdrvMainState_InfraSta)
            {
                // keepaliveの無効化をしておく
                m_pWlanFuncs->SetStaKeepAlive(0);
                m_WdrvState.SetMain( WdrvMainState_InfraIdle );
            }
            else if(m_WdrvState.GetMain() == WdrvMainState_LocalClient)
            {
                m_WdrvState.SetMain( WdrvMainState_LocalClientIdle );
                m_StaticAesMode = false;
            }
            else if(m_WdrvState.GetMain() == WdrvMainState_LocalSpectator)
            {
                m_WdrvState.SetMain( WdrvMainState_LocalSpectatorIdle );
                m_StaticAesMode = false;
            }

            // update sub state
            switch( m_WdrvState.GetSub() )
            {
            case WdrvSubState_Disconnecting:
                PostWaitingEventList(WLAN_DISASSOCIATE, pevnbuff->Result);
            case WdrvSubState_Connected:
                m_WdrvState.SetSub( WdrvSubState_Idle );
                break;
            case WdrvSubState_Scanning:
            case WdrvSubState_CancelingScan:
                WLAN_LOG_INFO("Infra or local sta receives disassoc indication during scan.\n");
                break;
            default:
                // No way
                NN_SDK_ASSERT(false, "BcnLoss link down event occurs in an unexpected state[%s, %s]",
                        MainStateStr[m_WdrvState.GetMain()], SubStateStr[m_WdrvState.GetSub()]);
            }

            m_Authorized = false;

            // 既にDisassocCompやDeauthIndを受けてstateとcauseが更新されている可能性があるのでチェック
            {
                ConnectionStatus status;
                GetConnectionStatus(&status);
                if( status.state != ConnectionState_Idle )
                {
                    // update connection status
                    // (SIGLO-64396)の問題で4way-shandshake失敗からのdisconnectの場合があるのでチェック
                    if( status.cause != CauseOfInfo_ConnectFailureAfterAssoc )
                    {
                        status.aid = 0;
                        status.beaconInterval = 0;
                        status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
                        status.capabilityInfo = 0;
                        status.cause = (linkInfo->reason == LinkChangeReason_BcnLoss) ? CauseOfInfo_BeaconLost : CauseOfInfo_RecieveDisconnect;
                        status.ssid.Set("");
                        status.statusReasonCode = 0;
                        status.channel = 0;
                    }
                    status.state = ConnectionState_Idle;
                    UpdateConnectionStatus(status);
                }
                nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

                // [Workaround]サプリカントスレッドの切断処理が回りきるようにwaitを入れておく
                // TODO wpa supplicantでの根本解決
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            break;
        case WdrvMainState_InfraIdle:
            if( m_WdrvState.GetSub() == WdrvSubState_Connecting )
            {
                // 4way-handshake失敗でwpaサプリカントが自分から切断した場合にここへ来る
                WLAN_LOG_INFO("Link down event comes before wpa failure event. 4way-handshake failed.\n");
                m_IsLinkDownBeforeWpaFailureEv = true;
                // wpaサプリカントから失敗を示すイベントが後で来るので、そこでステート遷移やConnectionStatusの処理を行う。
            }
            break;
        case WdrvMainState_LocalMasterBss:
        case WdrvMainState_LocalMasterIdle:
            // Can be ignored, but we set assertion code in order to analyze error.
            NN_SDK_ASSERT(false, "Link down event occurs in an unexpected state[%d, %d]",
                    m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        default:
            WLAN_LOG_DEBUG("do nothing because already in idle state.\n");
            // Can be ignored.
            break;
        }
        break;
    case LinkChangeReason_AssocRecFail:
        break;
    case LinkChangeReason_BsscfgDown:
        // 自身でBSSを破棄したことによるリンクダウン
        break;
    default:
        NN_SDK_ASSERT(false, "Unknown reason %d", linkInfo->reason);
    }
} //NOLINT(impl/function_size)

void StateMachine::IfChangeIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_DetectIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
            WLAN_LOG_DEBUG("IF UP event comes\n");
            // ActoinFrame受信モードのセット
            if( m_WdrvState.GetMain() == WdrvMainState_DetectIdle )
            {
                m_WdrvState.SetMain( WdrvMainState_Detect );
            }
            else
            {
                m_pWlanFuncs->SetActionFrameRecvMode(ActionFrameRecvMode_Normal);
            }
            PostWaitingEventList(WLAN_SET_IFUPDOWN, true);
            m_WdrvState.SetSub( WdrvSubState_Idle );
            break;
        default:
            // do nothing
            break;
        }
        break;
    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }
}

/* WPA connection success event ---------------------------------------------- */
void StateMachine::WpaConnectionSuccessEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    ConnectionStatus status;
    int16_t channel;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
            WLAN_LOG_INFO("Connection event from supplicant comes.[SUCCESS]\n");

            // Get ConnectionStatus from Mezcal class
            m_pWlanFuncs->GetConnectionStatus(&status);
            m_WdrvState.SetSub( WdrvSubState_Connected );
            m_WdrvState.SetMain( WdrvMainState_InfraSta );

            status.state = ConnectionState_Connected;
            status.cause = CauseOfInfo_ConnectRequest;
            status.statusReasonCode = Dot11StatusCode_Success;
            m_Authorized = true;

            m_pWlanFuncs->GetChannel(&channel);
            status.channel = channel;
#if !defined(WFA_CERTIFICATION)
            if( channel <= WirelessChannel_13ch )
            {
                // 2.4GHzAPに接続していた場合は、BTのCOEXによる影響があるため
                // txchain数とrxchain数を1本使用モードにセットしておく
                m_pWlanFuncs->SetTxChain(RxAntennaPattern_1);
                m_pWlanFuncs->SetRxChain(RxAntennaPattern_1);
                WLAN_LOG_INFO("Set txchain and rxchain to ANT_1 only\n");
#if 0  // SIGLO-63147の問題があるため、それが解決するまでセットしないようにしておく
                // CoexのSimultaneousTx設定をアンテナ1本用に変更しておく
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);
#endif
            }
#endif

            // (SIGLO-644964) KeepAlive間隔の設定
            m_pWlanFuncs->SetStaKeepAlive(STA_KEEPALIVE_TIME);

            UpdateConnectionStatus(status);

            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            break;
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

/* WPA connection failure event ---------------------------------------------- */
void StateMachine::WpaConnectionFailureEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    // This event means connection failure or disconnection occurs.

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
        {
            ConnectionStatus status;
            WlanDisconnectInfo* pinfo = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pinfo);
            WLAN_LOG_INFO("Connection event from supplicant comes.[FAILURE]\n");

            m_WdrvState.SetSub( WdrvSubState_Idle );
            // Get ConnectionStatus from Mezcal class
            m_pWlanFuncs->GetConnectionStatus(&status);
            if( status.state == ConnectionState_Idle )
            {
                if( status.statusReasonCode == Dot11StatusCode_Sized16 )
                {
                    WLAN_LOG_INFO("Found no network\n");
                    status.cause = CauseOfInfo_NoBss;
                    status.statusReasonCode = Dot11StatusCode_UnspecifiedFailure;
                }
                else
                {
                    WLAN_LOG_INFO("Found target network, but association was failed.\n");
                    status.cause = CauseOfInfo_ConnectFailure;
                    NN_SDK_LOG("status.statusReasonCode = %d\n", pinfo->reason); // SIGLO-65104 不要になったら削除します
                }
            }
            else
            {
                WLAN_LOG_INFO("Association is success but something wrong happens after the association.\n");
                // Associationには成功したが、その後の処理（4way-handshakeなど）に失敗。
                if( m_IsLinkDownBeforeWpaFailureEv == true )
                {
                    // 既にwpaサプリカントによる切断またはAPからの切断が完了済みを意味する
                    WLAN_LOG_INFO("Disconnection has already done.\n");
                    m_IsLinkDownBeforeWpaFailureEv = false;
                    status.state = ConnectionState_Idle;
                    status.cause = CauseOfInfo_ConnectFailureAfterAssoc;
                    status.statusReasonCode = pinfo->reason;
                }
                else
                {
                    // 無線FW層では接続済み状態となっているので一旦成功にしておく。
                    // ライブラリ側で直ぐに切断を呼ぶようにしておく。
                    WLAN_LOG_INFO("Disconnection will be done soon.\n");
                    m_WdrvState.SetMain( WdrvMainState_InfraSta );
                    m_WdrvState.SetSub( WdrvSubState_Connected );
                    status.state = ConnectionState_Connected;
                    status.cause = CauseOfInfo_ConnectFailureAfterAssoc;  // ライブラリから切断を呼んでもらうためのcause
                    status.statusReasonCode = pinfo->reason;
                    status.aid = 0;
                    UpdateConnectionStatus(status);
                    nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
                    // m_ConnectionStatusEventはシグナルさせない。上位層が動くのを防ぎ、ライブラリから切断を呼ばせるため。
                    break;
                }
            }
            status.aid = 0;

            UpdateConnectionStatus(status);

            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            break;
        }
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_InfraSta:
        // do nothing
        break;
    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

/* WPS connection failure event ---------------------------------------------- */
void StateMachine::WpsConnectionFailureEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    // This event means connection failure or disconnection occurs.

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
        {
            WlanDisconnectInfo* pinfo = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pinfo);
            WLAN_LOG_INFO("WPS failed due to time-out.[FAILURE]\n");

            m_WdrvState.SetSub( WdrvSubState_Idle );

            ConnectionStatus status;
            std::memset(&status, 0, sizeof(ConnectionStatus));
            status.cause = CauseOfInfo_WpsTimeout;
            status.state = ConnectionState_Idle;
            status.ssid.Set("");
            status.statusReasonCode = pinfo->reason;

            UpdateConnectionStatus(status);

            nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            break;
        }
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;
    case WdrvMainState_InfraSta:
        // do nothing
        break;
    default:
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::GtkSetEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    m_IsGtkSet = pevnbuff->Result;
    PostWaitingEventList(WLAN_SET_GTK_INFO, pevnbuff->Result);

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::GtkClearEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    if( pevnbuff->Result == true )
    {
        WLAN_LOG_INFO("Registered gtk is cleared.\n");
        m_IsGtkSet = false;
    }
    PostWaitingEventList(WLAN_CLEAR_GTK_INFO, pevnbuff->Result);

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::StartToConnectEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    PostWaitingEventList(WLAN_JOIN_NETWORK_STA, pevnbuff->Result);

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::StartToCancelConnectEv(WlanEvent* pevnbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    PostWaitingEventList(WLAN_CANCEL_JOIN_NETWORK, pevnbuff->Result);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
        {
            m_WdrvState.SetSub( WdrvSubState_None );
            m_Authorized = false;
            m_StaticAesMode = false;

            // Connect完了通知イベントをシグナルさせる
            nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            WLAN_LOG_INFO("Cancel connect done.\n");
            break;
        }
        default:
            DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        // Cancelの前に接続状態になった
        DisplayIlligalMessage(pevnbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

/* DHDから受信したイベントを処理する --------------------------------------- */
bool StateMachine::EventProcess(int signal) NN_NOEXCEPT
{
    bool ret;
    uintptr_t  msg;
    WlanEvent* pevebuff;

    if( signal == 0 )
    {
        ret = nn::os::TryReceiveMessageQueue(&msg, &m_EventQueue);

        // sleepステートでは処理しない
        if( ret == true && m_WdrvState.GetMain() == WdrvMainState_Sleep )
        {
            // sleep時用キューへ移し変え
            pevebuff = reinterpret_cast<WlanEvent*>(msg);
            PostEventSleepMessage(pevebuff);
            return true;
        }
    }
    else if( signal == 1 )
    {
        ret = nn::os::TryReceiveMessageQueue(&msg, &m_EventSleepQueue);
    }
    else
    {
        NN_ABORT("[WLAN]%s: Unexpected signal value[%d]\n", __FUNCTION__, signal);
    }

    if( ret == true )
    {
        pevebuff = reinterpret_cast<WlanEvent*>(msg);

        switch( pevebuff->id )
        {
        case WLAN_EVENT_SCAN_COMPLETE:
            ScanComplete( pevebuff );
            break;

        case WLAN_EVENT_SCAN_INDICATION:
            ScanIndication( pevebuff );
            break;

        case WLAN_EVENT_CONNECTED:
            ConnectionIndicate( pevebuff );
            break;

        case WLAN_EVENT_DISASSOCIATE:
            DisassocCompleteEv( pevebuff );
            break;

        case WLAN_EVENT_DEAUTHENTICATE:
            DeauthCompleteEv( pevebuff );
            break;

        case WLAN_EVENT_DISASSOCIATE_IND:
            DisassocIndicateEv( pevebuff );
            break;

        case WLAN_EVENT_DEAUTHENTICATE_IND:
            DeauthIndicateEv( pevebuff );
            break;

        case WLAN_EVENT_ROAM:
            // ローミング発生
            // do something
            break;

        case WLAN_EVENT_BMISS:
            BmissIndicate( pevebuff );
            break;

        case WLAN_EVENT_LINKCHANGE:
            LinkChangeIndicate( pevebuff );
            break;

        case WLAN_EVENT_IF_UP:
            IfChangeIndicate( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_SUCCESS:
            WpaConnectionSuccessEv( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_FAIURE:
            WpaConnectionFailureEv( pevebuff );
            break;

        case WLAN_WPS_NOTIFY_FAIURE:
            WpsConnectionFailureEv( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_GTK_SET:
            GtkSetEv( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_GTK_CLEAR:
            GtkClearEv( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_CONNECT_START:
            StartToConnectEv( pevebuff );
            break;

        case WLAN_WPA_NOTIFY_CONNECT_ABORT:
            StartToCancelConnectEv( pevebuff );
            break;

        default:
            WLAN_LOG_INFO("WDMS: Unknown Event[%ld]\n", pevebuff->id);
            break;
        }


        WlanReleaseEventBuff( pevebuff );
    }

    return ret;
}

/* ドライバ初期化処理 ------------------------------------------------------ */
bool StateMachine::InitalizeWlanDriver(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Stop:
    case WdrvMainState_Sleep:
        // もし既に無線ドライバーの初期化に失敗したことがあるのであれば、何も行わずエラーを返しておく。
        if( m_IsDrvInitFail == true )
        {
            WLAN_LOG_INFO("Wireless driver has failed to be initialized. Skip the initialization.\n");
            m_WdrvState.SetMain( WdrvMainState_OutOfService );
            m_WdrvState.SetSub( WdrvSubState_None );
            pcmdbuff->nnResult = ResultWlanDeviceAbnormal();
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
            break;
        }
        if( m_pWlanFuncs->InitializeDriver() == false )
        {
            WLAN_LOG_ERROR("StateMachine::InitializeDriver() failed\n");
            m_IsDrvInitFail = true;
            m_pWlanFuncs->FinalizeDriver();  // 終了処理を走らせておく
            // 無線ドライバーの初期化に失敗し、さらに固有MACアドレスもセットする前であった場合
            // ここでNANDから固有MACアドレスを読み出し、ステートマシーンに記憶させておく
            if( m_ownMacAddr == MacAddress::CreateZeroMacAddress() )
            {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
                nn::settings::factory::MacAddress wirelessUniqueMac;
                nn::Result result = nn::settings::factory::GetWirelessLanMacAddress(&wirelessUniqueMac);
                if( result.IsSuccess() )
                {
                    m_ownMacAddr.Set(&wirelessUniqueMac.octets[0]);
                }
                else
                {
                    m_ownMacAddr.SetDirect(0x40, 0xd2, 0x8a, 0x01, 0x23, 0x45);
                }
#else
                m_ownMacAddr.SetDirect(0x40, 0xd2, 0x8a, 0x01, 0x23, 0x45);
#endif
            }
            m_WdrvState.SetMain( WdrvMainState_OutOfService );
            m_WdrvState.SetSub( WdrvSubState_None );
            pcmdbuff->nnResult = ResultWlanDeviceAbnormal();
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
            break;
        }
        m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
        m_WdrvState.SetMain( WdrvMainState_Ready );
        m_WdrvState.SetSub( WdrvSubState_None );
        pcmdbuff->Result = true;

        // MACアドレスを取得しておく
        m_pWlanFuncs->GetMacAddress(m_ownMacAddr.GetMacAddressData());
        // Save allowed channels
        m_pWlanFuncs->GetAllowedChannels(&m_AllowedChannel);
        pcmdbuff->nnResult = ResultSuccess();
        m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
        m_pSuppFuncs->SetSleepFlag(false);
#endif
        break;
    case WdrvMainState_OutOfService:
        pcmdbuff->Result = true;
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
        {
            nn::settings::factory::MacAddress wirelessUniqueMac;
            nn::Result result = nn::settings::factory::GetWirelessLanMacAddress(&wirelessUniqueMac);
            if( result.IsSuccess() )
            {
                m_ownMacAddr.Set(&wirelessUniqueMac.octets[0]);
            }
            else
            {
                m_ownMacAddr.SetDirect(0x40, 0xd2, 0x8a, 0x01, 0x23, 0x45);
            }
        }
#else
        m_ownMacAddr.SetDirect(0x40, 0xd2, 0x8a, 0x01, 0x23, 0x45);
#endif
        pcmdbuff->nnResult = ResultSuccess();
        m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
        m_pSuppFuncs->SetSleepFlag(false);
#endif
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* ドライバ初期化処理(skip_boot=trueの時だけ利用可能)(SIGLO-67650) -------------------------------- */
bool StateMachine::InitalizeWlanDriverOnSkipBoot(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_OutOfService:
        // skip_boot=true状態か確認
        if( nn::wlan::FirmwareDebugSettings::IsSkipBoot() == false )
        {
            // skip_boot=trueではないのに、OutOfServiceにいるということは、通常起動で無線ドライバー初期化に失敗している状態
            WLAN_LOG_ERROR("Wireless driver is in abnormal state. Please reboot the system.\n");
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        if( m_pWlanFuncs->InitializeDriver() == false )
        {
            WLAN_LOG_ERROR("StateMachine::InitializeDriverOnSkipBoot() failed\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultWlanDeviceAbnormal());
        }
        m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
        m_WdrvState.SetMain( WdrvMainState_Ready );
        m_WdrvState.SetSub( WdrvSubState_None );
        pcmdbuff->Result = true;

        // MACアドレスを取得しておく
        m_pWlanFuncs->GetMacAddress(m_ownMacAddr.GetMacAddressData());
        // Save allowed channels
        m_pWlanFuncs->GetAllowedChannels(&m_AllowedChannel);
        pcmdbuff->nnResult = ResultSuccess();
        m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
        m_pSuppFuncs->SetSleepFlag(false);
#endif
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* ドライバ終了処理 -------------------------------------------------------- */
bool StateMachine::FinalizeWlanDriver(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    if(m_WdrvState.GetMain() != WdrvMainState_Stop)
    {
        m_pWlanFuncs->FinalizeDriver();
        m_WdrvState.SetMain( WdrvMainState_Stop );
        pcmdbuff->Result = true;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* 各動作モードのMainステートをIdleへ変更 ---------------------------------- */
void StateMachine::SetEachStateToIdle() NN_NOEXCEPT
{
    /* Main StateをIdleへ */
    if(m_WdrvState.GetMain() == WdrvMainState_InfraSta)
    {
        m_WdrvState.SetMain( WdrvMainState_InfraIdle );
    }
    else if(m_WdrvState.GetMain() == WdrvMainState_LocalMasterBss)
    {
        m_WdrvState.SetMain( WdrvMainState_LocalMasterIdle );
    }
    else if(m_WdrvState.GetMain() == WdrvMainState_LocalClient)
    {
        m_WdrvState.SetMain( WdrvMainState_LocalClientIdle );
    }
    else if(m_WdrvState.GetMain() == WdrvMainState_LocalSpectator)
    {
        m_WdrvState.SetMain( WdrvMainState_LocalSpectatorIdle );
    }
    else if(m_WdrvState.GetMain() == WdrvMainState_Detect)
    {
        m_WdrvState.SetMain( WdrvMainState_DetectIdle );
    }

    m_WdrvState.SetSub( WdrvSubState_None );
    m_Authorized = false;
    m_StaticAesMode = false;
}

/* 無線モードの設定 -------------------------------------------------------- */
bool StateMachine::SetWlanMode(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter[id=%ld]\n", __FUNCTION__, pcmdbuff->id);

    bool ret = false;
    uint32_t mode = 0;

    // Readyステートからでしか各モードに遷移は出来ない
    // 既に目的のステートにいる場合は何もせず成功を返す
    // Readyステートへの遷移は各モードのIDLEステートからでしか出来ない

    pcmdbuff->nnResult = ResultSuccess();

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_OutOfService:
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    case WdrvMainState_Sleep:
        // Readyステートへの遷移に限り、チェックする。
        // 遅れてCloseModeが呼ばれる可能性があるため。
        if( pcmdbuff->id == WLAN_CHANGE_STATE_READY )
        {
            if( m_WdrvStateBeforeSleep.GetMain() != WdrvMainState_Ready )
            {
                pcmdbuff->nnResult = ResultInvalidState();
            }
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    case WdrvMainState_Ready:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_READY ||
                (pcmdbuff->id == WLAN_CHANGE_STATE_READY_FROM_LCS_AP && m_IsLcsMode == true)||
                (pcmdbuff->id == WLAN_CHANGE_STATE_READY_FROM_LCS_STA && m_IsLcsMode == true) )
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
            break;
        }

        if(pcmdbuff->id == WLAN_CHANGE_STATE_INFRA)
        {
            mode = m_pWlanFuncs->WlanModeInfraSta;
            m_WdrvState.SetMain( WdrvMainState_InfraIdle );
            m_IsAntennaChangedByUser = false;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_AP)
        {
            mode = m_pWlanFuncs->WlanModeLocalAp;
            m_WdrvState.SetMain( WdrvMainState_LocalMasterIdle );
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_STA)
        {
            mode = m_pWlanFuncs->WlanModeLocalClient;
            m_WdrvState.SetMain( WdrvMainState_LocalClientIdle );
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_SPECTATOR)
        {
            mode = m_pWlanFuncs->WlanModeLocalSpectator;
            m_WdrvState.SetMain( WdrvMainState_LocalSpectatorIdle );
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_LCS_AP)
        {
            mode = m_pWlanFuncs->WlanModeLocalLcsAp;
            m_WdrvState.SetMain( WdrvMainState_LocalMasterIdle );
            m_IsLcsMode = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_LCS_STA)
        {
            mode = m_pWlanFuncs->WlanModeLocalLcsClient;
            m_WdrvState.SetMain( WdrvMainState_LocalClientIdle );
            m_IsLcsMode = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_DETECT)
        {
            mode = m_pWlanFuncs->WlanModeDetect;
            m_WdrvState.SetMain( WdrvMainState_DetectIdle );
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }

        ResetSystemEventsAndConnectionStatus();

        ret = m_pWlanFuncs->ChangeWirelessMode( mode );
        pcmdbuff->Result = true;
        // RxEntryを有効化しておく
        ActivateRxBuffer();
        ActivateRxActionFrame();
        m_pWlanFuncs->InitializeEventmask(mode);
#if defined(USE_WPA_SUPPLICANT)
        if( mode == m_pWlanFuncs->WlanModeInfraSta )
        {
            m_pSuppFuncs->InitializeSupplicant();
        }
#endif
        break;
    case WdrvMainState_InfraIdle:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_INFRA)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            pcmdbuff->Result = true;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
            m_pSuppFuncs->FinalizeSupplicant();
#endif

            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            m_IsAntennaChangedByUser = false;
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    case WdrvMainState_LocalMasterIdle:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_AP &&
                m_IsLcsMode == false)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_LCS_AP &&
                m_IsLcsMode == true)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY &&
                m_IsLcsMode == false)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            // ActionFrameの送出フラグをoffにしておく
            m_IsBeaconActionFrameSet = false;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsMatchingDataRegistered = false;
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY_FROM_LCS_AP &&
                m_IsLcsMode == true)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            // ActionFrameの送出フラグをoffにしておく
            m_IsBeaconActionFrameSet = false;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsMatchingDataRegistered = false;
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
            m_IsLcsMode = false;
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    case WdrvMainState_LocalClientIdle:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_STA &&
                m_IsLcsMode == false)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_LCS_STA &&
                m_IsLcsMode == true)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY &&
                m_IsLcsMode == false)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsMatchingDataRegistered = false;
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY_FROM_LCS_STA &&
                m_IsLcsMode == true)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsMatchingDataRegistered = false;
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
            m_IsLcsMode = false;
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    case WdrvMainState_LocalSpectatorIdle:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_LOCAL_SPECTATOR)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    case WdrvMainState_DetectIdle:
        if(pcmdbuff->id == WLAN_CHANGE_STATE_DETECT)
        {
            // 既に目的のステートにいるので何もせず終わる
            pcmdbuff->Result = true;
        }
        else if(pcmdbuff->id == WLAN_CHANGE_STATE_READY)
        {
            // IFのdownおよびパラメータのリセット
            m_pWlanFuncs->ResetParams();
            m_pWlanFuncs->InitializeEventmask(m_pWlanFuncs->WlanModeInfraSta);
            pcmdbuff->Result = true;

            m_WdrvState.SetMain( WdrvMainState_Ready );
            m_WdrvState.SetSub( WdrvSubState_None );
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            m_Authorized = false;
            m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(false);
#endif
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d %d]", __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}  // NOLINT(impl/function_size)

/* 無線I/Fの Up/Down ------------------------------------------------------- */
bool StateMachine::SetWlanInterfaceUpDown(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    bool ret = true;
    uint32_t up_down;

    up_down = *(reinterpret_cast<uint32_t*>(pcmdbuff->Args));

    pcmdbuff->nnResult = ResultSuccess();

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:

    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:

    case WdrvMainState_DetectIdle:
    case WdrvMainState_Detect:

        /* Up と Down の関数に後で分ける */
        if(up_down == 1)
        {
            /* Interface down => Up */
            switch( m_WdrvState.GetSub() )
            {
            case WdrvSubState_None:
                pcmdbuff->Result = m_pWlanFuncs->SetIfUpDown( 1 );
                if(pcmdbuff->Result == true)
                {
                    m_Authorized = false;
                }
                ret = false;  // IF upイベントを待つ
                break;
            default:
                // 何もしない
                pcmdbuff->Result = true;
                break;
            }
        }
        else
        {
            /* Interface Up => Down */
            switch( m_WdrvState.GetSub() )
            {
            case WdrvSubState_None:
                // 何もしない
                pcmdbuff->Result = true;
                break;
            default:
                pcmdbuff->Result = m_pWlanFuncs->SetIfUpDown( 0 );
                if(pcmdbuff->Result == true)
                {
                    SetEachStateToIdle();
                }
                break;
            }
        }
        break;

    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return ret;
}

/* 無線I/Fの Up/Down 状態の取得 -------------------------------------------- */
bool StateMachine::GetWlanInterfaceUpDown(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    pcmdbuff->nnResult = ResultSuccess();

    if( m_WdrvState.GetMain() == WdrvMainState_OutOfService )
    {
        pcmdbuff->nnResult = ResultInvalidState();
        return true;
    }

    uint32_t *p_up_down;
    p_up_down = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
    pcmdbuff->Result = m_pWlanFuncs->GetIfUpDown( p_up_down );

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* DHDに対するイベントマスク設定 ------------------------------------------- */
bool StateMachine::SetWlanEventMask(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    if( m_WdrvState.GetMain() == WdrvMainState_OutOfService )
    {
        pcmdbuff->nnResult = ResultInvalidState();
        return true;
    }

    uint8_t mask;

    mask = *(reinterpret_cast<uint8_t*>(pcmdbuff->Args));
    pcmdbuff->Result = m_pWlanFuncs->SetEventMask( mask );

    WLAN_LOG_INFO("WDMS: %s[rsulult=%s] leave\n", __FUNCTION__, PRN_BOOL(pcmdbuff->Result));

    return true;
}

/* DHDからのイベントマスク取得 --------------------------------------------- */
bool StateMachine::GetWlanEventMask(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    if( m_WdrvState.GetMain() == WdrvMainState_OutOfService )
    {
        pcmdbuff->nnResult = ResultInvalidState();
        return true;
    }

    uint8_t *pmask;

    pmask = reinterpret_cast<uint8_t*>(pcmdbuff->Args);
    pcmdbuff->Result = m_pWlanFuncs->GetEventMask( pmask );

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* AP動作設定 -------------------------------------------------------------- */
bool StateMachine::SetApConfigration(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    WlanLocalBssConfiguration* pbssinfo;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:  // APの開始チャンネルのセットを行うが、それはI/Fがdown状態でないと受け付けない
            pbssinfo = reinterpret_cast<WlanLocalBssConfiguration*>(pcmdbuff->Args);
            // Check the channel of ap
            if( pbssinfo->channel != -1 )
            {
                int i;
                for( i = 0; i < m_AllowedChannel.count; i++ )
                {
                    if( static_cast<uint32_t>(pbssinfo->channel) == m_AllowedChannel.channel[i] )
                    {
                        break;
                    }
                }
                if( i >= m_AllowedChannel.count )
                {
                    WLAN_LOG_ERROR("The channel you selected is not allowed.\n");
                    pcmdbuff->nnResult = ResultInvalidArgument();
                    return true;
                }
            }
            pcmdbuff->Result = m_pWlanFuncs->SetApConfigration( pbssinfo );
            if(pcmdbuff->Result == true)
            {
                WLAN_LOG_INFO("WDMS: SetApConfigration Success\n");
                pcmdbuff->nnResult = ResultSuccess();
            }
            else
            {
                WLAN_LOG_INFO("WDMS: SetApConfigration fail\n");
                pcmdbuff->nnResult = ResultCommandFailure();
            }
            break;

        default:
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }
    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* Scan Request ------------------------------------------------------------ */
bool StateMachine::ScanRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    bool mez_ret;
    WlanScanParameters* scan_params;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:

    // Master状態でもScan可能
    // だが、何のケアもせずにScanさせて良いかは要検討
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
        switch( m_WdrvState.GetSub() )
        {
        // I/Fがupしている必要があるが、そのケアはimplでやっておく
        case WdrvSubState_Idle:
        case WdrvSubState_Connected:
            BssInfoListClear();
            m_pWlanFuncs->SetAcsdFlag(false);  // Disable ACSD flag

            // Coex機能の1つであるSimultaneous Txを無効化しておく(SIGLO-51407)
            // インフラ、ローカル通信両方とも常に無効化しておくので、有効化に戻すなどの処理は不要。
            // ただし、使用するアンテナ本数によってパラメータは異なる。
            // 既接続スキャンの場合は、SimultaneousTx設定は設定済みなので処理不要
            switch( m_WdrvState.GetMain() )
            {
            case WdrvMainState_InfraIdle:
                if( m_IsAntennaChangedByUser == false ) // ユーザーにアンテナ設定が変更されていない場合
                {
                    WLAN_LOG_INFO("Change antenna settings before scan.\n");
                    // インフラモードでスキャンする場合、アンテナ使用数は最大数にセットする (SIGLO-61889)
                    m_pWlanFuncs->SetTxChain(RxAntennaPattern_Both);
                    m_pWlanFuncs->SetRxChain(RxAntennaPattern_Both);
                    m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);  // (SIGLO-70288)WoWL安定化のため1Antenna用の値にしておく
                }
                break;
            case WdrvMainState_LocalClientIdle:
            case WdrvMainState_LocalSpectatorIdle:
            case WdrvMainState_LocalMasterIdle:
                if( m_IsLcsMode == false )
                {
                    // ローカルモードでスキャンする場合、アンテナ使用数はモード遷移時に1にセットしているので、SimultaneousTx設定も1本用にしておく。
                    m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);
                }
                else
                {
                    // LCSモードではアンテナ2本使用なので、SimultaneousTx設定も2本用にしておく。
                    m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
                }
                break;
            default:
                break;
            }

            // Clear scan complete event
            nn::os::ClearSystemEvent(&m_ScanCompletionEvent);

            scan_params = reinterpret_cast<WlanScanParameters*>(pcmdbuff->Args);
            if( (m_WdrvState.GetMain()==WdrvMainState_InfraIdle) || (m_WdrvState.GetMain()==WdrvMainState_InfraSta) )
            {
#if defined(USE_WPA_SUPPLICANT)
                // WPA_SUPP:
                mez_ret = m_pSuppFuncs->ScanRequest( scan_params );
#else
                mez_ret = m_pWlanFuncs->ScanRequest( scan_params );
#endif
            }
            else
            {
                mez_ret = m_pWlanFuncs->ScanRequest( scan_params );
            }

            if(mez_ret == true)
            {
                m_WdrvState.SetSub( WdrvSubState_Scanning );
                pcmdbuff->nnResult = ResultSuccess();
            }
            else
            {
                pcmdbuff->Result = false;
                pcmdbuff->nnResult = ResultCommandFailure();
            }
            break;
        default:
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            pcmdbuff->Result = false;
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->Result = false;
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    /*
     * ScanCompleteは待たずにIPCは返してしまう
     */
    return true;
}

/* スキャンのキャンセル処理 ------------------------------------------------ */
bool StateMachine::CancelScan(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    pcmdbuff->nnResult = ResultSuccess();

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Scanning:
            if( (m_WdrvState.GetMain()==WdrvMainState_InfraSta) || (m_WdrvState.GetMain()==WdrvMainState_InfraSta) )
            {
#if defined(USE_WPA_SUPPLICANT)
                // WPA_SUPP: cancel scan using WPA supplicant only on infrastructure mode
                // disabled until verified fully
                m_pSuppFuncs->CancelScan();
#else
                pcmdbuff->Result = m_pWlanFuncs->CancelScan();
#endif
            }
            else
            {
                pcmdbuff->Result = m_pWlanFuncs->CancelScan();
            }
            if(pcmdbuff->Result == true)
            {
                m_WdrvState.SetSub( WdrvSubState_CancelingScan );
            }
            break;
        default:
            // do nothing
            break;
        }
        break;
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* Mac Addressのセット ------------------------------------------------------ */
bool StateMachine::SetMacAddress(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    if( m_WdrvState.GetMain() == WdrvMainState_Ready ||
            m_WdrvState.GetMain() == WdrvMainState_DetectIdle )
    {
        WlanMacAddressData macAddr;
        MacAddress* pMac = static_cast<MacAddress*>(pcmdbuff->Args);
        if( *pMac == MacAddress::CreateZeroMacAddress() ||
                *pMac == MacAddress::CreateBroadcastMacAddress() )
        {
            std::memcpy(&macAddr.addr[0], &m_ownMacAddr.GetMacAddressData()[0], MacAddress::MacAddressSize);
        }
        else
        {
            std::memcpy(&macAddr.addr[0], &pMac->GetMacAddressData()[0], MacAddress::MacAddressSize);
        }
        m_pWlanFuncs->SetMacAddress( &macAddr );

        // MACアドレスを取得しておく
        m_pWlanFuncs->GetMacAddress(m_ownMacAddr.GetMacAddressData());
    }
    else
    {
        NN_SDK_ASSERT(false, "%s called in an invalid state", __FUNCTION__);
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* Mac Addressの取得 ------------------------------------------------------- */
bool StateMachine::GetMacAddress(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    pcmdbuff->nnResult = ResultSuccess();
    uint8_t* pmac = static_cast<uint8_t*>(pcmdbuff->Args);
    std::memcpy(pmac, m_ownMacAddr.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* Execute ACSD  --------------------------------------------------------- */
bool StateMachine::ExecuteAcsd(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            m_pWlanFuncs->SetAcsdFlag(true);
            // スキャンがこの後走るのでSimultaneousTxのセットをしておく。
            if( m_IsLcsMode == false )
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);
            }
            else
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
            }
            m_pWlanFuncs->ExecuteAcsd();
            pcmdbuff->nnResult = ResultSuccess();
            break;
        default:
            NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d].",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;
    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d].",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_DEBUG("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* APの開始 ---------------------------------------------------------------- */
bool StateMachine::CreateAp(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    WlanSsidInfo* pSsid;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            // CoexのSimultaneousTx設定を変更しておく
            if( m_IsLcsMode == false )
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);
            }
            else
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
            }
            pSsid = reinterpret_cast<WlanSsidInfo*>(pcmdbuff->Args);
            pcmdbuff->Result = m_pWlanFuncs->CreateAp(pSsid);

            // 即座にリンクチェンジイベントが発生するが、一応待っておく (TODO イベント待ちにする)
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));

            if( m_IsMatchingDataRegistered == true )
            {
                // マッチング用データが登録済みの場合のみ、認証済みと判断する
                m_Authorized = true;
            }
            WLAN_LOG_INFO("WDMS: CreateAp success\n");
            m_WdrvState.SetMain( WdrvMainState_LocalMasterBss );
            m_WdrvState.SetSub( WdrvSubState_Connected );

            if( m_StaticAesMode == true )
            {
                // StaticAESが選択されている場合、ここでBroadcast用のStaticAES鍵を登録しておく
                WlanStaticAesInfo keyInfo;
                keyInfo.keyMode = 1;  // Broadcast用は1
                std::memset(&keyInfo.peerMacAddr[0], 0x00, MacAddress::MacAddressSize);
                std::memcpy(&keyInfo.key[0], &m_StaticAesKey[0], 16);
                m_pWlanFuncs->SetStaticAesKey(&keyInfo);
            }

            // Beacon用途ActionFrameがセット済みの場合、送信開始する
            if( m_IsBeaconActionFrameSet == true )
            {
                uint32_t interval;
                m_pWlanFuncs->GetBeaconInterval(&interval);
                m_pWlanFuncs->PutActionFramePeriodically(
                        m_BeaconActionFrame.data, m_BeaconActionFrame.size,
                        m_BeaconActionFrame.bssid, 0, interval);
                m_IsBeaconActionFrameTransferring = true;
            }

            m_pWlanFuncs->SetAcsdFlag(false);

            // store channel infromation
            {
                ConnectionStatus status = {
                        ConnectionState_Connected,
                        CauseOfInfo_NoInfo,
                        0,
                        Ssid(pSsid->ssid, pSsid->length),
                        m_ownMacAddr,
                        0,
                        0,
                        0,
                        0
                };
                int16_t channel;
                m_pWlanFuncs->GetChannel(&channel);
                status.channel = channel;
                UpdateConnectionStatus(status);
            }

            pcmdbuff->nnResult = ResultSuccess();
            break;
        default:
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d].",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d].",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

bool StateMachine::DestroyAp(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterBss:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
            // ActionFrame送信を停止＆クリアする
            if( m_IsBeaconActionFrameTransferring == true )
            {
                m_pWlanFuncs->CancelPutActionFramePeriodically();
                m_IsBeaconActionFrameTransferring = false;
            }
            m_IsBeaconActionFrameSet = false;

            // I/Fをdownする。Clientが接続しているかどうかはノーケア。Impl側の処理でそれを担保する。
            m_pWlanFuncs->DestroyAp();  // 内部でI/FのdownおよびBANDやChの初期化を行っている。
            m_Authorized = false;
            m_WdrvState.SetMain( WdrvMainState_LocalMasterIdle );
            m_WdrvState.SetSub( WdrvSubState_None );

            // 接続台数制限をデフォルト値に
            m_pWlanFuncs->SetMaxAssociationNumber(7);

            // ConnectionEventをクリアしておく
            nn::os::ClearSystemEvent(&m_ConnectionStatusEvent);
            pcmdbuff->nnResult = ResultSuccess();
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            break;
        }
        break;

    default:
        pcmdbuff->nnResult = ResultInvalidState();
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        break;
    }


    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* ネットワークへの接続(STA) ---------------------------------------------------- */
bool StateMachine::JoinNetworkSta(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WlanConnectinoParameters* pConnectParam;

    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    pcmdbuff->nnResult = ResultSuccess();

    bool ret = true;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            nn::os::ClearSystemEvent(&m_ConnectCompletionEvent);
            m_pWlanFuncs->InitializeConnectionStatus();
            pConnectParam = static_cast<WlanConnectinoParameters*>(pcmdbuff->Args);

            if( m_WdrvState.GetMain()==WdrvMainState_InfraIdle )
            {
                // インフラモードで接続する場合、アンテナ使用数は最大数にセットしてから接続開始する
                m_pWlanFuncs->SetTxChain(RxAntennaPattern_Both);
                m_pWlanFuncs->SetRxChain(RxAntennaPattern_Both);
                // Coex機能の1つであるSimultaneous Txを無効化しておく(SIGLO-51407)
                // インフラ、ローカル通信両方とも常に無効化しておくので、有効化に戻すなどの処理は不要。
                // ただし、使用するアンテナ本数によってパラメータは異なる。(SIGLO-61889)
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna); // (SIGLO-70288)WoWL安定化のため1Antenna用の値にしておく
#if defined(USE_WPA_SUPPLICANT)
                m_IsLinkDownBeforeWpaFailureEv = false;  // SILGO-64396問題に対応するためのフラグ
                // WPA_SUPP
                // use wpa_supplicant function instead
                WLAN_LOG_INFO("<<<<<<<<<<<<<WPA JOIN >>>>>>>>>>>>>\n");
                m_pSuppFuncs->JoinNetwork(pConnectParam);
                ret = false; // WPAサプリカントを使用する場合は、サプリカントスレッドとステートを合わせるためにサプリカントからのイベント待ちをする。
#else
                m_pWlanFuncs->JoinNetworkSta( pConnectParam );
#endif
            }
            else
            {
                if( pConnectParam->channel <= WirelessChannel_13ch )
                {
                    if( m_IsLcsMode == false )
                    {
                        // ローカル通信ではアンテナ本数1本用のSimultaneousTxパラメータをセットしておく
                        m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna);
                        m_pWlanFuncs->EnableCoexMc2();  // ローカル通信ではCoexのMC2モードを有効にしておく
                        m_IsEnabledMc2 = true;
                        WLAN_LOG_INFO("Enabled coex mc2 mode\n");
                    }
                    else
                    {
                        // LCSモードでは、2本用設定にし、MC2モードは有効化しない
                        m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
                        m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, LCS_AC_BE_CWMIN, LCS_AC_BE_CWMAX);
                        WLAN_LOG_INFO("Set CW params for LCS\n");
                        m_pWlanFuncs->SetAciMitigation(AciMitigation_LocalLcs);
                        m_pWlanFuncs->DisableFixedDesense();
                    }
                }

                m_pWlanFuncs->JoinNetworkSta( pConnectParam );
            }

            m_WdrvState.SetSub( WdrvSubState_Connecting );
            pcmdbuff->Result = true;

            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    // Connect完了は待たずに、IPCは返してしまう。
    // ただし、インフラでwpaサプリカントを使用する場合はサプリカントのイベントを待つ。
    return ret;
}

/* ネットワークへの接続WPS(STA) ---------------------------------------------------- */
bool StateMachine::JoinNetworkStaWithWps(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    WlanWpsParameters* pWpsParam;

#if defined(USE_WPA_SUPPLICANT)
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            nn::os::ClearSystemEvent(&m_ConnectCompletionEvent);
            pWpsParam = static_cast<WlanWpsParameters*>(pcmdbuff->Args);

            // インフラモードで接続する場合、アンテナ使用数は最大数にセットしてから接続開始する
            m_pWlanFuncs->SetTxChain(RxAntennaPattern_Both);
            m_pWlanFuncs->SetRxChain(RxAntennaPattern_Both);
            m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_1Antenna); // (SIGLO-70288)WoWL安定化のため1Antenna用の値にしておく
            // WPA_SUPP
            // use wpa_supplicant function instead
            WLAN_LOG_INFO("<<<<<<<<<<<<<WPA JOIN with WPS>>>>>>>>>>>>>\n");
            m_pSuppFuncs->JoinNetworkWithWps(pWpsParam);

            m_WdrvState.SetSub( WdrvSubState_Connecting );
            pcmdbuff->Result = true;
            pcmdbuff->nnResult = ResultSuccess();

            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        break;
    }
#else
    pcmdbuff->nnResult = ResultNotImplemented();
#endif

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    // Connect完了は待たずに、IPCは返してしまう。
    return true;
}

/* ネットワークへの同期(SPECTATOR) ---------------------------------------------------- */
bool StateMachine::JoinNetworkSpectator(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    bool mez_ret;
    WlanConnectinoParameters* pConnectParam;

    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalSpectatorIdle:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            nn::os::ClearSystemEvent(&m_ConnectCompletionEvent);
            pConnectParam = static_cast<WlanConnectinoParameters*>(pcmdbuff->Args);
            if( pConnectParam->channel <= WirelessChannel_13ch )
            {
                m_pWlanFuncs->EnableCoexMc2();  // ローカル通信ではCoexのMC2モードを有効にしておく
                m_IsEnabledMc2 = true;
                WLAN_LOG_INFO("Enabled coex mc2 mode\n");
            }
            m_pWlanFuncs->JoinNetworkSpectator( pConnectParam );
            // Spectatorの場合、WLC_E_SSIDイベントだけが発生する
            m_WdrvState.SetSub( WdrvSubState_Connecting );
            pcmdbuff->Result = true;

            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* ネットワークへの接続中止 ---------------------------------------------------- */
bool StateMachine::CancelJoinNetwork(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    bool ret = true;
    pcmdbuff->nnResult = ResultSuccess();
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connecting:
            if( m_WdrvState.GetMain()==WdrvMainState_InfraIdle )
            {   // インフラ通信モードの場合
#if defined(USE_WPA_SUPPLICANT)
                // WPA_SUPP: use supplicant function instead
                m_pSuppFuncs->CancelJoinNetwork();
                // サプリカントとのステートズレ防止のために、イベント通知を待つ。
                // サブステートの変更やConnect完了通知イベントのシグナルもイベント通知を受けてから行う。
                ret = false;
#else
                pcmdbuff->Result = m_pWlanFuncs->SetIfUpDown( 0 );
                SetEachStateToIdle();

                // Connect完了通知イベントをシグナルさせる
                nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
#endif
            }
            else
            {   // ローカル通信モードの場合
                if( m_IsEnabledMc2 == true )
                {
                    m_pWlanFuncs->DisableCoexMc2();
                    m_IsEnabledMc2 = false;
                }
                if( m_IsLcsMode == true )
                {
                    m_pWlanFuncs->SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
                    m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                }
                pcmdbuff->Result = m_pWlanFuncs->SetIfUpDown( 0 );
                SetEachStateToIdle();

                // Connect完了通知イベントをシグナルさせる
                nn::os::SignalSystemEvent(&m_ConnectCompletionEvent);
            }
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return ret;
}

/* ネットワークからの離脱 -------------------------------------------------- */
bool StateMachine::DisconnectRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    bool ret = true;
    bool mez_ret;

    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    pcmdbuff->nnResult = ResultSuccess();

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
        // 特に何もせず抜ける
        break;
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
            if( m_WdrvState.GetMain()==WdrvMainState_InfraSta )
            {
#if defined(USE_WPA_SUPPLICANT)
                // WPA_SUPP, we will use instead WPA supp disassoc API here
                mez_ret = m_pSuppFuncs->Disassociate();
                NN_ABORT_UNLESS(mez_ret, "WPA disassociate failed");
#else
                mez_ret = m_pWlanFuncs->Disassociate();
#endif
            }
            else
            {
                mez_ret = m_pWlanFuncs->Disassociate();
            }

            if( mez_ret == true )
            {
                // 接続中なので、切断ステートに移行する
                m_WdrvState.SetSub( WdrvSubState_Disconnecting );
                ret = false;
            }
            else
            {
                // 接続中でないので、そのまま何もせず抜ける
                // ただし、StateMachineのステートはConnectedのままなので、切断済みに更新しておく
                if( m_WdrvState.GetMain() == WdrvMainState_InfraSta )
                {
                    // インフラモードではここに来ることは無いはず。
                    WLAN_LOG_ERROR("%s: cmd queue of wpa supp may be full.\n", __FUNCTION__);
                    m_WdrvState.SetMain( WdrvMainState_InfraIdle );
                }
                else if( m_WdrvState.GetMain() == WdrvMainState_LocalClient )
                {
                    m_WdrvState.SetMain( WdrvMainState_LocalClientIdle );
                }
                else if( m_WdrvState.GetMain() == WdrvMainState_LocalSpectator )
                {
                    m_WdrvState.SetMain( WdrvMainState_LocalSpectatorIdle );
                }
                m_WdrvState.SetSub( WdrvSubState_Idle );

                // ConnectionStatusを更新し、通知しておく
                ConnectionStatus status;
                status.aid = 0;
                status.state = ConnectionState_Idle;
                status.channel = 0;
                status.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
                status.cause = CauseOfInfo_DisconnectReq;
                status.ssid.Set("");
                status.statusReasonCode = 0;
                status.beaconInterval = 0;
                status.capabilityInfo = 0;
                UpdateConnectionStatus(status);

                // 接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);

                m_Authorized = false;
                ret = true;
            }
            break;
        default:
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d %d]", __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return ret;
}

bool StateMachine::DeauthRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    WlanDisconnectInfo* deauthParam;
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
        // スルーしておく
        pcmdbuff->nnResult = ResultSuccess();
        break;
    case WdrvMainState_LocalMasterBss:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
            // Clientの切断は無線ドライバ関数を呼んだ時点で成功としておく。
            // 切断完了イベントが発生しないパターンもあるので、常にイベント待ちには出来ないため。
            deauthParam = static_cast<WlanDisconnectInfo*>(pcmdbuff->Args);
            pcmdbuff->nnResult = m_pWlanFuncs->Deauthenticate(deauthParam); // ClientStatusの更新が内部で行われる
            if( pcmdbuff->nnResult.IsSuccess() )
            {
                // CLIENTの接続状態に変化が起きたのでConnectionStatusEventをシグナルする
                nn::os::SignalSystemEvent(&m_ConnectionStatusEvent);
            }
            else if( ResultNoClients::Includes(pcmdbuff->nnResult) )
            {
                pcmdbuff->nnResult = ResultSuccess();
            }
            pcmdbuff->Result = true;
            break;
        default:
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

/* Management frameへのVIE追加 ----------------------------------------------- */
bool StateMachine::AddIe(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    WlanIeContainer* pContainer;

    // IEの追加はいつでも可能ではあるが、WDMのレイヤーでステート制限を設けるべき？
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connected:
            pContainer = reinterpret_cast<WlanIeContainer*>(pcmdbuff->Args);
            pcmdbuff->Result = m_pWlanFuncs->AddIe(&pContainer->index, pContainer);
            pcmdbuff->nnResult = ResultSuccess();
            break;
        default:
            NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    // 即時実行完了形のコマンドなので、イベントをシグナルさせて良い
    return true;
}

/* VIE削除 ----------------------------------------------- */
bool StateMachine::DeleteIe(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    uint32_t* pIeIndex;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:

        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connected:
            pIeIndex = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
            pcmdbuff->Result = m_pWlanFuncs->DeleteIe(*pIeIndex);
            if( pcmdbuff->Result == true )
            {
                pcmdbuff->nnResult = ResultSuccess();
            }
            else
            {
                pcmdbuff->nnResult = ResultInvalidArgument();
            }
            break;
        default:
            NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;

    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::AddMatchingDataInfoToRxEntry(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    // ローカル通信の時のみ使用されるのでステートチェック

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:
        {
            WlanMatchingDataInfo* pMatchInfo = reinterpret_cast<WlanMatchingDataInfo*>(pcmdbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pMatchInfo->pInfo);
            pcmdbuff->nnResult = m_RxEntries.AddMatchingInfo(pMatchInfo->rxId, static_cast<const ReceivedDataMatchInfo&>(*pMatchInfo->pInfo));

            if( pcmdbuff->nnResult.IsSuccess() )
            {
                // Removeでマッチングデータが空になっても、それを判断する術がないので、
                // 一度m_IsMatchingDataRegisteredがtrueになったらtrueになりっぱなし。仕方ないので仕様としておく。
                m_IsMatchingDataRegistered = true;

                if( m_WdrvState.GetSub() == WdrvSubState_Connected )
                {
                    // CONNECTED状態で呼ばれた場合m_Authorizedをtrueにしておく
                    m_Authorized = true;
                }
            }
        }
        break;
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]", m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::RemoveMatchDataInfoToRxEntry(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    // ローカル通信の時のみ使用されるのでステートチェック

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_LocalSpectator:
    {
        WlanMatchingDataInfo* pMatchInfo = reinterpret_cast<WlanMatchingDataInfo*>(pcmdbuff->Args);
        NN_SDK_ASSERT_NOT_NULL(pMatchInfo->pInfo);
        pcmdbuff->nnResult = m_RxEntries.RemoveMatchInfo(pMatchInfo->rxId, static_cast<const ReceivedDataMatchInfo&>(*pMatchInfo->pInfo));
        break;
    }
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::SetStaticAesMode(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    // ローカル通信の時のみ使用される

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
    {
        bool* pMode = reinterpret_cast<bool*>(pcmdbuff->Args);
        m_pWlanFuncs->SetStaticAesMode(*pMode);
        m_StaticAesMode = *pMode;  // 接続後にUnicastKeyのセットをする必要があるので、StateMachineクラスで覚えておく
        pcmdbuff->nnResult = ResultSuccess();
        break;
    }
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::SetClientTimeout(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    // ローカル通信MASTERの時のみ使用される

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterBss:
    {
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
        {
            uint32_t* pTime = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
            m_pWlanFuncs->SetClientTimeoutPeriod(*pTime);
            pcmdbuff->nnResult = ResultSuccess();
            break;
        }
        default:
            DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;
    }
    default:
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::SetBeaconLostTimeout(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
    {
        int32_t* pTime = reinterpret_cast<int32_t*>(pcmdbuff->Args);
        m_pWlanFuncs->SetBeaconLostTimeout(static_cast<int>(*pTime));
        pcmdbuff->nnResult = ResultSuccess();
        break;
    }
    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d %d].", __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }
    return true;
}

bool StateMachine::SetRxChain(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
    {
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
        case WdrvSubState_Connected:
        {
            int32_t* pChain = reinterpret_cast<int32_t*>(pcmdbuff->Args);
            m_pWlanFuncs->SetRxChain(static_cast<int>(*pChain));
            if( *pChain == RxAntennaPattern_Both )
            {
                WLAN_LOG_INFO("Use 2 antennas.\n");
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
            }
            else
            {
                WLAN_LOG_INFO("Use %s antenna.\n", (*pChain == RxAntennaPattern_0) ? "ANT_0" : "ANT_1" );
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_Default);  // (SIGLO-66348)
            }
            m_IsAntennaChangedByUser = true;
            pcmdbuff->nnResult = ResultSuccess();
            break;
        }
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            break;
        }
        break;
    }
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }
    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

bool StateMachine::SetMaxAssociationNumber(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    {
        int* pMaxNum = reinterpret_cast<int*>(pcmdbuff->Args);
        {
            bool ret;
            ret = m_pWlanFuncs->SetMaxAssociationNumber(*pMaxNum);
            if( ret == true )
            {
                pcmdbuff->nnResult = ResultSuccess();
            }
            else
            {
                pcmdbuff->nnResult = ResultInvalidArgument();
            }
        }
        break;
    }
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }
    return true;
}

bool StateMachine::SetActionFrameWithBeacon(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    // ローカル通信MASTERの時のみ使用される

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
    case WdrvMainState_LocalMasterBss:
    {
        // データ保存
        WlanPeriodicActionFrameInfo* pInfo = reinterpret_cast<WlanPeriodicActionFrameInfo*>(pcmdbuff->Args);
        NN_SDK_REQUIRES_NOT_NULL(pInfo);
        NN_SDK_REQUIRES_NOT_NULL(pInfo->pData);
        NN_SDK_REQUIRES_LESS_EQUAL(pInfo->size, BeaconActionFrameSizeMax);

        // ActionFrameデータのコピー
        std::memcpy(&m_BeaconActionFrame.data[0], pInfo->pData, pInfo->size);
        m_BeaconActionFrame.size = static_cast<uint16_t>(pInfo->size);
        std::memcpy(&m_BeaconActionFrame.bssid[0], &pInfo->bssid[0], MacAddress::MacAddressSize);

        if( m_WdrvState.GetMain() == WdrvMainState_LocalMasterBss )
        {
            // 既にBSS展開済みなので、直ぐに送信開始する

            uint32_t interval;
            m_pWlanFuncs->GetBeaconInterval(&interval);
            m_pWlanFuncs->PutActionFramePeriodically(
                    m_BeaconActionFrame.data, m_BeaconActionFrame.size,
                    m_BeaconActionFrame.bssid, 0, interval);
            m_IsBeaconActionFrameTransferring = true;
        }

        m_IsBeaconActionFrameSet = true;

        pcmdbuff->nnResult = ResultSuccess();
        break;
    }
    default:
        NN_SDK_ASSERT(false, "%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        DisplayIlligalMessage(pcmdbuff->id, __FUNCTION__);
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::CancelActionFrameWithBeacon(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    // ローカル通信MASTERの時のみ使用される

    pcmdbuff->nnResult = ResultSuccess();
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalMasterIdle:
        // BSS展開しないとクリア出来ないのは不便なのでIdle状態でもクリア出来るようにしておく。ただし送信停止は無線ドライバの仕様上出来ない。
        m_IsBeaconActionFrameSet = false;
        break;
    case WdrvMainState_LocalMasterBss:
    {
        if( m_IsBeaconActionFrameTransferring == true )
        {
            m_pWlanFuncs->CancelPutActionFramePeriodically();
            m_IsBeaconActionFrameTransferring = false;
        }
        else
        {
            WLAN_LOG_INFO("[WLAN]Periodic action frame is not sent.\n");
        }
        m_IsBeaconActionFrameSet = false;
        break;
    }
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

bool StateMachine::GetRssi(WlanCommand* pcmdbuff ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    int32_t* pRssi = reinterpret_cast<int32_t*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pRssi);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalSpectator:
        pcmdbuff->Result = m_pWlanFuncs->GetRssiForSta(pRssi);
        if( pcmdbuff->Result == true )
        {
            pcmdbuff->nnResult = ResultSuccess();
        }
        else
        {
            *pRssi = -128;
            pcmdbuff->nnResult = ResultCommandFailure();
        }
        break;
    default:
        *pRssi = -128;
        pcmdbuff->nnResult = ResultInvalidState();
        break;
    }

    return true;
}

void StateMachine::GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutDeltaTime);

    m_pWlanFuncs->GetDeltaTimeBetweenSystemAndTsf(pOutDeltaTime);
}

bool StateMachine::GetState(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
    NN_ABORT_UNLESS_NOT_NULL(state);

    state->mainState = m_WdrvState.GetMain();
    state->subState  = m_WdrvState.GetSub();
    if( m_IsLcsMode == true )
    {
        if( state->mainState == WdrvMainState_LocalMasterIdle )
        {
            state->mainState = WdrvMainState_LocalLcsMasterIdle;
        }
        else if( state->mainState == WdrvMainState_LocalMasterBss )
        {
            state->mainState = WdrvMainState_LocalLcsMasterBss;
        }
        else if( state->mainState == WdrvMainState_LocalClientIdle )
        {
            state->mainState = WdrvMainState_LocalLcsClientIdle;
        }
        else if( state->mainState == WdrvMainState_LocalClient )
        {
            state->mainState = WdrvMainState_LocalLcsClient;
        }
    }
    pcmdbuff->Result = true;

    return true;
}

bool StateMachine::GetStateBeforeSleep(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WlanStates* state = reinterpret_cast<WlanStates*>(pcmdbuff->Args);
    NN_SDK_ASSERT_NOT_NULL(state);

    state->mainState = m_WdrvStateBeforeSleep.GetMain();
    state->subState  = m_WdrvStateBeforeSleep.GetSub();
    pcmdbuff->Result = true;

    return true;
}

bool StateMachine::SetTsfTimerEventmask(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    bool* pFlag = reinterpret_cast<bool*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pFlag);

    m_pWlanFuncs->SetTsfTimerEventmask(*pFlag);
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::GetFwVersion(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Stop:
    case WdrvMainState_Sleep:
        pcmdbuff->nnResult = ResultInvalidState();
        return true;
    case WdrvMainState_OutOfService:
        pcmdbuff->nnResult = ResultWlanDeviceAbnormal();
        return true;
    default:
        break;
    }

    WlanIoctlResult* pOutResult = reinterpret_cast<WlanIoctlResult*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pOutResult);

    m_pWlanFuncs->GetFwVersion(pOutResult);
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::RequestSleep(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    bool ret = true;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_OutOfService:
        WLAN_LOG_INFO("Start: SLEEP PROCESSING for normal sleep\n");
        // RxEntryを無効化しておく
        InactivateRxBuffer();
        InactivateRxActionFrame();

        ResetSystemEventsAndConnectionStatus();

        // スリープ直前のステートを保存
        m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
        m_WdrvStateBeforeSleep.SetSub( WdrvSubState_None );
        // スリープステート時に来たイベント数を数えるカウンタを0にしておく
        m_EventMessageSleepCnt = 0;

        // 現在のステートをスリープステートへ
        m_WdrvState.SetMain( WdrvMainState_OutOfService );
        m_WdrvState.SetSub( WdrvSubState_None );
        m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
        m_pSuppFuncs->SetSleepFlag(true);
#endif

        // 無線ドライバーの終了処理
        if( m_IsDrvInitFail == false && nn::wlan::FirmwareDebugSettings::IsSkipBoot() == false )
        {
            // イベントマスクのクリア (SIGLO-76491)
            m_pWlanFuncs->ClearEventMask();
            m_pWlanFuncs->FinalizeDriver();
        }

        // WoWL起床要因をリセットしておく
        m_WowlWakeReason = 0;
        RemoveTcpSessionInfo();
        // スリープ復帰のシーケンスは回しておく必要があるので成功を返しておく
        pcmdbuff->nnResult = ResultSuccess();
        m_sleepMode = SleepMode_Normal;
        break;
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
        WLAN_LOG_INFO("Finish: SLEEP PROCESSING for normal sleep\n");
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
        case WdrvSubState_Idle:
#if defined(USE_WPA_SUPPLICANT)
            // WoWL用のGTKが登録されたままの場合があるのでチェック。見逃してしまうとSIGLO-43733の問題が起きてしまう。
            if( m_WdrvState.GetMain() == WdrvMainState_InfraIdle && m_IsGtkSet == true )
            {
                WLAN_LOG_INFO("GTK has been registerd in spite of normal sleep.\n");
                pcmdbuff->nnResult = ResultSleepFailure();
                break;
            }
#endif
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            // スリープ直前のステートを保存
            m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
            m_WdrvStateBeforeSleep.SetSub( WdrvSubState_None );
            // スリープステート時に来たイベント数を数えるカウンタを0にしておく
            m_EventMessageSleepCnt = 0;

            // 現在のステートをスリープステートへ
            m_WdrvState.SetMain( WdrvMainState_Sleep );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(true);
#endif
            // APと繋がっていないので完全に終了させる
            // 念のためI/Fはdownしておく
            m_pWlanFuncs->SetIfUpDown(0);

#if defined(USE_WPA_SUPPLICANT)
            if( m_WdrvState.GetMain() == WdrvMainState_InfraIdle )
            {
                // WPAサプリカントの終了処理
                m_pSuppFuncs->FinalizeSupplicant();
            }
#endif
            // イベントマスクのクリア (SIGLO-76491)
            m_pWlanFuncs->ClearEventMask();
            // 無線ドライバーの終了処理
            m_pWlanFuncs->FinalizeDriver();
            // WoWL起床要因をリセットしておく
            m_WowlWakeReason = 0;
            RemoveTcpSessionInfo();
            pcmdbuff->nnResult = ResultSuccess();
            m_sleepMode = SleepMode_Normal;
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    case WdrvMainState_InfraSta:
        WLAN_LOG_INFO("Start: SLEEP PROCESSING for WoWL\n");
        // WoWLによるスリープへ
        // 完全に接続を保ったままスリープに入ることを保証させるためにイベント待ちにしておく
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Connected:
#if defined(USE_WOWL_WAKEPATTERN) && !defined(ENABLE_WOWL_AGING_DEBUG)
            // WoWLのためにTCPセッション情報が登録されているかチェック
            if( m_IsSetTcpInfoWoWL == false )
            {
                // TCPセッション情報が登録されていないのでAPから切断する。
                // Impl層からやってもらう。
                WLAN_LOG_INFO("TCP session information is not registered. Disconnect from AP\n");
                pcmdbuff->nnResult = ResultSleepFailure();
                break;
            }
#endif
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();
            // パケット送信を制限するためのフラグ
            m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(true);
#endif
            // WoWL用のパラメータ設定
            m_pWlanFuncs->SetupWowlParams(&m_WowlParams);
            // イベントマスクのクリア (SIGLO-75396)
            m_pWlanFuncs->ClearEventMask();

            // WoWLモード開始
            if( m_pWlanFuncs->ActivateWowl() == false )
            {
                // WoWLモード遷移失敗
                WLAN_LOG_INFO("[SM]ActivateWowl failed.\n");

                // WoWL用パラメータクリア
                m_pWlanFuncs->ClearWowlParams();
                m_IsSetTcpInfoWoWL = false;
                RemoveTcpSessionInfo();

                // 未接続状態に移行させる。Impl層からやってもらう。
                pcmdbuff->nnResult = ResultSleepFailure();

                // RxEntryを有効化しておく
                ActivateRxBuffer();
                ActivateRxActionFrame();
                m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
                m_pSuppFuncs->SetSleepFlag(false);
#endif
                break;
            }
            else
            {
                WLAN_LOG_INFO("[SM]ActivateWowl success.\n");
                // 無線チップをWoWL用の省電力モードへ。
                m_pWlanFuncs->SetDevicePowerState(false);
            }

            // スリープ直前のステートを保存
            m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
            m_WdrvStateBeforeSleep.SetSub( m_WdrvState.GetSub() );
            // スリープステート時に来たイベント数を数えるカウンタを0にしておく
            m_EventMessageSleepCnt = 0;
            // 現在のステートをスリープステートへ
            m_WdrvState.SetMain( WdrvMainState_Sleep );
            m_WdrvState.SetSub( WdrvSubState_None );
            // 起床要因のリセット
            m_WowlWakeReason = 0;
            pcmdbuff->nnResult = ResultSuccess();
            m_sleepMode = SleepMode_Wowl;
            WLAN_LOG_INFO("[SUCCESS]Entered wowl sleep mode.\n");
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
    WLAN_LOG_INFO("Finish: SLEEP PROCESSING\n");

    return ret;
}  //NOLINT(impl/function_size)

bool StateMachine::RequestWakeUp(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    WLAN_LOG_INFO("Start: WAKE-UP PROCESSING\n");

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Sleep:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
            m_WowlWakeReason = 0;
            // スリープからの復帰処理可能
            // WoWLスリープからの復帰かチェック
            if( m_WdrvStateBeforeSleep.GetMain() == WdrvMainState_InfraSta &&
                    m_WdrvStateBeforeSleep.GetSub() == WdrvSubState_Connected )
            {
                // WoWLからの復帰
                WLAN_LOG_INFO("Wake up from Wowl sleep.\n");

                // RxEntryを有効化しておく
                ActivateRxBuffer();
                ActivateRxActionFrame();

                // 無線チップをWoWLの省電力モードから復帰させる
                m_pWlanFuncs->SetDevicePowerState(true);

                // 起床要因を取得
                {
                    m_pWlanFuncs->GetWakeupReasonRaw(&m_WowlWakeReason, &m_WowlWakeCount);
                    if( ConvertWakeupReason(m_WowlWakeReason) == WowlWakeReason_Linkdown )
                    {
                        // APからの切断、または切断した方が良い理由での起床
                        pcmdbuff->nnResult = ResultWakeupWithLinkDown();
                    }
                    else
                    {
                        // パケット受信による起床またはユーザー操作やシステムの自発的起床
                        pcmdbuff->nnResult = ResultSuccess();
                    }
                }
                // スタッツの取得
                m_pWlanFuncs->GetWowlSleepStats(&m_WowlSleepStats);
                // WoWLの解除。FWからイベントやデータパケットが送られ始める。
                m_pWlanFuncs->DisableWowl();
                // WoWL用パラメータクリア
                m_pWlanFuncs->ClearWowlParams();

                // ステートをスリープ前に戻しておく
                m_WdrvState.SetMain( m_WdrvStateBeforeSleep.GetMain() );
                m_WdrvState.SetSub( m_WdrvStateBeforeSleep.GetSub() );

                m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
                m_pSuppFuncs->SetSleepFlag(false);
#endif
            }
            else
            {
                // WoWLスリープからの復帰ではない
                WLAN_LOG_INFO("Wake up from normal sleep.\n");
                // システム起因の起床カウントなので該当カウンタを増やす
                m_WowlWakeCount.system += 1;
                pcmdbuff->nnResult = ResultWakeupNormalSleep();
            }

            // スリープからの復帰時点でTCPセッション情報はクリアしておく
            m_IsSetTcpInfoWoWL = false;
            RemoveTcpSessionInfo();

            // イベント処理を止めているので再開させる
            SignalEventMessage();
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        pcmdbuff->nnResult = ResultAlreadyAwake();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
    WLAN_LOG_INFO("Finish: WAKE-UP PROCESSING\n");

    return true;
}

bool StateMachine::GetWakeupReasonRaw(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    uint32_t* pReason = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pReason);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        *pReason = m_WowlWakeReason;
        pcmdbuff->nnResult = ResultSuccess();
        break;
    default:
        *pReason = 0;
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    return true;
}

WowlWakeReason StateMachine::ConvertWakeupReason(uint32_t reason) NN_NOEXCEPT
{
    return m_pWlanFuncs->ConvertWakeupReason(reason);
}

void StateMachine::GetAllowedChannels(int16_t* pBuf, size_t length, uint32_t* pCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuf);
    NN_SDK_ASSERT_NOT_NULL(pCount);

    size_t cnt = length;
    if( length > driver::ChannelCountMax )
    {
        cnt = driver::ChannelCountMax;
    }
    std::memcpy(pBuf, &m_AllowedChannel.channel[0], sizeof(int16_t) * cnt);
    *pCount = m_AllowedChannel.count;
}

Result StateMachine::CheckChannels(const int16_t* pBuf, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuf);

    for( int i = 0; i < length; i++ )
    {
        int j;
        for( j = 0; j < m_AllowedChannel.count; j++ )
        {
            if( pBuf[i] == m_AllowedChannel.channel[j] )
            {
                break;
            }
        }
        if( j >= m_AllowedChannel.count )
        {
            WLAN_LOG_ERROR("%s: invalid channel [%d]\n", __FUNCTION__, pBuf[i]);
            return ResultInvalidArgument();
        }
    }

    return ResultSuccess();
}

bool StateMachine::SetWakeupReasonRaw(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    uint32_t* pReason = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pReason);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        m_pWlanFuncs->SetWakeupReasonRaw(*pReason);
        pcmdbuff->nnResult = ResultSuccess();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    return true;
}

bool StateMachine::EnableWowlFeatures(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    uint32_t* pFeatures = reinterpret_cast<uint32_t*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pFeatures);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        m_pWlanFuncs->EnableWowlFeatures(*pFeatures);
        pcmdbuff->nnResult = ResultSuccess();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    return true;
}

bool StateMachine::GetWowlWakeCount(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    WowlWakeCount* pCount = reinterpret_cast<WowlWakeCount*>(pcmdbuff->Args);
    NN_SDK_ASSERT_NOT_NULL(pCount);

    std::memcpy(pCount, &m_WowlWakeCount, sizeof(WowlWakeCount));
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::GetWowlSleepStats(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    WowlSleepStats* pStats = reinterpret_cast<WowlSleepStats*>(pcmdbuff->Args);
    NN_SDK_ASSERT_NOT_NULL(pStats);

    std::memcpy(pStats, &m_WowlSleepStats, sizeof(WowlSleepStats));
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::ClearWowlWakeCount(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    std::memset(&m_WowlWakeCount, 0, sizeof(WowlWakeCount));
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::ClearWowlSleepStats(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    std::memset(&m_WowlSleepStats, 0, sizeof(WowlSleepStats));
    pcmdbuff->nnResult = ResultSuccess();
    return true;
}

bool StateMachine::SetGtkInfo(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    bool ret = true;
    pcmdbuff->Result = false;

#if defined(USE_WPA_SUPPLICANT)
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraSta:
        // WoWLスリープ条件が揃っているときのみ、GTKをFWにセット
        if( m_WdrvState.GetSub() == WdrvSubState_Connected && m_IsSetTcpInfoWoWL == true )
        {
            // GTK情報をFWに登録
            WLAN_LOG_INFO("Set GTK info because wowl condition is met.\n");
            m_pSuppFuncs->SetGtkKeyInfo();
            pcmdbuff->nnResult = ResultSuccess();
            ret = false;
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
#else
    pcmdbuff->nnResult = ResultInvalidState();
#endif

    return ret;
}

bool StateMachine::ClearGtkInfo(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    bool ret = true;
    pcmdbuff->Result = false;

#if defined(USE_WPA_SUPPLICANT)
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        // GTK登録済みの場合のみクリアを実行する
        if( m_IsGtkSet == true )
        {
            // GTK情報をクリア
            WLAN_LOG_INFO("Clear GTK info because it is stored in FW.\n");
            m_pSuppFuncs->ClearGtkKeyInfo();
            pcmdbuff->nnResult = ResultSuccess();
            ret = false;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
#else
    pcmdbuff->nnResult = ResultInvalidState();
#endif

    return ret;
}

bool StateMachine::EmulateDriverInitFail(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    m_IsDrvInitFail = true;

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

bool StateMachine::GetChannelStats(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
    WlanTotalChannelStats* pTotalStats = reinterpret_cast<WlanTotalChannelStats*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pTotalStats);

    bool ret = m_pWlanFuncs->GetChannelStats(
            pTotalStats->stats,
            pTotalStats->length,
            pTotalStats->count);
    if( ret == true )
    {
        pcmdbuff->nnResult = ResultSuccess();
    }
    else
    {
        pcmdbuff->nnResult = ResultCommandFailure();
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);

    return true;
}

bool StateMachine::PutOneShotActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_LocalClient:
    case WdrvMainState_LocalClientIdle:
    case WdrvMainState_LocalMasterBss:
    case WdrvMainState_LocalSpectator:
    case WdrvMainState_LocalSpectatorIdle:
    case WdrvMainState_Detect:
        // IdleまたはConnected時のみ送信を許可する
        if( m_WdrvState.GetSub() == WdrvSubState_Idle || m_WdrvState.GetSub() == WdrvSubState_Connected )
        {
            WlanOneShotActionFrameInfo* pInfo = reinterpret_cast<WlanOneShotActionFrameInfo*>(pcmdbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pInfo);
            pcmdbuff->nnResult = m_pWlanFuncs->PutActionFrameOneShot(pInfo->pData, static_cast<uint16_t>(pInfo->size),
                    pInfo->dstMac, pInfo->bssid, pInfo->channel, pInfo->dwellTime);
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
    return true;
}

bool StateMachine::StartPeriodicActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Detect:
        // Idle時のみ開始を許可する
        if( m_WdrvState.GetSub() == WdrvSubState_Idle )
        {
            WlanPeriodicActionFrameInfo* pInfo = reinterpret_cast<WlanPeriodicActionFrameInfo*>(pcmdbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pInfo);

            DetectPeriodicAfCycle ptn;
            memcpy(&ptn, &m_DetectHdComuPat, sizeof(DetectPeriodicAfCycle));
            ptn.txInterval = pInfo->interval;
            m_pWlanFuncs->PutActionFramePeriodicallyForDetect(
                    pInfo->pData, static_cast<uint16_t>(pInfo->size), ptn);
            pcmdbuff->nnResult = ResultSuccess();
        }
        else
        {
            pcmdbuff->nnResult = ResultInvalidState();
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
    return true;
}

bool StateMachine::CancelPeriodicActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Detect:
        // Cancelはいつでも可能
        m_pWlanFuncs->CancelPutActionFramePeriodically();
        pcmdbuff->nnResult = ResultSuccess();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
    return true;
}

/* StateMachineが保持しているSystemEventのReadableHandleを返す --------------------------- */
nn::os::NativeHandle StateMachine::GetSystemEventReadableHandle(detail::SystemEventType type) NN_NOEXCEPT
{
    nn::os::NativeHandle retHandle;

    if( type == detail::SystemEventType_ConnectionEvent )
    {
        retHandle = nn::os::GetReadableHandleOfSystemEvent(&m_ConnectionStatusEvent);
    }
    else if( type == detail::SystemEventType_ScanComplete )
    {
        retHandle = nn::os::GetReadableHandleOfSystemEvent(&m_ScanCompletionEvent);
    }
    else if( type == detail::SystemEventType_ConnectionComplete )
    {
        retHandle = nn::os::GetReadableHandleOfSystemEvent(&m_ConnectCompletionEvent);
    }
    else
    {
        NN_ABORT("SystemEventType is invalid\n");
    }

    return retHandle;
}

/* StateMachineが保持しているConnectionStatusを返す --------------------------- */
void StateMachine::GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT
{
    // ステートチェックした方が良い？
    NN_SDK_ASSERT_NOT_NULL(pOutStatus);

    nn::os::LockMutex(&m_ConnectionStatusMutex);
    *pOutStatus = m_ConnectionStatus;
    nn::os::UnlockMutex(&m_ConnectionStatusMutex);
}

/* ClientStatusManagerが保持しているClientStatusを返す --------------------------- */
void StateMachine::GetClientStatus(WlanClientStatusInfo* pOutInfo) NN_NOEXCEPT
{
    // ステートチェックした方が良い？
    NN_SDK_ASSERT_NOT_NULL(pOutInfo);

    m_pWlanFuncs->GetClientStatusList(pOutInfo->status, sizeof(pOutInfo->status) / sizeof(ClientStatus), &pOutInfo->updatedIndexBitMap, true);
}

/* WoWL時のTCPセッション維持のための情報を登録しておく ----------------------------------- */
nn::Result StateMachine::RegisterTcpSessionInfo(WlanWowlSetupParams params) NN_NOEXCEPT
{
    std::memcpy(&m_WowlParams, &params, sizeof(WlanWowlSetupParams));
    m_IsSetTcpInfoWoWL = true;

    return ResultSuccess();
}

/* WoWL時のTCPセッション維持のための情報を削除する ------------------------------------- */
nn::Result StateMachine::RemoveTcpSessionInfo() NN_NOEXCEPT
{
#ifndef ENABLE_WOWL_AGING_DEBUG
    std::memset(&m_WowlParams, 0, sizeof(WlanWowlSetupParams));
    m_IsSetTcpInfoWoWL = false;
#endif
    return ResultSuccess();
}

/* Stateマシーンへの処理要求 ----------------------------------------------- */
bool StateMachine::CommandProcess() NN_NOEXCEPT
{
    bool ret;

    /*
     * コマンド処理結果を、OSのイベントオブジェクトで待っている呼び出し手に対して、
     * そのイベントオブジェクトをシグナルさせるかどうか判断するためのflag
     */
    bool post_fin_event = false;

    uintptr_t pmsg;
    WlanCommand* pcmdbuff;

    ret = nn::os::TryReceiveMessageQueue(&pmsg, &m_CommandQueue);
    if(ret == true)
    {
        NN_SDK_ASSERT( pmsg != NULL );

        pcmdbuff = reinterpret_cast<WlanCommand*>(pmsg);
#if defined (WLANTYPE_DEBUG)
        NN_SDK_ASSERT(pcmdbuff->_magicWord == WlanTypeMagic);
#endif
        /* コマンドバッファ初期化時に false となっているはずだが、念のため */
        pcmdbuff->Result = false;

        switch( pcmdbuff->id )
        {
        case WLAN_INITIALIZE:
            post_fin_event = InitalizeWlanDriver( pcmdbuff );
            break;

        case WLAN_INITIALIZE_ON_SKIPBOOT:
            post_fin_event = InitalizeWlanDriverOnSkipBoot( pcmdbuff );
            break;

        case WLAN_FINALIZE:
            post_fin_event = FinalizeWlanDriver( pcmdbuff );
            break;

        case WLAN_CHANGE_STATE_READY:
        case WLAN_CHANGE_STATE_INFRA:
        case WLAN_CHANGE_STATE_LOCAL_AP:
        case WLAN_CHANGE_STATE_LOCAL_STA:
        case WLAN_CHANGE_STATE_LOCAL_SPECTATOR:
        case WLAN_CHANGE_STATE_LOCAL_LCS_AP:
        case WLAN_CHANGE_STATE_LOCAL_LCS_STA:
        case WLAN_CHANGE_STATE_READY_FROM_LCS_AP:
        case WLAN_CHANGE_STATE_READY_FROM_LCS_STA:
        case WLAN_CHANGE_STATE_DETECT:
            post_fin_event = SetWlanMode( pcmdbuff );
            break;

        case WLAN_SET_IFUPDOWN:
            post_fin_event = SetWlanInterfaceUpDown( pcmdbuff );
            break;

        case WLAN_GET_IFUPDOWN:
            post_fin_event = GetWlanInterfaceUpDown( pcmdbuff );
            break;

        case WLAN_SET_EVENTMASK:
            post_fin_event = SetWlanEventMask( pcmdbuff );
            break;

        case WLAN_GET_EVENTMASK:
            post_fin_event = GetWlanEventMask( pcmdbuff );
            break;

        case WLAN_SET_APCONFIGRATION:
            post_fin_event = SetApConfigration( pcmdbuff );
            break;

        case WLAN_SCAN_REQUEST:
            post_fin_event = ScanRequest( pcmdbuff );
            break;

        case WLAN_SCANLIST_REQUEST:
            post_fin_event = BssInfoListGet( pcmdbuff );
            break;

        case WLAN_CANCEL_SCAN:
            post_fin_event = CancelScan( pcmdbuff );
            break;

        case WLAN_SET_MACADDRESS:
            post_fin_event = SetMacAddress( pcmdbuff );
            break;

        case WLAN_GET_MACADDRESS:
            post_fin_event = GetMacAddress( pcmdbuff );
            break;

        case WLAN_CREATE_AP:
            post_fin_event = CreateAp( pcmdbuff );
            break;

        case WLAN_DESTROY_AP:
            post_fin_event = DestroyAp( pcmdbuff );
            break;

        case WLAN_JOIN_NETWORK_STA:
            post_fin_event = JoinNetworkSta( pcmdbuff );
            break;

        case WLAN_JOIN_NETWORK_STA_WITH_WPS:
            post_fin_event = JoinNetworkStaWithWps( pcmdbuff );
            break;

        case WLAN_JOIN_NETWORK_SPECTATOR:
            post_fin_event = JoinNetworkSpectator( pcmdbuff );
            break;

        case WLAN_CANCEL_JOIN_NETWORK:
            post_fin_event = CancelJoinNetwork( pcmdbuff );
            break;

        case WLAN_DISASSOCIATE:
            post_fin_event = DisconnectRequest( pcmdbuff );
            break;

        case WLAN_DEAUTHENTICATE:
            post_fin_event = DeauthRequest( pcmdbuff );
            break;

        case WLAN_ADD_IE:
            post_fin_event = AddIe( pcmdbuff );
            break;

        case WLAN_DELETE_IE:
            post_fin_event = DeleteIe( pcmdbuff );
            break;

        case WLAN_SET_STATICAES_MODE:
            post_fin_event = SetStaticAesMode( pcmdbuff );
            break;

        case WLAN_ADD_MATCHING_DATA:
            post_fin_event = AddMatchingDataInfoToRxEntry( pcmdbuff );
            break;

        case WLAN_REMOVE_MATCHING_DATA:
            post_fin_event = RemoveMatchDataInfoToRxEntry( pcmdbuff );
            break;

        case WLAN_SET_BEACON_ACTION_FRAME:
            post_fin_event = SetActionFrameWithBeacon( pcmdbuff );
            break;

        case WLAN_CANCEL_BEACON_ACTION_FRAME:
            post_fin_event = CancelActionFrameWithBeacon( pcmdbuff );
            break;

        case WLAN_SET_CLIENT_TIMEOUT:
            post_fin_event = SetClientTimeout( pcmdbuff );
            break;

        case WLAN_SET_BEACONLOST_TIMEOUT:
            post_fin_event = SetBeaconLostTimeout( pcmdbuff );
            break;

        case WLAN_EXECUTE_ACSD:
            post_fin_event = ExecuteAcsd( pcmdbuff );
            break;

        case WLAN_GET_RSSI:
            post_fin_event = GetRssi( pcmdbuff );
            break;

        case WLAN_SET_MAX_ASSOC_NUM:
            post_fin_event = SetMaxAssociationNumber( pcmdbuff );
            break;

        case WLAN_GET_STATE:
            /*
             * コマンド順に処理してステート情報を取得できるように
             * コマンドでステートを取得できるようにする
             */
            post_fin_event = GetState( pcmdbuff );
            break;

        case WLAN_SET_TSF_TIMER_EVENT:
            post_fin_event = SetTsfTimerEventmask( pcmdbuff );
            break;

        case WLAN_SET_RX_CHAIN:
            post_fin_event = SetRxChain( pcmdbuff );
            break;

        case WLAN_GET_FW_VER:
            post_fin_event = GetFwVersion( pcmdbuff );
            break;

        case WLAN_REQUEST_SLEEP:
            post_fin_event = RequestSleep( pcmdbuff );
            break;

        case WLAN_REQUEST_WAKEUP:
            post_fin_event = RequestWakeUp( pcmdbuff );
            break;

        case WLAN_GET_STATE_BEFORE_SLEEP:
            post_fin_event = GetStateBeforeSleep( pcmdbuff );
            break;

        case WLAN_GET_WAKEUP_REASON:
            post_fin_event = GetWakeupReasonRaw( pcmdbuff );
            break;

        case WLAN_SET_WAKEUP_REASON:
            post_fin_event = SetWakeupReasonRaw( pcmdbuff );
            break;

        case WLAN_ENABLE_WOWL_FEATURES:
            post_fin_event = EnableWowlFeatures( pcmdbuff );
            break;

        case WLAN_GET_WOWL_WAKE_COUNT:
            post_fin_event = GetWowlWakeCount( pcmdbuff );
            break;

        case WLAN_GET_WOWL_SLEEP_STATS:
            post_fin_event = GetWowlSleepStats( pcmdbuff );
            break;

        case WLAN_CLEAR_WOWL_WAKE_COUNT:
            post_fin_event = ClearWowlWakeCount( pcmdbuff );
            break;

        case WLAN_CLEAR_WOWL_SLEEP_STATS:
            post_fin_event = ClearWowlSleepStats( pcmdbuff );
            break;

        case WLAN_SET_GTK_INFO:
            post_fin_event = SetGtkInfo( pcmdbuff );
            break;

        case WLAN_CLEAR_GTK_INFO:
            post_fin_event = ClearGtkInfo( pcmdbuff );
            break;

        case WLAN_EMULATE_DRIVER_INIT_FAIL:
            post_fin_event = EmulateDriverInitFail( pcmdbuff );
            break;

        case WLAN_GET_CHANNEL_STATS:
            post_fin_event = GetChannelStats( pcmdbuff );
            break;

        case WLAN_PUT_ONE_SHOT_ACTION_FRAME:
            post_fin_event = PutOneShotActionFrame( pcmdbuff );
            break;

        case WLAN_START_PERIODIC_ACTION_FRAME:
            post_fin_event = StartPeriodicActionFrame( pcmdbuff );
            break;

        case WLAN_CANCEL_PERIODIC_ACTION_FRAME:
            post_fin_event = CancelPeriodicActionFrame( pcmdbuff );
            break;

        case WLAN_REQUEST_SLEEP_FOR_ND:
            post_fin_event = RequestSleepForDetect( pcmdbuff );
            break;

        case WLAN_REQUEST_WAKEUP_FOR_ND:
            post_fin_event = RequestWakeUpForDetect( pcmdbuff );
            break;

        case WLAN_SET_DETECT_RECV_CONF:
            post_fin_event = SetDetectRecvConfig( pcmdbuff );
            break;

        case WLAN_CLEAR_DETECT_SA_TOTAL_RECV_CNT:
            post_fin_event = ClearDetectSaTotalRecvCnt( pcmdbuff );
            break;

        case WLAN_SET_MULTICAST_LIST:
            post_fin_event = SetMulticastList( pcmdbuff );
            break;

        default:
            WLAN_LOG_INFO("WDMS: Unknown Cmd = %ld\n", pcmdbuff->id);
            post_fin_event = true;
            break;
        }

        /* 即時に応答できないコマンドは、イベント側でマッチングする */
        if(post_fin_event == true )
        {
//          pcmdbuff->Result = result;
            nn::os::SignalEvent( &pcmdbuff->_finishEvent );
        }
        else
        {
            AppendWaitEventList( pcmdbuff );
        }

//        WLAN_LOG_INFO("WDMS: => StatusBit = %0lx\n", GetState());

        ret = true;
    }

    return ret;
}  //NOLINT(impl/function_size)

/* State Macineのメイン処理部 ---------------------------------------------- */
void StateMachine::ProcessStateMachine() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WLSF: State machine main process starts\n");
    bool ret;

    /* メッセージバッファの初期化 */
    m_exitStateMachine = false;

    InitMessageSystem();
    BssInfoListInitialize();
    InitializeSystemEvent();
    // TODO 上記の処理はコンストラクタに移動させる

    // 初期化が完了したので、通知する
    nn::os::SignalEvent(&m_StateMachineInitEvent);

    while( 1 )
    {
        /* メッセージがキューイングされると同時にセマフォがポストされる。
         * それにより WaitMessage() から処理が抜ける。
         * EventとCommandを同じセマフォで受信待ちしているが、コマンドは
         * 頻繁にこないので、処理するべきイベントがない場合のみ、コマンドを
         * 見に行く仕様としても、処理負荷が増えないのでこうしている。
         *
         * 1ループに処理するメッセージはイベント、コマンド合わせて1個のみ
         * イベント処理中に次のイベントがあった場合、イベントを優先して
         * 処理する設計思想なのでこうしている。
         */
//        WLAN_LOG_INFO("WDMS: Wait Cmd/Event\n");
        int signal = WaitMessage();
//        WLAN_LOG_INFO("WDMS: State Machine Loop\n");

        if(m_exitStateMachine == true)
        {
            break;
        }

        ret = EventProcess(signal);
        if(ret == true)
        {
            continue;
        }

        ret = CommandProcess();
        if(ret == true)
        {
            continue;
        }

        /* ここにはこないはず */
        WLAN_LOG_INFO("WDMS: State machine error(1) NazeKuru?\n");
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );
    }

    /* メッセージバッファシステムの破棄 */
    BssInfoListFinalize();
    FinalizeMessageSystem();
    FinalizeSystemEvent();
}


/* State Machine の初期化処理 ---------------------------------------------- */
void StateMachine::Initialize(int num_evn, int num_cmd,
                              nn::wlan::WlanBaseFunctions* pfuncs,
                              nn::wlan::WpaSupplicant* pSuppFuncs) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    m_EventMessageMax     = num_evn;
    m_CommandMessageMax   = num_cmd;
    m_MessageSemaphoreMax = (m_EventMessageMax + m_CommandMessageMax);

    m_pWlanFuncs = pfuncs;
    m_pSuppFuncs = pSuppFuncs;
    m_IsInit = true;

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

void StateMachine::Finalize() NN_NOEXCEPT
{
    m_pWlanFuncs = NULL;
    m_pSuppFuncs = NULL;
    m_IsInit = false;
}

void StateMachine::WaitInitialization() NN_NOEXCEPT
{
    nn::os::WaitEvent(&m_StateMachineInitEvent);
}

/* ステートマシーンへの、メッセージポスト ---------------------------------- */
bool StateMachine::PostEventMessage(WlanEvent* pevebuff) NN_NOEXCEPT
{
    bool ret = false;
//    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    NN_SDK_ASSERT( pevebuff != NULL );

//    WLAN_LOG_INFO("Append Message = %lp\n", pevebuff);

    PostMassage(&m_EventQueue, reinterpret_cast<uintptr_t>(pevebuff));
    nn::os::ReleaseSemaphore( &this->m_MessageSemaphore );

    return ret;
}

/* ステートマシーンへの、メッセージポスト ---------------------------------- */
bool StateMachine::PostCommandMessage(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    bool ret = false;
//    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    NN_SDK_ASSERT( pcmdbuff != NULL );

//    WLAN_LOG_INFO("Append Message = %lp\n", pcmdbuff);

    PostMassage(&m_CommandQueue, reinterpret_cast<uintptr_t>(pcmdbuff));
    nn::os::ReleaseSemaphore( &this->m_MessageSemaphore );

    if(pcmdbuff->EvnWait == WAIT_CMD_FINISH)
    {
        /* 終了イベント待ち(終了待ちしないものは、StateMachineスレッドで解放 */
        nn::os::WaitEvent( &pcmdbuff->_finishEvent );
    }

    return ret;
}

/* ステートマシーンへの、メッセージポスト ---------------------------------- */
bool StateMachine::PostEventSleepMessage(WlanEvent* pevebuff) NN_NOEXCEPT
{
    bool ret = false;
//    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    NN_SDK_ASSERT( pevebuff != NULL );

//    WLAN_LOG_INFO("Append Message = %lp\n", pevebuff);

    PostMassage(&m_EventSleepQueue, reinterpret_cast<uintptr_t>(pevebuff));
    m_EventMessageSleepCnt++;

    return ret;
}

/* イベントメッセージがスルーされていた場合にシグナルします */
void StateMachine::SignalEventMessage() NN_NOEXCEPT
{
    if( m_EventMessageSleepCnt > 0 )
    {
        WLAN_LOG_FORCE("Event message was posted during sleep state[%d].\n", m_EventMessageSleepCnt);
        nn::os::ReleaseSemaphore(&m_MessageSleepSemaphore, m_EventMessageSleepCnt);
        m_EventMessageSleepCnt = 0;
    }
}

/* BssInfo List周りの処理 -------------------------------------------------- */
void StateMachine::BssInfoListInitialize() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    m_BssInfoList = NULL;
    m_BssInfoListCnt = 0;
    nn::os::InitializeMutex(&m_BssInfoListMutex, false, 0);
}

void StateMachine::BssInfoListFinalize() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s\n", __FUNCTION__);

    BssInfoListClear();
    nn::os::FinalizeMutex( &m_BssInfoListMutex );
}

void StateMachine::BssInfoListClear() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    nn::os::LockMutex( &m_BssInfoListMutex );

    if(m_BssInfoList != NULL)
    {
        WlanBssInfo *pbssinfo = m_BssInfoList;
        WlanBssInfo *pbssinfo_next;

        do{
            pbssinfo_next = pbssinfo->NextItem;
            FreeScanBuf(pbssinfo);
            pbssinfo = pbssinfo_next;
        } while(pbssinfo_next != NULL);

        m_BssInfoList = NULL;
        m_BssInfoListCnt = 0;
    }

    // スキャン結果管理情報を初期化
    m_ScanResultInfo.minRssiValue = InvalidRssiValue;  // 0は意味を持ってしまうので、あり得ない値で初期化しておく
    m_ScanResultInfo.pMinRssiInfo = NULL;

    nn::os::UnlockMutex( &m_BssInfoListMutex );

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
}

/* BSS Info の登録 --------------------------------------------------------- */
void StateMachine::BssInfoListAdd(WlanBssInfo* pbssinfo_src) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("WDMS: %s enter\n", __FUNCTION__);

    WlanBssInfo* pbssinfo_dst;
    int ie_len;

    ie_len = pbssinfo_src->IeLength;

    // ScanBufに空きがなければ、スキャン結果の中で最小RSSI値のものとRSSI値を比較し、
    // 新規結果の方が大きければ最小RSSI値のbssinfoを解放し、バッファに空きを作る。
    uint8_t* buffer = NULL;
    while( 1 )
    {
        buffer = reinterpret_cast<uint8_t*>(MallocScanBuf(sizeof(WlanBssInfo) + ie_len));
        if( buffer != NULL )
        {
            break;
        }
        else
        {
            WLAN_LOG_INFO("Scan buffer starves. Try to exchange the minimum RSSI info.\n");
            if( pbssinfo_src->Rssi > m_ScanResultInfo.minRssiValue && m_ScanResultInfo.minRssiValue != InvalidRssiValue )
            {
                nn::os::LockMutex( &m_BssInfoListMutex );

                NN_ABORT_UNLESS_NOT_NULL(m_ScanResultInfo.pMinRssiInfo);
                if( m_BssInfoListCnt == 1 )
                {
                    // 最後の1個が解放対象となっていたことを示す。スキャンバッファのリークが疑われる
                    NN_SDK_ASSERT(false, "Maybe scan buffer starves.\n");

                    // とりあえず入れ替え作業は止めて抜ける
                    nn::os::UnlockMutex( &m_BssInfoListMutex );
                    return;
                }

                // 最小RSSIノードの解放前に、ノードの付け替え
                if( m_ScanResultInfo.pMinRssiInfo->PrevItem != NULL )
                {
                    m_ScanResultInfo.pMinRssiInfo->PrevItem->NextItem = m_ScanResultInfo.pMinRssiInfo->NextItem;
                    if( m_ScanResultInfo.pMinRssiInfo->NextItem != NULL )
                    {
                        m_ScanResultInfo.pMinRssiInfo->NextItem->PrevItem = m_ScanResultInfo.pMinRssiInfo->PrevItem;
                    }
                }
                else
                {
                    // 先頭ノードの解放
                    m_BssInfoList = m_ScanResultInfo.pMinRssiInfo->NextItem;
                    m_BssInfoList->PrevItem = NULL;
                }
                // 最小RSSI値を格納していたWlanBssInfoを解放
                FreeScanBuf(m_ScanResultInfo.pMinRssiInfo);
                m_BssInfoListCnt--;

                // 最小RSSI値の更新
                m_ScanResultInfo.minRssiValue = 0;
                for( WlanBssInfo* pNode = m_BssInfoList; pNode != NULL; pNode = pNode->NextItem )
                {
                    if( m_ScanResultInfo.minRssiValue >= pNode->Rssi )
                    {
                        m_ScanResultInfo.minRssiValue = pNode->Rssi;
                        m_ScanResultInfo.pMinRssiInfo = pNode;
                    }
                }

                nn::os::UnlockMutex( &m_BssInfoListMutex );
            }
            else
            {
                // 最小RSSI値より小さいRSSI値だったので無視する
                WLAN_LOG_INFO("Ignore bss info because scan buffer starves and RSSI value[%d] is less than the smallest one[%d] in the scanresults\n",
                        pbssinfo_src->Rssi, m_ScanResultInfo.minRssiValue);
                return;
            }
        }
    }

    pbssinfo_dst = reinterpret_cast<WlanBssInfo*>(buffer);

    if(pbssinfo_dst != NULL)
    {
        nn::os::LockMutex( &m_BssInfoListMutex );

        /* pbssinfo_src は戻った先で解放される */
        std::memcpy(pbssinfo_dst, pbssinfo_src, sizeof(WlanBssInfo));

        /* IE copy */
        if(ie_len > 0)
        {
            pbssinfo_dst->IeData = reinterpret_cast<uint8_t*>(pbssinfo_dst) + sizeof(WlanBssInfo);
            std::memcpy(pbssinfo_dst->IeData, pbssinfo_src->IeData, ie_len);
        }

        if(m_BssInfoList == NULL)
        {
            /* 最初の1個 */
            m_BssInfoList = pbssinfo_dst;
            m_BssInfoList->PrevItem = NULL;
            m_BssInfoList->NextItem = NULL;
            m_BssInfoListCnt = 1;
            m_ScanResultInfo.minRssiValue = pbssinfo_dst->Rssi;
            m_ScanResultInfo.pMinRssiInfo = pbssinfo_dst;

            WLAN_LOG_DEBUG("----->1: pbssinfo_dst=%p, next=%p\n", pbssinfo_dst, pbssinfo_dst->NextItem);
        }
        else
        {
            /* 2個目以降 */
            WlanBssInfo* pbsslist = m_BssInfoList;
            while( 1 ){
                WLAN_LOG_DEBUG("----->2: pbssinfo_dst=%p, next=%p\n", pbsslist, pbsslist->NextItem);

                /* BSSID を使い重複確認 */
                int bssid_chk_result;
                bssid_chk_result = std::memcmp(pbsslist->Bssid,
                                               pbssinfo_dst->Bssid,
                                               MacAddress::MacAddressSize);
                if(bssid_chk_result == 0)
                {
                    if (pbsslist->Rssi < pbssinfo_dst->Rssi)
                    {
                        // 重複ノードと入れ替えで追加
                        pbssinfo_dst->NextItem = pbsslist->NextItem;
                        pbssinfo_dst->PrevItem = pbsslist->PrevItem;
                        if (pbsslist->NextItem != NULL)
                        {
                            pbsslist->NextItem->PrevItem = pbssinfo_dst;
                        }
                        if (pbsslist->PrevItem == NULL)
                        {
                            /* リストの先頭は追加で調整 */
                            m_BssInfoList = pbssinfo_dst;
                        }
                        else
                        {
                            pbsslist->PrevItem->NextItem = pbssinfo_dst;
                        }

                        // 解放するノードが最小RSSIのbssだった場合、m_ScanResultInfoの更新が必要
                        if( m_ScanResultInfo.pMinRssiInfo == pbsslist )
                        {
                            // 最小RSSI値の更新
                            m_ScanResultInfo.minRssiValue = 0;
                            for( WlanBssInfo* pNode = m_BssInfoList; pNode != NULL; pNode = pNode->NextItem )
                            {
                                if( m_ScanResultInfo.minRssiValue >= pNode->Rssi )
                                {
                                    m_ScanResultInfo.minRssiValue = pNode->Rssi;
                                    m_ScanResultInfo.pMinRssiInfo = pNode;
                                }
                            }
                        }
                        FreeScanBuf(pbsslist);
                    }
                    else
                    {
                        // リストの BSS より RSSI 値が小さいので破棄
                        FreeScanBuf(pbssinfo_dst);
                        pbssinfo_dst = NULL;
                    }
                    break;
                }

                if(pbsslist->NextItem == NULL)
                {
                    // 末尾に追加
                    pbsslist->NextItem = pbssinfo_dst;
                    pbssinfo_dst->PrevItem = pbsslist;
                    pbssinfo_dst->NextItem = NULL;
                    m_BssInfoListCnt ++;
                    break;
                }

                pbsslist = pbsslist->NextItem;
            }
        }

        // 最小RSSI値の更新があるか調べる
        if( pbssinfo_dst != NULL && pbssinfo_dst->Rssi < m_ScanResultInfo.minRssiValue )
        {
            m_ScanResultInfo.minRssiValue = pbssinfo_dst->Rssi;
            m_ScanResultInfo.pMinRssiInfo = pbssinfo_dst;
        }

        nn::os::UnlockMutex( &m_BssInfoListMutex );
    }

    WLAN_LOG_DEBUG("WDMS: %s leave\n", __FUNCTION__);
}  //NOLINT(impl/function_size)

bool StateMachine::BssInfoListGet(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);

    // RSSI順にソートしたものを渡す
    if( m_BssInfoListCnt > 1)
    {
        WlanBssInfo* pInsert = NULL;
        WlanBssInfo* pFind = NULL;
        WlanBssInfo* pInsertNext = NULL;

        for( pInsert = m_BssInfoList->NextItem; pInsert != NULL; pInsert = pInsertNext )
        {
            pInsertNext = pInsert->NextItem;

            for( pFind = m_BssInfoList; pFind != pInsert; pFind = pFind->NextItem )
            {
                if( pInsert->Rssi > pFind->Rssi )
                {
                    // swap
                    // pInsertが最後尾かどうか
                    if( pInsert->NextItem != NULL )
                    {
                        pInsert->NextItem->PrevItem = pInsert->PrevItem;
                    }
                    pInsert->PrevItem->NextItem = pInsert->NextItem;

                    pInsert->PrevItem = pFind->PrevItem;
                    pInsert->NextItem = pFind;

                    // pFindが先頭要素かどうか
                    if( pFind->PrevItem == NULL )
                    {
                        // 先頭要素の入れ替え
                        m_BssInfoList = pInsert;
                    }
                    else
                    {
                        pFind->PrevItem->NextItem = pInsert;
                    }
                    pFind->PrevItem = pInsert;

                    break;
                }
            }
        }
    }

    WlanBssInfo** pbsslist = static_cast<WlanBssInfo**>(pcmdbuff->Args);
    *pbsslist = m_BssInfoList;
    pcmdbuff->Result = true;

    WLAN_LOG_INFO("WDMS: %s leave\n", __FUNCTION__);
    return true;
}

void StateMachine::UpdateConnectionStatus(const ConnectionStatus& status) NN_NOEXCEPT
{
    nn::os::LockMutex( &m_ConnectionStatusMutex );

    m_ConnectionStatus.aid = status.aid;
    m_ConnectionStatus.state = status.state;
    m_ConnectionStatus.channel = status.channel;
    m_ConnectionStatus.bssid.Set(status.bssid.GetMacAddressData());
    m_ConnectionStatus.cause = status.cause;
    m_ConnectionStatus.ssid.Set(status.ssid.GetSsidData(), status.ssid.GetLength());
    m_ConnectionStatus.statusReasonCode = status.statusReasonCode;
    m_ConnectionStatus.beaconInterval = status.beaconInterval;
    m_ConnectionStatus.capabilityInfo = status.capabilityInfo;

    nn::os::UnlockMutex( &m_ConnectionStatusMutex );
}

void StateMachine::ResetSystemEventsAndConnectionStatus() NN_NOEXCEPT
{
    // Clear system events
    nn::os::ClearSystemEvent(&m_ConnectionStatusEvent);
    nn::os::ClearSystemEvent(&m_ConnectCompletionEvent);
    nn::os::ClearSystemEvent(&m_ScanCompletionEvent);

    // Initialize connection status
    nn::os::LockMutex( &m_ConnectionStatusMutex );

    m_ConnectionStatus.state = ConnectionState_Idle;
    m_ConnectionStatus.cause = CauseOfInfo_NoInfo;
    m_ConnectionStatus.channel = 0;
    m_ConnectionStatus.ssid.Set("");
    m_ConnectionStatus.bssid.SetDirect(0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
    m_ConnectionStatus.aid = 0;
    m_ConnectionStatus.statusReasonCode = Dot11ReasonCode_Reserved;
    m_ConnectionStatus.capabilityInfo = 0;
    m_ConnectionStatus.beaconInterval = 0;

    nn::os::UnlockMutex( &m_ConnectionStatusMutex );

    m_Authorized = false;
    m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
    m_pSuppFuncs->SetSleepFlag(false);
#endif
}

bool StateMachine::RequestSleepForDetect(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);
    bool ret = true;

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_OutOfService:
        WLAN_LOG_INFO("Start: SLEEP PROCESSING for normal sleep\n");
        // RxEntryを無効化しておく
        InactivateRxBuffer();
        InactivateRxActionFrame();

        ResetSystemEventsAndConnectionStatus();

        // スリープ直前のステートを保存
        m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
        m_WdrvStateBeforeSleep.SetSub( WdrvSubState_None );

        // 現在のステートをスリープステートへ
        m_WdrvState.SetMain( WdrvMainState_OutOfService );
        m_WdrvState.SetSub( WdrvSubState_None );
        m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
        m_pSuppFuncs->SetSleepFlag(true);
#endif

        // 無線ドライバーの終了処理
        if( m_IsDrvInitFail == false && nn::wlan::FirmwareDebugSettings::IsSkipBoot() == false )
        {
            m_pWlanFuncs->FinalizeDriver();
        }

        // 起床要因のリセット
        m_DetectWakeReason = 0;
        // スリープ復帰のシーケンスは回しておく必要があるので成功を返しておく
        pcmdbuff->nnResult = ResultSuccess();
        m_sleepMode = SleepMode_Normal;
        break;
    case WdrvMainState_Ready:
    case WdrvMainState_DetectIdle:
        // 通常スリープ
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
            WLAN_LOG_INFO("Start: SLEEP PROCESSING for normal sleep\n");
            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();

            // スリープ直前のステートを保存
            m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
            m_WdrvStateBeforeSleep.SetSub( m_WdrvState.GetSub() );

            // 現在のステートをスリープステートへ
            m_WdrvState.SetMain( WdrvMainState_Sleep );
            m_WdrvState.SetSub( WdrvSubState_None );
            m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(true);
#endif
            // 念のためI/Fはdownしておく
            m_pWlanFuncs->SetIfUpDown(0);

            // 無線ドライバーの終了処理
            m_pWlanFuncs->ClearEventMask();
            m_pWlanFuncs->FinalizeDriver();
            // 起床要因のリセット
            m_DetectWakeReason = 0;
            pcmdbuff->nnResult = ResultSuccess();
            m_sleepMode = SleepMode_Normal;
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    case WdrvMainState_Detect:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            WLAN_LOG_INFO("Start: SLEEP PROCESSING for Detect\n");
            // すれちがい予約済みであることは保証されている

            // RxEntryを無効化しておく
            InactivateRxBuffer();
            InactivateRxActionFrame();
            // パケット送信を制限するためのフラグ
            m_IsSleepState = true;
#if defined(USE_WPA_SUPPLICANT)
            m_pSuppFuncs->SetSleepFlag(true);
#endif

            // すれちがい用のパラメータ設定
            m_pWlanFuncs->SetupDetectSleepParams(&m_DetectSetupParams);
            // すれちがいスリープモード開始
            m_pWlanFuncs->ActivateDetectSleep();
            WLAN_LOG_INFO("[SM]Activate Neighbor Detect SA mode.\n");
            // 無線チップをWoWL用の省電力モードへ。
            m_pWlanFuncs->SetDevicePowerState(false);

            // 起床要因のリセット
            m_DetectWakeReason = 0;
            // スリープ直前のステートを保存
            m_WdrvStateBeforeSleep.SetMain( m_WdrvState.GetMain() );
            m_WdrvStateBeforeSleep.SetSub( m_WdrvState.GetSub() );
            // 現在のステートをスリープステートへ
            m_WdrvState.SetMain( WdrvMainState_Sleep );
            m_WdrvState.SetSub( WdrvSubState_None );
            pcmdbuff->nnResult = ResultSuccess();
            m_sleepMode = SleepMode_Detect;
            WLAN_LOG_INFO("[SUCCESS]Entered Neighbor Detect SA mode.\n");
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    return ret;
}  //NOLINT(impl/function_size)

bool StateMachine::RequestWakeUpForDetect(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Sleep:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_None:
            m_DetectWakeReason = 0;
            // スリープからの復帰処理可能
            // すれちがいスリープからの復帰かチェック
            if( m_WdrvStateBeforeSleep.GetMain() == WdrvMainState_Detect &&
                    m_WdrvStateBeforeSleep.GetSub() == WdrvSubState_Idle )
            {
                // すれちがいスリープからの復帰
                WLAN_LOG_INFO("Wake up from Neighbor Detection sleep.\n");

                // RxEntryを有効化しておく
                ActivateRxBuffer();
                ActivateRxActionFrame();

                // 無線チップをWoWLの省電力モードから復帰させる
                m_pWlanFuncs->SetDevicePowerState(true);

                // 起床要因を取得
                m_pWlanFuncs->GetWakeupReasonRawDetectSleep(&m_DetectWakeReason, &m_DetectWakeCount);

                // TODO スタッツの取得
                // m_pWlanFuncs->GetWowlSleepStats(&m_WowlSleepStats);

                // すれちがいスリープの解除。FWからイベントやデータパケットが送られ始める。
                m_pWlanFuncs->DisableDetectSleep();
                // WoWL用パラメータクリア
                m_pWlanFuncs->ClearDetectSleepParams();
                // FWに溜まっているActionFrameを吸い上げる
                m_pWlanFuncs->PullActionFrameFromDongle();

                // ステートをスリープ前に戻しておく
                m_WdrvState.SetMain( m_WdrvStateBeforeSleep.GetMain() );
                m_WdrvState.SetSub( m_WdrvStateBeforeSleep.GetSub() );

                m_IsSleepState = false;
#if defined(USE_WPA_SUPPLICANT)
                m_pSuppFuncs->SetSleepFlag(false);
#endif
                pcmdbuff->nnResult = ResultSuccess();
            }
            else
            {
                // 通常スリープ復帰
                WLAN_LOG_INFO("Wake up from normal sleep.\n");
                pcmdbuff->nnResult = ResultWakeupNormalSleep();
            }

#if !defined(ENABLE_DETECT_AGING_DEBUG)
            // スリープからの復帰時点ですれちがい用のデータはクリアしておく
            m_IsReservedDetectSleep = false;
            ClearDetectData();
#endif
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
    return true;
}

void StateMachine::ClearDetectData() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    // sleepParamを退避
    DetectPeriodicAfCycle tmp;
    memcpy(&tmp, &m_DetectSetupParams.sleepParam, sizeof(DetectPeriodicAfCycle));
    if( m_DetectSetupParams.hashList.pHashList != NULL )
    {
        delete m_DetectSetupParams.hashList.pHashList;
    }
    memset(&m_DetectSetupParams, 0, sizeof(WlanDetectSetupParams));

    // sleepParamを書き直し
    memcpy(&m_DetectSetupParams.sleepParam, &tmp, sizeof(DetectPeriodicAfCycle));
}

void StateMachine::InitializeDetectData() NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    if( m_DetectSetupParams.hashList.pHashList != NULL )
    {
        delete m_DetectSetupParams.hashList.pHashList;
    }
    memset(&m_DetectSetupParams, 0, sizeof(WlanDetectSetupParams));
    m_DetectSetupParams.sleepParam.txInterval = DetectSleepTxInterval;
    m_DetectSetupParams.sleepParam.txCount = DetectSleepTxCount;
    m_DetectSetupParams.sleepParam.idleCount = DetectSleepIdleCount;
    m_DetectSetupParams.sleepParam.rxStartCount = DetectSleepRxStartCount;
    m_DetectSetupParams.sleepParam.rxCount = DetectSleepRxCount;
    m_DetectSetupParams.cacheCntMax = DetectCacheCnt;

    m_DetectHdComuPat.txInterval = 100;
    m_DetectHdComuPat.txCount = 1;
    m_DetectHdComuPat.idleCount = 0;
    m_DetectHdComuPat.rxStartCount = 0;
    m_DetectHdComuPat.rxCount = 1;
}

void StateMachine::SetActionFrameForDetectSleep(uint8_t subtype, DetectHash hash, const char* pData, size_t size) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    if( size > DetectActionFramePayloadMax )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return;
    }

    NintendoActionFrameHeader ninAfHeader = {
            0x7f,
            { 0x00, 0x22, 0xAA },
            subtype,
            0
    };
    DetectHeader ndHeader = {
            DhpMajor,
            DhpMiner,
            DhpCmdPeriodic,
            0,
            hash
    };
    memcpy(&m_DetectSetupParams.actionFrame[0], &ninAfHeader, sizeof(NintendoActionFrameHeader));
    memcpy(&m_DetectSetupParams.actionFrame[sizeof(NintendoActionFrameHeader)], &ndHeader, sizeof(DetectHeader));
    memcpy(&m_DetectSetupParams.actionFrame[sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader)], pData, size);
    m_DetectSetupParams.length = static_cast<uint16_t>(size + sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader));
}

nn::Result StateMachine::SetHashListForDetectSleep(const char* pHashList, size_t size) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    if( pHashList == NULL )
    {
        NN_SDK_REQUIRES(false, "%s: pHashList must not be null\n", __FUNCTION__);
        return ResultInvalidArgument();
    }
    if( size > DetectSleepFilterMax * sizeof(uint64_t) || size == 0 )
    {
        NN_SDK_REQUIRES(false, "%s: size is out of range\n", __FUNCTION__);
        return ResultInvalidArgument();
    }
    if( m_DetectSetupParams.hashList.pHashList != NULL )
    {
        delete m_DetectSetupParams.hashList.pHashList;
    }
    m_DetectSetupParams.hashList.pHashList = new uint64_t[size / sizeof(uint64_t)];
    if( m_DetectSetupParams.hashList.pHashList == NULL )
    {
        return ResultBufferTooShort();  // TODO メモリ不足エラーを定義する
    }
    memcpy(reinterpret_cast<char*>(m_DetectSetupParams.hashList.pHashList), pHashList, size);
    m_DetectSetupParams.hashList.numOfHash = size / sizeof(uint64_t);

    return ResultSuccess();
}

void StateMachine::SetPeriodicActionFrameCycle(const DetectPeriodicAfCycle& param, DetectPeriodicAfCycleTarget target) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    if( target == DetectPeriodicAfCycleTarget_Both )
    {
        memcpy(&m_DetectSetupParams.sleepParam, &param, sizeof(DetectPeriodicAfCycle));
        memcpy(&m_DetectHdComuPat, &param, sizeof(DetectPeriodicAfCycle));
    }
    else if( target == DetectPeriodicAfCycleTarget_Hd )
    {
        memcpy(&m_DetectHdComuPat, &param, sizeof(DetectPeriodicAfCycle));
    }
    else if( target == DetectPeriodicAfCycleTarget_Sa )
    {
        memcpy(&m_DetectSetupParams.sleepParam, &param, sizeof(DetectPeriodicAfCycle));
    }
}

bool StateMachine::SetDetectRecvConfig(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    pcmdbuff->nnResult = ResultSuccess();

    ActionFrameRecvMode* pMode;
    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Detect:
        switch( m_WdrvState.GetSub() )
        {
        case WdrvSubState_Idle:
            pMode = reinterpret_cast<ActionFrameRecvMode*>(pcmdbuff->Args);
            NN_SDK_ASSERT_NOT_NULL(pMode);
            if( *pMode == ActionFrameRecvMode_DetectWake )
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
                m_pWlanFuncs->DisableFixedDesense();
                m_pWlanFuncs->SetAciMitigation(AciMitigation_Detect);
                m_pWlanFuncs->SetActionFrameRecvMode(ActionFrameRecvMode_DetectWake);
            }
            else if( *pMode == ActionFrameRecvMode_DetectSleep )
            {
                m_pWlanFuncs->SetSimultaneousTx(SimultaneousTxParam_2Antenna);
                m_pWlanFuncs->DisableFixedDesense();
                m_pWlanFuncs->SetAciMitigation(AciMitigation_Detect);
                m_pWlanFuncs->SetActionFrameRecvMode(ActionFrameRecvMode_DetectSleep);
            }
            else
            {
                m_pWlanFuncs->SetAciMitigation(AciMitigation_Default);
                m_pWlanFuncs->SetActionFrameRecvMode(ActionFrameRecvMode_Normal);
            }
            break;
        default:
            pcmdbuff->nnResult = ResultInvalidState();
            WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                    __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
            break;
        }
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
    return true;
}

bool StateMachine::ClearDetectSaTotalRecvCnt(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    pcmdbuff->nnResult = ResultSuccess();

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_OutOfService:
        pcmdbuff->nnResult = ResultWlanDeviceAbnormal();
        break;
    case WdrvMainState_DetectIdle:
    case WdrvMainState_Detect:
        m_pWlanFuncs->ClearDetectSaTotalRecvCnt();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }
    return true;
}

bool StateMachine::SetMulticastList(WlanCommand* pcmdbuff) NN_NOEXCEPT
{
    WLAN_LOG_INFO("WDMS: %s enter\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pcmdbuff);

    WlanAllowedMulticastList* pList = reinterpret_cast<WlanAllowedMulticastList*>(pcmdbuff->Args);
    NN_SDK_REQUIRES_NOT_NULL(pList);

    switch( m_WdrvState.GetMain() )
    {
    case WdrvMainState_Ready:
    case WdrvMainState_InfraIdle:
    case WdrvMainState_InfraSta:
        m_pWlanFuncs->SetMulticastList(pList);
        pcmdbuff->nnResult = ResultSuccess();
        break;
    default:
        pcmdbuff->nnResult = ResultInvalidState();
        WLAN_LOG_ERROR("%s called in an invalid state[%d, %d]",
                __FUNCTION__, m_WdrvState.GetMain(), m_WdrvState.GetSub());
        break;
    }

    return true;
}

/* WDM State Machine Thread ------------------------------------------------ */
void StateMachineThreadFunc(void *pArg) NN_NOEXCEPT
{
    nn::wlan::StateMachine* pST = static_cast<nn::wlan::StateMachine*>(pArg);

    WLAN_LOG_INFO("WDM: %s start\n", __FUNCTION__);

    pST->ProcessStateMachine();

    WLAN_LOG_INFO("WDM: %s end\n", __FUNCTION__);
}

/* name space */
}}
