﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/util/util_FormatString.h>
#include <nn/htcs.h>
#include <nn/htcs/htcs_LibraryPrivate.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "../wlan_MemoryInit.h"
#include "wlan_Htcs.h"
#include "wlan_DbgConsole.h"

nn::os::ThreadType  g_HtcsThread;
const size_t ThreadStackSize = 16 * 1024;              // Thread stack size
NN_OS_ALIGNAS_THREAD_STACK uint8_t g_HtcsThreadStack[ ThreadStackSize ];

nn::diag::LogObserverHolder g_LogObserverHolder;

namespace nn { namespace wlan { namespace dbg{
const size_t wlanHtcsTmpBuffSize = 512;         // printfの一時展開用
wlanDbgConsole *g_whtcs;


/* Alloc/Free functions */
void* Allocate(size_t size) NN_NOEXCEPT
{
    return nnwlanMallocNormal(size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nnwlanFreeNormal(p);
}

// \nは、LFのみなのでLF+CRに置き換える
static int ConvertLfToCrlf(char *dst, const char *src)
{
    int dst_cnt = 0;
    int n;

    for(n = 0; n < (wlanHtcsTmpBuffSize / 2); n++)
    {
        dst[dst_cnt] = src[n];
        dst_cnt ++;
        if(src[n] == '\n')
        {
            dst[dst_cnt] = '\r';
            dst_cnt ++;
        }
        if(src[n] == '\0')
        {
            break;
        }
    }

    return dst_cnt;
}

/* Constructor */
wlanDbgConsole::wlanDbgConsole()
{
    nn::diag::InitializeLogObserverHolder(&g_LogObserverHolder, LogObserver, this);
    nn::diag::RegisterLogObserver(&g_LogObserverHolder);
}

wlanDbgConsole::~wlanDbgConsole()
{
    nn::diag::UnregisterLogObserver(&g_LogObserverHolder);
}

extern "C" int wlu_process_cmdline_iput(char *line, int length);
/* Input callback (strings. max. 512chars(include CR/LF/Terminate) */
void wlanDbgConsole::InputCallBack(char* buff, int length) NN_NOEXCEPT
{
    // Modify here
    buff[length] = '\0';
//  NN_SDK_LOG("%s\n", "<<wlanDbgConsole says hello>>");
//  NN_SDK_LOG("%s\n", __FUNCTION__);
//  NN_SDK_LOG("%s\n", buff);
    wlu_process_cmdline_iput(buff, length);
}

/* Terminate Callback */
void wlanDbgConsole::TerminateCallback() NN_NOEXCEPT
{
    NN_SDK_LOG("%s\n", __FUNCTION__);
}

/* LogObserver for put message from wl commands to HtcsConsole */

void wlanDbgConsole::LogObserver(const nn::diag::LogMetaData& metaInfo, const nn::diag::LogBody& logBody, void* argument)
{

    if (!*metaInfo.sourceInfo.fileName && !*metaInfo.sourceInfo.functionName && !*metaInfo.moduleName)
    {
        static_cast<wlanDbgConsole*>(argument)->SendStrings(logBody.message, logBody.messageBytes);
    }
}

/* Console Daemon thread */
void HtcsThread(void* arg) NN_NOEXCEPT
{
    NN_SDK_LOG("\n\nHTCS thread start\n\n");
    while(1)
    {
        nn::htcs::Initialize(Allocate, Deallocate, 1);
        {
            wlanDbgConsole whtcs;
            g_whtcs = &whtcs;

            whtcs.WlanHtcsInit();
            whtcs.WlanHtcsMain();
        }
        nn::htcs::Finalize();
    }
}

/* Debug console initialized */
void wlan_DbgConsoleInit() NN_NOEXCEPT
{
    nn::Result result;

    NN_SDK_LOG("\nWlanConsole App(Create thread)\n");

    /* Create Console thread and run it */
    result = nn::os::CreateThread(&g_HtcsThread,
                                  HtcsThread,
                                  NULL,
                                  g_HtcsThreadStack,
                                  ThreadStackSize,
                                  NN_SYSTEM_THREAD_PRIORITY(wlan, HtcsConsole) );
    NN_SDK_ASSERT( result.IsSuccess(), "Cannot create g_HtcsThread." );
    nn::os::SetThreadNamePointer(&g_HtcsThread, NN_SYSTEM_THREAD_NAME(wlan, HtcsConsole));
    nn::os::StartThread( &g_HtcsThread );
}

void wlan_HtcsPreprocessSendStrings(const char *fmt, va_list args)
{
    int len;
    char *src;
    char *dst;
    static const char *AllocFailMsg = "Alloc Fail\r\n";

    src = static_cast<char*>(Allocate(wlanHtcsTmpBuffSize));
    dst = static_cast<char*>(Allocate(wlanHtcsTmpBuffSize));

    if((src != NULL) && (dst != NULL))
    {
        memset(src, 0x00, wlanHtcsTmpBuffSize);
        // 全て\nで来た時の対策で半分までしか展開しない
        nn::util::VSNPrintf(src, (wlanHtcsTmpBuffSize / 2), fmt, args);

        len = ConvertLfToCrlf(dst, src);
        g_whtcs->SendStrings(dst, len);
    }
    else
    {
        len = strlen(AllocFailMsg);
        g_whtcs->SendStrings(AllocFailMsg, len);
    }

    if(src != NULL)
    {
        Deallocate(src, wlanHtcsTmpBuffSize);
    }
    if(dst != NULL)
    {
        Deallocate(dst, wlanHtcsTmpBuffSize);
    }
}

}}}

extern "C" void wlan_HtcsCPrintf(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    nn::wlan::dbg::wlan_HtcsPreprocessSendStrings(fmt, ap);
    va_end(ap);
}
