﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "server/ssl_MemoryManager.h"
#include "server/ssl_NssCore.h"
#include "server/ssl_SslServiceManager.h"
#include "server/ssl_NssConfigFsManager.h"
#include "server/ssl_Crl.h"
#include "server/ssl_EvCertUtilManager.h"
#include "debug/ssl_DebugUtil.h"

using namespace nn::ssl;
using namespace nn::ssl::detail;


namespace
{
    std::aligned_storage<BuildParameter_MemoryManagerHeapSizeHorizonProcess>::type g_MemoryManagerHeapBuf;
    SslServiceManager                                                              g_srvMgr;
    uintptr_t                                                                      g_HeapAddr = 0;
    nn::socket::SystemConfigLightDefaultWithMemory
    <
        1, // only enough for one TCP socket, since SSL uses shared sockets
        0  // no UDP sockets
    > g_SocketConfigWithMemory;
}

// START ------------------------------------------------------------------------------------------
// Overriding operators to avoid exception related code being linked to reduce
// the code size.
void* operator new(std::size_t size)
{
    return malloc(size);
}

void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete(void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

void* operator new[](std::size_t size)
{
    return malloc(size);
}

void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void  operator delete[](void* ptr) NN_NOEXCEPT
{
    free(ptr);
}
// END Overriding operators------------------------------------------------------------------------

extern "C"
void nninitStartup()
{
    NN_SDK_LOG("[ssl] nninitStartup at %p\n", nninitStartup);

    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 4 * nn::os::MemoryHeapUnitSize;
    auto result = nn::os::SetMemoryHeapSize(MemoryHeapSize);

    result = nn::os::AllocateMemoryBlock(&g_HeapAddr, MemoryHeapSize);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[ssl] FAILED to alloc memory for heap\n");
        NN_ABORT("[ssl] FAILED to alloc memory for heap\n");
    }

    NN_SDK_LOG("[ssl] init heap with %u bytes\n", MemoryHeapSize);

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator(reinterpret_cast<void *>(g_HeapAddr), MemoryHeapSize);
}

extern "C"
void nndiagStartup()
{
}

extern "C"
void nnMain()
{
    nn::Result                  result;
    nn::os::SystemEvent         shutdownEvent(EventClearMode_AutoClear, true);

    NN_DETAIL_SSL_DBG_UTIL_INITIALIZE();

    do
    {
        //  Set our main thread name
        nn::os::ThreadType  *mainThread;
        mainThread = nn::os::GetCurrentThread();
        nn::os::SetThreadNamePointer(mainThread, NN_SYSTEM_THREAD_NAME(ssl, Main));

        //  Init the socket library
        NN_SDK_LOG("[ssl] init socket...\n");
        // Socket concurrency does not have to be higher than number of SSL server threads
        g_SocketConfigWithMemory.SetConcurrencyCountMax( g_srvMgr.GetSessionThreadCount() );
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ssl] failed to init socket lib\n");
            break;
        }

        //  Init memory management
        NN_SDK_LOG("[ssl] init mem mgmt...\n");
        SslMemoryManager::Initialize(
            reinterpret_cast<char*>(&g_MemoryManagerHeapBuf),
            BuildParameter_MemoryManagerHeapSizeHorizonProcess);

        //  Mount the config filesystem area
        result = NssConfigFsManager::Initialize();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to mount file system\n");
            break;
        }

        //  Startup the NSS core used to power SSL
        NN_SDK_LOG("[ssl] init SSL core...\n");
        result = NssCore::Initialize();
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ssl] failed to init NSS\n");
            break;
        }

        //  Initialize CRL manager which loads built-in CRL too
        result = CrlManager::Initialize();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to initialize CRL manager\n");
            break;
        }

        //  Initialize built-in policy OIDs
        result = EvCertUtilManager::Initialize();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to initialize built-in policy IDs\n");
        }

        //  Init the SSL service manager which listens for incoming clients
        NN_SDK_LOG("[ssl] init service manager...\n");
        result = g_srvMgr.Initialize();
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ssl] failed to init SSL service manager\n");
            break;
        }

        //  Now that all the immediate init is done, tell NssCore to
        //  kick off any deferred init
        NssCore::StartDeferredInit();

        //  Loop and wait for exit
        NN_SDK_LOG("[ssl] running...\n");
        shutdownEvent.Wait();
        NN_SDK_LOG("[ssl] shutting down\n");

        //  Cleanup the service manager, memory management and NSS
        result = g_srvMgr.Finalize();
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ssl] failed to shutdown SSL service manager\n");

            //  pass thru, finish cleaning up
        }

        EvCertUtilManager::Finalize();

        if (CrlManager::Finalize().IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to finalize CrlManager\n");
        }

        SslMemoryManager::Finalize();

        result = NssCore::Finalize();
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ssl] failed to shutdown SSL core\n");

            //  pass thur, finish cleaning up
        }

        result = NssConfigFsManager::Finalize();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to unmount host fs\n");
        }

        NN_SDK_LOG("[ssl] exiting\n");
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_UTIL_FINALIZE();
}

