﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <cstring>
#include <cctype>
#include <nn/htcs.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include "shell_Console.h"
#include "shell_TelnetConsole.h"
#include "shell_CommandExecutorSet.h"
#include "shell_Shell.h"
#include "shell_TelnetServer.h"

namespace nn { namespace shell {
    namespace
    {
        const int MaxConnectionCount    = 4;

        const char* g_PortName = "@Shell";

        nn::os::ThreadType g_ServerThread;
        nn::os::ThreadType g_ClientThread[MaxConnectionCount];
        nn::os::MessageQueueType g_AcceptQueue;

        class TelnetConnection
        {
        public:
            explicit TelnetConnection(int fd) NN_NOEXCEPT : m_SocketFd(fd) {}
            ~TelnetConnection() NN_NOEXCEPT
            {
                nn::htcs::Close(m_SocketFd);
            }
            void Run() NN_NOEXCEPT
            {
                TelnetConsole console(m_SocketFd);
                CommandExecutorSet commandExecutorSet;

                console.Negotiate();

                Shell shell(&console, &commandExecutorSet);
                shell.Run();
            }
        private:
            int m_SocketFd;
        };

        void ServerThreadHtcs() NN_NOEXCEPT
        {
            while (NN_STATIC_CONDITION(true))
            {
                // PC と接続していない時に nn::htcs::Socket() を呼ぶと失敗するので、成功するまで繰り返す。
                // この nn::htcs::Socket() の仕様は将来改善される予定。
                // TODO: 仕様が変わったら、nn::htcs::Socket() の失敗で ABORT するように変更。
                int fd;
                while ((fd = nn::htcs::Socket()) == -1)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

                nn::htcs::SockAddrHtcs server;
                server.family = nn::htcs::HTCS_AF_HTCS;
                server.peerName = nn::htcs::GetPeerNameAny();
                std::strcpy(server.portName.name, g_PortName);

                if (nn::htcs::Bind(fd, &server) == -1)
                {
                    //NN_ABORT("[shell] nn::htcs::Bind() failed (%d).\n", nn::htcs::GetLastError());
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    htcs::Close(fd);
                    continue;
                }

                if (nn::htcs::Listen(fd, 1) == -1)
                {
                    //NN_ABORT("[shell] nn::htcs::Listen() failed (%d).\n", nn::htcs::GetLastError());
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    htcs::Close( fd );
                    continue;
                }

                for (;;)
                {
                    int clientFd = nn::htcs::Accept(fd, nullptr);
                    if (clientFd != -1)
                    {
                        nn::os::SendMessageQueue(&g_AcceptQueue, static_cast<uintptr_t>(clientFd));
                    }
                    else
                    {
                        // 同時接続可能なクライアント数を超えた場合
                        if(nn::htcs::GetLastError() == nn::htcs::HTCS_EMFILE)
                        {
                            NN_SDK_LOG("[shell] can not create htcs client because of the upper limit\n");
                            continue;
                        }
                        break; // Accept でその他の失敗 = TMS との接続が切れた
                    }
                }

                // TMS との接続が切れたため、ソケットの作成からやり直す
                htcs::Close( fd );
            }
        }

        void ServerThread(void* p) NN_NOEXCEPT
        {
            ServerThreadHtcs();
        }

        void ClientThread(void* p) NN_NOEXCEPT
        {
            for (;;)
            {
                uintptr_t l;
                nn::os::ReceiveMessageQueue(&l, &g_AcceptQueue);
                int fd = static_cast<int>(l);
                if (fd < 0)
                {
                    break;
                }

                {
                    TelnetConnection client(fd);
                    client.Run();
                }
            }
        }
    }

    void InitializeTelnetServer() NN_NOEXCEPT
    {
        const size_t StackSize = nn::os::StackRegionAlignment * 4;
        static uintptr_t s_AcceptQueueBuffer[MaxConnectionCount];
        NN_ALIGNAS(nn::os::StackRegionAlignment) static char s_ServerThreadStack[StackSize];
        NN_ALIGNAS(nn::os::StackRegionAlignment) static char s_ClientThreadStack[MaxConnectionCount][StackSize];

        nn::os::InitializeMessageQueue(&g_AcceptQueue, s_AcceptQueueBuffer, sizeof(s_AcceptQueueBuffer) / sizeof(*s_AcceptQueueBuffer));
        nn::os::CreateThread(&g_ServerThread, &ServerThread, 0, reinterpret_cast<void*>(s_ServerThreadStack), StackSize, nn::os::HighestThreadPriority - 1);
        nn::os::StartThread(&g_ServerThread);
        for (size_t i = 0; i < sizeof(g_ClientThread) / sizeof(*g_ClientThread); i++)
        {
            nn::os::CreateThread(&g_ClientThread[i], &ClientThread, 0, reinterpret_cast<void*>(s_ClientThreadStack[i]), StackSize, nn::os::HighestThreadPriority - 1);
            nn::os::StartThread(&g_ClientThread[i]);
        }
    }

    void FinalizeTelnetServer() NN_NOEXCEPT
    {
    }

}}  // namespace nn::shell
