﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cctype>
#include <cstdarg>
#include <nn/htcs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include "shell_TelnetConsole.h"
#include "shell_TelnetServer.h"

// TORIAEZU: TeraTerm で使えればよいことにする。
//
// オプションは、以下のものに決め打ちする。
// * ECHO              有効（サーバー側）
// * SUPPRESS-GO-AHEAD 有効（サーバー側、クライアント側）
// * その他            無効
//
// TeraTerm は以下のオプションも送ってくるが拒否する（DON'T）。
// * Will Terminal Type
// * Will Negotiate About Window Size

namespace nn { namespace shell {

namespace
{
    const int TelnetIac = 255;    // "Interpret as Command"

    enum TelnetCommand
    {
        TelnetCommand_Se        = 240,  // Subnegotiation Parameters End
        TelnetCommand_Nop       = 241,  // No Operation
        TelnetCommand_DataMark  = 242,
        TelnetCommand_Break     = 243,
        TelnetCommand_Ip        = 244,  // Interrupt Process
        TelnetCommand_Ao        = 245,  // Abort Output
        TelnetCommand_Ayt       = 246,  // Are You There
        TelnetCommand_Ec        = 247,  // Erase Character
        TelnetCommand_El        = 248,  // Erase Line
        TelnetCommand_Ga        = 249,  // Go Ahead
        TelnetCommand_Will      = 251,
        TelnetCommand_Wont      = 252,
        TelnetCommand_Do        = 253,
        TelnetCommand_Dont      = 254,
        TelnetCommand_Escaped   = 255
    };

    enum TelnetCommandOption
    {
        TelnetCommandOption_Echo            = 1,
        TelnetCommandOption_SuppressGoAhead = 3
    };
}

const int TelnetConsole::BufferLength;

TelnetConsole::TelnetConsole(int streamId) NN_NOEXCEPT
    : m_StreamId(streamId), m_Writer(streamId, m_WriteBuffer, sizeof(m_WriteBuffer))
{
}

TelnetConsole::~TelnetConsole() NN_NOEXCEPT
{
}

bool TelnetConsole::Negotiate() NN_NOEXCEPT
{
    m_Writer.PutString(
        // IAC WILL ECHO
        "\xff\xfb\x01"
        // IAC WILL SUPPRESS-GO-AHEAD
        "\xff\xfb\x03"
        // IAC DO SUPPRESS-GO-AHEAD
        "\xff\xfd\x03"
        );
    m_Writer.Flush();
    return true;
}

void TelnetConsole::PutChar(int c) NN_NOEXCEPT
{
    if (c == '\n')
    {
        m_Writer.PutString("\r\n");
        m_Writer.Flush();
    }
    else if (c == TelnetIac)
    {
        m_Writer.PutString("\xff\xff");
    }
    else
    {
        m_Writer.PutChar(c);
    }
}

int TelnetConsole::GetChar() NN_NOEXCEPT
{
    while (true)
    {
        int c = GetCharRaw();
        if (c == Console::EndOfStream)
        {
            return Console::EndOfStream;
        }

        // TELNETコマンドではない通常の文字なので返す
        if (c != TelnetIac)
        {
            return c;
        }

        int command = GetCharRaw();
        // エスケープされた TelnetIac 文字
        if (command == TelnetIac)
        {
            return TelnetIac;
        }

        ProcessCommand(command);
    }
}

// IAC より後の TELNET コマンドを取得して処理する
void TelnetConsole::ProcessCommand(int command) NN_NOEXCEPT
{
    switch (command)
    {
    // 「送信側はオプションを無効化した」
    case TelnetCommand_Wont:
        {
            int optionCode =  GetCharRaw();
            switch (optionCode)
            {
            case TelnetCommandOption_SuppressGoAhead:
                // GO AHEAD には対応しない
                break;
            default:
                break;
            }
            break;
        }
    // 「受信側はオプションを無効化せよ」
    case TelnetCommand_Dont:
        {
            int optionCode = GetCharRaw();
            switch (optionCode)
            {
            case TelnetCommandOption_SuppressGoAhead:
                // GO AHEAD には対応しない
                break;
            case TelnetCommandOption_Echo:
                // 本当は、こちら側でのエコーを無効にすべき
                break;
            default:
                break;
            }
            break;
        }
    // 「送信側はオプションを有効化したい」提案。DO で了承、DON'T で拒否。
    case TelnetCommand_Will:
        {
            int optionCode =  GetCharRaw();
            switch (optionCode)
            {
            case TelnetCommandOption_SuppressGoAhead:
                // NegotiationでDOを送ってある
                break;
            default:
                // 知らないオプションは拒否する
                // IAC DON'T optionCode
                m_Writer.PutString("\xff\xfe");
                m_Writer.PutChar(optionCode);
                m_Writer.Flush();
                break;
            }
            break;
        }
    // 「受信側でオプションを有効化せよ」要請。WILL で了承、WON'T で拒否。
    case TelnetCommand_Do:
        {
            int optionCode =  GetCharRaw();
            switch (optionCode)
            {
            case TelnetCommandOption_SuppressGoAhead:
            case TelnetCommandOption_Echo:
                // NegotiationでWILLを送ってある
                break;
            default:
                // 知らないオプションは拒否する
                // IAC WON'T optionCode
                m_Writer.PutString("\xff\xfc");
                m_Writer.PutChar(optionCode);
                m_Writer.Flush();
                break;
            }
            break;
        }
    default:
        break;
    }
}

int TelnetConsole::GetCharRaw() NN_NOEXCEPT
{
    return GetCharRawHtcs();
}

int TelnetConsole::GetCharRawHtcs() NN_NOEXCEPT
{
    // TORIAEZU: 1文字ずつRecvする。
    // この実装だと、ホストとの通信が切れた時（LANケーブル抜け、PCハングアップ）にタイムアウトしない
    uint8_t c;
    int result = nn::htcs::Recv(m_StreamId, &c, sizeof(uint8_t), 0);
    if (result <= 0)
    {
        if (nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN)
        {
            NN_SDK_LOG("shell: telnet recv error (%d)", nn::htcs::GetLastError());
        }

        return Console::EndOfStream;
    }
    else
    {
        return c;
    }
}

void TelnetConsole::FlushCore() NN_NOEXCEPT
{
    m_Writer.Flush();
}

TelnetBufferedWriter::TelnetBufferedWriter(int streamId, char* writeBuffer, int writeBufferLength) NN_NOEXCEPT
    : BufferedWriter(writeBuffer, writeBufferLength), m_StreamId(streamId)
{
}

bool TelnetBufferedWriter::Write(char* pBuffer, int length) NN_NOEXCEPT
{
    int socketResult;

    socketResult = nn::htcs::Send(m_StreamId, pBuffer, length, 0);
    return socketResult >= 0;
}

}}  // namespace nn { namespace shell {
