﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/util/util_FormatString.h>
#include <cstring>
#include <cctype>
#include <cstdarg>

#include "shell_Console.h"

namespace nn { namespace shell {

const int Console::EndOfStream;

bool Console::ReadLine(char *pBuffer, int bufferLength) NN_NOEXCEPT
{
    int n = 0;

    while (true)
    {
        int c = GetChar();

        if (c == Console::EndOfStream)
        {
            return false;
        }
        else if (std::iscntrl(c))
        {
            switch (c)
            {
            case 0x7f:
            case '\b':
                {
                    if (n <= 0)
                    {
                        PutChar('\a');
                        Flush();
                    }
                    else
                    {
                        PutString("\b\x1b[K");
                        Flush();
                        n --;
                    }
                    break;
                }
            case '\r':
                {
                    PutChar('\n');
                        Flush();
                    pBuffer[n] = '\0';
                    return true;
                }
            case 0x1b:  // ESC
                {
                    // TORIAEZU: VT100コントロールシーケンスを読み飛ばす
                    char c2 = GetChar();
                    if (c2 == '[')
                    {
                        // CSI
                        char c3 = GetChar();
                        NN_UNUSED(c3);
                    }
                    else if (0x40 <= c2 && c2 <= 0x5f)
                    {
                        // C1
                    }
                    NN_UNUSED(c2);
                }
                break;
            default:
                break;
            }
        }
        else
        {
            if (n >= bufferLength  - 1)
            {
                PutChar('\a');
                Flush();
            }
            else
            {
                PutChar(c);
                Flush();
                pBuffer[n] = c;
                n ++;
            }
        }
    }
}

int Console::TPrintf(const char *pFormat, ...) NN_NOEXCEPT
{
    static const int ShellPrintfBufferSize = 256;
    char buf[ShellPrintfBufferSize];

    va_list vlist;
    va_start(vlist, pFormat);
    int count = nn::util::VSNPrintf(buf, sizeof(buf), pFormat, vlist);
    PutString(buf);
    va_end(vlist);

    return count;
}

void Console::PutString(const char *pString) NN_NOEXCEPT
{
    for (const char* p = pString; *p != '\0'; p ++)
    {
        PutChar(*p);
    }
}

BufferedWriter::BufferedWriter(char* writeBuffer, int writeBufferLength) NN_NOEXCEPT
    : m_WriteBuffer(writeBuffer), m_WriteBufferLength(writeBufferLength), m_StoredLength(0)
{
}

void BufferedWriter::PutChar(int c) NN_NOEXCEPT
{
    m_WriteBuffer[m_StoredLength] = c;
    m_StoredLength ++;

    if (m_StoredLength >= m_WriteBufferLength)
    {
        Flush();
    }
}

void BufferedWriter::PutString(const char* pString) NN_NOEXCEPT
{
    for (const char* p = pString; *p != '\0'; p ++)
    {
        PutChar(*p);
    }
}

void BufferedWriter::Flush() NN_NOEXCEPT
{
    Write(m_WriteBuffer, m_StoredLength);
    m_StoredLength = 0;
}

}}  // namespace nn { namespace shell {
