﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/nn_Common.h>

namespace nn { namespace shell {

class Console;

const int MaxCommandArgumentCount = 128;

struct InputCommand
{
    int         argc;
    const char* argv[MaxCommandArgumentCount];

    const char* pCommandLine;
    const char* pArgumentPositions[MaxCommandArgumentCount];
    size_t      commandLineBytes;
};

class CommandParser
{
    NN_DISALLOW_COPY(CommandParser);
    NN_DISALLOW_MOVE(CommandParser);
public:
    enum ReadResult
    {
        ReadResult_Ok,
        ReadResult_TooManyArguments,
        ReadResult_UnmatchedQuote,
        ReadResult_Empty
    };

public:
    // 作業用バッファとコマンド行を指定して構築する
    // pBuffer は pCommandLine に格納された文字列以上の長さが必要です
    CommandParser(char* pBuffer, const char* pCommandLine) NN_NOEXCEPT;

    // コマンド行からコマンドを1つ読み込む。
    // 成功（空のコマンドも成功に含む）すれば true 、エラーなら false を返す
    ReadResult ReadCommand(InputCommand* pOutCommand) NN_NOEXCEPT;

    // 次のコマンドが存在するかどうか（;で区切られている）
    bool HasNext() const NN_NOEXCEPT
    {
        return m_HasNext;
    }
private:
    ReadResult ReadToken() NN_NOEXCEPT;
    ReadResult ReadQuotedToken() NN_NOEXCEPT;

    int Peek() NN_NOEXCEPT;
    void SkipAndNext() NN_NOEXCEPT;
    void UseAndNext() NN_NOEXCEPT;

    ReadResult BeginArg() NN_NOEXCEPT;
    void EndArg() NN_NOEXCEPT;
    void NullTerminateArgs() NN_NOEXCEPT;

    const char*     m_pSrc; // 読み取りポインタ
    char*           m_pDst; // 引数の書き込みポインタ
    char*           m_pArgEnd[MaxCommandArgumentCount];  // 引数の終端位置
    InputCommand*   m_pOutCommand;
    bool            m_HasNext;
};

}} // namespace nn { namespace shell {
