﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/mem.h>
#include <nn/nn_SdkAssert.h>
#include <nn/htc/tenv/htc_Tenv.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>

#include "scs_TenvServiceManager.h"

namespace nn { namespace scs {

    namespace
    {
        const size_t ExpHeapSize = 48 * 1024;
        NN_ALIGNAS(4096) uint8_t g_ExpHeapStack[ExpHeapSize];
        nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;
        struct StaticMutex
        {
            nn::os::MutexType mutex;
            void lock() NN_NOEXCEPT
            {
                nn::os::LockMutex(&mutex);
            }
            void unlock() NN_NOEXCEPT
            {
                nn::os::UnlockMutex(&mutex);
            }
        } g_ExpHeapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

        void InitializeExpHeap()
        {
            std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
            if( g_ExpHeapHandle == nullptr )
            {
                g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill);
                NN_ABORT_UNLESS(g_ExpHeapHandle);
            }
        }

        void* Allocate(size_t size) NN_NOEXCEPT
        {
            std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
            return nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
        }

        void Deallocate(void* p, size_t size) NN_NOEXCEPT
        {
            std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
            NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
        }
    }   // anonymous namespace

    void InitializeTenvServiceManager() NN_NOEXCEPT
    {
        InitializeExpHeap();
        nn::htc::tenv::Initialize(Allocate, Deallocate);
    }

    void NotifyTenvMountHostSuccess() NN_NOEXCEPT
    {
        nn::htc::tenv::NotifyMountHostSuccess();
    }

}}  // namespace nn::scs
