﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/htcs.h>
#include <cstring>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/os.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_Service.h>
#include <nn/ns/ns_Result.h>

#include "scs_ShellServer.h"
#include "scs_Shell.h"

namespace nn { namespace scs {

    namespace
    {
        int CreateSocket()
        {
            for(;;)
            {
                int socket = htcs::Socket();
                if (socket >= 0)
                {
                    return socket;
                }

                os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
            }
        }

        int Bind(int socket, htcs::HtcsPortName htcsPortName)
        {
            htcs::SockAddrHtcs sah;
            sah.family = htcs::HTCS_AF_HTCS;
            sah.peerName = htcs::GetPeerNameAny();
            std::strcpy(sah.portName.name, htcsPortName.name);

            auto result = htcs::Bind(socket, &sah);
            return result;
        }

        // size バイト受信するまでブロックする
        // htcs::Recv に HTCS_MSG_WAITALL を渡すと切断されたときに返らないのでその W/A
        htcs::ssize_t Receive(int socket, void* pBuffer, size_t size)
        {
            Bit8* p8 = reinterpret_cast<Bit8*>(pBuffer);
            size_t receivedSize = 0;

            while( receivedSize < size )
            {
                auto ret = htcs::Recv(socket, p8 + receivedSize, size - receivedSize, 0);
                if( ret <= 0 )
                {
                    return ret;
                }

                receivedSize += ret;
            }

            return static_cast<htcs::ssize_t>(size);
        }

        bool ReceiveCommand(CommandHeader* pHeaderBuffer, void* pBuffer, size_t bufferSize, int socket)
        {
            htcs::ssize_t receiveSize;

            receiveSize = Receive(socket, pHeaderBuffer, sizeof(*pHeaderBuffer));
            if( receiveSize != sizeof(*pHeaderBuffer) )
            {
                return false;
            }

            if( pHeaderBuffer->dataSize >= bufferSize )
            {
                NN_SDK_LOG("[cs] too large command data size (%d)\n", pHeaderBuffer->dataSize);
                return false;
            }

            receiveSize = Receive(socket, pBuffer, pHeaderBuffer->dataSize);
            if( receiveSize != static_cast<htcs::ssize_t>(pHeaderBuffer->dataSize) )
            {
                return false;
            }

            return true;
        }
    }
    // 匿名名前空間

    void ShellServer::Initialize(const char* portName, void* pStack, size_t stackSize, CommandProcessor* pCommandProcessor)
    {
        m_pCommandProcessor = pCommandProcessor;
        std::strcpy(m_HtcsPortName.name, portName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            os::CreateThread(
                &m_Thread,
                ThreadFunc,
                this,
                pStack,
                stackSize,
                NN_SYSTEM_THREAD_PRIORITY(scs, ShellServer) ));

        os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(scs, ShellServer));
    }

    void ShellServer::Start()
    {
        nn::os::StartThread(&m_Thread);
    }

    void ShellServer::ThreadFunc(void* arg)
    {
        reinterpret_cast<ShellServer*>(arg)->DoShellServer();
    }

    void ShellServer::DoShellServer()
    {
        int result;

        for(;;)
        {
            const int listenSocket = CreateSocket();

            const int BindResult = Bind(listenSocket, m_HtcsPortName);

            if(BindResult == 0)
            {
                for(;;)
                {
                    result = htcs::Listen(listenSocket, 0);

                    if(result != 0)
                    {
                        NN_SDK_LOG("[cs] listen failed. value = %d\n", result);
                        break;
                    }

                    int socket;

                    {
                        htcs::SockAddrHtcs temp;
                        socket = htcs::Accept(listenSocket, &temp);
                        if( socket <= 0 )
                        {
                            NN_SDK_LOG("[cs] accept failed\n");
                            continue;
                        }
                    }

                    RegisterSocket(socket);

                    for(;;)
                    {
                        bool isSuccess;
                        CommandHeader ch;

                        isSuccess = ReceiveCommand(&ch, m_Buffer, sizeof(m_Buffer), socket);
                        if( ! isSuccess )
                        {
                            break;
                        }

                        isSuccess = m_pCommandProcessor->ProcessCommand(ch, m_Buffer, socket);
                        if( ! isSuccess )
                        {
                            break;
                        }
                    }

                    UnregisterSocket(socket);
                    htcs::Close(socket);
                }
            }

            htcs::Close(listenSocket);
        }
    }

}}  // namespace nn::scs
