﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/htcs.h>
#include <cstring>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/os.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ldr/ldr_Result.public.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/htc/tenv/htc_Tenv.h>
#include <nn/util/util_StringUtil.h>

#include "scs_Shell.h"
#include "scs_Result.public.h"

namespace nn { namespace scs {

    namespace
    {
        const int ProgramEntryCountMax = 64;
        const int SocketEntryCountMax = 2;

        struct ProgramInfo
        {
            os::ProcessId       processId;
            int64_t             commandId;
            int                 socket;
            bool                isLaunchedByCs;
            bool                isDebugEnabled;
        };

        struct SocketInfo
        {
            int64_t         commandId;
            int             socket;
            bool            isSubsribeProcessEvent;
        };

        NN_OS_ALIGNAS_THREAD_STACK Bit8 g_Stack[4096];
        os::ThreadType                  g_Thread;
        os::Mutex                       g_Mutex(false);
        ProgramInfo                     g_Programs[ProgramEntryCountMax];
        SocketInfo                      g_Sockets[SocketEntryCountMax];
        ProcessEventHandler             g_CommonStartHandler;
        ProcessEventHandler             g_CommonExitHandler;
        ProcessEventHandler             g_CommonExceptionHandler;

        void ClearProgramInfo()
        {
            for( auto& e: g_Programs )
            {
                e.processId = os::ProcessId::GetInvalidId();
            }
        }

        ProgramInfo* FindProgram(os::ProcessId id)
        {
            for( auto& e: g_Programs )
            {
                if( e.processId == id )
                {
                    return &e;
                }
            }

            return NULL;
        }

        void ClearSocketInfo()
        {
            for( auto& e: g_Sockets )
            {
                e.socket = -1;
                e.isSubsribeProcessEvent = false;
            }
        }

        SocketInfo* FindSocket(int socket)
        {
            for( auto& e: g_Sockets )
            {
                if( e.socket == socket )
                {
                    return &e;
                }
            }

            return NULL;
        }

        // pTo には toSize 以内で NUL 終端された文字列が格納されている
        bool CatenateString(char* pTo, size_t toSize, const char* pFrom, size_t fromSize)
        {
            auto stringLength = std::strlen(pTo);
            if( stringLength + fromSize + 1 > toSize )
            {
                return false;
            }

            std::memcpy(pTo + stringLength, pFrom, fromSize);
            pTo[stringLength + fromSize] = '\0';

            return true;
        }

        Result GetProgramPath(lr::Path* pOutPath, const char* pPath, size_t pathSize) NN_NOEXCEPT
        {
            const char HostMountName[] = "@Host:/";

            std::strcpy(pOutPath->string, HostMountName);
            bool isSuccess = CatenateString(pOutPath->string, sizeof(pOutPath->string), pPath, pathSize);
            if( ! isSuccess )
            {
                return ResultTooLongPath();
            }

            return ResultSuccess();
        }

        void ProcessExitEvent(const ns::ShellEventInfo& sei)
        {
            nn::htc::tenv::UnregisterDefinitionFilePath(sei.processId);

            ProgramInfo pi;
            pi.processId = os::ProcessId::GetInvalidId();

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);

                auto pFound = FindProgram(sei.processId);
                if( pFound != NULL )
                {
                    pi = *pFound;
                    pFound->processId = os::ProcessId::GetInvalidId();
                }
            }

            if( pi.processId != os::ProcessId::GetInvalidId() )
            {
                g_CommonExitHandler(pi.commandId, pi.socket, pi.processId);
            }

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);
                for( auto& e : g_Sockets )
                {
                    if( e.isSubsribeProcessEvent )
                    {
                        g_CommonExitHandler(e.commandId, e.socket, sei.processId);
                    }
                }
            }
        }

        void ProcessStartedEvent(const ns::ShellEventInfo& sei)
        {
            ProgramInfo pi;
            pi.processId = os::ProcessId::GetInvalidId();

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);

                auto pFound = FindProgram(sei.processId);
                if( pFound != NULL )
                {
                    pi = *pFound;
                    pFound->isDebugEnabled = true;
                }
            }

            if( pi.processId != os::ProcessId::GetInvalidId() )
            {
                std::lock_guard<os::Mutex> lock(g_Mutex);
                auto pFoundSocket = FindSocket(pi.socket);
                if( pFoundSocket != NULL && pFoundSocket->isSubsribeProcessEvent )
                {
                    g_CommonStartHandler(pi.commandId, pi.socket, pi.processId);
                }
            }
            else
            {
                std::lock_guard<os::Mutex> lock(g_Mutex);
                RegisterProcessEventHandler(sei.processId, 0, -1, false, true);
            }

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);
                for( auto& e : g_Sockets )
                {
                    if( e.isSubsribeProcessEvent )
                    {
                        g_CommonStartHandler(e.commandId, e.socket, sei.processId);
                    }
                }
            }
        }

        void ProcessExceptionEvent(const ns::ShellEventInfo& sei)
        {
            ProgramInfo pi;
            pi.processId = os::ProcessId::GetInvalidId();

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);

                auto pFound = FindProgram(sei.processId);
                if( pFound != NULL )
                {
                    pi = *pFound;
                }
            }

            if( pi.processId != os::ProcessId::GetInvalidId() )
            {
                g_CommonExceptionHandler(pi.commandId, pi.socket, pi.processId);
            }

            {
                std::lock_guard<os::Mutex> lock(g_Mutex);
                for( auto& e : g_Sockets )
                {
                    if( e.isSubsribeProcessEvent )
                    {
                        g_CommonExceptionHandler(e.commandId, e.socket, sei.processId);
                    }
                }
            }
        }

        void EventHandlerThread(void*)
        {
            os::SystemEvent shellEvent;
            ns::GetShellEvent(&shellEvent);

            for(;;)
            {
                shellEvent.Wait();

                for (;;)
                {
                    ns::ShellEventInfo sei;
                    Result result = ns::GetShellEventInfo(&sei);
                    if (result.IsFailure())
                    {
                        NN_SDK_LOG("[cs] Break from ShellEvent loop (%08X). \n", result.GetInnerValueForDebug());
                        break;
                    }

                    NN_SDK_LOG("[cs] Receive event: %d\n", sei.event);

                    if( sei.event == ns::ShellEvent_Exit )
                    {
                        ProcessExitEvent(sei);
                    }
                    else if( sei.event == ns::ShellEvent_Started )
                    {
                        ProcessStartedEvent(sei);
                    }
                    else if( sei.event == ns::ShellEvent_Exception )
                    {
                        ProcessExceptionEvent(sei);
                    }
                }
            }
        }

        void StartEventHandlerThread()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                os::CreateThread(
                    &g_Thread,
                    EventHandlerThread,
                    NULL,
                    g_Stack,
                    sizeof(g_Stack),
                    NN_SYSTEM_THREAD_PRIORITY(scs, ShellEventHandler) ));

            os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(scs, ShellEventHandler));
            os::StartThread(&g_Thread);
        }
    }
    // 匿名名前空間


    void InitializeShell()
    {
        ClearProgramInfo();
        ClearSocketInfo();

        StartEventHandlerThread();
    }

    Result PrepareToLaunchProgram(ns::ProgramLaunchProperty* pOutProperty, const char* pPath, size_t pathSize, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
    {
        Result result;

        lr::Path path;
        result = GetProgramPath(&path, pPath, pathSize);
        if( result.IsFailure() )
        {
            return result;
        }

        ns::ProgramLaunchProperty launchProperty;
        result = nn::ns::PrepareLaunchProgramFromHost(&launchProperty, path.string);
        if( result.IsFailure() )
        {
            return result;
        }

        result = nn::ldr::SetProgramArgument(launchProperty.programId, pArgument, argumentSize);
        if( result <= ResultArgumentCountOverflow() )
        {
            result = nn::ldr::FlushArguments();
            if( result.IsSuccess() )
            {
                result = nn::ldr::SetProgramArgument(launchProperty.programId, pArgument, argumentSize);
            }
        }
        if( result.IsFailure() )
        {
            return result;
        }

        *pOutProperty = launchProperty;

        return ResultSuccess();
    }

    Result PrepareToLaunchProgram(ncm::ProgramId programId, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
    {
        Result result;

        result = nn::ldr::SetProgramArgument(programId, pArgument, argumentSize);
        if( result <= ResultArgumentCountOverflow() )
        {
            result = nn::ldr::FlushArguments();
            if( result.IsSuccess() )
            {
                result = nn::ldr::SetProgramArgument(programId, pArgument, argumentSize);
            }
        }
        if( result.IsFailure() )
        {
            return result;
        }

        return ResultSuccess();
    }

    void FlushProgramArgument(ncm::ProgramId programId) NN_NOEXCEPT
    {
        // 直前の引数を残さないように argv[0] に既定値の空文字列を上書きしておく
        nn::ldr::SetProgramArgument(programId, "", 1);
    }

    os::Mutex* GetProcessManagementMutex()
    {
        return &g_Mutex;
    }

    bool CanRegisterProcessEventHandler()
    {
        return FindProgram((os::ProcessId){0}) != NULL;
    }

    void RegisterCommonProcessEventHandler(
            ProcessEventHandler pStartHandler,
            ProcessEventHandler pExitHandler,
            ProcessEventHandler pExceptionHandler )
    {
        g_CommonStartHandler    = pStartHandler;
        g_CommonExitHandler     = pExitHandler;
        g_CommonExceptionHandler = pExceptionHandler;
    }

    bool RegisterProcessEventHandler(
            os::ProcessId       processId,
            int64_t             commandId,
            int                 socket,
            bool                isLaunchedByCs,
            bool                isDebugEnabled )
    {
        NN_SDK_ASSERT( g_Mutex.IsLockedByCurrentThread() );

        auto pFound = FindProgram((os::ProcessId){0});
        if( pFound == NULL )
        {
            return false;
        }

        pFound->processId        = processId;
        pFound->commandId        = commandId;
        pFound->socket           = socket;
        pFound->isLaunchedByCs   = isLaunchedByCs;
        pFound->isDebugEnabled   = isDebugEnabled;

        return true;
    }

    Result TerminateProcessAny(int64_t* pOutRegisteredCommandId)
    {
        std::lock_guard<os::Mutex> lock(g_Mutex);

        for( auto& e: g_Programs )
        {
            if( e.processId != os::ProcessId::GetInvalidId() && e.isLaunchedByCs )
            {
                return ns::TerminateProcess(e.processId);
            }
        }

        return ResultNoProcess();
    }

    void RaiseLaunchedEventAny(int64_t commandId, int socket, ProcessEventHandler pLaunchHandler)
    {
        std::lock_guard<os::Mutex> lock(g_Mutex);

        for( auto& e : g_Programs )
        {
            if( e.processId != os::ProcessId::GetInvalidId() && e.isDebugEnabled )
            {
                pLaunchHandler(e.commandId, socket, e.processId);
            }
        }
    }

    bool RegisterSocket(int socket)
    {
        std::lock_guard<os::Mutex> lock(g_Mutex);

        auto pFound = FindSocket(-1);
        if(pFound == NULL)
        {
            return false;
        }
        pFound->socket = socket;
        return true;
    }

    void UnregisterSocket(int socket)
    {
        std::lock_guard<os::Mutex> lock(g_Mutex);

        auto pFound = FindSocket(socket);
        pFound->socket = -1;
        pFound->isSubsribeProcessEvent = false;
    }

    Result SubscribeProcessEvent(int socket, bool isRegister, int64_t commandId)
    {
        std::lock_guard<os::Mutex> lock(g_Mutex);

        auto pFound = FindSocket(socket);
        if( pFound == NULL )
        {
            // TODO : 適切なエラー、但しここに来ることはない
            return ResultNoProcess();
        }
        pFound->isSubsribeProcessEvent = isRegister;
        pFound->commandId = commandId;
        return ResultSuccess();
    }

}}  // namespace nn::scs
