﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/htcs.h>
#include <cstring>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/htc/tenv/htc_Tenv.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Bis.h>
#include <nn/bpc/bpc_BoardPowerControl.h>

#include "scs_CommandProcessor.h"
#include "scs_Shell.h"
#include "scs_Result.public.h"

namespace nn { namespace scs {
    namespace
    {
        // This will lock htcs::Send calls to all of the sockets (@$cs and @$csForRunnerTools).
        // Htcs is processed in series as it does not support priorities so locking does not matter so much.
        // The only problem is that small packet will be blocked by long packet but long packet will be used
        // only when taking screenshot so we accept this design.
        nn::os::SdkRecursiveMutexType g_HtcsSendMutex = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();

        enum LaunchFlags
        {
            LaunchFlags_EnableException     = (1 << 2),
            LaunchFlags_DisableAslr         = (1 << 3),
        };

        int MakeNsLaunchFlags(int csFlags)
        {
            int nsFlags = 0;

            if( (csFlags & LaunchFlags_EnableException) != 0 )
            {
                nsFlags |= ns::LaunchProgramFlags_NotifyException;
            }
            if( (csFlags & LaunchFlags_DisableAslr) != 0 )
            {
                nsFlags |= ns::LaunchProgramFlags_DisableAslr;
            }

            return nsFlags;
        }
    }

    nn::os::SdkRecursiveMutexType* GetHtcsSendMutex()
    {
        return &g_HtcsSendMutex;
    }

    void CommandProcessor::SendErrorResult(int socket, int64_t commandId, Result result)
    {
        ResponseError re =
        {
            {
                commandId,
                Response_Error,
                sizeof(ResponseError) - sizeof(ResponseHeader)
            },
            result.GetInnerValueForDebug()
        };

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, &re, sizeof(re), htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::SendExited(int socket, int64_t commandId, Bit64 processIndex)
    {
        NN_SDK_LOG("[cs] Send Exit Event.\n");

        ResponseProgramExited rpe =
        {
            {
                commandId,
                Response_ProgramExited,
                8
            },
            processIndex
        };

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, &rpe, sizeof(rpe), htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::SendJitDebug(int socket, int64_t commandId)
    {
        NN_SDK_LOG("[cs] Send Jit Debug Event.\n");

        ResponseHeader rh =
        {
            commandId,
            Response_JitDebug,
            0
        };

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, &rh, sizeof(rh), htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::SendLaunched(int socket, int64_t commandId, Bit64 processIndex)
    {
        NN_SDK_LOG("[cs] Send Launched Event.\n");

        ResponseProgramLaunched rpl =
        {
            {
                commandId,
                Response_ProgramLaunched,
                8
            },
            processIndex
        };

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, &rpl, sizeof(rpl), htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::SendTitleName(const CommandHeader& header, const void* pBody, int socket)
    {
        // scs は TitleName の取得に非対応だが、
        // Target Manager をごまかすために長さ 0 の文字列をタイトル名として返す
        uint32_t titleLength = 0;
        ResponseHeader rh =
        {
            header.id,
            Response_TitleName,
            4 + titleLength
        };

        const size_t dataSizeMax = sizeof(rh) + sizeof(titleLength) + (sizeof(nn::ns::ApplicationTitle::name));
        static Bit8 s_Data[dataSizeMax];

        size_t dataSize = sizeof(rh) + sizeof(titleLength) + (sizeof(char) * titleLength);
        Bit8* pData(new(&s_Data) Bit8[dataSize]);
        std::memcpy(pData, &rh, sizeof(rh));
        std::memcpy(pData + sizeof(rh), &titleLength, sizeof(titleLength));
        //std::memcpy(pData + sizeof(titleLength) + sizeof(rh), buffer, sizeof(char) * titleLength);

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, pData, dataSize, htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::OnProcessExit(int64_t commandId, int socket, os::ProcessId processId)
    {
        SendExited(socket, commandId, processId.value);
    }

    void CommandProcessor::OnProcessRaiseJitDebug(int64_t commandId, int socket, os::ProcessId processId)
    {
        NN_UNUSED(processId);
        SendJitDebug(socket, commandId);
    }

    void CommandProcessor::OnProcessStart(int64_t commandId, int socket, os::ProcessId processId)
    {
        SendLaunched(socket, commandId, processId.value);
    }

    void CommandProcessor::ProcessCommandLaunchApplication(
        const CommandHeader& header,
        const void* pBody,
        int socket )
    {
        // 受信したデータからパラメータ取り出し
        CommandLaunchApplicationBody clab;
        {
            std::memcpy(&clab, pBody, sizeof(clab));
            const size_t requiredSize = sizeof(clab) + clab.pathSize + clab.argumentSize + clab.envPathSize;
            if( header.dataSize != requiredSize )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, ResultInvalidCommand());
                return;
            }
        }
        const char* pPath     = reinterpret_cast<const char*>(pBody) + sizeof(clab);
        const char* pArgument = pPath + clab.pathSize;
        const char* pEnvPath = pPath + clab.pathSize + clab.argumentSize;

        os::ProcessId processId ;
        Result result;

        if( ! CanRegisterProcessEventHandler() )
        {
            SendErrorResult(socket, header, ResultOutOfResource());
            return;
        }

        // 起動準備
        ns::ProgramLaunchProperty launchProperty;
        result = PrepareToLaunchProgram(&launchProperty, pPath, clab.pathSize, pArgument, clab.argumentSize);
        if( result.IsFailure() )
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        {
            std::lock_guard<os::Mutex> lock(*GetProcessManagementMutex());

            // 起動
            const int32_t flags = ( MakeNsLaunchFlags(clab.flags)
                                  | ns::LaunchProgramFlags_NotifyStart
                                  | ns::LaunchProgramFlags_NotifyExit
                                  | ns::LaunchProgramFlags_NotifyDebug );
            result = ns::LaunchProgram(&processId, launchProperty, flags);

            FlushProgramArgument(launchProperty.programId);
            if( result.IsFailure() )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }

            NN_SDK_LOG("[cs] Program launched.\n");

            // プログラム情報を格納
            NN_ABORT_UNLESS(
                RegisterProcessEventHandler(processId, header.id, socket, true, false) );

            NN_SDK_LOG("[cs] Registered event handler.\n");

            // ターゲット環境変数定義ファイルの設定
            if( clab.envPathSize != 0 )
            {
                nn::htc::tenv::RegisterDefinitionFilePath(processId, pEnvPath, clab.envPathSize);
            }
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandLaunchInstalledApplication(
        const CommandHeader& header,
        const void* pBody,
        int socket )
    {
        // 受信したデータからパラメータ取り出し
        CommandLaunchInstalledApplicationBody clab;
        {
            std::memcpy(&clab, pBody, sizeof(clab));
            const size_t requiredSize = sizeof(clab) + clab.argumentSize + clab.envPathSize;
            if( header.dataSize != requiredSize )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, ResultInvalidCommand());
                return;
            }
        }
        const char* pArgument = reinterpret_cast<const char*>(pBody) + sizeof(clab);
        const char* pEnvPath = reinterpret_cast<const char*>(pBody) + sizeof(clab) + clab.argumentSize;

        os::ProcessId processId ;
        Result result;
        ncm::ProgramId programId = { clab.programId };

        if( ! CanRegisterProcessEventHandler() )
        {
            SendErrorResult(socket, header, ResultOutOfResource());
            return;
        }

        // 起動準備
        result = PrepareToLaunchProgram(programId, pArgument, clab.argumentSize);
        if( result.IsFailure() )
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        {
            std::lock_guard<os::Mutex> lock(*GetProcessManagementMutex());

            // 起動
            ncm::ApplicationId applicationId = { programId.value };
            const int32_t flags = ( MakeNsLaunchFlags(clab.flags)
                                  | ns::LaunchProgramFlags_NotifyStart
                                  | ns::LaunchProgramFlags_NotifyExit
                                  | ns::LaunchProgramFlags_NotifyDebug );
            result = nn::ns::LaunchApplicationForDevelop(&processId, applicationId, flags);

            FlushProgramArgument(programId);
            if( result.IsFailure() )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }

            // プログラム情報を格納
            NN_ABORT_UNLESS(
                RegisterProcessEventHandler(processId, header.id, socket, true, false) );

            // ターゲット環境変数定義ファイルの設定
            if( clab.envPathSize != 0 )
            {
                nn::htc::tenv::RegisterDefinitionFilePath(processId, pEnvPath, clab.envPathSize);
            }
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandLaunchGameCardApplication(
        const CommandHeader& header,
        const void* pBody,
        int socket )
    {
        // 受信したデータからパラメータ取り出し
        CommandLaunchInstalledApplicationBody clab;
        {
            std::memcpy(&clab, pBody, sizeof(clab));
            const size_t requiredSize = sizeof(clab) + clab.argumentSize + clab.envPathSize;
            if( header.dataSize != requiredSize )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, ResultInvalidCommand());
                return;
            }
        }
        const char* pArgument = reinterpret_cast<const char*>(pBody) + sizeof(clab);
        const char* pEnvPath = reinterpret_cast<const char*>(pBody) + sizeof(clab) + clab.argumentSize;

        if( ! CanRegisterProcessEventHandler() )
        {
            SendErrorResult(socket, header, ResultOutOfResource());
            return;
        }

        // アプリを記録からリストアップする
        nn::ncm::ProgramId programId = { };
        nn::ns::ApplicationRecord record;
        int index = 0;
        int count = nn::ns::ListApplicationRecord(&record, 1, index);
        while(count != 0)
        {
            nn::ns::ApplicationView view;
            Result result = GetApplicationView(&view, &record.id, 1);

            if( result.IsFailure() )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }

            if( view.IsGameCard() )
            {
                programId = record.id;
                break;
            }

            index += 1;
            count = nn::ns::ListApplicationRecord(&record, 1, index);
        }

        os::ProcessId processId ;
        Result result;

        // 起動準備
        result = PrepareToLaunchProgram(programId, pArgument, clab.argumentSize);
        if( result.IsFailure() )
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        {
            std::lock_guard<os::Mutex> lock(*GetProcessManagementMutex());

            // 起動
            const int32_t flags = ( MakeNsLaunchFlags(clab.flags)
                                  | ns::LaunchProgramFlags_NotifyStart
                                  | ns::LaunchProgramFlags_NotifyExit
                                  | ns::LaunchProgramFlags_NotifyDebug );
            nn::ncm::ApplicationId applicationId = { programId.value };
            result = nn::ns::LaunchApplicationWithStorageIdForDevelop(
                &processId, applicationId, flags, ncm::StorageId::Card, ncm::StorageId::Any);

            FlushProgramArgument(programId);
            if( result.IsFailure() )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }

            // プログラム情報を格納
            NN_ABORT_UNLESS(
                RegisterProcessEventHandler(processId, header.id, socket, true, false) );

            // ターゲット環境変数定義ファイルの設定
            if( clab.envPathSize != 0 )
            {
                nn::htc::tenv::RegisterDefinitionFilePath(processId, pEnvPath, clab.envPathSize);
            }
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    bool CommandProcessor::IsAllowedSystemProcess(ncm::ProgramId programId)
    {
        ncm::ProgramId whiteList[] =
        {
            { 0x0100000000002101 }, // DevMenuCommand
        };
        for (auto allowed : whiteList)
        {
            if (allowed == programId)
            {
                return true;
            }
        }
        return false;
    }

    void CommandProcessor::ProcessCommandLaunchInstalledSystemProcess(
        const CommandHeader& header,
        const void* pBody,
        int socket)
    {
        // 受信したデータからパラメータ取り出し
        CommandLaunchInstalledSystemProcessBody clsb;
        {
            std::memcpy(&clsb, pBody, sizeof(clsb));
            const size_t requiredSize = sizeof(clsb) + clsb.argumentSize;
            if (header.dataSize != requiredSize)
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, ResultInvalidCommand());
                return;
            }
        }

        const char* pArgument = reinterpret_cast<const char*>(pBody) + sizeof(clsb);

        Result result;
        ncm::ProgramId programId = { clsb.programId };

        if (!IsAllowedSystemProcess(programId))
        {
            SendErrorResult(socket, header, ResultInvalidCommand());
            return;
        }

        if (!CanRegisterProcessEventHandler())
        {
            SendErrorResult(socket, header, ResultOutOfResource());
            return;
        }

        // 起動準備
        result = PrepareToLaunchProgram(programId, pArgument, clsb.argumentSize);
        if (result.IsFailure())
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        {
            std::lock_guard<os::Mutex> lock(*GetProcessManagementMutex());

            os::ProcessId processId;
            // 起動
            const int32_t flags = (ns::LaunchProgramFlags_NotifyStart | ns::LaunchProgramFlags_NotifyExit);
            ns::ProgramLaunchProperty plp = { programId, 0, ncm::StorageId::BuiltInSystem}; // システムプロセス起動時の version は任意値でよい
            result = nn::ns::LaunchProgram(&processId, plp, flags);

            FlushProgramArgument(programId);
            if (result.IsFailure())
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }

            // プログラム情報を格納
            NN_ABORT_UNLESS(
                RegisterProcessEventHandler(processId, header.id, socket, true, false));
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandTerminateProcess(
        const CommandHeader& header,
        int socket )
    {
        for(;;)
        {
            int64_t commandId;
            Result result = TerminateProcessAny(&commandId);

            if( result.IsFailure() )
            {
                if( result <= ResultNoProcess() )
                {
                    break;
                }

                SendErrorResult(socket, commandId, result);
            }
        }

        // 終了をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandRegisterTenvDefinitionFilePath(
        const CommandHeader& header,
        const void* pBody,
        int socket )
    {
        // 受信したデータからパラメータ取り出し
        CommandRegisterTenvDefinitionFilePath cmd;
        {
            std::memcpy(&cmd, pBody, sizeof(cmd));
            const size_t requiredSize = sizeof(cmd) + cmd.pathSize;
            if( header.dataSize != requiredSize )
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, ResultInvalidCommand());
                return;
            }
        }
        const char* pPath = reinterpret_cast<const char*>(pBody) + sizeof(cmd);
        os::ProcessId processId = { cmd.processId };

        // ターゲット環境変数定義ファイルの設定
        if( cmd.pathSize != 0 )
        {
            auto result = nn::htc::tenv::RegisterDefinitionFilePath(processId, pPath, cmd.pathSize);
            if (result.IsFailure())
            {
                // 失敗をクライアントに通知
                SendErrorResult(socket, header, result);
                return;
            }
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandTerminateApplication(
        const CommandHeader& header,
        int socket )
    {
        // アプリケーションを終了
        Result result = ns::TerminateApplication();

        if( result.IsFailure() )
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandSubscribeProcessEvent(
        const CommandHeader& header,
        const void* pBody,
        int socket,
        int64_t commandId )
    {
        bool isRegister = *(static_cast<const bool*>(pBody));
        Result result = SubscribeProcessEvent(socket, isRegister, commandId);
        if( result.IsFailure() )
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        // 成功をクライアントに通知
        SendSuccess(socket, header);

        // 起動しているプロセスを通知
        if( isRegister )
        {
            RaiseLaunchedEventAny(header.id, socket, OnProcessStart);
        }
    }

    void CommandProcessor::ProcessCommandSetSafeMode(
        const CommandHeader& header,
        int socket )
    {
        nn::Result result;

        {
            // ストレージをマウント
            std::unique_ptr<nn::fs::IStorage> pBootPartition1;
            result = nn::fs::OpenBisPartition(&pBootPartition1, nn::fs::BisPartitionId::BootPartition1Root);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                return;
            }

            // BCT をつぶす（512バイト０埋めで書き込む）
            const size_t FillSize = 512;
            std::unique_ptr<uint8_t[]> fillData(new uint8_t[FillSize]);
            std::memset(fillData.get(), 0, FillSize);
            result = pBootPartition1->Write(0, fillData.get(), FillSize);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                return;
            }
        }

        // 終了をクライアントに通知
        SendSuccess(socket, header);
    }

    void CommandProcessor::ProcessCommandShutdown(
        const CommandHeader& header,
        int socket )
    {
        // 終了をクライアントに通知
        SendSuccess(socket, header);

        // シャットダウン
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nn::bpc::InitializeBoardPowerControl();
        nn::bpc::ShutdownSystem();
    }

    void CommandProcessor::ProcessCommandReboot(
        const CommandHeader& header,
        int socket )
    {
        // 終了をクライアントに通知
        SendSuccess(socket, header);

        // 再起動
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        nn::bpc::InitializeBoardPowerControl();
        nn::bpc::RebootSystem();
    }

    void CommandProcessor::Initialize()
    {
        RegisterCommonProcessEventHandler(OnProcessStart, OnProcessExit, OnProcessRaiseJitDebug);
    }

    bool CommandProcessor::ProcessCommand(const CommandHeader& header, const Bit8* pBody, int socket)
    {
        switch(header.command)
        {
        // プロセス起動
        case Command_LaunchApplication:
            {
                ProcessCommandLaunchApplication(header, pBody, socket);
            }
            break;

        // プロセス強制終了
        case Command_TerminateProcess:
            {
                ProcessCommandTerminateProcess(header, socket);
            }
            break;

        // ターゲット環境変数定義ファイル登録
        case Command_RegisterTenvDefinitionFilePath:
            {
                ProcessCommandRegisterTenvDefinitionFilePath(header, pBody, socket);
            }
            break;

        // アプリ強制終了
        case Command_TerminateApplication:
            {
                ProcessCommandTerminateApplication(header, socket);
            }
            break;

        // プロセスイベント購読
        case Command_SubscribeProcessEvent:
            {
                ProcessCommandSubscribeProcessEvent(header, pBody, socket, header.id);
            }
            break;

        // タイトル文字列取得
        case Command_GetTitleName:
            {
                SendTitleName(header, pBody, socket);
            }
            break;

        // プロセス起動
        case Command_LaunchInstalledApplication:
            {
                ProcessCommandLaunchInstalledApplication(header, pBody, socket);
            }
            break;

        case Command_LaunchGameCardApplication:
            {
                ProcessCommandLaunchGameCardApplication(header, pBody, socket);
            }
            break;

        case Command_LaunchInstalledSystemProcess:
            {
                ProcessCommandLaunchInstalledSystemProcess(header, pBody, socket);
            }
            break;

        // セーフモード
        case Command_SetSafeMode:
            {
                ProcessCommandSetSafeMode(header, socket);
            }
            break;

        // シャットダウン
        case Command_Shutdown:
            {
                ProcessCommandShutdown(header, socket);
            }
            break;

        // 再起動
        case Command_Reboot:
            {
                ProcessCommandReboot(header, socket);
            }
            break;

        // 知らないコマンド
        default:
            {
                SendErrorResult(socket, header, ResultUnknownCommand());
            }
            break;
        }

        return true;
    }

    void CommandProcessor::SendErrorResult(int socket, const CommandHeader& command, Result result)
    {
        SendErrorResult(socket, command.id, result);
    }

    void CommandProcessor::SendSuccess(int socket, const CommandHeader& command)
    {
        ResponseHeader rh =
        {
            command.id,
            Response_Success,
            0
        };

        std::lock_guard<os::SdkRecursiveMutexType> lock(*GetHtcsSendMutex());
        htcs::Send(socket, &rh, sizeof(rh), htcs::HTCS_MSG_WAITALL);
    }

}}  // namespace nn::scs
