﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#include <nn/profiler/profiler_Module.private.h>
#include "profiler_LibModule.h"

#include <new>

#include <nn/profiler/profiler_Api.h>
#include <nn/ro/ro_Api.h>
#include <nn/ro/ro_Types.h>
#include <nn/ro/detail/ro_RoModule.h>

#include "profiler_Defines.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_TargetApplication.h"
#include "profiler_Time.h"

#include <nn/svc/svc_Base.h>

namespace nn { namespace profiler {


namespace /*anonymous*/ {

    const char* GetState(nn::svc::MemoryState state) NN_IS_UNUSED_MEMBER;
    const char* GetPermission(nn::svc::MemoryPermission permission) NN_IS_UNUSED_MEMBER;

    const char* GetState(nn::svc::MemoryState state)
    {
        switch (state)
        {
        case nn::svc::MemoryState_Free:                 return "     Free";
        case nn::svc::MemoryState_Io:                   return "       IO";
        case nn::svc::MemoryState_Static:               return "   Static";
        case nn::svc::MemoryState_Code:                 return "     Code";
        case nn::svc::MemoryState_CodeData:             return " CodeData";
        case nn::svc::MemoryState_Normal:               return "   Normal";
        case nn::svc::MemoryState_Shared:               return "   Shared";
        case nn::svc::MemoryState_Alias:                return "    Alias";
        case nn::svc::MemoryState_AliasCode:            return "    ACode";
        case nn::svc::MemoryState_AliasCodeData:        return "ACodeData";
        case nn::svc::MemoryState_Ipc:                  return "      IPC";
        case nn::svc::MemoryState_Stack:                return "    Stack";
        case nn::svc::MemoryState_ThreadLocal:          return "      TLS";
        case nn::svc::MemoryState_Transfered:           return " Transfrd";
        case nn::svc::MemoryState_SharedTransfered:     return "STransfrd";
        case nn::svc::MemoryState_SharedCode:           return "    SCode";
        case nn::svc::MemoryState_Inaccessible:         return " NoAccess";
        case nn::svc::MemoryState_NonSecureIpc:         return "  NoS IPC";
        case nn::svc::MemoryState_NonDeviceIpc:         return "  NoD IPC";
        case nn::svc::MemoryState_Kernel:               return "   Kernel";
        case nn::svc::MemoryState_GeneratedCode:        return "  GenCode";
        case nn::svc::MemoryState_CodeOut:              return "  CodeOut";
        default:                                        return "ERRORERRO";
        }
    }


    const char* GetPermission(nn::svc::MemoryPermission permission)
    {
        switch (permission)
        {
        case nn::svc::MemoryPermission_None:        return "----";
        case nn::svc::MemoryPermission_Read:        return "R---";
        case nn::svc::MemoryPermission_Write:       return "-W--";
        case nn::svc::MemoryPermission_Execute:     return "--X-";
        case nn::svc::MemoryPermission_ReadWrite:   return "RW--";
        case nn::svc::MemoryPermission_ReadExecute: return "R-X-";
        case nn::svc::MemoryPermission_DontCare:    return "DNC-";
        default:                                    return "ERR-";
        }
    }


    // TODO: Module API
    void LogMemoryRegion(nn::svc::MemoryInfo& memInfo) NN_IS_UNUSED_MEMBER;
    void LogMemoryRegion(nn::svc::MemoryInfo& memInfo)
    {
        NN_UNUSED(memInfo);

        INFO_LOG("Region: 0x%p - 0x%p: %s, %s\n",
            static_cast<uintptr_t>(memInfo.baseAddress),
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size),
            GetState(memInfo.state),
            GetPermission(memInfo.permission));
    }


    //void DumpModule(nn::ro::Module* pModule)
    //{
    //    DEBUG_LOG("Module %p\n", pModule);
    //    DEBUG_LOG("  RoModule* = %p\n", pModule->_module);
    //    DEBUG_LOG("  State = %d\n", pModule->_state);
    //    DEBUG_LOG("  FileAddr = %p\n", pModule->_fileAddress);
    //    DEBUG_LOG("  BuffAddr = %p\n", pModule->_bufferAddress);
    //    DEBUG_LOG("  ModuleSize = %d\n", int(pModule->_moduleSize));
    //}

} // anonymous


RoModuleList gActiveModules;
RoModuleList gPastModules;


void ModuleInitialize()
{
    new(&gActiveModules) RoModuleList;
    new(&gPastModules) RoModuleList;
}


void ModuleFinalize()
{
    // Nothing to do
}


RoModuleDetails::RoModuleDetails(
    uintptr_t codeData,
    uint64_t address,
    uint64_t size,
    uint64_t loadTime)
{
    Address = address;
    Size = size;
    LoadTime = loadTime;
    UnloadTime = 0;
    memset(Name, 0, sizeof(Name));

    ModulePointerStart = codeData;
    ModulePointerEnd = static_cast<uintptr_t>(address) - codeData; // TODO: Improve this as there may be empty pages

    nn::svc::MemoryInfo memInfo;
    nn::svc::PageInfo pageInfo;
    nn::Result result;
    NN_UNUSED(result);

    TargetApplication* app = TargetApplication::GetCurrent();

    result = app->QueryMemory(&memInfo, &pageInfo, codeData);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_SDK_ASSERT(memInfo.permission == nn::svc::MemoryPermission_ReadWrite &&
        (memInfo.state == nn::svc::MemoryState_CodeData || memInfo.state == nn::svc::MemoryState_AliasCodeData));
    ModulePointerStart = static_cast<uintptr_t>(memInfo.baseAddress);
    ModulePointerEnd = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);

    result = app->QueryMemory(&memInfo, &pageInfo, static_cast<uintptr_t>(address + size));
    NN_SDK_ASSERT(result.IsSuccess());
    NN_SDK_ASSERT(memInfo.permission == nn::svc::MemoryPermission_Read &&
        (memInfo.state == nn::svc::MemoryState_AliasCode || memInfo.state == nn::svc::MemoryState_Code));

    app->FindModuleName(
        static_cast<uintptr_t>(memInfo.baseAddress),
        static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size),
        Name);

    app->FindModuleBuildId(
        static_cast<uintptr_t>(memInfo.baseAddress),
        static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size),
        &BuildId);
}


void RegisterLoadModule(uintptr_t codeData, uint64_t addr, uint64_t size, uint64_t loadTime)
{
    void* memory = Memory::GetInstance()->Allocate<RoModuleDetails>();
    RoModuleDetails* details = new (memory) RoModuleDetails(codeData, addr, size, loadTime);
    DEBUG_LOG(
        "Module: %s\n\t\tCode Data: %p\n\t\tBase Addr: %p\n\t\tBase Size: %llu\n\t\tLoad Time: %llu\n",
        details->Name,
        codeData,
        addr,
        size,
        loadTime);

    gActiveModules.push_back(*details);
}


// TODO: Module API
//void RegisterLoadModule(nn::ro::detail::RoModule* pModule, uint64_t loadTime)
//{
//    NN_SDK_ASSERT(pModule != nullptr);
//
//    NN_STATIC_ASSERT(offsetof(nn::ro::Module, _module) == 0);
//
//    uintptr_t address = pModule->GetBase();
//
//    nn::svc::MemoryInfo memInfo;
//    nn::svc::PageInfo pageInfo;
//    nn::Result result;
//
//    result = nn::svc::QueryMemory(&memInfo, &pageInfo, address);
//    NN_SDK_ASSERT(result.IsSuccess());
//    NN_SDK_ASSERT(memInfo.permission == nn::svc::MemoryPermission_ReadExecute &&
//        (memInfo.state == nn::svc::MemoryState_AliasCode || memInfo.state == nn::svc::MemoryState_Code));
//    LogMemoryRegion(memInfo);
//    NN_UNUSED(result);
//
//    RegisterLoadModule(reinterpret_cast<uintptr_t>(pModule), memInfo.baseAddress, memInfo.size, loadTime);
//}
//
//
//
//void RecordLoadModule(nn::ro::Module* pModule)
//{
//    INFO_LOG("Module Recorded: %p\n");
//    if (pModule == nullptr ||
//        GetProfilerStatus() != ProfilerStatus_Profiling)
//    {
//        return;
//    }
//
//    NN_SDK_ASSERT(pModule->_state == nn::ro::Module::State_Loaded);
//
//    DumpModule(pModule);
//
//    RegisterLoadModule(pModule->_module, GetCurrentTime());
//}



void RegisterUnloadModule(RoModuleDetails* pModule, uint64_t unloadTime)
{
    NN_SDK_ASSERT(pModule != nullptr);

    pModule->UnloadTime = unloadTime;

    gActiveModules.erase(gActiveModules.iterator_to(*pModule));
    gPastModules.push_back(*pModule);
}



// TODO: Module API
//void RecordUnloadModule(nn::ro::Module* pModule)
//{
//    if (pModule == nullptr ||
//        GetProfilerStatus() != ProfilerStatus_Profiling)
//    {
//        return;
//    }
//
//    NN_SDK_ASSERT(pModule->_module != nullptr);
//    NN_SDK_ASSERT(pModule->_state == nn::ro::Module::State_Unloaded);
//
//    DumpModule(pModule);
//
//    RoModuleDetails* ro = nullptr;
//    for (auto& x : gActiveModules)
//    {
//        uintptr_t m = reinterpret_cast<uintptr_t>(pModule->_module);
//        if (x.ModulePointerStart <= m && m < x.ModulePointerEnd)
//        {
//            ro = &x;
//            break;
//        }
//    }
//    if (ro != nullptr)
//    {
//        RegisterUnloadModule(ro, GetCurrentTime());
//    }
//}



void ClearModuleLists()
{
    for (auto& x : gActiveModules)
    {
        Memory::GetInstance()->Free(&x);
    }
    gActiveModules.clear();

    for (auto& x : gPastModules)
    {
        Memory::GetInstance()->Free(&x);
    }
    gPastModules.clear();
}



void BuildActiveModules(nn::svc::MemoryInfo *pMemoryInfo, int count)
{
    uintptr_t addr = 0;
    uint64_t moduleCodeBase = 0;
    uint64_t moduleCodeSize = 0;

    for (int i = 0; i < count; ++i)
    {
        nn::svc::MemoryInfo &memInfo = pMemoryInfo[i];
        if (memInfo.permission == nn::svc::MemoryPermission_ReadExecute &&
            memInfo.state == nn::svc::MemoryState_AliasCode)
        {
            // Found a dynamic module
            moduleCodeBase = memInfo.baseAddress;
            moduleCodeSize = memInfo.size;
        }
        else if (memInfo.permission == nn::svc::MemoryPermission_ReadWrite &&
            memInfo.state == nn::svc::MemoryState_AliasCodeData &&
            moduleCodeBase != 0)
        {
            RegisterLoadModule(static_cast<uintptr_t>(memInfo.baseAddress), moduleCodeBase, moduleCodeSize, 0);
            moduleCodeBase = 0;
        }
        else if (memInfo.state == nn::svc::MemoryState_Inaccessible ||
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size) <= addr)
        {
            break;
        }

        addr = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);
    }
}



void BuildStaticModules(nn::svc::MemoryInfo *pMemoryInfo, int count)
{
    uintptr_t addr = 0;
    uint64_t moduleCodeBase = 0;
    uint64_t moduleCodeSize = 0;

    for (int i = 0; i < count; ++i)
    {
        nn::svc::MemoryInfo &memInfo = pMemoryInfo[i];
        if (memInfo.permission == nn::svc::MemoryPermission_ReadExecute &&
            memInfo.state == nn::svc::MemoryState_Code)
        {
            // Found a dynamic module
            moduleCodeBase = memInfo.baseAddress;
            moduleCodeSize = memInfo.size;
        }
        else if (memInfo.permission == nn::svc::MemoryPermission_ReadWrite &&
            memInfo.state == nn::svc::MemoryState_CodeData &&
            moduleCodeBase != 0)
        {
            RegisterLoadModule(static_cast<uintptr_t>(memInfo.baseAddress), moduleCodeBase, moduleCodeSize, 0);
            RoModuleDetails& y = gActiveModules.back();
            RegisterUnloadModule(&y, UINT64_MAX);
            moduleCodeBase = 0;
        }
        else if (memInfo.state == nn::svc::MemoryState_Inaccessible ||
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size) <= addr)
        {
            break;
        }

        addr = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);
    }
}



void CheckForNewModules()
{
    uintptr_t addr = 0;
    uint64_t moduleCodeBase = 0;
    uint64_t moduleCodeSize = 0;

    const int memoryInfoSize = 1024;
    nn::svc::MemoryInfo *memoryInfo = reinterpret_cast<nn::svc::MemoryInfo*>(
        Memory::GetInstance()->Allocate(
            memoryInfoSize * sizeof(nn::svc::MemoryInfo),
            alignof(nn::svc::MemoryInfo)));
    int count = 0;
    nn::Result result = TargetApplication::GetCurrent()->GetProcessMemoryInfo(&count, memoryInfo, memoryInfoSize);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        count = 0; // Ensure that this is actually 0
    }

    for (int i = 0; i < count; ++i)
    {
        nn::svc::MemoryInfo &memInfo = memoryInfo[i];
        if (memInfo.permission == nn::svc::MemoryPermission_ReadExecute &&
            memInfo.state == nn::svc::MemoryState_AliasCode)
        {
            // Found a dynamic module
            moduleCodeBase = memInfo.baseAddress;
            moduleCodeSize = memInfo.size;
        }
        else if (memInfo.permission == nn::svc::MemoryPermission_ReadWrite &&
            memInfo.state == nn::svc::MemoryState_AliasCodeData &&
            moduleCodeBase != 0)
        {
            bool found = false;
            for (auto& x : gActiveModules)
            {
                if (memInfo.baseAddress == x.ModulePointerStart)
                {
                    found = true;
                    break;
                }
            }
            if (!found)
            {
                RegisterLoadModule(static_cast<uintptr_t>(memInfo.baseAddress), moduleCodeBase, moduleCodeSize, 0);
            }
            moduleCodeBase = 0;
        }
        else if (memInfo.state == nn::svc::MemoryState_Inaccessible ||
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size) <= addr)
        {
            break;
        }

        addr = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);
    }

    Memory::GetInstance()->Free(memoryInfo);
}



void CloseActiveModules()
{
    while (gActiveModules.size() > 0)
    {
        RoModuleDetails& x = gActiveModules.front();
        RegisterUnloadModule(&x, UINT64_MAX);
    }
}



void DumpMemoryInfo()
{
#if MINIMUM_LOG_TIER <= LOG_AS_INFO
    uintptr_t addr = 0;

    const int memoryInfoSize = 1024;
    nn::svc::MemoryInfo *memoryInfo = reinterpret_cast<nn::svc::MemoryInfo*>(
        Memory::GetInstance()->Allocate(
            memoryInfoSize * sizeof(nn::svc::MemoryInfo),
            alignof(nn::svc::MemoryInfo)));
    int count = 0;
    nn::Result result = TargetApplication::GetCurrent()->GetProcessMemoryInfo(&count, memoryInfo, memoryInfoSize);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        return;
    }

    INFO_LOG("Target Memory Info:\n");
    INFO_LOG("----\n");
    for (int i = 0; i < count; ++i)
    {
        nn::svc::MemoryInfo &memInfo = memoryInfo[i];

        LogMemoryRegion(memInfo);

        if (memInfo.state == nn::svc::MemoryState_Inaccessible ||
            static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size) <= addr)
        {
            break;
        }

        addr = static_cast<uintptr_t>(memInfo.baseAddress + memInfo.size);
    }
    INFO_LOG("----\n");

    Memory::GetInstance()->Free(memoryInfo);
#endif
}


}} // nn::profiler
