﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/profiler/profiler_Result.h>

#include "profiler_CodeRewriting.h"
#include "profiler_Logging.h"
#include "profiler_ResultPrivate.h"
#include "profiler_TargetApplication.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const int InstrumentationTrampolineCount = 1;
    const uint32_t BranchInstructionBase = 0x14000000;
    const uint32_t NopInstruction = 0xd503201f;

    struct Globals
    {
        uint64_t trampolineAddresses[InstrumentationTrampolineCount];
        uint64_t functionAddresses[InstrumentationTrampolineCount];

        uint32_t activeSlots;
        bool canInstrument;
    };
    Globals globals;
}


void ResetInstrumentation()
{
    memset(&globals, 0, sizeof(globals));
}


void SetTrampolineAddresses(uint64_t* trampolines, int count)
{
    NN_SDK_ASSERT(count <= InstrumentationTrampolineCount);
    int available = count;
    for (int i = 0; i < count; ++i)
    {
        globals.trampolineAddresses[i] = trampolines[i];
        if (!IsValidCodeAddress(trampolines[i]))
        {
            --available;
        }
    }

    if (available <= 0)
    {
        ERROR_LOG("Instrumentation will not be available.\n");
        globals.canInstrument = false;
        return;
    }

    globals.canInstrument = true;
}


nn::Result InstrumentFunction(uint64_t address, int slot)
{
    nn::Result result;

    NN_SDK_ASSERT(slot < InstrumentationTrampolineCount);
    NN_SDK_ASSERT((globals.activeSlots & (1 << slot)) == 0);
    NN_SDK_ASSERT(IsValidCodeAddress(globals.trampolineAddresses[slot]));

    if (!IsValidCodeAddress(address))
    {
        return nn::profiler::ResultInstrumentInvalidAddress();
    }

    auto app = TargetApplication::GetCurrent();

    // Check if this is a valid code region for the target application
    result = app->IsValidCodeAddress(static_cast<uintptr_t>(address));
    if (result.IsFailure())
    {
        ERROR_LOG("Could not validate address %p as code\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultInstrumentationFailure();
    }

    uint64_t targetAddress = address;
    uint64_t trampolineAddress = globals.trampolineAddresses[slot];
    uint64_t nopOffset = 15 * 4;
    uint64_t overwriteAddress = trampolineAddress + nopOffset;

    FORCE_LOG("Target Address = 0x%016llx\n", targetAddress);
    FORCE_LOG("Trampoline Address = 0x%016llx\n", trampolineAddress);
    FORCE_LOG("Nop Offset = 0x%016llx\n", nopOffset);
    FORCE_LOG("Overwrite Address = 0x%016llx\n", overwriteAddress);

    uint32_t codeBuffer[2];
    uintptr_t codePtr = reinterpret_cast<uintptr_t>(codeBuffer);

    // Get the first instruction of the function being instrumented.
    result = app->ReadMemory(codePtr, targetAddress, sizeof(uint32_t));
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to read instrumentation target address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultInstrumentationFailure(); // TODO: More specific?
    }

    int64_t toTargetOffset = ((int64_t)targetAddress + 4 - ((int64_t)overwriteAddress + 4)) / 4;
    uint32_t branchToTarget = BranchInstructionBase | (uint32_t)(toTargetOffset & 0x03FFFFFF);
    // codeBuffer[0] = <instrumented function first instruction>
    codeBuffer[1] = branchToTarget;

    // Write the first instruction and branch back to instrumented function into the trampoline
    result = app->WriteMemory(codePtr, overwriteAddress, sizeof(uint32_t) * 2);
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to write instrumentation trampoline address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultInstrumentationFailure(); // TODO: More specific?
    }

    int64_t toTrampolineOffset = ((int64_t)trampolineAddress - (int64_t)targetAddress) / 4;
    uint32_t branchToTrampoline = BranchInstructionBase | (uint32_t)(toTrampolineOffset & 0x03FFFFFF);
    codeBuffer[0] = branchToTrampoline;

    // Write the branch to the trampoline into the instrumented function.
    result = app->WriteMemory(codePtr, targetAddress, sizeof(uint32_t));
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to write instrumentation target address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        return nn::profiler::ResultInstrumentationFailure();
    }

    globals.functionAddresses[slot] = address;
    globals.activeSlots |= (1 << slot);

    return nn::ResultSuccess();
}


nn::Result InstrumentRemoval(int slot)
{
    nn::Result result;

    NN_SDK_ASSERT(slot < InstrumentationTrampolineCount);

    if ((globals.activeSlots & (1 << slot)) == 0)
    {
        return nn::ResultSuccess();
    }

    uint64_t targetAddress = globals.functionAddresses[slot];
    uint64_t trampolineAddress = globals.trampolineAddresses[slot];
    uint64_t nopOffset = 15 * 4;
    uint64_t overwriteAddress = trampolineAddress + nopOffset;

    FORCE_LOG("Target Address = 0x%016llx\n", targetAddress);
    FORCE_LOG("Trampoline Address = 0x%016llx\n", trampolineAddress);
    FORCE_LOG("Nop Offset = 0x%016llx\n", nopOffset);
    FORCE_LOG("Overwrite Address = 0x%016llx\n", overwriteAddress);

    uint32_t codeBuffer[2];
    uintptr_t codePtr = reinterpret_cast<uintptr_t>(codeBuffer);

    auto app = TargetApplication::GetCurrent();

    // Get the original first instruction of the instrumented function.
    result = app->ReadMemory(codePtr, overwriteAddress, sizeof(uint32_t));
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to read memory of instrumentation trampoline address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // Put back the original first instruction into the instrumented function. (replaces the branch to trampoline)
    result = app->WriteMemory(codePtr, targetAddress, sizeof(uint32_t));
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to write memory of instrumentation target address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // Put the NOP instructions back at the end of the trampoline.
    codeBuffer[0] = NopInstruction;
    codeBuffer[1] = NopInstruction;
    result = app->WriteMemory(codePtr, overwriteAddress, sizeof(uint32_t) * 2);
    if (result.IsFailure())
    {
        ERROR_LOG("Failed to write memory of instrumentation trampoline address.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    globals.activeSlots &= ~(1 << slot);
    return nn::ResultSuccess();
}


uint64_t GetFunctionAddressForSlot(int slot)
{
    NN_SDK_ASSERT(slot < InstrumentationTrampolineCount);
    return globals.functionAddresses[slot];
}


}} // namespace nn::profiler
