﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <curl/curl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_TimeSpan.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/os/os_Thread.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/time/time_Api.h>
#include <nn/util/util_ScopeExit.h>

#include "olsc_OlscServer.h"

// 例外関連コードのリンク抑制のためのオーバーライド
// 参考：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-34162
void* operator new(std::size_t size) { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT { free(ptr); }

void* operator new[](std::size_t size) { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT{ return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT{ free(ptr); }

namespace
{
#define HEAP_INITIALIZER_INITIALIZER {false}
struct HeapInitializer
{
    bool _initialized;
    nn::lmem::HeapHandle _handle;
    char _buffer[512 * 1024];

    void Initialize() NN_NOEXCEPT
    {
        _handle = nn::lmem::CreateExpHeap(_buffer, sizeof(_buffer), nn::lmem::CreationOption_NoOption);
        _initialized = true;
    }
    nn::lmem::HeapHandle GetHandle() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(_initialized);
        return _handle;
    }
} g_HeapInitializer = HEAP_INITIALIZER_INITIALIZER;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapInitializer.GetHandle(), size);
}
void Deallocate(void* ptr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapInitializer.GetHandle(), ptr);
}

// libcurl が使用する malloc/free 用のバッファ
NN_ALIGNAS(4096) char g_BufferForStandardAllocator[1 * 1024 * 1024];

// Socket が要求するメモリプール
const int SocketCount = 3;
const int SocketConcurrencyCount = SocketCount;
nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> g_SocketConfigWithMemory(SocketConcurrencyCount);

void MainLoop() NN_NOEXCEPT
{
    NN_DETAIL_OLSC_TRACE("[Olsc] MainLoop\n");

    while (NN_STATIC_CONDITION(true))
    {
        // TODO: bgtc とか psc ハンドリング
        nn::os::SleepThread(nn::TimeSpan::FromDays(1));
    }
}
} // ~namespace <anonymous>

extern "C" void nninitStartup()
{
    // malloc, free の初期化
    nn::init::InitializeAllocator(g_BufferForStandardAllocator, sizeof(g_BufferForStandardAllocator));
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    NN_DETAIL_OLSC_TRACE("[Olsc] nnMain\n");
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(olsc, Main));

    nn::account::InitializeForSystemService();

    // 環境の初期化 ---------------------------------------------------------------------------------
    // fs サービスを複数セッションで初期化
    nn::fs::InitializeWithMultiSessionForSystem();

    // ヒープ
    g_HeapInitializer.Initialize();
    nn::fs::SetAllocator(Allocate, Deallocate);

    // 依存するサービスの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

    // サービスの初期化 --------------------------------------------------------------------------
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::olsc::srv::StartOlscServer());

    // olsc ライブラリの初期化 --------------------------------------------------------------------------
    nn::olsc::srv::InitializeOlscLibraryForDfc();

    // ループ  -----------------------------------------------------------------------------------
    MainLoop();

    // 終了処理 -----------------------------------------------------------------------------------
}
