﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/npns.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>

#include "npns_AsyncManager.h"

// TODO
#include <strophe.h>

namespace nn{ namespace npns{

//!< XMPP の標準的な処理を行うクラス.(NPNS の独自のプロトコル実装は nn::npns::Client クラスへ実装する)
class Xmpp
{
    NN_DISALLOW_COPY( Xmpp );
    NN_DISALLOW_MOVE( Xmpp );

public:
    class Stanza;
    class StanzaText;
    enum
    {
        HandlerResult_OneShot = 0,
        HandlerResult_NotOneShot = 1
    } HandlerResult;

    Xmpp(const char* pAgenaName, const char* pVersion);

    void Initialize();
    void Finalize();
    void Reinitialize();

    Result Connect(const char* jid, const char* password);
    void Disconnect();

    void SetPresence(const char* pShowString)
    {
        SetPresenceWithStatus(pShowString, NULL);
    }
    void SetPresenceWithStatus(const char* pShowString, const char* pStatusString);
    void EnableStreamManagement();

    Result PubsubSubscribe(const char* pNodeName);
    Result PubsubUnsubscribe(const char* pNodeName);
    Result PubsubGet(const char* pNodeName, int32_t maxItems);

    virtual void RunOnce(unsigned long timeout);

    bool IsRunning() const;
    bool IsConnected() const;
    bool IsConnectionInProgress() const;
    bool HasPendingData() const;
    Result GetLastError() const;

    bool IsUsingProxy() const;
    int GetSocket() const;
    void ActivateKeepAlive();
    void DeactivateKeepAlive();

    static const char* PresenceAway;
    static const char* PresenceChat;
    static const char* PresenceDoNotDisturb;
    static const char* PresenceExtendedAway;

protected:
    class Text
    {
        NN_DISALLOW_COPY(Text);
        NN_DISALLOW_MOVE(Text);
    public:
        Text();
        ~Text();
        void Set(xmpp_ctx_t* pContext, char* pText);
        const char* Get() const;
        size_t GetLength() const;
        void Free();

    private:
        xmpp_ctx_t* m_pContext;
        char*       m_pText;
    };

    typedef AsyncManager<Xmpp::Stanza> XmppAsyncManager;

    virtual void OnChangeConnectionStatus(const xmpp_conn_event_t status,
        const int error, xmpp_stream_error_t * const stream_error);

    virtual void OnReceiveVersion(const Stanza& stanzaIncoming);
    virtual void OnReceiveMessage(const Stanza& stanzaIncoming);
    virtual void OnReceiveMessageBody(const Text& text);
    virtual void OnReceiveResultAndError(const Stanza& stanzaIncoming);
    virtual void OnReceiveDisconnectError(const Stanza& stanzaIncoming);
    virtual void OnReceiveAuthFailure(const Stanza& stanzaIncoming);
    virtual void OnAckNeeded();
    virtual void OnConnected();
    virtual void OnDisconnected();
    virtual void OnBeforeDisconnect();
    virtual void OnFlush();

    void Send(Stanza& stanza);
    Result SendAndWait(Stanza& stanza)
    {
        return SendAndWait(nullptr, stanza, nullptr);
    }
    Result SendAndWait(Stanza* pReceiveStanza, Stanza& stanza, nn::os::Event* pEventInterrupt);
    void SendRawString(const char* pString);
    void Flush();

    const char* GetConnectedDomain() const;
    const char* GetConnectedJid() const;
    xmpp_ctx_t* GetContext() const;

    Result EnsureConnected();
    int SetTcpKeepAlive(bool enable, int idle, int interval, int count) const;
    int SetTcpNoDelay(bool bValid) const;
    void AddUserHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type, void * const userdata);
    void DeleteHandler(xmpp_handler handler);

    static Result ConvertErrorToResult(int error);
    static Result ConvertErrorStanzaToResult(const Stanza& stanzaError);

protected:
    void AddHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type);
    void AddSystemHandler(xmpp_handler handler, const char * const ns, const char * const name, const char * const type);

private:
    static void ConnectionHandlerEntry(xmpp_conn_t * const conn, const xmpp_conn_event_t status,
        const int error, xmpp_stream_error_t * const stream_error,
        void * const userdata);
    static int VersionHandlerEntry(xmpp_conn_t * const conn, xmpp_stanza_t * const stanza, void * const userdata);
    static int MessageHandlerEntry(xmpp_conn_t * const conn, xmpp_stanza_t * const stanza, void * const userdata);
    static int ResultHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata);
    static int ErrorHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata);
    static int AckHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata);
    static int StreamManagementHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata);
    static int AuthFailureHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const stanza, void * const userdata);

    void MakePubsubStanza(Stanza& stanzaOut, Stanza& stanzaChild);

    mutable nn::os::Mutex   m_mutex;

    xmpp_ctx_t*     m_pContext;
    xmpp_conn_t*    m_pConnection;
    const xmpp_log_t* m_pLog;
    const char*     m_pAgentName;
    const char*     m_pVersion;
    uint32_t        m_countAckRequiredMeesages;
    bool            m_bUsingProxy;
    XmppAsyncManager m_asyncManager;
    bool            m_bAuthFailureReceived;
};

}}
