﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/npns.h>
#include <nn/bgsu/bgsu_ThreadTemplate.h>
#include <nn/bgsu/bgsu_MultiClientSystemEvent.h>
#include <mutex>

#include "npns_StateMachine.h"
#include "npns_TypedMessageQueue.h"

namespace nn {
namespace npns {

    class StateMachineThread : public bgsu::ThreadTemplate<StateMachineThread, NN_NPNS_THREAD_STACK_SIZE_STATEMACHINE>,
                               private StateMachine
    {
    public:
        typedef bgsu::ThreadTemplate<StateMachineThread, NN_NPNS_THREAD_STACK_SIZE_STATEMACHINE> ThreadBase;
        static const int32_t REQUEST_QUEUE_DEPTH = 8;

        class Request
        {
        public:
            typedef StateMachineThread::StateMachine StateMachine;

            Request(State stateTarget, bool bUser, Result resultCancel)
                : m_mutexWait(true)
                , m_event(os::EventClearMode_ManualClear)
                , m_resultCancel(resultCancel)
                , m_stateTarget(stateTarget)
                , m_bUserDriven(bUser)
                , m_bFinished(false)
            {
            }

            Request(State stateTarget, Result resultCancel)
                : m_mutexWait(true)
                , m_event(os::EventClearMode_ManualClear)
                , m_resultCancel(resultCancel)
                , m_stateTarget(stateTarget)
                , m_bUserDriven(true)
                , m_bFinished(false)
            {
            }

            Request& operator=(const Request& req)
            {
                m_event.Clear();
                m_resultCancel  = req.m_resultCancel;
                m_stateTarget   = req.m_stateTarget;
                m_bUserDriven   = req.m_bUserDriven;
                m_bFinished     = req.m_bFinished;
                return *this;
            }

            ~Request()
            {
                std::lock_guard<os::Mutex> lock(m_mutexWait);
            }

            Result Wait()
            {
                Result result;
                {
                    std::lock_guard<os::Mutex> lock(m_mutexWait);
                    m_event.Wait();
                    result = m_result;
                }
                return result;
            }

            State GetTargetState() const
            {
                return m_stateTarget;
            }

            void ReturnResult(Result result)
            {
                m_result = result;
                m_bFinished = true;
                m_event.Signal();
            }

            bool IsUserDriven() const
            {
                return m_bUserDriven;
            }

            Result GetResult() const
            {
                return m_result;
            }

            Result GetCancelResult() const
            {
                return m_resultCancel;
            }

            bool IsFinished() const
            {
                return m_bFinished;
            }

            os::Event& GetEvent()
            {
                return m_event;
            }

        private:
            nn::os::Mutex       m_mutexWait;
            nn::os::Event       m_event;
            Result              m_result;
            Result              m_resultCancel;
            State               m_stateTarget;
            bool                m_bUserDriven;
            bool                m_bFinished;
        };

        NN_IMPLICIT StateMachineThread(Controller& controller);
        virtual ~StateMachineThread();

        typedef ThreadTemplate<StateMachineThread> Base;

        Result Initialize();
        virtual void Finalize();

        Result SendRequest(Request& request, bool bAsync = false);
        Result RequestChangeState(State stateTarget, Result resultCancel = ResultCanceledByOtherRequest());
        Result RequestChangeStateAsync(State stateTarget, Result resultCancel = ResultCanceledByOtherRequest());
        Result RequestChangeStateForceAsync(State stateTarget, Result resultCancel);
        Result RequestChangeStateForceTimed(State stateTarget, Result resultCancel, const TimeSpan& timeout, os::SystemEvent* pCancelEvent = nullptr);

        void CancelRequest(Result result);

        bool IsProcessingRequest() const;
        bool IsProcessingForceRequest() const;

        const char* GetCurrentStateString() const;

        // TODO: 別のクラスに分離
        void OnFullAwakeLeave();
        void OnFullAwakeEnter();
        void OnSleepEnter(bool bKeepSession);
        void OnShutdown();
        void OnHalfAwakeEnter(os::SystemEvent* pEventCancel, bool bPerformPingPong, bool* pIsRetryProcessed = nullptr);
        static State DecideTargetStateOnSleepEnter(State stateCurrent, bool bKeepSession);

        void AttachStateChangeEvent(bgsu::MultiClientSystemEvent::Node* pNode);
        void DetachStateChangeEvent(bgsu::MultiClientSystemEvent::Node* pNode);

        // StateMachine クラスのメソッドを公開
        using StateMachine::GetCurrentState;
        using StateMachine::GetTargetState;
        using StateMachine::GetTargetStateOfLastFailure;
        using StateMachine::IsTransitionComplete;
        using StateMachine::IsInProgress;
        using StateMachine::GetStateString;

        using StateMachine::SetAutonomyEnabled;
    protected:
        virtual void OnTransitionComplete();
        virtual bool OnTransitionFailure(Result result);
        virtual void OnEntered(State state);

        virtual void ThreadBody();
        virtual void RequestExit();

        Result TransitionLoop();
        bool HandleNewRequest();
        void AcceptRequest(Request* pRequest);
        void FinishRequest(Result result);
        void CleanupPendingRequests(Result result);
        void CleanupCurrentRequest(Result result);

        Request& GetAsyncRequestStorage();
        Result RequestChangeStateAsyncImpl(State stateTarget, bool bForce, Result resultCancel);
        bool PollState(State state, const TimeSpan& timeout);
        bool PollStateWithCancelEvent(State state, const TimeSpan& timeout, os::SystemEvent* pCancelEvent);
        bool PollStateSettle(const TimeSpan& initialWait, const TimeSpan& AdditionalWait, os::SystemEvent* pCancelEvent);

    private:
        typedef TypedMessageQueue<Request*, REQUEST_QUEUE_DEPTH> RequestQueue;
        nn::os::Mutex m_csAsyncLock;
        nn::os::Mutex m_csRequestLock;
        nn::os::Event m_eventRequest;
        nn::os::Event m_eventStateChange;
        RequestQueue  m_queueRequest;
        Request       m_AsyncRequest;
        Request*      m_pCurrentRequest;
        bgsu::MultiClientSystemEvent m_MultiClientSystemEvent;
        bool          m_bAutonomyEnabledFullAwake;
    };

    inline bool StateMachineThread::IsProcessingRequest() const
    {
        return m_pCurrentRequest != NULL;
    }

    inline bool StateMachineThread::IsProcessingForceRequest() const
    {
        return m_pCurrentRequest != NULL && IsUserDriven();
    }

    inline const char* StateMachineThread::GetCurrentStateString() const
    {
        return GetStateString(GetCurrentState());
    }

    inline StateMachineThread::Request& StateMachineThread::GetAsyncRequestStorage()
    {
        return m_AsyncRequest;
    }
}
}
