﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/bgtc.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_ResultMenu.h>
#include <nn/util/util_ScopeExit.h>

#include "npns_Common.h"
#include "npns_ClientThread.h"
#include "npns_StateMachineThread.h"

namespace nn {
namespace npns {

namespace {
#if NN_NPNS_ENABLE_ONESHOT_NETWORK
    Result SubmitAndWaitOnlineRequest(nifm::NetworkConnection* pNetworkConnection, os::SystemEvent * pEventInterrupt)
    {
        if (pNetworkConnection->IsAvailable())
        {
            return ResultSuccess();
        }

        os::SystemEvent& eventConnection = pNetworkConnection->GetSystemEvent();
        eventConnection.Clear();
        pNetworkConnection->SubmitRequest();

        int index = os::WaitAny(
            pNetworkConnection->GetSystemEvent().GetBase(),
            pEventInterrupt->GetBase()
        );
        switch (index)
        {
        case 0:
            eventConnection.Clear();
            if (!pNetworkConnection->IsAvailable())
            {
                return pNetworkConnection->GetResult();
            }
            break;

        case 1:
            // pEventInterrupt は他のイベントを流用しているのでクリアしてはいけない
            return ResultInterruptByRequest();

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return ResultSuccess();
    }
#endif
}

StateMachineThread::StateMachineThread(Controller& controller)
    : ThreadBase(NN_NPNS_THREAD_CONFIG(StateMachine))
    , StateMachine(controller, m_eventRequest)
    , m_csAsyncLock(true)
    , m_csRequestLock(true)
    , m_eventRequest(os::EventClearMode_AutoClear)
    , m_eventStateChange(os::EventClearMode_AutoClear)
    , m_AsyncRequest(State_Initial, ResultUnexpected())
    , m_pCurrentRequest(nullptr)
{
}

StateMachineThread::~StateMachineThread()
{
    // Finalize 確認
    NN_SDK_ASSERT(!IsAlive(), "");
}

Result StateMachineThread::Initialize()
{

    Result result;

    result = StateMachine::Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = ThreadBase::Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    // 非同期用のリクエストバッファを明示的に使用済みにする
    m_AsyncRequest.ReturnResult(Result());

    return ResultSuccess();
}

void StateMachineThread::Finalize()
{
    ThreadBase::Finalize();

    StateMachine::Finalize();
}

bool StateMachineThread::HandleNewRequest()
{
    std::lock_guard<os::Mutex> locker(m_csRequestLock);

    Request* pRequest = NULL;

    m_eventRequest.Clear();
    if (!m_queueRequest.TryReceive(&pRequest))
    {
        // 新しいリクエストなし、ステート遷移処理へ
        return false;
    }

    if (pRequest->IsFinished())
    {
        // 使用済みのリクエストブロック。おそらくリクエストが処理する前にキャンセルされた
        NN_NPNS_INFO("Already cancelled [%s](%p).\n", GetStateString(pRequest->GetTargetState()), pRequest);
        return true;
    }

    NN_NPNS_TRACE("Received new request [%s](%p).\n", GetStateString(pRequest->GetTargetState()), pRequest);
    if (IsProcessingForceRequest() && !pRequest->IsUserDriven())
    {
        // 処理中のリクエストよりもこのリクエストは優先度が低いので拒否する
        NN_NPNS_INFO("Forced request is processing.\n");
        pRequest->ReturnResult(ResultProcessingPriorityRequest());

        // リクエストを処理したことを返す
        return true;
    }

    if (IsProcessingRequest())
    {
        // 既に処理中のリクエストがあるならそれをキャンセル
        NN_NPNS_INFO("Canceling previous request [%s](%p).\n",
            m_pCurrentRequest ? GetStateString(m_pCurrentRequest->GetTargetState()) : "none",
            m_pCurrentRequest);
        CancelRequest(pRequest->GetCancelResult());
    }

    if (!IsInProgress()     // これからステートが変わる可能性がある？
        && GetCurrentState() == pRequest->GetTargetState())
    {
        // 既に要求されたステート
        NN_NPNS_INFO("Already current state [%s](%p).\n", GetStateString(pRequest->GetTargetState()), pRequest);
        pRequest->ReturnResult(ResultSuccess());
        OnTransitionComplete();

        // リクエストを処理したことを返す
        return true;
    }

    // リクエストを受け付ける
    AcceptRequest(pRequest);

    // リクエストを受け付けたことを返す
    return true;
}

void StateMachineThread::AcceptRequest(Request* pRequest)
{
    std::lock_guard<os::Mutex> locker(m_csRequestLock);

    NN_SDK_ASSERT(m_pCurrentRequest == NULL || m_pCurrentRequest == pRequest);
    NN_SDK_ASSERT(nn::os::GetThreadId(nn::os::GetCurrentThread()) == GetThreadId());
    m_pCurrentRequest = pRequest;

    SetTargetState(pRequest->GetTargetState(), pRequest->IsUserDriven());
    NN_NPNS_TRACE("AcceptRequest: accepted request(%p).\n", m_pCurrentRequest);
}

void StateMachineThread::FinishRequest(Result result)
{
    std::lock_guard<os::Mutex> locker(m_csRequestLock);
    if (m_pCurrentRequest)
    {
        NN_NPNS_TRACE("FinishRequest: returning result to current request(%p)...\n", m_pCurrentRequest);
        // 処理中のリクエストを一回空にしてから呼び出し元に結果を返す
        Request* pRequest = m_pCurrentRequest;
        m_pCurrentRequest = NULL;
        pRequest->ReturnResult(result);
    }
}

void StateMachineThread::CleanupPendingRequests(Result result)
{
    std::lock_guard<os::Mutex> locker(m_csRequestLock);

    Request* pRequest = NULL;
    while (m_queueRequest.TryReceive(&pRequest) && pRequest)
    {
        NN_NPNS_TRACE("FinishRequest: returning result to request(%p)...\n", pRequest);
        pRequest->ReturnResult(result);
    }
}

void StateMachineThread::CleanupCurrentRequest(Result result)
{
    std::lock_guard<os::Mutex> locker(m_csRequestLock);

    if (m_pCurrentRequest)
    {
        FinishRequest(result);
    }
}

void StateMachineThread::ThreadBody()
{
    Result      result;

    // すべての初期化が終わったので、ステート変更リクエストのディスパッチ開始
    NN_NPNS_INFO("StateMachineThread ready. (autonomy = %s)\n", IsAutonomyEnabled() ? "enable" : "disable");
    while (!IsExiting())
    {
        // リクエストを処理したら true。キューが空なら false を返す
        if (HandleNewRequest())
        {
            continue;
        }

        result = TransitionLoop();
    }
    NN_NPNS_INFO("StateMachineThread exit.\n");
}

void StateMachineThread::RequestExit()
{
    ThreadBase::RequestExit();
    m_eventRequest.Signal();
}

Result StateMachineThread::TransitionLoop()
{
    Result result;

    result = TransitState();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    if (IsTransitionComplete() && m_queueRequest.IsEmpty() && m_pCurrentRequest == nullptr)
    {
        WaitAnyEvents();
    }

    return ResultSuccess();
}

Result StateMachineThread::SendRequest(Request& request, bool bAsync)
{
    NN_NPNS_TRACE("Sending request [%s](%p)\n", GetStateString(request.GetTargetState()), &request);

    // ステート変更要求をキューに積む
    m_queueRequest.Send(&request);
    // 同時待ちしている別のスレッドにもステート変更要求を通知
    m_eventRequest.Signal();

    if (bAsync)
    {
        // 非同期ならここで完了
        return ResultSuccess();
    }

    // ステートマシンのスレッドからは呼び出し不可
    NN_SDK_ASSERT(nn::os::GetThreadId(nn::os::GetCurrentThread()) != GetThreadId());

    // 同期ならリクエストの完了を待ち、実行結果を得る
    Result result = request.Wait();
    if (result.IsFailure())
    {
        NN_NPNS_INFO("RequestChangeState(stateTarget = [%s], bForce = %s) failed.\n",
            GetStateString(request.GetTargetState()), request.IsUserDriven() ? "true" : "false");
#ifdef NN_NPNS_INFO_ENABLE
        NN_DBG_TPRINT_RESULT_(result);
#endif
    }
    return result;
}


Result StateMachineThread::RequestChangeState(State stateTarget, Result resultCancel)
{
    Request request(stateTarget, false, resultCancel);
    return SendRequest(request);
}

Result StateMachineThread::RequestChangeStateAsync(State stateTarget, Result resultCancel)
{
    return RequestChangeStateAsyncImpl(stateTarget, false, resultCancel);
}

Result StateMachineThread::RequestChangeStateForceAsync(State stateTarget, Result resultCancel)
{
    return RequestChangeStateAsyncImpl(stateTarget, true, resultCancel);
}

Result StateMachineThread::RequestChangeStateForceTimed(State stateTarget, Result resultCancel, const TimeSpan & timeout, os::SystemEvent * pCancelEvent)
{
    Result result;
    Request request(stateTarget, false, resultCancel);
    {
        std::lock_guard<os::Mutex> locker(m_csRequestLock);

        CleanupPendingRequests(ResultInterruptByRequest());
        CleanupCurrentRequest(ResultInterruptByRequest());

        result = SendRequest(request, true);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
    }

    int index;
    if (pCancelEvent)
    {
        index = os::TimedWaitAny(
            timeout,
            request.GetEvent().GetBase(),
            pCancelEvent->GetBase()
        );
    }
    else
    {
        index = os::TimedWaitAny(
            timeout,
            request.GetEvent().GetBase()
        );
    }
    switch (index)
    {
    case  0: // request.GetEvent()
        return request.GetResult();

    case  1: // pCancelEvent
        result = request.GetCancelResult();
        break;

    case -1: // timeout
        result = ResultRequestTimeout();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    {
        std::lock_guard<os::Mutex> locker(m_csRequestLock);

        CleanupPendingRequests(result);
        CleanupCurrentRequest(result);

        NN_SDK_ASSERT(request.IsFinished());
    }
    return result;
}

Result StateMachineThread::RequestChangeStateAsyncImpl(State stateTarget, bool bForce, Result resultCancel)
{
    std::lock_guard<os::Mutex> locker(m_csAsyncLock);

    // 非同期リクエスト用のバッファを取得
    Request& request = GetAsyncRequestStorage();

    if (bForce)
    {
        CleanupPendingRequests(ResultInterruptByRequest());
        CleanupCurrentRequest(ResultInterruptByRequest());
        NN_SDK_ASSERT(m_pCurrentRequest == nullptr);
    }
    else if (!request.IsFinished())
    {
        return ResultProcessingPriorityRequest();
    }

    request = Request(stateTarget, bForce, resultCancel);

    // 実行完了は待たない
    return SendRequest(request, true);
}

bool StateMachineThread::PollState(State state, const TimeSpan & timeout)
{
    return PollStateWithCancelEvent(state, timeout, nullptr);
}

bool StateMachineThread::PollStateWithCancelEvent(State state, const TimeSpan & timeout, os::SystemEvent* pCancelEvent)
{
    NN_NPNS_INFO("PollStateWithCancelEvent: %s %dsec\n", GetStateString(state), timeout.GetSeconds());
    os::Tick tickStart = os::GetSystemTick();
    while (GetCurrentState() != state)
    {
        os::Tick tickWait = os::Tick(timeout) - (os::GetSystemTick() - tickStart);
        if (tickWait.GetInt64Value() <= 0)
        {
            NN_NPNS_INFO("PollStateWithCancelEvent: timeout\n");
            break;
        }

        if (pCancelEvent)
        {
            int index = os::TimedWaitAny(
                tickWait.ToTimeSpan(),
                m_eventStateChange.GetBase(),
                pCancelEvent->GetBase()
            );
            switch (index)
            {
            case 0:
                m_eventStateChange.Clear();
                break;

            case 1:
                NN_NPNS_INFO("PollStateWithCancelEvent: cancelled\n");
                return false;

            case -1:
                // timeout
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            m_eventStateChange.TimedWait(tickWait.ToTimeSpan());
        }
    }
    return GetCurrentState() == state;
}

bool StateMachineThread::PollStateSettle(const TimeSpan & initialWait, const TimeSpan & additionalWait, os::SystemEvent * pCancelEvent)
{
    // pCancelEvent のシグナル状態をキープする
    int index = os::TimedWaitAny(
        initialWait,
        pCancelEvent->GetBase()
    );
    if (index == 0)
    {
        NN_NPNS_INFO("PollStateSettle: cancelled\n");
        return false;
    }
    return PollStateWithCancelEvent(GetTargetState(), additionalWait, pCancelEvent);
}

void StateMachineThread::CancelRequest(Result result)
{
    OnTransitionFailure(result);
}

void StateMachineThread::OnTransitionComplete()
{
    NN_SDK_ASSERT(nn::os::GetThreadId(nn::os::GetCurrentThread()) == GetThreadId());

    StateMachine::OnTransitionComplete();

    // リクエストの処理完了
    FinishRequest(ResultSuccess());

    if (GetCurrentState() == State_Exit)
    {
        RequestExit();
    }
}

bool StateMachineThread::OnTransitionFailure(Result result)
{
    bool bResult = StateMachine::OnTransitionFailure(result);
    if (bResult)
    {
        return true;
    }

    // リクエストの処理失敗
    FinishRequest(result);
    return false;
}

void StateMachineThread::OnEntered(State state)
{
    m_eventStateChange.Signal();
    m_MultiClientSystemEvent.Signal();
}

void StateMachineThread::OnFullAwakeLeave()
{
    m_bAutonomyEnabledFullAwake = SetAutonomyEnabled(false);
}

void StateMachineThread::OnFullAwakeEnter()
{
    SetAutonomyEnabled(m_bAutonomyEnabledFullAwake);
    RequestChangeStateAsync(State_Connected);
    g_Daemon.GetClientThread().ActivateKeepAlive();
    g_Daemon.GetClientThread().TriggerCheckConnection();
}

void StateMachineThread::OnSleepEnter(bool bKeepSession)
{
    // スリープ遷移時には必ず自律動作を無効にする
    (void)SetAutonomyEnabled(false);

    Result result;
    State stateTarget = DecideTargetStateOnSleepEnter(GetCurrentState(), bKeepSession);
    if (GetCurrentState() != stateTarget)
    {
        result = RequestChangeStateForceTimed(
            stateTarget,
            ResultCanceledByHardwareEvents(),
            TimeSpan::FromMilliSeconds(NN_NPNS_TIMEOUT_MS_SLEEP_IN)
        );
    }
    else
    {
        result = ResultSuccess();
    }

    if (result.IsSuccess())
    {
        if (stateTarget == State_ConnectedOnHalfAwake)
        {
            g_Daemon.GetClientThread().DeactivateKeepAlive();
        }
    }
    else if (ResultRequestTimeout::Includes(result))
    {
        NN_NPNS_WARN("State change for sleep is timeout. (stale network connection?)(timeout %d ms)\n", NN_NPNS_TIMEOUT_MS_SLEEP_IN);
        if (stateTarget == State_ConnectedOnHalfAwake)
        {
            NN_NPNS_RECORD_EVENT(SleepStateConnectedOnHalfawakeTimeout);
        }
        else
        {
            NN_NPNS_RECORD_EVENT(SleepStateSuspendTimeout);
        }
    }
}

void StateMachineThread::OnShutdown()
{
    RequestChangeStateForceAsync(State_Exit, ResultCanceledByHardwareEvents());
    if (!PollState(State_Exit, TimeSpan::FromSeconds(1)))
    {
        NN_NPNS_WARN("State change for shutdown is timeout. (stale network connection?)\n");
    }
}

void StateMachineThread::OnHalfAwakeEnter(os::SystemEvent* pEventCancel, bool bPerformPingPong, bool* pIsRetryProcessed)
{
    const TimeSpan timeout = GetSleepProcessingTimeout();
    NN_NPNS_INFO("Halfawake processing is started. (timeout: %dsec, autonomy = %s)\n", timeout.GetSeconds(), IsAutonomyEnabled() ? "enable" : "disable");
    if (pIsRetryProcessed)
    {
        *pIsRetryProcessed = false;
    }

    // NPNS ではステートマシンの状態と BGTC にアドバタイズするタスク状態は一致しない。
    Result result = bgtc::NotifyTaskStarting();
    if (result.IsFailure())
    {
        NN_NPNS_WARN("NotifyTaskStarting failed.(0x%08x)\n", result.GetInnerValueForDebug());
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        bgtc::NotifyTaskFinished();
    };

    bool bNeedConfirmation = false;
#if NN_NPNS_ENABLE_WOWLAN
    // 切断検出のコマンドを明示的に発行してサーバの応答を待つ
    if (GetCurrentState() == State_ConnectedOnHalfAwake)
    {
        ClientThread& client = g_Daemon.GetClientThread();
        if (bPerformPingPong)
        {
            // サーバから明示的な応答を期待するので確実に切断を検出できる
            result = client.CheckConnectionAndWait(pEventCancel, TimeSpan::FromSeconds(2));
            if (result.IsFailure())
            {
                NN_NPNS_WARN("Failed to receive pong from server.(0x%08x)\n", result.GetInnerValueForDebug());
                bNeedConfirmation = true;

                // OneShotRequest が存在しない状態で Online にしてはいけない（ステートマシンで無限待ちに入る）
                RequestChangeStateForceTimed(State_Suspend, ResultCanceledByOtherRequest(), TimeSpan::FromSeconds(5), pEventCancel);

                NN_NPNS_RECORD_EVENT(PingPongFailure);
            }
            else
            {
                NN_NPNS_RECORD_EVENT(PingPongSuccess);
            }
        }
        else
        {
            // サーバが応答を返さない分軽いが、失敗時の RST 受信のみに頼るので WAN 側の切断は検出できない
            g_Daemon.GetClientThread().TriggerCheckConnection();
        }
    }

    // 切断検出のためステートマシンの状態が落ち着くまで待つ
    NN_NPNS_INFO("Wait for StateMachine to settle...\n");
    if (!PollStateSettle(TimeSpan::FromMilliSeconds(500), TimeSpan::FromMilliSeconds(ClientThread::DisconnectionTimeOutInMs + 2000), pEventCancel))
    {
        NN_NPNS_WARN("Failed to wait for StateMachine to settle. Refreshing connection may be fail.\n");
    }
#endif

    State stateBefore = GetCurrentState();

#if NN_NPNS_ENABLE_ONESHOT_NETWORK
    NN_NPNS_INFO("Submit a oneshot request...\n");
    // Persistent な接続要求をステートマシン側で出す前に、ワンショットの接続要求で接続を準備しておく
    // （タイムアウトを NIFM 側に管理させることで、接続設定が存在しない場合などに無駄な待ちを防ぐ）
    nifm::NetworkConnection scopedOneshotConnection;
    nifm::SetRequestRequirementPreset(scopedOneshotConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable);

    if (bNeedConfirmation)
    {
        nifm::SetRequestConnectionConfirmationOption(scopedOneshotConnection.GetRequestHandle(), nifm::ConnectionConfirmationOption_Forced);
        nifm::SetRequestRawPriority(scopedOneshotConnection.GetRequestHandle(), 0);
    }

    result = SubmitAndWaitOnlineRequest(&scopedOneshotConnection, pEventCancel);
    if (result.IsFailure())
    {
        NN_NPNS_WARN("SubmitAndWaitOnlineRequest failed.(0x%08x)\n", result.GetInnerValueForDebug());
        return;
    }
    else
    {
        NN_NPNS_INFO("SubmitAndWaitOnlineRequest successful. Starting periodic tasks\n");
        if (bNeedConfirmation)
        {
            NN_NPNS_RECORD_EVENT(PingPongFailureThenConfirmationPass);
        }
    }
#endif

    result = RequestChangeStateForceTimed(State_ConnectedOnHalfAwake, ResultCanceledByHardwareEvents(), timeout, pEventCancel);
    if (result.IsSuccess())
    {
        if (stateBefore != State_ConnectedOnHalfAwake && pIsRetryProcessed)
        {
            // この半起床によって接続が復旧された
            *pIsRetryProcessed = true;
        }

        // リクエスト処理中に全起床に遷移していないか念のため確認
        if (bgtc::IsInHalfAwake())
        {
            if (bgtc::WillStayHalfAwakeInsteadSleep())
            {
                // 半起床滞留有りなら全起床時と同じ自律動作モードにする
                SetAutonomyEnabled(m_bAutonomyEnabledFullAwake);
            }
            else
            {
                // 有線 → 無線や電池残量増では FullAwake を経由するのでこの処理は不要なはず
                // SetAutonomyEnabled(false);
            }
        }
    }
    else if (ResultRequestTimeout::Includes(result))
    {
        NN_NPNS_WARN("State change for halfawake is timeout.\n");
    }
    else if (ResultNoMessageArrived::Includes(result))
    {
        NN_NPNS_INFO("No message is arrived this time.\n");
    }
    else if (result.IsFailure())
    {
        NN_NPNS_WARN("Periodical arrival check failed.(0x%08x)\n", result.GetInnerValueForDebug());
    }
    // bgtc にはタスク終了を通知するものの、接続していた場合は接続を維持する
}

State StateMachineThread::DecideTargetStateOnSleepEnter(State stateCurrent, bool bKeepSession)
{
    if (bKeepSession)
    {
        switch (stateCurrent)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            return State_ConnectedOnHalfAwake;

        default:
            return State_Suspend;
        }
    }
    else
    {
         return State_Suspend;
    }
}

void StateMachineThread::AttachStateChangeEvent(bgsu::MultiClientSystemEvent::Node * pNode)
{
    m_MultiClientSystemEvent.Attach(pNode);
}

void StateMachineThread::DetachStateChangeEvent(bgsu::MultiClientSystemEvent::Node * pNode)
{
    m_MultiClientSystemEvent.Detach(pNode);
}

}
}

