﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/http.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>

#include "npns_DeviceTokenConsumer.h"
#include "npns_DeviceTokenHolder.h"
#include "npns_JidConsumer.h"
#include "npns_NotificationTokenConsumer.h"
#include "npns_ArrivalConsumer.h"
#include "npns_PersistentData.h"
#include "npns_SaveDataManager.h"
#include "npns_Client.h"

namespace nn{ namespace npns{

class ClientThread;
class Controller
{
    NN_DISALLOW_COPY(Controller);
    NN_DISALLOW_MOVE(Controller);
public:
    static const uint64_t NdasClientId = 0x83b72b05dc3278d7;

    NN_IMPLICIT Controller(ClientThread& client);
    ~Controller();

    Result Initialize();
    void Finalize();

    Result XmppConnect();
    void XmppEnableWhiteSpacePing(os::Event* pEventInterrupt, bool isPowerModeNormal);
    void XmppSetupConnectionForSleep();
    void XmppDisconnect();
    bool XmppIsConnected();
    bool XmppIsConnectionInProgress();
    bool XmppIsUsingProxy();
    Result XmppGetLastError() const;
    void XmppSetStatus(Client::Status status);

    Result RequestJid();
    Result DestroyJid();
    Result SetJid(const Credential& credential);
    Result GetJid(Credential* pCredential);
    Result ClearJid();
    void InvalidateJid();
    Result CreateNotificationToken(NotificationToken* pOutToken, const ReceiverId& uid, ApplicationId applicationId);
    Result DestroyNotificationToken(const ReceiverId& uid, ApplicationId applicationId);
    Result CheckMessageArrived(bool* pIsArrived, os::Event* pEventInterrupt);
    void RequestPlayReportSubmissionIfRequired();
    os::NativeHandle GetPlayReportRequestEventReadableHandle();

    bool HasValidJid();
    Result WaitForOnline(os::Event* pEventInterrupt, bool bEnableTimeout);
    Result WaitAnyEvents(os::Event* pEventInterrupt, const TimeSpan& time);
    Result WaitAnyEvents(os::Event* pEventInterrupt);

    void SubmitOnlineRequest();
    void CancelOnlineRequest();
    bool IsOnline();

    Result FlushSaveData();
    const char* GetCurrentEnvironment();

    TimeSpan GetBackoffWaitFromServer() const;
    static bool IsEnabledRealTimePushDuringSleep();

    inline int32_t GetLastBrokerCheckingIntervalFromServer() const
    {
        return m_LastBrokerCheckingIntervalFromServer;
    }
    inline int32_t GetLastKeepSessionTryCountFromServer() const
    {
        return m_LastKeepSessionTryCountFromServer;
    }

private:
    enum ActivityMode
    {
        ActivityMode_SystemDefault   = 0,
        ActivityMode_Periodical      = 1,
        ActivityMode_RealTime        = 2
    };

    nifm::NetworkConnection& GetNetworkConnection()
    {
        return *reinterpret_cast<nifm::NetworkConnection*>(&m_pContinuousNetworkConnection);
    }

    Result EnsureDeviceToken();

    os::Event m_eventIntrrupt;
    os::SystemEvent m_eventPlayReportRequest;
    os::Tick m_tickLastRequestPlayReport;
    os::Mutex m_mutexJid;
    ClientThread& m_Client;
    typedef util::TypedStorage<nifm::NetworkConnection, sizeof(nifm::NetworkConnection), sizeof(void*)> NetworkConnectionStorage;
    NetworkConnectionStorage m_pContinuousNetworkConnection;
    http::ConnectionBroker m_ConnectionBroker;

    DeviceTokenHolder m_DeviceTokenHolder;
    DeviceTokenConsumer m_DeviceTokenConsumer;
    JidConsumer m_JidConsumer;
    NotificationTokenConsumer m_NotificationTokenConsumer;
    ArrivalConsumer m_ArrivalConsumer;

    SaveDataManager m_SaveDataManager;
    PersistentData m_PersistentData;

    int32_t m_LastBrokerCheckingIntervalFromServer;
    int32_t m_LastKeepSessionTryCountFromServer;
};


}}


