﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>

#include "npns_Client.h"
#include "npns_Config.h"
#include "npns_Common.h"
#include "npns_ClientSessionMaintainer.h"

namespace nn {
namespace npns {

ClientSessionMaintainer::ClientSessionMaintainer(Client* pClient)
    : m_Client(*pClient)
    , m_tickLastSent(0)
    , m_PingInterval(0)
    , m_eventPongReceive(os::EventClearMode_AutoClear)
    , m_PongReceiveCount(0)
{
}

void ClientSessionMaintainer::Initialize()
{
    m_PingInterval = 0;
    m_tickLastSent = os::GetSystemTick();

    m_Client.AddUserHandler(OnReceivePongHandlerEntry, nullptr, "pong", nullptr, this);
}

void ClientSessionMaintainer::SendWhiteSpacePingIfNeeded()
{
    if (!m_Client.IsConnected() || !IsWhiteSpacePingEnabled()
        || !IsTimeElapsed(m_PingInterval))
    {
        return;
    }

    m_Client.SendRawString(" ");
    m_tickLastSent = os::GetSystemTick();
}

void ClientSessionMaintainer::SendDoubleSpaceKeepAliveIfNeeded()
{
    if (!m_Client.IsConnected()
        || !IsTimeElapsed(TimeSpan::FromSeconds(60)))
    {
        return;
    }

    // 接続が失われていればサーバから RST が返って来て別途検出される
    m_Client.SendRawString("  " /* サーバ側の WSP 検出機能を回避するため 2 bytes */);
    m_tickLastSent = os::GetSystemTick();
}

void ClientSessionMaintainer::EnableWhiteSpacePing(TimeSpan interval)
{
    m_PingInterval = interval;
}

bool ClientSessionMaintainer::IsTimeElapsed(TimeSpan elapsed) const
{
    os::Tick tickDiff = os::GetSystemTick() - m_tickLastSent;
    if (0 <= tickDiff.GetInt64Value() && tickDiff.ToTimeSpan() < elapsed)
    {
        return false;
    }
    else
    {
        return true;
    }
}

Result ClientSessionMaintainer::PerformPingPong(os::SystemEvent * pCancelEvent, TimeSpan timeout)
{
    Result result;

    if (!m_Client.IsConnected())
    {
        return ResultNotConnected();
    }

    int previousPongReceiveCount = m_PongReceiveCount;
    m_eventPongReceive.Clear();

    SendPingAsync();

    int index = os::TimedWaitAny(
        timeout,
        m_eventPongReceive.GetBase(),
        pCancelEvent->GetBase()
    );
    switch (index)
    {
    case -1:
        return ResultCommandTimeOut();

    case 0:
        if (previousPongReceiveCount == m_PongReceiveCount)
        {
            return ResultNotReceived();
        }
        break;

    case 1:
        return ResultCanceledByOtherRequest();

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return ResultSuccess();
}

int ClientSessionMaintainer::OnReceivePongHandlerEntry(xmpp_conn_t * const, xmpp_stanza_t * const, void * const userdata)
{
    static_cast<ClientSessionMaintainer*>(userdata)->OnReceivePong();
    return Xmpp::HandlerResult_NotOneShot;
}

void ClientSessionMaintainer::SendPingAsync()
{
    m_Client.SendRawString("<ping/>");
}

void ClientSessionMaintainer::OnReceivePong()
{
    ++m_PongReceiveCount;
    m_eventPongReceive.Signal();
}

}
}
