﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/lbl/detail/lbl_Log.h>
#include <nn/lbl/impl/lbl.h>
#include <nn/lbl/server/lbl_LblHipcServer.h>

#include <nn/fatal/fatal_Api.h>

namespace {

const size_t LblServerStackSize = 4096;

NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_LblServerStack[LblServerStackSize];

nn::os::ThreadType g_LblServerThread;

void InitializeHipcServer() NN_NOEXCEPT
{
    // LBL ドライバライブラリ自体の Initialize
    nn::lbl::impl::Initialize();

    // LBL 用サーバの開始
    nn::lbl::server::StartServer();

    // LBL 用サーバ処理スレッドの作成
    auto result = nn::os::CreateThread(&g_LblServerThread, &nn::lbl::server::LoopServer,
        nullptr, g_LblServerStack, sizeof(g_LblServerStack), NN_SYSTEM_THREAD_PRIORITY(lbl, IpcServer));
    NN_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
    nn::os::SetThreadNamePointer(&g_LblServerThread, NN_SYSTEM_THREAD_NAME(lbl, IpcServer));
    nn::os::StartThread(&g_LblServerThread);

    // LBL 用サーバーのループ処理
    NN_DETAIL_LBL_TRACE("Start LBL server.\n");
}

void FinalizeHipcServer() NN_NOEXCEPT
{
    // サーバ終了をリクエスト

    // LBL サーバー
    nn::lbl::server::RequestStopServer();

    // LBL 用サーバ処理スレッド終了待機と破棄
    nn::os::WaitThread(&g_LblServerThread);
    nn::os::DestroyThread(&g_LblServerThread);

    // LBL ライブラリ自体の Finalize
    nn::lbl::impl::Finalize();

    // LBL サーバの終了処理
    nn::lbl::server::FinalizeServer();
}

//-----------------------------------------------------------------------------

}   // namespace

extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}

//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    InitializeHipcServer();

    // TORIAEZU:メインスレッドが終了しないように待機
    nn::os::WaitThread(&g_LblServerThread);

    // TORIAEZU:終了はしない
    NN_UNUSED(FinalizeHipcServer);
}

//-----------------------------------------------------------------------------
