﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "friends_PowerStateEventManager.h"

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/bgtc.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/friends/service/friends_Server.h>

namespace
{
    void NotifyBgtcEventTriggered(bool isInHalfAwake) NN_NOEXCEPT
    {
        nn::friends::service::NotifyBgtcEventTriggered(isInHalfAwake);
    }

    void NotifySystemFullAwaked() NN_NOEXCEPT
    {
        nn::friends::service::NotifySystemFullAwaked();
    }

    void NotifySystemMinimumAwaked(bool isWakeSequence) NN_NOEXCEPT
    {
        nn::friends::service::NotifySystemMinimumAwaked(isWakeSequence);
    }

    void NotifySystemSleepReady() NN_NOEXCEPT
    {
        nn::friends::service::NotifySystemSleepReady();
    }

    void NotifySystemShutdownReady() NN_NOEXCEPT
    {
        nn::friends::service::NotifySystemShutdownReady();
    }

    void WaitForSleepReady() NN_NOEXCEPT
    {
        nn::friends::service::WaitForSleepReady();
    }

    void WaitForShutdownReady() NN_NOEXCEPT
    {
        nn::friends::service::WaitForShutdownReady();
    }
}

void PowerStateEventManager::LoopAuto() NN_NOEXCEPT
{
    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Bgtc};

    nn::psc::PmModule module;
    NN_ABORT_UNLESS_RESULT_SUCCESS(module.Initialize(nn::psc::PmModuleId_Friends,
        dependencies, NN_ARRAY_SIZE(dependencies), nn::os::EventClearMode_ManualClear));

    NN_UTIL_SCOPE_EXIT
    {
        module.Finalize();
    };

    nn::os::SystemEvent& bgtcTriggerEvent = nn::bgtc::GetTriggerEvent();

    nn::psc::PmState prevState = nn::psc::PmState_FullAwake;

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(bgtcTriggerEvent.GetBase(), module.GetEventPointer()->GetBase());

        if (bgtcTriggerEvent.TryWait())
        {
            bgtcTriggerEvent.Clear();

            NotifyBgtcEventTriggered(nn::bgtc::IsInHalfAwake());
        }
        if (module.GetEventPointer()->TryWait())
        {
            module.GetEventPointer()->Clear();

            nn::psc::PmState state;
            nn::psc::PmFlagSet flags;
            NN_ABORT_UNLESS_RESULT_SUCCESS(module.GetRequest(&state, &flags));

            switch (state)
            {
            case nn::psc::PmState_MinimumAwake:
                {
                    // FullAwake ⇒ MinimumAwake の遷移はスリープシーケンスである。
                    if (prevState == nn::psc::PmState_FullAwake)
                    {
                        NotifySystemMinimumAwaked(false);
                    }
                }
                break;
            case nn::psc::PmState_SleepReady:
                {
                    NotifySystemSleepReady();
                    WaitForSleepReady();
                }
                break;
            case nn::psc::PmState_ShutdownReady:
                {
                    NotifySystemShutdownReady();
                    WaitForShutdownReady();
                }
                break;
            default:
                break;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(module.Acknowledge(state, nn::ResultSuccess()));

            // システムの起床をより速くするため、起床系のイベントは Acknowledge より後で通知する。
            switch (state)
            {
            case nn::psc::PmState_FullAwake:
                {
                    NotifySystemFullAwaked();
                }
                break;
            case nn::psc::PmState_MinimumAwake:
                {
                    // EssentialServicesAwake ⇒ MinimumAwake の遷移は起床シーケンスである。
                    if (prevState == nn::psc::PmState_EssentialServicesAwake)
                    {
                        NotifySystemMinimumAwaked(true);
                    }
                }
                break;
            default:
                break;
            }

            prevState = state;
        }
    }
}
