﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/dbg/dbg_Api.h>
#include <nn/dbg/dbg_Result.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/osdbg.h>

#include <nn/fs/fs_Directory.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/ncm/ncm_ProgramLocation.h>
#include <algorithm>
#include <cstring>
#include <nn/ldr/ldr_DebugMonitorApi.h>

#include "dmnt_Server.h"
#include <nn/dmnt/dmnt_ThreadData.h>
#include <nn/dmnt/dmnt_Result.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/svc/svc_Result.h>
//==============================================================================

//#define TRACE_RESULT( result ) if( result.IsFailure() ) { NN_SDK_LOG( "File %s Line %d Module %d Description %d\n", __FILE__, __LINE__, result.GetModule(), result.GetDescription() ); }
#define TRACE_RESULT( result ) (void)result

//==============================================================================

namespace
{
bool                        g_LocationResolverInitialized = false;
nn::lr::LocationResolver    g_LocationResolver;

// MultiCoreThread data objects

bool                        g_MultiCoreThreadInitialized = false;

NN_ALIGNAS(nn::os::StackRegionAlignment)
nn::Bit8                    g_MultiCoreStack[ 16 * 1024 ];
nn::os::ThreadType          g_MultiCoreThread;

nn::os::MessageQueueType    g_MultiCoreRequestMQ;
nn::os::MessageQueueType    g_MultiCoreResultMQ;

uintptr_t                   g_MultiCoreRequestQueue[ 2 ];
uintptr_t                   g_MultiCoreResultQueue[  2 ];
}

//==============================================================================

void Server::Initialize() NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::Initialize\n" );
}

//==============================================================================

nn::Result Server::BreakDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::BreakDebugProcess\n" );

    nn::Result result = nn::dbg::BreakDebugProcess( debug );
    TRACE_RESULT( result );

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Server::TerminateDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::TerminateDebugProcess\n" );

    nn::Result result = nn::dbg::TerminateDebugProcess( debug );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::CloseHandle(nn::svc::Handle debug) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::CloseHandle\n" );

    nn::Result result = nn::dbg::CloseHandle( debug );
    TRACE_RESULT( result );

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Server::LoadImage( nn::sf::Out<uint32_t> outHandle, const nn::sf::InBuffer& FileName, const nn::sf::InBuffer& Args ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::LoadImage\n" );

    const char* pFileName = FileName.GetPointerUnsafe();
    const char* pArgs     = Args    .GetPointerUnsafe();

//    NN_SDK_LOG( "FileName: '%s'\n", pFileName );
//    NN_SDK_LOG( "Args    : '%s'\n", pArgs     );

    if( !g_LocationResolverInitialized )
    {
        nn::Result result;

        // NN_ABORT_UNLESS(nn::pm::InitializeForDebugMonitor().IsSuccess());
        nn::lr::Initialize();
        NN_ABORT_UNLESS(nn::lr::OpenLocationResolver(&g_LocationResolver, nn::ncm::StorageId::Host).IsSuccess());

        g_LocationResolverInitialized = true;
    }

    nn::lr::Path path;
    std::memset( path.string, 0, sizeof(path.string) );

    const char HostMountName[] = "@Host:/";
    const char Fs0HostArchiveName[] = "host:";

    if( std::strncmp(pFileName, Fs0HostArchiveName, std::strlen(Fs0HostArchiveName)) != 0 )
    {
        std::memcpy( path.string, HostMountName, std::strlen(HostMountName) );
    }
    std::memcpy( path.string + std::strlen(path.string), pFileName, std::min(strlen(pFileName), sizeof(path.string) - std::strlen(path.string) - 1) );

    // Setup the arguments.
    size_t argumentSize = strlen(pArgs);

    // Start the process.
    nn::svc::Handle hProcess;
    nn::Result result = nn::dbg::DebugNewProcessFromHost(&hProcess, path.string, pArgs, argumentSize );

    // Make sure we can debug
    if (result.IsFailure())
    {
        if (result.CanAccept(nn::dbg::ResultCannotDebug()))
        {
            return nn::dmnt::ResultDebuggingDisabledForProcess();
        }
        return result;
    }

    *outHandle = nnHandle(hProcess).value;
    return result;
}

//==============================================================================

nn::Result Server::AttachByProgramId( nn::sf::Out<uint32_t> outHandle, nn::sf::Out<nn::Bit64> pPID, nn::Bit64 processId ) NN_NOEXCEPT
{
    nn::svc::Handle hProcess;
    nn::ncm::ProgramId attachProgramId;
    attachProgramId.value = processId;

    nn::Result result = ( attachProgramId == nn::ncm::ProgramId::GetInvalidId() ) ?
        nn::dbg::DebugActiveApplication( &hProcess ) :
        nn::dbg::DebugActiveProcess( &hProcess, attachProgramId );

    if( result.IsSuccess() )
    {
        nn::Bit64 pid;
        result = nn::dbg::GetProcessId( &pid, hProcess );
        pPID.Set( pid );
        *outHandle = nnHandle(hProcess).value;
    }

    return result;
}

//==============================================================================

nn::Result Server::AttachOnLaunch( nn::sf::Out<uint32_t> outHandle, nn::sf::Out<nn::Bit64> pPID, nn::Bit64 processId ) NN_NOEXCEPT
{
    nn::svc::Handle hProcess;
    nn::ncm::ProgramId attachProgramId;
    attachProgramId.value = processId;

    nn::Result result = ( attachProgramId == nn::ncm::ProgramId::GetInvalidId() ) ?
                            nn::dbg::DebugNextApplicationProcess( &hProcess ) :
                            nn::dbg::DebugNextProcess( &hProcess, attachProgramId );

    if( result.IsSuccess() )
    {
        nn::Bit64 pid;
        result = nn::dbg::GetProcessId( &pid, hProcess );
        pPID.Set( pid );
    }

    *outHandle = nnHandle(hProcess).value;
    return result;
}

//==============================================================================

nn::Result Server::GetDebugMonitorProcessId( nn::sf::Out<nn::Bit64> out ) NN_NOEXCEPT
{
    nn::Bit64 pid;
    nn::Result result = nn::dbg::GetCurrentProcessId( &pid );
    out.Set( pid );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetProcessId( nn::sf::Out<nn::Bit64> out, nn::svc::Handle process ) NN_NOEXCEPT
{
    nn::Bit64 pid;
    nn::Result result = nn::dbg::GetProcessId( &pid, process );
    out.Set( pid );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetProcessHandle( nn::sf::Out<uint32_t> outHandle, nn::Bit64 processId) NN_NOEXCEPT
{
    nn::svc::Handle Out;
    nn::Result result = nn::dbg::DebugActiveProcess( &Out, processId );
    *outHandle = nnHandle(Out).value;
    return result;
}

//==============================================================================

nn::Result Server::GetProcessList( nn::sf::Out<int32_t> pNumProcesses, const nn::sf::OutBuffer& outProcessIds ) NN_NOEXCEPT
{
    nn::Bit64* pProcessIds = (nn::Bit64*)outProcessIds.GetPointerUnsafe();
    int32_t NumProcesses = 0;
    int32_t arraySize = outProcessIds.GetSize() / sizeof(nn::Bit64);
    nn::Result result = nn::dbg::GetProcessList( &NumProcesses, pProcessIds, arraySize );
    pNumProcesses.Set( NumProcesses );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::WaitSynchronization( nn::svc::Handle handle, nn::Bit64 ns ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::WaitSynchronization\n" );

    int32_t iHandle = -1;
    nn::Result result = nn::dbg::WaitSynchronization( &iHandle, &handle, 1, ns );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetDebugEvent( const nn::sf::OutBuffer& pEvent, nn::svc::Handle process ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::GetDebugEvent\n" );

    char* pData = pEvent.GetPointerUnsafe();
    nn::Result result = nn::dbg::GetDebugEvent( (nn::svc::DebugEventInfo*)pData, process );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetProcessModuleInfo( nn::sf::Out<int> outCount, const nn::sf::OutBuffer& outModules, int num, nn::os::ProcessId pid ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::GetProcessModuleInfo\n" );

    nn::dbg::ModuleInfo* pModules = (nn::dbg::ModuleInfo*)outModules.GetPointerUnsafe();
    int nModules = 0;
    nn::Result result = nn::dbg::GetProcessModuleInfo( &nModules, pModules, num, pid );
    *outCount = nModules;
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetThreadList( nn::sf::Out<int32_t> pNumThreads, const nn::sf::OutBuffer& outThreads, nn::svc::Handle domain ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::GetThreadList\n" );

    char* pData = outThreads.GetPointerUnsafe();
    int32_t nThreads = 0;
    int32_t arraySize = outThreads.GetSize() / sizeof(nn::Bit64);
    nn::Result result = nn::dbg::GetThreadList( &nThreads, (nn::Bit64*)pData, arraySize, domain );
    pNumThreads.Set( nThreads );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetDebugThreadContext( const nn::sf::OutBuffer& outContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::GetDebugThreadContext\n" );

    nn::svc::ThreadContext* pContext = (nn::svc::ThreadContext*)outContext.GetPointerUnsafe();
    nn::Result result = nn::dbg::GetDebugThreadContext( pContext, debug, threadId, controlFlags );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetAllDebugThreadInfo( nn::svc::Handle debug, const nn::sf::OutBuffer& outData, const nn::sf::InBuffer& threadInfo, nn::Bit64 arraySize ) NN_NOEXCEPT
{
    nn::Bit64 r1 = 0;
    nn::Bit32 r2 = 0;
    int32_t nThreads = 0;
    nn::Bit64 pThreadIds[arraySize];
    nn::svc::ThreadContext nnThreadContext;
    nn::Result result = nn::dbg::GetThreadList( &nThreads, (nn::Bit64*)pThreadIds, arraySize, debug );
    nn::dmnt::ThreadData* pData = (nn::dmnt::ThreadData*)outData.GetPointerUnsafe();
    nn::osdbg::ThreadInfo* pInfo = (nn::osdbg::ThreadInfo*)threadInfo.GetPointerUnsafe();
    while( nThreads-- )
    {
        nn::dmnt::ThreadData Thread = pData[nThreads];
        Thread.m_ThreadId = pThreadIds[nThreads];
        nn::Result result = nn::dbg::GetDebugThreadContext( &nnThreadContext, debug, Thread.m_ThreadId, nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl );

        if( result.IsSuccess() )
        {
            Thread.m_IP = nnThreadContext.pc;
            Thread.m_SP = nnThreadContext.sp;
            Thread.m_Context = nnThreadContext;
        }
        else
        {
            memset( &pData[nThreads], 0, sizeof( nn::dmnt::ThreadData ) );
            Thread.m_ThreadId = pThreadIds[nThreads];

            if( result.CanAccept( nn::svc::ResultTerminateRequested() ) == false )
            {
                NN_SDK_LOG( "[dmnt] GetAllDebugThreadInfo:  dbg::GetDebugThreadContext FAILED for thread %d:  Error = 0x%x\n", Thread.m_ThreadId, result.GetInnerValueForDebug() );
            }
            else
            {
                // This thread is dead.
                Thread.m_Status = 'M';
            }
            continue;
        }
        Thread.m_Priority = nn::os::DefaultThreadPriority;
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debug, Thread.m_ThreadId, nn::svc::DebugThreadParam_Priority );
        if( result.IsSuccess() )
        {
            //Cleanse this for public consumption.
            Thread.m_Priority = r2 - 28;
        }

        Thread.m_Status = 'S'; //Default to runnable suspended
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debug, Thread.m_ThreadId, nn::svc::DebugThreadParam_State );
        if( result.IsSuccess() )
        {
            switch( r2 )
            {
                default:
                case nn::svc::ThreadState::ThreadState_Runnable:
                    Thread.m_Status = ((r1 & nn::svc::ThreadSuspend_Debug)? 'S': 'R');
                    break;
                case nn::svc::ThreadState::ThreadState_Wait:
                    Thread.m_Status = 'W';
                    break;
                case nn::svc::ThreadState::ThreadState_Terminated:
                    Thread.m_Status = 'M';
                    break;
            }
        }

        Thread.m_Core          = 0;
        Thread.m_IdealCore     = 0;
        Thread.m_AffinityMask  = 0;
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debug, Thread.m_ThreadId, nn::svc::DebugThreadParam_CurrentCore );
        if( result.IsSuccess() )
        {
            Thread.m_Core = r2;
        }
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debug, Thread.m_ThreadId, nn::svc::DebugThreadParam_IdealProcessor );
        if( result.IsSuccess() )
        {
            Thread.m_IdealCore = r2;
        }
        result = nn::dbg::GetDebugThreadParam( &r1, &r2, debug, Thread.m_ThreadId, nn::svc::DebugThreadParam_AffinityMask );
        if( result.IsSuccess() )
        {
            Thread.m_AffinityMask = (nn::Bit32)r1;
        }
        pData[nThreads] = Thread;
        pData[nThreads].m_Stack = 0;
        pData[nThreads].m_StackSize = 0;

        nn::osdbg::ThreadInfo Info;
        memset( pData[nThreads].m_Name, '\0', nn::os::ThreadNameLengthMax );
        for( int32_t Count = 0; Count < arraySize; Count++ )
        {
            if( pInfo[Count]._debugInfoCreateThread.id == Thread.m_ThreadId )
            {
                Info = pInfo[Count];
                break;
            }
        }
        if( Info._debugInfoCreateThread.id == Thread.m_ThreadId )
        {
            result = nn::osdbg::InitializeThreadInfo( &Info, debug, &Info._debugInfoCreateProcess, &Info._debugInfoCreateThread );
            if( result.IsSuccess() )
            {
                pData[nThreads].m_Stack = Info._stack;
                pData[nThreads].m_StackSize = Info._stackSize;
                char ThreadName[nn::os::ThreadNameLengthMax];
                if( nn::dbg::ReadDebugProcessMemory( (uintptr_t)ThreadName, debug, Info._namePointer, sizeof(ThreadName) ).IsSuccess() )
                {
                    ThreadName[nn::os::ThreadNameLengthMax - 1] = 0;
                    memcpy( pData[nThreads].m_Name, ThreadName, nn::os::ThreadNameLengthMax );
                }
            }
            else
            {
                NN_SDK_LOG("[dmnt] GetAllDebugThreadInfo:  nn::osdbg::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n", Thread.m_ThreadId, result.GetInnerValueForDebug() );
            }
        }
        else
        {
            Thread.m_Status = 'I';
        }
    }

    return nn::ResultSuccess();
} // NOLINT(impl/function_size)

//==============================================================================

nn::Result Server::ContinueDebugEvent( nn::svc::Handle debug, nn::Bit32 flags, const nn::sf::InBuffer& threadIds, nn::Bit32 size ) NN_NOEXCEPT
{
    nn::Bit64* pData = (nn::Bit64*)threadIds.GetPointerUnsafe();
    nn::Result result = nn::dbg::ContinueDebugEvent( debug, flags, pData, size );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::ReadDebugProcessMemory( const nn::sf::OutBuffer& buf, nn::svc::Handle debug, uint64_t addr, uint64_t size ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::ReadDebugProcessMemory\n" );

    char* pData = (char*)buf.GetPointerUnsafe();
    nn::Result result = nn::dbg::ReadDebugProcessMemory( uintptr_t(pData), debug, uintptr_t(addr), size_t(size) );
    //TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::WriteDebugProcessMemory( nn::svc::Handle debug, const nn::sf::InBuffer& buf, uint64_t addr, uint64_t size ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::WriteDebugProcessMemory\n" );

    char* pData = (char*)buf.GetPointerUnsafe();
    nn::Result result = nn::dbg::WriteDebugProcessMemory( debug, uintptr_t(pData), uintptr_t(addr), size_t(size) );
    //TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::SetDebugThreadContext( nn::svc::Handle debug, nn::Bit64 threadId, const nn::sf::InBuffer& context, nn::Bit32 controlFlags ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::SetDebugThreadContext\n" );

    nn::svc::ThreadContext* pContext = (nn::svc::ThreadContext*)context.GetPointerUnsafe();
    nn::Result result = nn::dbg::SetDebugThreadContext( debug, threadId, *pContext, controlFlags );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetDebugThreadParam( nn::sf::Out<nn::Bit64> pOut1, nn::sf::Out<nn::Bit32> pOut2, nn::svc::Handle debug, nn::Bit64 threadId, uint32_t select ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::GetDebugThreadParam\n" );

    nn::Bit64 out1;
    nn::Bit32 out2;
    nn::Result result = nn::dbg::GetDebugThreadParam( &out1, &out2, debug, threadId, (nn::svc::DebugThreadParam)select );
    TRACE_RESULT( result );

    pOut1.Set( out1 );
    pOut2.Set( out2 );

    return result;
}

//==============================================================================

nn::Result Server::InitializeThreadInfo( const nn::sf::OutBuffer& threadInfo, nn::svc::Handle debug, const nn::sf::InBuffer& createProcess, const nn::sf::InBuffer& createThread ) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::InitializeThreadInfo\n" );

    nn::osdbg::ThreadInfo*              pThreadInfo     = (nn::osdbg::ThreadInfo*          )threadInfo   .GetPointerUnsafe();
    nn::svc::DebugInfoCreateProcess*    pCreateProcess  = (nn::svc::DebugInfoCreateProcess*)createProcess.GetPointerUnsafe();
    nn::svc::DebugInfoCreateThread*     pCreateThread   = (nn::svc::DebugInfoCreateThread* )createThread .GetPointerUnsafe();
    nn::Result result = nn::osdbg::InitializeThreadInfo( pThreadInfo, debug, pCreateProcess, pCreateThread );

    return result;
}

//==============================================================================

nn::Result Server::QueryDebugProcessMemory( const nn::sf::OutBuffer& BlockInfo, const nn::sf::OutBuffer& PageInfo, nn::svc::Handle debug, uint64_t addr) NN_NOEXCEPT
{
//    NN_SDK_LOG( "dmnt::QueryDebugProcessMemory\n" );

    nn::svc::MemoryInfo*                pBlockInfo  = (nn::svc::MemoryInfo*)BlockInfo.GetPointerUnsafe();
    nn::svc::PageInfo*                  pPageInfo   = (nn::svc::PageInfo*)PageInfo.GetPointerUnsafe();

    nn::Result result = nn::dbg::QueryDebugProcessMemory( pBlockInfo, pPageInfo, debug, uintptr_t(addr) );

    TRACE_RESULT( result );

    return result;
}

//==============================================================================
// MultiCore thread

void Server::MultiCoreThread( void* pArg ) NN_NOEXCEPT
{
    NN_SDK_LOG( "MultiCoreThread starting on Core %d, AvailableCoreMask = %llx\n", nn::svc::GetCurrentProcessorNumber(), nn::os::GetThreadAvailableCoreMask() );

    nn::Result  Result = nn::ResultSuccess();
    int32_t     CurCore;
    nn::Bit64   Mask;
    nnHandle    Handle;
    Handle.value = 0xFFFF8000;

#ifdef DEBUG_MULTICORE
    Result = nn::svc::GetThreadCoreMask( &CurCore, &Mask, static_cast<nn::svc::Handle>( Handle ) );

    NN_SDK_LOG( "GetThreadCoreMask( %x ): Core = %x, Mask = %llx\n", Handle.value, CurCore, Mask );

    if( ! Result.IsSuccess() )
    {
        NN_SDK_LOG( "GetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
    }
#endif

    while( true )
    {
        uintptr_t pRequest;
        nn::os::ReceiveMessageQueue( &pRequest, &g_MultiCoreRequestMQ );
        uint64_t* Request = (uint64_t*)pRequest;

#ifdef DEBUG_MULTICORE
        NN_SDK_LOG( "MultiCoreRequest: %llx, %llx, %llx, %llx\n", Request[0], Request[1], Request[2], Request[3] );
#endif

        if( Request[0] == 1 )
        {
            for(int32_t Core = 0; Core < 4; Core++)
            {
                Mask    = 1 << Core;
                Result  = nn::svc::SetThreadCoreMask( static_cast<nn::svc::Handle>( Handle ), Core, Mask );

#ifdef DEBUG_MULTICORE
                NN_SDK_LOG( "SetThreadCoreMask( %d, %d, %llx )\n", Handle, Core, Mask );
#endif

                if( ! Result.IsSuccess() )
                {
                    NN_SDK_LOG( "SetThreadCoreMask( %d, %d, %llx ) failed: %08x\n", Handle, Core, Mask, Result.GetInnerValueForDebug() );
                    break;
                }

#ifdef DEBUG_MULTICORE
                Mask        = 0;
                CurCore     = -1;;

                Result = nn::svc::GetThreadCoreMask( &CurCore, &Mask, static_cast<nn::svc::Handle>( Handle ) );

                NN_SDK_LOG( "GetThreadCoreMask( %x ): Core = %x, Mask = %llx\n", Handle.value, CurCore, Mask );

                if( ! Result.IsSuccess() )
                {
                    NN_SDK_LOG( "GetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
                }

                NN_SDK_LOG( "GetThreadCoreMask() = %d, CurrentProcessor = %d\n", CurCore, nn::svc::GetCurrentProcessorNumber() );

                NN_SDK_LOG( "SendMultiCoreRequest( %d, %d, %llx, %llx )\n", Core, (nn::svc::HardwareBreakPointRegisterName)Request[1], Request[2], Request[3] );
#endif

                Result = nn::dbg::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)Request[1], Request[2], Request[3] );

                if( ! Result.IsSuccess() )
                {
#ifdef DEBUG_MULTICORE
                    NN_SDK_LOG( "SetHardwareBreakPoint failed: %08x\n", Result.GetInnerValueForDebug() );
#endif
                    break;
                }
            }
        }
        nn::os::SendMessageQueue( &g_MultiCoreResultMQ, (uintptr_t)&Result );
    }

    NN_SDK_LOG( "MultiCoreThread ending\n" );
}

//==============================================================================

void* Server::SendMultiCoreRequest( void* pMultiCoreRequest ) NN_NOEXCEPT
{
    if( ! g_MultiCoreThreadInitialized )
    {
        nn::os::InitializeMessageQueue( &g_MultiCoreRequestMQ, g_MultiCoreRequestQueue, sizeof(g_MultiCoreRequestQueue) / sizeof(g_MultiCoreRequestQueue[ 0 ]) );
        nn::os::InitializeMessageQueue( &g_MultiCoreResultMQ , g_MultiCoreResultQueue,  sizeof(g_MultiCoreResultQueue)  / sizeof(g_MultiCoreResultQueue[  0 ]) );

        nn::Result Result = nn::os::CreateThread( &g_MultiCoreThread, &MultiCoreThread, nullptr, g_MultiCoreStack, sizeof(g_MultiCoreStack), NN_SYSTEM_THREAD_PRIORITY(dmnt, MultiCoreBP) );
        if( ! Result.IsSuccess() )
        {
            NN_SDK_LOG( "SetThreadCoreMask failed: %08x\n", Result.GetInnerValueForDebug() );
        }
        nn::os::SetThreadName( &g_MultiCoreThread, NN_SYSTEM_THREAD_NAME(dmnt, MultiCoreBP) );
        nn::os::StartThread( &g_MultiCoreThread );

        g_MultiCoreThreadInitialized = true;
    }

#ifdef DEBUG_MULTICORE
    uint64_t* Request = (uint64_t*)pMultiCoreRequest;

    NN_SDK_LOG( "SendMultiCoreRequest( %llx, %llx, %llx, %llx )\n", Request[0], Request[1], Request[2], Request[3] );
#endif

    nn::os::SendMessageQueue( &g_MultiCoreRequestMQ, (uintptr_t)pMultiCoreRequest );

    uintptr_t RetVal;

    nn::os::ReceiveMessageQueue( &RetVal, &g_MultiCoreResultMQ );

    return( (void*)RetVal );
}

//==============================================================================

nn::Result Server::SetHardwareBreakPoint( /*nn::svc::HardwareBreakPointRegisterName*/ uint32_t regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT
{
    //NN_SDK_LOG( "dmnt::SetHardwareBreakPoint\n" );

    uint64_t Request[4];

    Request[0] = 1;
    Request[1] = regNo;
    Request[2] = control;
    Request[3] = value;

    // nn::dbg::SetHardwareBreakPoint( (nn::svc::HardwareBreakPointRegisterName)regNo, control, value );

    void* RetVal = SendMultiCoreRequest( (void*)Request );

#ifdef DEBUG_MULTICORE
    NN_SDK_LOG( "SetHardwareBreakPoint: SendMultiCoreRequest() = %08x\n", (*(nn::Result*)RetVal).GetInnerValueForDebug() );
#endif

    return( *(nn::Result*)RetVal );
}

//==============================================================================

nn::Result Server::GetProcessMemoryDetails( nn::sf::Out<int32_t> pNumMemoryBlocks, const nn::sf::OutBuffer& outData, nn::svc::Handle debugHandle ) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();
    nn::svc::MemoryInfo* pBlocks = (nn::svc::MemoryInfo*)outData.GetPointerUnsafe();
    int32_t nEntries = 0;
    int32_t arraySize = outData.GetSize() / sizeof(nn::svc::MemoryInfo);

    //======================================================================================================
    // Walk the memory block list.
    //======================================================================================================
    for (uintptr_t v = 0; nEntries < arraySize; )
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        result = nn::dbg::QueryDebugProcessMemory( &blockInfo, &pageInfo, debugHandle, v );
        if( result.IsFailure() )
        {
            break;
        }
        if( blockInfo.state == nn::svc::MemoryState_Inaccessible )
        {
            break;
        }

        if( blockInfo.baseAddress + blockInfo.size < v )
        {
            break;
        }

        if( blockInfo.permission != nn::svc::MemoryPermission_None )
        {
            memcpy( &pBlocks[nEntries], &blockInfo, sizeof(blockInfo) );
            nEntries += 1;
        }

        v = blockInfo.baseAddress + blockInfo.size;
    }

    pNumMemoryBlocks.Set( nEntries );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================

nn::Result Server::GetJitDebugProcessList( nn::sf::Out<int32_t> pNumProcesses, const nn::sf::OutBuffer& outData ) NN_NOEXCEPT
{
    //nn::svc::MemoryInfo* pBlocks = (nn::svc::MemoryInfo*)outData.GetPointerUnsafe();
    nn::os::ProcessId* pProcessIds = (nn::os::ProcessId*)outData.GetPointerUnsafe();
    int32_t nEntries = 0;
    int32_t arraySize = outData.GetSize() / sizeof(nn::os::ProcessId);

    nn::Result result = nn::dbg::GetJitDebugProcessList( &nEntries, pProcessIds, arraySize );

    pNumProcesses.Set( nEntries );
    TRACE_RESULT( result );

    return result;
}

//==============================================================================
