﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>

#include <nn/crypto/crypto_Csrng.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>

#include <nn/erpt.h>
#include <nn/util/util_FormatString.h>

#include <nn/err/err_Result.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/creport/creport_Result.h>

#include "creport_Constant.h"
#include "creport_CrashReport.h"

#include <nn/nn_SdkLog.h>

namespace nn { namespace creport {

    void CrashReport::CollectDebugEvent() NN_NOEXCEPT
    {
        if (m_DebugHandle == svc::INVALID_HANDLE_VALUE)
        {
            return;
        }

        svc::DebugEventInfo dei;
        while (svc::GetDebugEvent(&dei, m_DebugHandle).IsSuccess())
        {
            switch (dei.event)
            {
            case svc::DebugEvent_Exception:
                HandleDebugEventException(dei);
                break;
            case svc::DebugEvent_CreateProcess:
                HandleDebugEventCreateProcess(dei);
                break;
            case svc::DebugEvent_CreateThread:
            case svc::DebugEvent_ExitThread:
            case svc::DebugEvent_ExitProcess:
            default:
                break;
            }
        }

        m_Modules.CollectModuleInformation(m_DebugHandle, m_ExceptionThread.GetPc(), m_ExceptionThread.GetLr());
        m_Threads.CollectThreadInformation(m_DebugHandle, m_Is64Bit);
    }

    void CrashReport::CollectDyingMessage() NN_NOEXCEPT
    {
        if (m_DyingMessageAddress == 0 || m_DyingMessageAddress % os::MemoryPageSize != 0 ||
            m_DyingMessageSize > DyingMessageMaxSize || m_DyingMessageSize > sizeof(m_DyingMessage))
        {
            return;
        }

        if (m_DebugHandle == svc::INVALID_HANDLE_VALUE)
        {
            return;
        }

        if (GetResult() <= ResultInvalidResult())
        {
            // JIT Debug に入っていなかった可能性が高いので何もせず返る
            return;
        }

        svc::MemoryInfo block;
        svc::PageInfo   page;

        // 対象のメモリ領域が妥当かチェック
        auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, m_DyingMessageAddress);
        if (result.IsFailure())
        {
            return;
        }

        if (block.permission != svc::MemoryPermission_Read && block.permission != svc::MemoryPermission_ReadWrite)
        {
            return;
        }

        if (m_DyingMessageAddress < block.baseAddress || block.baseAddress + block.size < m_DyingMessageAddress + m_DyingMessageSize)
        {
            return;
        }

        // メモリ読み込み
        result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(m_DyingMessage), m_DebugHandle, m_DyingMessageAddress, m_DyingMessageSize);
        if (result.IsFailure())
        {
            return;
        }
    }

    void CrashReport::RecordException(Result result, Bit64 extra, const svc::DebugEventInfo& dei) NN_NOEXCEPT
    {
        m_Result = result;
        m_ExceptionCode = dei.info.exception.exceptionCode;
        m_ExceptionAddress = dei.info.exception.exceptionAddress;
        m_ExceptionExtra = extra;
        m_ExceptionThreadId = dei.threadId;
        m_ExceptionThread.GetThreadInfo(m_DebugHandle, dei.threadId, m_Is64Bit);
    }

    Result CrashReport::GetUserBreakResult(uintptr_t address, size_t size) NN_NOEXCEPT
    {
        svc::MemoryInfo block;
        svc::PageInfo   page;

        Result userResult = ResultUserBreak();

        if (address == 0)
        {
            return userResult;
        }

        // 今は固定サイズにする
        size = sizeof(Result);

        // 対象のメモリ領域が妥当かチェック
        auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, address);
        if (result.IsFailure())
        {
            return userResult;
        }

        if (block.permission != svc::MemoryPermission_Read && block.permission != svc::MemoryPermission_ReadWrite)
        {
            return userResult;
        }

        if (address < block.baseAddress || block.baseAddress + block.size < address + size)
        {
            return userResult;
        }

        // メモリ読み込み
        result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(&userResult), m_DebugHandle, address, size);
        if (result.IsFailure())
        {
            return userResult;
        }
        return userResult;
    }

    void CrashReport::HandleDebugEventException(const svc::DebugEventInfo& dei) NN_NOEXCEPT
    {
        const svc::DebugInfo &info = dei.info;

        // 例外コード別の処理
        switch (info.exception.exceptionCode)
        {
        case svc::DebugException_UndefinedInstruction:
            RecordException(ResultUndefinedInstruction(), info.exception.detail.undefinedInstruction.code, dei);
            break;
        case svc::DebugException_UndefinedSystemCall:
            RecordException(ResultUndefinedSystemCall(), info.exception.detail.undefinedSystemCall.svcId, dei);
            break;
        case svc::DebugException_DataTypeMissaligned:
            RecordException(ResultDataTypeMissaligned(), info.exception.detail.dataTypeMissaligned.faultAddress, dei);
            break;
        case svc::DebugException_AccessViolationInstruction:
            RecordException(ResultAccessViolationInstruction(), 0, dei);
            break;
        case svc::DebugException_AccessViolationData:
            RecordException(ResultAccessViolationData(), info.exception.detail.accessViolationData.faultAddress, dei);
            break;
        case svc::DebugException_MemorySystemError:
            RecordException(ResultMemorySystemError(), 0, dei);
            break;
        case svc::DebugException_UserBreak:
            {
                Result result = GetUserBreakResult(dei.info.exception.detail.userBreak.data, dei.info.exception.detail.userBreak.size);
                RecordException(result, info.exception.detail.userBreak.reason, dei);
            }
            break;
        case svc::DebugException_BreakPoint:
        case svc::DebugException_DebuggerBreak:
        case svc::DebugException_AttachBreak:
        default:
            break;
        }
    }

    void CrashReport::HandleDebugEventCreateProcess(const svc::DebugEventInfo& dei) NN_NOEXCEPT
    {
        // プロセス情報を保存
        m_ProgramId = dei.info.createProcess.programId;
        m_Is64Bit = (dei.info.createProcess.flag & svc::CreateProcessParameterFlag_64Bit) ? true : false;
        m_IsApplication = (dei.info.createProcess.flag & svc::CreateProcessParameterFlag_IsApplication) ? true : false;

        if (!m_IsApplication)
        {
            // アプリ以外のダイイング・メッセージは見ない
            return;
        }

        // クラッシュレポートにアプリが任意に残す情報の場所を取得
        uintptr_t address = dei.info.createProcess.processRegionAddress + offsetof(svc::ProcessLocalRegion, dyingMessageRegionAddress);

        svc::MemoryInfo block;
        svc::PageInfo   page;
        Bit64           dmAddress;
        Bit64           dmSize;

        NN_STATIC_ASSERT(sizeof(dmAddress) == sizeof(svc::ProcessLocalRegion::dyingMessageRegionAddress));
        NN_STATIC_ASSERT(sizeof(dmSize)    == sizeof(svc::ProcessLocalRegion::dyingMessageRegionSize));

        auto result = svc::QueryDebugProcessMemory(&block, &page, m_DebugHandle, address);
        if (result.IsFailure())
        {
            return;
        }

        if (block.permission != svc::MemoryPermission_Read && block.permission != svc::MemoryPermission_ReadWrite)
        {
            return;
        }

        if (address < block.baseAddress || block.baseAddress + block.size < address + sizeof(dmAddress) + sizeof(dmSize))
        {
            return;
        }

        // ダイイング・メッセージのアドレス読み込み
        result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(&dmAddress), m_DebugHandle, address, sizeof(dmAddress));
        if (result.IsFailure())
        {
            return;
        }

        // アドレスチェック
        if (dmAddress == 0 || dmAddress % os::MemoryPageSize != 0)
        {
            return;
        }

        // ダイイング・メッセージのサイズ読み込み
        result = svc::ReadDebugProcessMemory(reinterpret_cast<uintptr_t>(&dmSize), m_DebugHandle, address + sizeof(dmAddress), sizeof(dmSize));
        if (result.IsFailure())
        {
            return;
        }

        // サイズチェック
        if (dmSize > DyingMessageMaxSize)
        {
            dmSize = DyingMessageMaxSize;
        }

        m_DyingMessageAddress = dmAddress;
        m_DyingMessageSize    = dmSize;
        NN_SDK_LOG("[creport] m_DyingMessageAddress = 0x%llx\n", m_DyingMessageAddress);
        NN_SDK_LOG("[creport] m_DyingMessageSize    = 0x%llx\n", m_DyingMessageSize);
    }

    void CrashReport::AppendToBuffer(InfoBuffer* pOutBuffer) NN_NOEXCEPT
    {
        Bit32 signature = 0x50455243;
        Bit32 version = 4;

        pOutBuffer->Append(&signature, sizeof(signature));
        pOutBuffer->Append(&version, sizeof(version));

        pOutBuffer->Append(&m_Is64Bit, sizeof(m_Is64Bit));
        pOutBuffer->Append(&m_ExceptionCode, sizeof(m_ExceptionCode));
        pOutBuffer->Append(&m_ExceptionAddress, sizeof(m_ExceptionAddress));
        pOutBuffer->Append(&m_ExceptionExtra, sizeof(m_ExceptionExtra));
        pOutBuffer->Append(&m_ExceptionThreadId, sizeof(m_ExceptionThreadId));

        m_Modules.AppendToBuffer(pOutBuffer);

        m_ExceptionThread.CalculateHash(m_Modules);

        // m_ExceptionThread の情報は m_Threads （全スレッドのダンプ）に含まれるので書き出さない
        // m_ExceptionThreadId のみ残せば十分
        //m_ExceptionThread.AppendToBuffer(pOutBuffer);

        m_Threads.AppendToBuffer(pOutBuffer);
    }

    void CrashReport::GenerateAesKey() NN_NOEXCEPT
    {
        // AES Key, Iv を生成
        nn::crypto::GenerateCryptographicallyRandomBytes(m_AesKey, AesKeySize);
        nn::crypto::GenerateCryptographicallyRandomBytes(m_AesIv,  AesIvSize);
    }

    void CrashReport::GetEncryptedKey(Bit8* pOutKey, size_t keySize) NN_NOEXCEPT
    {
        Bit8 aesKeyAndIv[AesKeySize + AesIvSize];

        // 暗号化に用いた AES 鍵を RSA で暗号化
        std::memcpy(aesKeyAndIv,              m_AesKey, AesKeySize);
        std::memcpy(aesKeyAndIv + AesKeySize, m_AesIv,  AesIvSize);

        Bit8 seed[RsaOaepSeedSize];
        nn::crypto::GenerateCryptographicallyRandomBytes(seed, sizeof(seed));

        int keyIndex = m_IsApplication ? 1 : 0;

        nn::crypto::RsaOaepEncryptor<RsaKeyLength, nn::crypto::Sha256Generator> rsaEncryptor;
        rsaEncryptor.Initialize(PublicKeyModulus[keyIndex], sizeof(PublicKeyModulus[keyIndex]), PublicKeyExponent, sizeof(PublicKeyExponent));
        rsaEncryptor.Encrypt(pOutKey, keySize, aesKeyAndIv, sizeof(aesKeyAndIv), seed, sizeof(seed));

        memset_s(aesKeyAndIv, sizeof(aesKeyAndIv), 1, sizeof(aesKeyAndIv));
    }

    void CrashReport::EraseAesKey() NN_NOEXCEPT
    {
        memset_s(m_AesKey, sizeof(m_AesKey), 1, sizeof(m_AesKey));
        memset_s(m_AesIv, sizeof(m_AesIv), 1, sizeof(m_AesIv));
    }

    void CrashReport::Encrypt(Bit8* source, size_t sourceSize) NN_NOEXCEPT
    {
        // AES CTR モードなので暗号元と出力先のバッファは同じでよい
        nn::crypto::EncryptAes128Ctr(source, sourceSize,
            m_AesKey, sizeof(m_AesKey),
            m_AesIv, sizeof(m_AesIv),
            source, sourceSize);
    }

    template <typename T, size_t N>
    bool CrashReport::FindArray(const T(&list)[N], const T& target) NN_NOEXCEPT
    {
        return (std::find(list, list + N, target) - list < N);
    }

    Result CrashReport::AddEncryptedDebugInformation(erpt::Context* context) NN_NOEXCEPT
    {
        static const erpt::FieldId FieldIdList[] = {
            erpt::FieldId::EncryptedExceptionInfo,
            erpt::FieldId::EncryptedExceptionInfo1,
            erpt::FieldId::EncryptedExceptionInfo2,
            erpt::FieldId::EncryptedExceptionInfo3,
        };

        // ProgramId がブラックリストにある場合にはデバッグ情報を残さない
        if (FindArray(BlackList, m_ProgramId) == true)
        {
            NN_RESULT_SUCCESS;
        }

        // アプリの場合、以下の場合はデバッグ情報を残さない
        // - ホワイトリストにない かつ m_IsDetailed == false
        if (m_IsApplication)
        {
            if (m_IsDetailed == false && FindArray(WhiteList, m_ProgramId) == false)
            {
                NN_RESULT_SUCCESS;
            }
        }

        // デバッグ情報をバッファに吐き出して暗号化
        AppendToBuffer(&m_Buffer);

        // AES 用の暗号鍵（乱数）を生成
        GenerateAesKey();
        GetEncryptedKey(m_EncryptedAesKey, sizeof(m_EncryptedAesKey));

        // 暗号化
        Encrypt(const_cast<Bit8*>(m_Buffer.GetBuffer()), m_Buffer.GetCurrentSize());
        if (m_DyingMessageSize > 0)
        {
            Encrypt(m_DyingMessage, m_DyingMessageSize);
        }

        // AES 用の暗号鍵（乱数）を削除
        EraseAesKey();

        // 暗号化用の鍵情報
        NN_RESULT_DO(context->Add(erpt::FieldId::EncryptionKey, m_EncryptedAesKey, sizeof(m_EncryptedAesKey)));

        // 暗号化された例外情報
        // Redshift の制約で 1 フィールド最大 16KB まで。最大 4 個使って全情報を保存する。
        size_t offset = 0;
        for (int i = 0; i < NN_ARRAY_SIZE(FieldIdList) && offset < m_Buffer.GetCurrentSize(); i++)
        {
            // erpt 内で配列・文字列は 16384 より小さいことをチェックしているので -16 しておく
            size_t size = std::min(static_cast<size_t>(16 * 1024 - 16), m_Buffer.GetCurrentSize() - offset);
            NN_RESULT_DO(context->Add(FieldIdList[i], m_Buffer.GetBuffer() + offset, size));
            offset += size;
        }

        NN_STATIC_ASSERT(erpt::MaxArrayBufferLength >= CrashReportBufferSize);

        // ダイイング・メッセージ
        if (m_DyingMessageSize > 0)
        {
            NN_RESULT_DO(context->Add(erpt::FieldId::EncryptedDyingMessage, m_DyingMessage, m_DyingMessageSize));
        }

        // スタックトレースのハッシュ
        NN_RESULT_DO(context->Add(erpt::FieldId::CrashReportHash, m_ExceptionThread.GetHashBuffer(), m_ExceptionThread.GetHashLength()));

        NN_RESULT_SUCCESS;
    }

    Result CrashReport::CreateErrorReport() NN_NOEXCEPT
    {
        static Bit8 s_ErptContextBuffer[CrashReportBufferSize];
        char programIdString[17];
        char errorCodeString[nn::err::ErrorCode::StringLengthMax];

        // アプリ以外の User break (=NN_ABORT) はエラーレポートを残さない
        if (m_IsApplication == false && IsUserBreak())
        {
            NN_RESULT_SUCCESS;
        }

        // エラーレポートを作成
        erpt::Context context(erpt::CategoryId::ErrorInfo, s_ErptContextBuffer, sizeof(s_ErptContextBuffer));

        // エラーコード
        auto errorCode = err::detail::ConvertResultToErrorCode(m_Result);
        nn::util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);
        NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));

        // ErrorInfo に ProgramId を登録
        util::SNPrintf(programIdString, sizeof(programIdString), "%016llx", m_ProgramId);
        NN_RESULT_DO(context.Add(erpt::FieldId::ProgramId, programIdString, static_cast<uint32_t>(sizeof(programIdString))));

        // Abort フラグ
        NN_RESULT_DO(context.Add(nn::erpt::FieldId::AbortFlag, true));
        if( m_IsApplication )
        {
            NN_RESULT_DO(context.Add(nn::erpt::FieldId::ApplicationAbortFlag, true));
        }
        else
        {
            // FatalFlag : エラー履歴での詳細表示抑制用
            NN_RESULT_DO(context.Add(nn::erpt::FieldId::FatalFlag, true));
            NN_RESULT_DO(context.Add(nn::erpt::FieldId::SystemAbortFlag, true));
        }

        // デバッグ情報を暗号化して追加
        NN_RESULT_DO(AddEncryptedDebugInformation(&context));

        // エラーレポート作成
        // NOTE: システムプロセスの UserBreak (NN_ABORT) はここに来ずに return するのでケアしない
        erpt::ReportType reportType = m_IsApplication ? erpt::ReportType_Invisible : erpt::ReportType_Visible;
        NN_RESULT_DO(context.CreateReport(reportType));

        NN_RESULT_SUCCESS;
    }

}}
