﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

#include "boot_I2cHelper.h"
#include "boot_PmicDriver.h"

namespace nn { namespace boot {

NN_NORETURN void PmicDriver::ShutdownSystem() NN_NOEXCEPT
{
    NN_SDK_LOG("[boot] Shutdown System.\n");
    ShutdownSystem(false);
}

NN_NORETURN void PmicDriver::RebootSystem() NN_NOEXCEPT
{
    NN_SDK_LOG("[boot] Reboot System.\n");
    ShutdownSystem(true);
}

nn::Result PmicDriver::GetAcOk(bool* pOutAcOk) NN_NOEXCEPT
{
    const uint8_t mask = 0x02;

    uint8_t onoffStatus;
    NN_RESULT_DO(GetOnOffStatus(&onoffStatus));

    *pOutAcOk = ((onoffStatus & mask) != 0);

    NN_RESULT_SUCCESS;
}

nn::Result PmicDriver::GetPowerButtonPressed(bool* pOutPowerButtonPressed) NN_NOEXCEPT
{
    const uint8_t mask = 0x04;

    uint8_t onoffStatus;
    NN_RESULT_DO(GetOnOffStatus(&onoffStatus));

    *pOutPowerButtonPressed = ((onoffStatus & mask) != 0);

    NN_RESULT_SUCCESS;
}

nn::Result PmicDriver::GetOnOffStatus(uint8_t* pOut) NN_NOEXCEPT
{
    const uint8_t address = 0x15;
    NN_RESULT_DO(ReadI2cRegister(pOut, m_I2cSession, &address, sizeof(address), sizeof(*pOut)));
    NN_RESULT_SUCCESS;
}

nn::Result PmicDriver::GetOnOffIrq(uint8_t* pOut) NN_NOEXCEPT
{
    const uint8_t address = 0x0B;
    NN_RESULT_DO(ReadI2cRegister(pOut, m_I2cSession, &address, sizeof(address), sizeof(*pOut)));
    NN_RESULT_SUCCESS;
}

nn::Result PmicDriver::GetNverc(uint8_t* pOut) NN_NOEXCEPT
{
    const uint8_t address = 0x0C;
    NN_RESULT_DO(ReadI2cRegister(pOut, m_I2cSession, &address, sizeof(address), sizeof(*pOut)));
    NN_RESULT_SUCCESS;
}

NN_NORETURN void PmicDriver::ShutdownSystem(bool isReboot) NN_NOEXCEPT
{
    const uint8_t onoffcnfg1Address = 0x41;
    const uint8_t onoffcnfg2Address = 0x42;
    const uint8_t sftRstMask = 0x80;
    const uint8_t sftRstWkMask = 0x80;
    const auto shutdownTimeout = nn::TimeSpan::FromMilliSeconds(5000);

    uint8_t onoffcnfg2Value = 0x00;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadI2cRegister(&onoffcnfg2Value, m_I2cSession, &onoffcnfg2Address, sizeof(onoffcnfg2Address), sizeof(onoffcnfg2Value)));
    if (isReboot)
    {
        onoffcnfg2Value = onoffcnfg2Value | sftRstWkMask;
    }
    else
    {
        onoffcnfg2Value = onoffcnfg2Value & (~sftRstWkMask);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteI2cRegister(m_I2cSession, &onoffcnfg2Address, sizeof(onoffcnfg2Address), &onoffcnfg2Value, sizeof(onoffcnfg2Value)));

    uint8_t onoffcnfg1Value = 0x00;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadI2cRegister(&onoffcnfg1Value, m_I2cSession, &onoffcnfg1Address, sizeof(onoffcnfg1Address), sizeof(onoffcnfg1Value)));
    onoffcnfg1Value = onoffcnfg1Value | sftRstMask;

    // nn::bpc::driver::detail::ShutdownSystemImpl ではここで RTC Alarm 2 をセットしているが、boot ライブラリではシャットダウンを給電状態で行うことがないため、不要

    // シャットダウン直前に充電状況に応じて電池残量計をシャットダウンするか決定する
    FuelgaugeDriver fuelgaugeDriver;
    FinalizeFuelgauge(&fuelgaugeDriver);

    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteI2cRegister(m_I2cSession, &onoffcnfg1Address, sizeof(onoffcnfg1Address), &onoffcnfg1Value, sizeof(onoffcnfg1Value)));

    nn::os::SleepThread(shutdownTimeout);
    NN_ABORT("[boot] Timeout of shutdown signal.");
}

void PmicDriver::FinalizeFuelgauge(FuelgaugeDriver* pFuelgaugeDriver) NN_NOEXCEPT
{
    // SIGLO-65162: SHDNTIMER が短く、緩和時間が確保できていない市場本体に対しての一括修正。
    pFuelgaugeDriver->SetShutdownTimer();

    bool isI2cShutdownEnabled;
    auto result = pFuelgaugeDriver->IsI2cShutdownEnabled(&isI2cShutdownEnabled);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] IsI2cShutdownEnabled() failure. (%08x)\n", result);
        return;
    }

    bool isAcOk;
    result = GetAcOk(&isAcOk);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[boot] GetAcOk() failure. (%08x)\n", result);
    }

    bool newIsI2cShutdownEnabled;
    // AcOk の取得失敗時にはシャットダウンモードにしない
    if (result.IsFailure() || isAcOk)
    {
        newIsI2cShutdownEnabled = false;
    }
    else
    {
        newIsI2cShutdownEnabled = true;
    }

    if (newIsI2cShutdownEnabled != isI2cShutdownEnabled)
    {
        result = pFuelgaugeDriver->SetI2cShutdownEnabled(newIsI2cShutdownEnabled);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[boot] SetI2cShutdownEnabled() failure. (%08x)\n", result);
        }
    }
}

}} // namespace nn::boot
