﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/dd.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/i2c/driver/i2c.h>

#include <t21x/arapb_misc.h>
#include <t21x/arapbpm.h>
#include <t21x/arclk_rst.h>
#include <t21x/ardsi.h>
#include "reg.h"

#include "boot_Display.h"
#include "boot_I2cHelper.h"
#include "boot_SplAccessor.h"

/* display initialization */
#define GPIO_PI_CNF     (NN_BLSM_REG_GPIO + 0x200)
#define GPIO_PI_OE      (NN_BLSM_REG_GPIO + 0x210)
#define GPIO_PI_OUT     (NN_BLSM_REG_GPIO + 0x220)

#define GPIO_PV_CNF     (NN_BLSM_REG_GPIO + 0x504)
#define GPIO_PV_OE      (NN_BLSM_REG_GPIO + 0x514)
#define GPIO_PV_OUT     (NN_BLSM_REG_GPIO + 0x524)


#define WRITE_REGISTERS(base, data)     WriteRegisters(base, data, (sizeof(data) / sizeof(*(data))))
#define WRITE_REGISTERS_DEPENDING_ON_SOC(base, data) \
    WriteRegisters(base, data ## Erista, sizeof(data ## Erista) / sizeof(*(data ## Erista)), data ## Mariko, sizeof(data ## Mariko) / sizeof(*(data ## Mariko)))


namespace nn { namespace boot {

    namespace
    {
        const uintptr_t FrameBufferAddress = 0xc0000000;
        const dd::PhysicalAddress PmcPhysicalAddress = 0x7000E400ull;

        const int DsiWaitForCommandMicroSecondsMax          = 250 * 1000;
        const int DsiWaitForCommandCompletionMicroSeconds   = 5;
        const int DsiStatusPollingDurationMicroSecondsMax   = 150 * 1000;

        const size_t DeviceAddressSpaceBlockSize    = (1 << 22);
        const size_t FrameBufferSize                = 768 * 1280 * 4;

        uintptr_t   NN_BLSM_REG_DSI;
        uintptr_t   NN_BLSM_REG_CAR;
        uintptr_t   NN_BLSM_REG_DISP1;
        uintptr_t   NN_BLSM_REG_APB_MISC;
        uintptr_t   NN_BLSM_REG_MIPI_CAL;
        uintptr_t   NN_BLSM_REG_GPIO;

        dd::DeviceAddressSpaceType  g_DeviceAddressSpace;
        Bit32*                      g_pFrameBuffer = nullptr;
        bool                        g_IsInitialized = false;

        Bit8                        g_FrameBuffer[FrameBufferSize + DeviceAddressSpaceBlockSize];


        enum LcdVendor
        {
            LcdVendor_Jdi       = 0x10,
            LcdVendor_Innolux   = 0x20,
            LcdVendor_Auo       = 0x30,
        };

        struct RegWrite
        {
            Bit32 offset;
            Bit32 value;
        };

        const RegWrite Dc01[] =
        {
            { 0x0040, 0 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0043, 0x00000054 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0042, 0x00000010 },
            { 0x0042, 0x00000020 },
            { 0x0042, 0x00000040 },
            { 0x0480, 0 },
            { 0x0403, 0 },
            { 0x0404, 0 },
            { 0x0036, 0x00050155 },
            { 0x0001, 0x00000100 },
            { 0x0028, 0x00000109 },
            { 0x0041, 0x00000f00 },
            { 0x0041, 0x0000000f },
            { 0x0040, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 },
            { 0x0042, 0x00000010 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0042, 0x00000020 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0042, 0x00000040 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0430, 0x00000008 },
            { 0x042f, 0 },
            { 0x0307, 0x01000000 },
            { 0x0309, 0 },
            { 0x04e4, 0 },
            { 0x0300, 0 },
            { 0x0041, 0x00000f00 },
            { 0x0041, 0x0000000f },
            { 0x0042, 0x00000010 },
            { 0x0716, 0x010000ff },
            { 0x0042, 0x00000020 },
            { 0x0716, 0x010000ff },
            { 0x0042, 0x00000040 },
            { 0x0716, 0x010000ff },
            { 0x0031, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0402, 0 },
            { 0x0032, 0 },
            { 0x0041, 0x00000f00 },
            { 0x0041, 0x0000000f },
        };

        const RegWrite Dc02[] =
        {
            { 0x0040, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 },
            { 0x0042, 0x00000010 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0042, 0x00000020 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            /* Connect to services. If services is not started yet, start it now */
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x070e, 0 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0042, 0x00000040 },
            { 0x0611, 0x000000f0 },
            { 0x0612, 0x0000012a },
            { 0x0613, 0 },
            { 0x0614, 0x00000198 },
            { 0x0615, 0x0000039b },
            { 0x0616, 0x0000032f },
            { 0x0617, 0x00000204 },
            { 0x0618, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0430, 0x00000008 },
            { 0x042f, 0 },
            { 0x0307, 0x01000000 },
            { 0x0309, 0 },
            { 0x04e4, 0 },
            { 0x0300, 0 },
            { 0x0041, 0x00000f00 },
            { 0x0041, 0x0000000f },
            { 0x0042, 0x00000010 },
            { 0x0716, 0x010000ff },
            { 0x0042, 0x00000020 },
            { 0x0716, 0x010000ff },
            { 0x0042, 0x00000040 },
            { 0x0716, 0x010000ff },
            { 0x0031, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0402, 0 },
            { 0x0032, 0 },
            { 0x0041, 0x00000f00 },
            { 0x0041, 0x0000000f },
            { 0x0040, 0 },
            { 0x0405, 0 },
            { 0x0406, 0x00010000 },
            { 0x0407, 0x00010048 },
            { 0x0408, 0x00090048 },
            { 0x0409, 0x050002d0 },
            { 0x040a, 0x000a0088 },
            { 0x0431, 0x00010001 },
            { 0x0303, 0 },
            { 0x0432, 0x00000005 },
            { 0x042f, 0 },
            { 0x042e, 0 },
            { 0x0031, 0 },
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0402, 0 },
            { 0x0032, 0x00000020 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0040, 0x00000005 },
            { 0x040a, 0x000a0088 },
            { 0x0040, 0 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0000, 0x00000301 },
            { 0x0000, 0x00000301 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0040, 0 },
            { 0x042e, 0x00000004 },
            { 0x0430, 0x00000008 },
            { 0x0031, 0 },
        };

#if 0 // 単色表示テスト用
        const RegWrite Dc03[] =
        {
            { 0x0042, 0x00000010 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0032, 0x00000020 }, /* refresh */
        };
#endif

        const RegWrite Dc04[] =
        {
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 }, /* window A, 0x20 for B, 0x40 for C, 0x80 for D */
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0703, 0x0000000d }, /* R8G8B8A8 */
            { 0x0700, 0 },
            { 0x0700, 0 },
            { 0x0704, 0 },
            { 0x0707, 0 },
            { 0x0708, 0 },
            { 0x0706, 0x05000b40 },
            { 0x0709, 0x10001000 },
            { 0x0705, 0x050002d0 },
            { 0x070a, 0x06000c00 },
            { 0x0702, 0 },
            { 0x080b, 0 },
            { 0x800,  FrameBufferAddress },
            { 0x0806, 0 },
            { 0x0808, 0 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0x40000000 },
            { 0x0032, 0x00000020 },
            { 0x0041, 0x00000300 },
            { 0x0041, 0x00000003 },
        };

        const RegWrite DsiHostDsiCtrlErista[] =
        {
            { 0x003c, 0x6070601 },
        };

        const RegWrite DsiHostDsiCtrlMariko[] =
        {
            { 0x003c, 0x6070603 },
        };

        const RegWrite Dsi01Top01[] =
        {
            { 0x000a, 0 },
            { 0x000c, 0 },
            { 0x000d, 0 },
            { 0x000e, 0 },
            { 0x001b, 0 },
            { 0x001c, 0 },
            { 0x001d, 0 },
            { 0x001e, 0 },
        };

        const RegWrite Dsi01Top02Erista[] =
        {
            { 0x005f, 0 },
        };

        const RegWrite Dsi01Top02Mariko[] =
        {
            { 0x0062, 0 },
        };

        const RegWrite Dsi01Top03[] =
        {
            { 0x0033, 0 },
            { 0x0023, 0 },
            { 0x0025, 0 },
            { 0x0027, 0 },
            { 0x0029, 0 },
            { 0x002b, 0 },
            { 0x002d, 0 },
            { 0x0024, 0 },
            { 0x0026, 0 },
            { 0x0028, 0 },
            { 0x002a, 0 },
            { 0x002c, 0 },
            { 0x002e, 0 },
            { 0x0010, 0 },
        };

        const RegWrite Dsi01MiddleErista[] =
        {
        };

        const RegWrite Dsi01MiddleMariko[] =
        {
            { 0x004f, 0 },
            { 0x0050, 0 },
            { 0x0051, 0 },
            { 0x0052, 0 },
            { 0x0053, 0 },
            { 0x0054, 0 },
            { 0x0055, 0 },
        };

        const RegWrite Dsi01Bottom01[] =
        {
            { 0x004c, 0 },
            { 0x0011, 0x00000018 },
            { 0x0012, 0x000001e0 },
            { 0x0013, 0 },
            { 0x001a, 0 },
            { 0x0034, 0 },
            { 0x0035, 0 },
            { 0x0036, 0 },
            { 0x0037, 0 },
            { 0x004f, 0 },
        };

        const RegWrite Dsi01Bottom02[] =
        {
            { 0x003d, 0x040a0e05 },
            { 0x003e, 0x00030109 },
            { 0x003f, 0x00190a14 },
            { 0x0044, 0x2000ffff },
            { 0x0045, 0x07652000 },
            { 0x0046, 0 },
            { 0x004b, 0 },
            { 0x000b, 0x00000001 },
            { 0x000b, 0x00000001 },
            { 0x000b, 0 },
            { 0x000b, 0 },
            { 0x004f, 0 },
        };

        const RegWrite Dsi01Bottom03[] =
        {
            { 0x003d, 0x040a0e05 },
            { 0x003e, 0x00030118 },
            { 0x003f, 0x00190a14 },
            { 0x0044, 0x2000ffff },
            { 0x0045, 0x13432000 },
            { 0x0046, 0 },
            { 0x000f, 0x00102003 },
            { 0x0010, 0x00000031 },
            { 0x000b, 0x00000001 },
            { 0x000b, 0x00000001 },
            { 0x0012, 0x00000040 },
            { 0x0013, 0 },
            { 0x0014, 0 },
            { 0x001a, 0 },
        };

        const RegWrite DsiExitSleepForJdi[] =
        {
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x9483ffb9 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x0000bd15 }, // SET_RBDh
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00001939 }, // SET_RD8h
            { 0x000a, 0xaaaaaad8 },
            { 0x000a, 0xaaaaaaeb },
            { 0x000a, 0xaaebaaaa },
            { 0x000a, 0xaaaaaaaa },
            { 0x000a, 0xaaaaaaeb },
            { 0x000a, 0xaaebaaaa },
            { 0x000a, 0x000000aa },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x0001bd15 }, // SET_RBDh
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00002739 }, // SET_RD8h
            { 0x000a, 0xffffffd8 },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0x00ffffff },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x0002bd15 }, // SET_RBDh
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000f39 }, // SET_RD8h
            { 0x000a, 0xffffffd8 },
            { 0x000a, 0xffffffff },
            { 0x000a, 0xffffffff },
            { 0x000a, 0x00ffffff },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x0000bd15 }, // SET_RBDh
            { 0x0013, 0x00000002 },
            { 0x000a, 0x0006D915 }, // SET_RD9h
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x000000b9 },
            { 0x0013, 0x00000002 },
        };

        const RegWrite Dsi0201[] =
        {
            { 0x004f, 0 },
        };

        const RegWrite Dsi0202[] =
        {
            { 0x003d, 0x040a0e05 },
            { 0x003e, 0x00030172 },
            { 0x003f, 0x00190a14 },
            { 0x0044, 0x20000a40 },
            { 0x0045, 0x5a2f2000 },
            { 0x0046, 0 },
            { 0x0023, 0x40000208 },
            { 0x0027, 0x40000308 },
            { 0x002b, 0x40000308 },
            { 0x0025, 0x40000308 },
            { 0x0029, 0x3f3b2b08 },
            { 0x002a, 0x000002cc },
            { 0x002d, 0x3f3b2b08 },
            { 0x002e, 0x000002cc },
            { 0x0034, 0x00ce0000 },
            { 0x0035, 0x087001a2 },
            { 0x0036, 0x00000190 },
            { 0x0037, 0x00000190 },
            { 0x000f, 0 },
        };

        const RegWrite Dsi03[] =
        {
            { 0x0013, 0 },
            { 0x0010, 0 },
            { 0x0011, 0x00000006 },
            { 0x0012, 0x000001e0 },
            { 0x000b, 0x00000001 },
            { 0x0010, 0x00103032 },
            { 0x000f, 0x00000033 },
            { 0x0010, 0x00103032 },
            { 0x000f, 0x00000003 },
            { 0x000f, 0x00000023 },
        };

        const RegWrite Dsi04Erista[] =
        {
            { 0x004f, 0 },
            { 0x0050, 0 },
            { 0x0051, 0x00003333 },
            { 0x0052, 0 },
        };

        const RegWrite Dsi04Mariko[] =
        {
            { 0x004f, 0 },
            { 0x0050, 0 },
            { 0x0051, 0 },
            { 0x0052, 0x00077777 },
            { 0x0053, 0x00077777 },
            { 0x0054, 0x00001111 },
            { 0x0055, 0 },
        };

        const RegWrite MipiCal01[] =
        {
            { 0x0060 / 4, 0 },
            { 0x0008 / 4, 0xf3f10000 },
            { 0x0058 / 4, 0 },
            { 0x0060 / 4, 0 },
        };

        const RegWrite MipiCal02Erista[] =
        {
            { 0x0060 / 4, 0x00010010 },
            { 0x005C / 4, 0x00000300 },
        };

        const RegWrite MipiCal02Mariko[] =
        {
            { 0x0060 / 4, 0x00010010 },
            { 0x005C / 4, 0 },
        };

        const RegWrite MipiCal03Erista[] =
        {
            { 0x0038 / 4, 0x00200200 },
            { 0x003C / 4, 0x00200200 },
            { 0x0064 / 4, 0x00200002 },
            { 0x0068 / 4, 0x00200002 },
            { 0x0014 / 4, 0 },
            { 0x0018 / 4, 0 },
        };

        const RegWrite MipiCal03Mariko[] =
        {
            { 0x0038 / 4, 0x00200006 },
            { 0x003C / 4, 0x00200006 },
            { 0x0064 / 4, 0x00260000 },
            { 0x0068 / 4, 0x00260000 },
            { 0x0014 / 4, 0x200000 },
            { 0x0018 / 4, 0x200000 },
        };

        const RegWrite MipiCal04[] =
        {
            { 0x001C / 4, 0 },
            { 0x0020 / 4, 0 },
            { 0x0024 / 4, 0 },
            { 0x0028 / 4, 0 },
            { 0x0040 / 4, 0 },
            { 0x0044 / 4, 0 },
            { 0x0068 / 4, 0 },
            { 0x0070 / 4, 0 },
            { 0x0074 / 4, 0 },
            { 0x0000 / 4, 0x2a000001 },
        };

        const RegWrite Plld01Erista[] =
        {
            { 0x0138 / 4, 0x40000000 }, /* select PLLD_OUT0 as DC clock */
            { 0x00d0 / 4, 0x4830a001 },
            { 0x00d8 / 4, 0x00000020 },
            { 0x00dc / 4, 0x002d0aaa },
        };

        const RegWrite Plld01Mariko[] =
        {
            { 0x0138 / 4, 0x40000000 }, /* select PLLD_OUT0 as DC clock */
            { 0x00d0 / 4, 0x4830a001 },
            { 0x00d8 / 4, 0 },
            { 0x00dc / 4, 0x002dfc00 },
        };

        const RegWrite Plld02Erista[] =
        {
            { 0x00d0 / 4, 0x4810c001 },
            { 0x00d8 / 4, 0x00000020 },
            { 0x00dc / 4, 0x002dfc00 },
        };

        const RegWrite Plld02Mariko[] =
        {
            { 0x00d0 / 4, 0x4810c001 },
            { 0x00d8 / 4, 0 },
            { 0x00dc / 4, 0x002dfc00 },
        };

#if !defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
        const RegWrite Dc0A[] =
        {
            { 0x040a, 0x000a0088 },
            { 0x0038, 0 },
            { 0x0040, 0 },
            { 0x0039, 0 },
            { 0x0028, 0 },
            { 0x0032, 0 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0000, 0x00000301 },
            { 0x0000, 0x00000301 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
            { 0x0036, 0 },
            { 0x0041, 0x00000100 },
            { 0x0041, 0x00000001 },
        };

        const RegWrite Dsi0A01[] =
        {
            { 0x000b, 0 },
            { 0x004f, 0 },
        };

        const RegWrite Dsi0A02[] =
        {
            { 0x003d, 0x040a0e05 },
            { 0x003e, 0x00030118 },
            { 0x003f, 0x00190a14 },
            { 0x0044, 0x2000ffff },
            { 0x0045, 0x13432000 },
            { 0x0046, 0 },
            { 0x000f, 0x00102003 },
            { 0x0010, 0x00000031 },
            { 0x000b, 0x00000001 },
            { 0x0012, 0x00000040 },
            { 0x0013, 0 },
            { 0x0014, 0 },
            { 0x001a, 0 },
        };

        const RegWrite DsiChangeVgl[] =
        {
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x9483ffb9 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00002139 }, // SET_RD5h
            { 0x000a, 0x191919d5 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x00000019 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000b39 }, // SET_RD1h
            { 0x000a, 0x4f0f41b1 },
            { 0x000a, 0xf179a433 },
            { 0x000a, 0x00002d81 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x000000b9 },
            { 0x0013, 0x00000002 },
        };

        const RegWrite DsiChangeVglAuo[] =
        {
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x9483ffb9 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00002c39 }, // SET_RD5h
            { 0x000a, 0x191919d5 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00002c39 }, // SET_RD6h
            { 0x000a, 0x191919d6 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x000a, 0x19191919 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000b39 }, // SET_RB1h
            { 0x000a, 0x711148b1 },
            { 0x000a, 0x71543209 },
            { 0x000a, 0x00114d31 },
            { 0x0013, 0x00000002 },
            { 0x000a, 0x00000439 }, // SET_RB9h
            { 0x000a, 0x000000b9 },
            { 0x0013, 0x00000002 },
        };
#endif

        LcdVendor    g_LcdVendor;
        spl::SocType g_SocType;



        void WriteRegisters(uintptr_t baseAddress, const RegWrite* pWrites, int numWrites)
        {
            for( int i = 0; i < numWrites; ++i )
            {
                reg::Write(baseAddress + (pWrites[i].offset * sizeof(Bit32)), pWrites[i].value);
            }
        }

        void WriteRegisters(uintptr_t baseAddress, const RegWrite* pWritesForErista, int numWritesForErista, const RegWrite* pWritesForMariko, int numWritesForMariko)
        {
            switch (g_SocType)
            {
                case spl::SocType_Erista:
                    WriteRegisters(baseAddress, pWritesForErista, numWritesForErista);
                    break;
                case spl::SocType_Mariko:
                    WriteRegisters(baseAddress, pWritesForMariko, numWritesForMariko);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }

        void WaitForDsiCommandModeReady()
        {
            os::Tick timeout = os::GetSystemTick() + os::ConvertToTick(TimeSpan::FromMicroSeconds(DsiWaitForCommandMicroSecondsMax));

            for(;;)
            {
                if( os::GetSystemTick() >= timeout )
                {
                    break;
                }
                if( reg::Read(NN_BLSM_REG_DSI + (DSI_DSI_TRIGGER_0 * 4)) == 0 )
                {
                    break;
                }
            }

            os::SleepThread(TimeSpan::FromMicroSeconds(DsiWaitForCommandCompletionMicroSeconds));
        }

        void WaitForDsiBta()
        {
            os::Tick timeout = os::GetSystemTick() + os::ConvertToTick(TimeSpan::FromMicroSeconds(DsiStatusPollingDurationMicroSecondsMax));

            for(;;)
            {
                if( os::GetSystemTick() >= timeout )
                {
                    break;
                }
                if( (reg::Read(NN_BLSM_REG_DSI + (DSI_HOST_DSI_CONTROL_0 * 4)) & 0x8) == 0 )
                {
                    break;
                }
            }
        }

        void BitBlt(int offsetX, int offsetY, int width, int height, const Bit32* buffer, size_t size) NN_NOEXCEPT
        {
            // const int DisplayWidth = 720;
            const int DisplayHeight = 1280;
            const int DisplayStride = 768;

            // NN_SDK_LOG("offsetX %d, offsetY %d, width %d, height %d, size %lld\n", offsetX, offsetY, width, height, size);
            NN_SDK_ASSERT(width * height * sizeof(Bit32) <= size);

            memset(g_pFrameBuffer, 0, FrameBufferSize);
            for( int y = 0; y < height; ++y )
            {
                for( int x = 0; x < width; ++x )
                {
                    int dispX = offsetY + y;
                    int dispY = DisplayHeight - (offsetX + x);
                    int frameBufferOffset = dispY * DisplayStride + dispX;
                    int bufferOffset = y * width + x;
                    // NN_SDK_LOG("dispX %d, dispY %d, frameBufferOffset %d, bufferOffset %d, buffer[bufferOffset] 0x%08x\n", dispX, dispY, frameBufferOffset, bufferOffset, buffer[bufferOffset]);
                    g_pFrameBuffer[frameBufferOffset] = buffer[bufferOffset];
                }
            }
            nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);
        }

        void SetupFrameBuffer() NN_NOEXCEPT
        {
            if (g_pFrameBuffer == nullptr) // 一度だけ実行
            {
                const uintptr_t frameBufferAddress = reinterpret_cast<uintptr_t>(g_FrameBuffer);
                const uintptr_t alingedFrameBufferAddress = util::align_up(frameBufferAddress, DeviceAddressSpaceBlockSize);

                g_pFrameBuffer = reinterpret_cast<Bit32*>(alingedFrameBufferAddress);

                // ゼロクリアされていなかった場合に表示直後ノイズが乗る可能性があるため、明示的にクリアしておく (IAAA-3561)
                memset(g_pFrameBuffer, 0, FrameBufferSize);
                nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::CreateDeviceAddressSpace(&g_DeviceAddressSpace, 0, (1ull << 32)) );

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::AttachDeviceAddressSpace(&g_DeviceAddressSpace, dd::DeviceName_Dc) );

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::MapDeviceAddressSpaceAligned(
                        &g_DeviceAddressSpace,
                        dd::GetCurrentProcessHandle(),
                        alingedFrameBufferAddress,
                        FrameBufferSize,
                        FrameBufferAddress,
                        dd::MemoryPermission_ReadOnly ));
            }
            else
            {
                memset(g_pFrameBuffer, 0, FrameBufferSize);
                nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);
            }
        }

        void TeardownFrameBuffer() NN_NOEXCEPT
        {
            if (g_pFrameBuffer != nullptr) // 一度だけ実行
            {
                const uintptr_t frameBufferAddress = reinterpret_cast<uintptr_t>(g_FrameBuffer);
                const uintptr_t alingedFrameBufferAddress = util::align_up(frameBufferAddress, DeviceAddressSpaceBlockSize);

                dd::UnmapDeviceAddressSpace(
                    &g_DeviceAddressSpace,
                    dd::GetCurrentProcessHandle(),
                    alingedFrameBufferAddress,
                    FrameBufferSize,
                    FrameBufferAddress);
                dd::DetachDeviceAddressSpace(&g_DeviceAddressSpace, dd::DeviceName_Dc);
                dd::DestroyDeviceAddressSpace(&g_DeviceAddressSpace);

                g_pFrameBuffer = nullptr;
            }
        }

        nn::Result WriteI2c(nn::i2c::driver::I2cSession& session, const uint8_t addr, const uint8_t data) NN_NOEXCEPT
        {
            return WriteI2cRegister(session, &addr, sizeof(addr), &data, sizeof(data));
        }
    }
    // anonymous namespace


    void SetupDisplay()
    {
        g_SocType = nn::boot::GetSocType();
        bool isMariko = (g_SocType == nn::spl::SocType_Mariko);

        NN_BLSM_REG_DISP1       = dd::QueryIoMappingAddress(0x54200000,  0x3000);
        NN_BLSM_REG_DSI         = dd::QueryIoMappingAddress(0x54300000,  0x1000);
        NN_BLSM_REG_CAR         = dd::QueryIoMappingAddress(0x60006000,  0x1000);
        NN_BLSM_REG_GPIO        = dd::QueryIoMappingAddress(0x6000D000,  0x1000);
        NN_BLSM_REG_APB_MISC    = dd::QueryIoMappingAddress(0x70000000,  0x1000);
        NN_BLSM_REG_MIPI_CAL    = dd::QueryIoMappingAddress(0x700E3000,  0x1000);

        SetupFrameBuffer();

        // #1 PMIC turn ON V_VDD-1V8

        nn::i2c::driver::I2cSession i2cSession;
        nn::i2c::driver::Initialize();
        nn::i2c::driver::OpenSession(&i2cSession, nn::i2c::I2cDevice_Max77620Pmic);

        if (isMariko)
        {
            WriteI2c(i2cSession, 0x18, 0x3A);
            WriteI2c(i2cSession, 0x1F, 0x71);
        }

        /* Enable DSI power */
        //Data = 0xd0;
        //NvTbootI2cDeviceWrite(NvTbootI2c5, MAX77620_PMU_I2C_SLAVE_ADDRESS, 0x23, &Data, sizeof(Data));
        WriteI2c(i2cSession, 0x23, 0xD0);

        nn::i2c::driver::CloseSession(i2cSession);
        nn::i2c::driver::Finalize();

        /* Taking care of reset and clock enables */
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_H_CLR,
            VALUE, CLR_MIPI_CAL_RST,    1,
            VALUE, CLR_DSI_RST,         1 );
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_ENB_H_SET,
            VALUE, SET_CLK_ENB_MIPI_CAL,1,
            VALUE, SET_CLK_ENB_DSI,     1 );

        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_L_CLR,
            VALUE, CLR_HOST1X_RST,      1,
            VALUE, CLR_DISP1_RST,       1 );
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_ENB_L_SET,
            VALUE, SET_CLK_ENB_HOST1X,  1,
            VALUE, SET_CLK_ENB_DISP1,   1 );

        /* UART_FST_MIPI_CAL */
        NN_REG_WRITE_1(CAR, CLK_RST_CONTROLLER_CLK_ENB_X_SET,
            VALUE,  SET_CLK_ENB_UART_FST_MIPI_CAL,  1);
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_SOURCE_UART_FST_MIPI_CAL,
            NAME,   UART_FST_MIPI_CAL_CLK_SRC,      PLLP_OUT3,
            VALUE,  UART_FST_MIPI_CAL_CLK_DIVISOR,  10 );

        /* DSIA_LP */
        NN_REG_WRITE_1(CAR, CLK_RST_CONTROLLER_CLK_ENB_W_SET, VALUE, SET_CLK_ENB_DSIA_LP, 1);
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_SOURCE_DSIA_LP,
            NAME,   DSIA_LP_CLK_SRC,        PLLP_OUT0,
            VALUE,  DSIA_LP_CLK_DIVISOR,    10 );

        /* DPD */
        dd::WriteIoRegister(PmcPhysicalAddress + APBDEV_PMC_IO_DPD_REQ_0,
            NN_REG_MAKE_BITS_BY_NAME(APBDEV_PMC_IO_DPD_REQ_0_CODE, DPD_OFF) );
        dd::WriteIoRegister(PmcPhysicalAddress + APBDEV_PMC_IO_DPD2_REQ_0,
            NN_REG_MAKE_BITS_BY_NAME(APBDEV_PMC_IO_DPD2_REQ_0_CODE, DPD_OFF) );

        /* LCD pinmux */
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_NFC_EN_0,    TRISTATE, PASSTHROUGH); // LCD_PWR_N_EN
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_NFC_INT_0,   TRISTATE, PASSTHROUGH); // LCD_PWR_P_EN

        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_LCD_BL_PWM_0,TRISTATE, PASSTHROUGH); // LCD_BL_PWM
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_LCD_BL_EN_0, TRISTATE, PASSTHROUGH); // LCD_BL_PWM_EN
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_LCD_RST_0,   TRISTATE, PASSTHROUGH); // LCD_RST

        /* LCD power */
        reg::ReadSetWrite(GPIO_PI_CNF, 0, 2, 0x3);
        reg::ReadSetWrite(GPIO_PI_OE,  0, 2, 0x3);

        // #2 turn ON PAVDD
        reg::ReadSetWrite(GPIO_PI_OUT, 0, 1, 0x1);

        // #3 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #4 trun ON NAVDD
        reg::ReadSetWrite(GPIO_PI_OUT, 1, 1, 0x1);

        // #5 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #6 DSI: LP-00 → LP-11

        /* LCD reset and backlight */
        reg::ReadSetWrite(GPIO_PV_CNF, 0, 3, 0x7);
        reg::ReadSetWrite(GPIO_PV_OE,  0, 3, 0x7);
        reg::ReadSetWrite(GPIO_PV_OUT, 1, 1, 0x1);

        /* Clear dsi pad power down */
        reg::Write(NN_BLSM_REG_MIPI_CAL + 0x060, 0);
        if (isMariko)
        {
            reg::Write(NN_BLSM_REG_MIPI_CAL + 0x058, 0);
            reg::Write(NN_BLSM_REG_APB_MISC + 0xac0, 0);
        }

        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_CAR,   Plld01);
        WRITE_REGISTERS(                 NN_BLSM_REG_DISP1, Dc01);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI,   Dsi01Top01);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_CAR,   Dsi01Top02);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI,   Dsi01Top03);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI,   Dsi01Middle);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI,   Dsi01Bottom01);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI,   DsiHostDsiCtrl);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI,   Dsi01Bottom02);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI,   DsiHostDsiCtrl);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI,   Dsi01Bottom03);

        // #7 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #8 RESET: L → H
        /* Reset L->H */
        reg::ReadSetWrite(GPIO_PV_OUT, 2, 1, 0x1);

        // #9 wait 60 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(60));

        // #10 issue "read_LCD_ID"

        reg::Write(NN_BLSM_REG_DSI + (0x3f * 4), 0x50204);
        reg::Write(NN_BLSM_REG_DSI + (0xa * 4), 0x000337);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);
        WaitForDsiCommandModeReady();

        reg::Write(NN_BLSM_REG_DSI + (0xa * 4), 0x000406);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);
        WaitForDsiCommandModeReady();

        reg::Write(NN_BLSM_REG_DSI + (0xf * 4), 0x200b);
        WaitForDsiBta();
        os::SleepThread(TimeSpan::FromMilliSeconds(5));

        Bit32 response[3];
        response[0] = reg::Read(NN_BLSM_REG_DSI + (0x9 * 4));
        response[1] = reg::Read(NN_BLSM_REG_DSI + (0x9 * 4));
        response[2] = reg::Read(NN_BLSM_REG_DSI + (0x9 * 4));

        /* Read back 1st byte of panel ID. 0x20 is for the 2nd panel (Innolux) */
        g_LcdVendor = static_cast<LcdVendor>(response[2] & 0xff);

        // #11 issue "exit_sleep" for JDI
        // #11 issue "exit_sleep_INX" for INX and AUO

        if( g_LcdVendor == LcdVendor_Jdi )
        {
            WRITE_REGISTERS(NN_BLSM_REG_DSI, DsiExitSleepForJdi);
        }

        reg::Write(NN_BLSM_REG_DSI + (0x0a * 4), 0x001105);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);

        // #12 wait 180 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(180));

        // #13 issue "set_display_on"
        reg::Write(NN_BLSM_REG_DSI + (0x0a * 4), 0x002905);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);

        // #14 wait 20 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(20));

        // #15 DSI: LP-11 → HS_mode
        // #16 start Video Mode transmission

        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_CAR, Plld02);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI, Dsi0201);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI, DsiHostDsiCtrl);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI, Dsi0202);

        /* Pixel clock divider */
        reg::Write(NN_BLSM_REG_DISP1 + (0x42e * 4), 0x4);

        WRITE_REGISTERS(NN_BLSM_REG_DSI,        Dsi03);
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        WRITE_REGISTERS(NN_BLSM_REG_MIPI_CAL,   MipiCal01);

        /* PrivDsiMipiCalibrationT21x */
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_MIPI_CAL, MipiCal02);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI,      Dsi04);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_MIPI_CAL, MipiCal03);
        WRITE_REGISTERS(                 NN_BLSM_REG_MIPI_CAL, MipiCal04);

        /* WAR B01: PrivDsiMipiCalibrationT21x */
        if (isMariko)
        {
            WRITE_REGISTERS(NN_BLSM_REG_MIPI_CAL, MipiCal02Mariko);
            WRITE_REGISTERS(NN_BLSM_REG_DSI,      Dsi04Mariko);
            WRITE_REGISTERS(NN_BLSM_REG_MIPI_CAL, MipiCal03Mariko);
            WRITE_REGISTERS(NN_BLSM_REG_MIPI_CAL, MipiCal04);
        }

        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        WRITE_REGISTERS(NN_BLSM_REG_DISP1,      Dc02);
        WRITE_REGISTERS(NN_BLSM_REG_DISP1,      Dc04);

        // #17 wait 35 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(35));

        g_IsInitialized = true;
    } // NOLINT(impl/function_size)

    void ShowDisplay(int offsetX, int offsetY, int width, int height, const Bit32* buffer, size_t size)
    {
        if (!g_IsInitialized)
        {
            return;
        }

        BitBlt(offsetX, offsetY, width, height, buffer, size);

        // #18 turn ON BL
        /* Turn backlight on */
        reg::ReadSetWrite(GPIO_PV_OUT, 0, 1, 0x1);
    }

    void ShutdownDisplay()
    {
        if (!g_IsInitialized)
        {
            return;
        }

        // #19

        // #20 BL: H → L
        /* Turn backlight off */
        reg::ReadSetWrite(GPIO_PV_OUT, 0, 1, 0x0);

        // #21 issue "set_display_off" command at blank line starting with 1st VS vblank
        reg::Write(NN_BLSM_REG_DSI + (0x4e * 4), 0x1);
        reg::Write(NN_BLSM_REG_DSI + (0xa * 4), 0x002805);

#if !defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
        // #22 wait 3 frame
        // #23 stop Video Mode transmittion at VFP first blank line; DSI: HS_mode → LP-11
        /* Wait for 4 frames */
        {
            const uintptr_t regAddrX = dd::QueryIoMappingAddress(0x500030a4,  4);
            Bit32 syncptValue = reg::Read(regAddrX);
            while( reg::Read(regAddrX) < (syncptValue + 5) ){}
        }

        reg::Write(NN_BLSM_REG_DISP1 + (0x40 * 4), 0x5);
        reg::Write(NN_BLSM_REG_DSI + (0x4e * 4), 0x0);
        WRITE_REGISTERS(NN_BLSM_REG_DISP1,  Dc0A);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_CAR, Plld01);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI, Dsi0A01);
        WRITE_REGISTERS_DEPENDING_ON_SOC(NN_BLSM_REG_DSI, DsiHostDsiCtrl);
        WRITE_REGISTERS(                 NN_BLSM_REG_DSI, Dsi0A02);

        // #24 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #25 issue "change_VGL" for JDI
        // #25 issue "change_VGL_AUO" for AUO
        /* Execute this sequence only for panel #1 */
        if (g_LcdVendor == LcdVendor_Jdi)
        {
            WRITE_REGISTERS(NN_BLSM_REG_DSI, DsiChangeVgl);
        }
        else if (g_LcdVendor == LcdVendor_Auo)
        {
            WRITE_REGISTERS(NN_BLSM_REG_DSI, DsiChangeVglAuo);
            os::SleepThread(TimeSpan::FromMilliSeconds(5));
        }

        // #26 issue "enter_sleep"
        reg::Write(NN_BLSM_REG_DSI + (0xa * 4), 0x001005);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);

        // #27 wait 50 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(50));

        // #28 RESER: H → L
        /* Reset H->L */
        reg::ReadSetWrite(GPIO_PV_OUT, 2, 1, 0x0);

        // #29 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #30 trun OFF NAVDD
        /* Turn NAVDD off */
        reg::ReadSetWrite(GPIO_PI_OUT, 1, 1, 0x0);

        // #31 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #32 rutn OFF PAVDD
        /* Turn PAVDD off */
        reg::ReadSetWrite(GPIO_PI_OUT, 0, 1, 0x0);

        // #33 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #34 DSI: LP-11 → LP-00

        /* Disable DSI clock */
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_H_SET,
            VALUE, SET_MIPI_CAL_RST,    1,
            VALUE, SET_DSI_RST,         1 );
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_ENB_H_CLR,
            VALUE, CLR_CLK_ENB_MIPI_CAL,1,
            VALUE, CLR_CLK_ENB_DSI,     1 );

        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_L_SET,
            VALUE, SET_HOST1X_RST,      1,
            VALUE, SET_DISP1_RST,       1 );
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_ENB_L_CLR,
            VALUE, CLR_CLK_ENB_HOST1X,  1,
            VALUE, CLR_CLK_ENB_DISP1,   1 );

        reg::Write(NN_BLSM_REG_DSI + (0x4b * 4), 0x10f010f);
        reg::Write(NN_BLSM_REG_DSI + (0xb * 4), 0x0);

        /* Configure the PV00 to PWM0 for backlight wpm functionality later on */
        reg::ReadSetWrite(GPIO_PV_CNF, 0, 1, 0x0);
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_LCD_BL_PWM_0, TRISTATE, TRISTATE);   // LCD_BL_PWM
        NN_REG_UPDATE_BY_NAME(APB_MISC, PINMUX_AUX_LCD_BL_PWM_0, PM,       PWM0);       // LCD_BL_PWM

        /* Note that 1.2V dsi power is left on */

        g_IsInitialized = false;
#endif

#if defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_L_SET,
            VALUE, SET_HOST1X_RST,      1,
            VALUE, SET_DISP1_RST,       1 );

        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_RST_DEV_L_CLR,
            VALUE, CLR_HOST1X_RST,      1,
            VALUE, CLR_DISP1_RST,       1 );
        NN_REG_WRITE_2(CAR, CLK_RST_CONTROLLER_CLK_ENB_L_SET,
            VALUE, SET_CLK_ENB_HOST1X,  1,
            VALUE, SET_CLK_ENB_DISP1,   1 );

        WRITE_REGISTERS(NN_BLSM_REG_DISP1,  Dc01);

        // #7 wait 10 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        // #13 issue "set_display_on"
        reg::Write(NN_BLSM_REG_DSI + (0x0a * 4), 0x002905);
        reg::Write(NN_BLSM_REG_DSI + (0x13 * 4), 0x2);

        /* Pixel clock divider */
        reg::Write(NN_BLSM_REG_DISP1 + (0x42e * 4), 0x4);

        os::SleepThread(TimeSpan::FromMilliSeconds(10));

        WRITE_REGISTERS(NN_BLSM_REG_DISP1,      Dc02);

        // #17 wait 35 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(35));
#endif

        TeardownFrameBuffer();
    } // NOLINT(readability/fn_size)

    void SetupDisplayForSafeMode()
    {
#if defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
        os::SleepThread(TimeSpan::FromSeconds(1));

        // #18 turn ON BL
        /* Turn backlight on */
        reg::ReadSetWrite(GPIO_PV_OUT, 0, 1, 0x1);
#endif
    }

}} // namespace nn::boot
