﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>

#include "xml_param.h"
#include <nn/settings/system/settings_BluetoothDevices.h>


/*
 * Definitions
 */
static tAPP_XML_CONFIG app_xml_config;
static int app_xml_config_ready = 0;

/*
 * Global Variables
 */
tAPP_XML_REM_DEVICE app_xml_remote_devices_db[APP_MAX_NB_REMOTE_STORED_DEVICES];

static tAPP_XML_REM_DEVICE *app_xml_add_dev_db(const BD_ADDR bd_addr);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
static int getDeviceCount()
{
    int deviceNum = 0;

    /* First look in database if this device already exist */
    for (int index = 0; index < APP_MAX_NB_REMOTE_STORED_DEVICES; index++)
    {
        if (app_xml_remote_devices_db[index].in_use != FALSE)
        {
            deviceNum++;
        }
    }

    return deviceNum;
}

const char* toString(const uint8_t *data, int len)
{
    const char dec2hex[] = "0123456789abcdef";
    static char string[2][16 * 3];
    static int si = 0;
    si ^= 1;
    int i;
    for (i=0; i<len && i<16; i++)
    {
        string[si][i * 3  ] = dec2hex[data[i]>>4];
        string[si][i * 3 + 1] = dec2hex[data[i]&0xF];
        string[si][i * 3 + 2] = ':';
    }
    string[si][i * 3 - 1] = 0;

    return string[si];
}
#endif

/*******************************************************************************
 **
 ** Function        app_xml_init
 **
 ** Description
 **
 ** Returns
 **
 *******************************************************************************/
void app_xml_init()
{
    memset(app_xml_remote_devices_db, 0, sizeof(app_xml_remote_devices_db));
    for (int i=0; i<APP_MAX_NB_REMOTE_STORED_DEVICES; i++)
        app_xml_remote_devices_db[i].handle = BSA_HH_INVALID_HANDLE;
}

/*******************************************************************************
 **
 ** Function        app_xml_add_db
 **
 ** Description
 **
 ** Returns
 **
 *******************************************************************************/
int app_xml_add_db(const nn::settings::system::BluetoothDevicesSettings* pBds)
{
    BD_ADDR nullBdAddr = {0, 0, 0, 0, 0, 0};

    if (isSameBdAddr(nullBdAddr, pBds->bd_addr))
    {
        NN_SDK_LOG("[bluetooth] %s: Null BD_ADDR - skipping", __func__);
        return -1;
    }

    tAPP_XML_REM_DEVICE *pDev = app_xml_add_dev_db(pBds->bd_addr);
    if (pDev == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: failed to add device", __func__);
        return -1;
    }

    memset(pDev, 0, sizeof(tAPP_XML_REM_DEVICE));

    memcpy(pDev->bd_addr,         pBds->bd_addr,         sizeof(pDev->bd_addr));
    memcpy(pDev->name,            pBds->device_name,     sizeof(pBds->device_name)); // pBds->device_name: 32, pDev->name: 249 bytes
    memcpy(pDev->class_of_device, pBds->class_of_device, sizeof(pDev->class_of_device));
    memcpy(pDev->link_key,        pBds->link_key,        sizeof(pDev->link_key));
    memcpy(pDev->brr,             pBds->brr,             sizeof(pDev->brr));
    pDev->link_key_present = pBds->link_key_present;
    pDev->trusted_services = pBds->trusted_services;
    pDev->vid              = pBds->vid;
    pDev->pid              = pBds->pid;
    pDev->version          = pBds->version;
    pDev->key_type         = pBds->key_type;
    pDev->device_type      = pBds->device_type;
    pDev->sub_class        = pBds->sub_class;
    pDev->attr_mask        = 0x8075; //pBds->attribute_mask;
    pDev->brr_size         = pBds->brr_size;

    if (pBds->descriptor_length <= sizeof(pDev->descriptor))
    {
        pDev->descriptor_size = pBds->descriptor_length;
        memcpy(pDev->descriptor, pBds->descriptor, sizeof(pDev->descriptor));
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: invalid desciptor_size (%d)", __func__, pBds->descriptor_length);
        pDev->descriptor_size = 0;
        memset(pDev->descriptor, 0, sizeof(pDev->descriptor));
    }

    pDev->handle = BSA_HH_INVALID_HANDLE;
    pDev->in_use = TRUE;
    return 0;

}

/*******************************************************************************
 **
 ** Function        app_xml_remove_db
 **
 ** Description
 **
 ** Returns
 **
 *******************************************************************************/
int app_xml_remove_db(const BD_ADDR bd_addr)
{
    tAPP_XML_REM_DEVICE *pDev = NULL;

    if ((pDev = app_xml_find_dev_db(bd_addr)) == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: failed to remove from table - not found\n", __func__);
        return -1;
    }
    else
    {
        memset(pDev, 0, sizeof(tAPP_XML_REM_DEVICE));
        pDev->handle = BSA_HH_INVALID_HANDLE;
        NN_SDK_LOG("[bluetooth] %s: device count=%d\n", __func__, getDeviceCount());

        return 0;
    }
}


/*******************************************************************************
 **
 ** Function        app_xml_add_dev_db
 **
 ** Description     Update link key information for a device
 **
 ** Returns         0 if unsuccessful, pointer to database otherwise
 **
 *******************************************************************************/
static tAPP_XML_REM_DEVICE *app_xml_add_dev_db(const BD_ADDR bd_addr)
{
    int index;
    tAPP_XML_REM_DEVICE *pDev = NULL;

    /* First look in Database if this device already exist */
    if ((pDev = app_xml_find_dev_db(bd_addr)) != NULL)
    {
        return pDev;
    }

    /* Look for a free location in database */
    for (index = 0; index < APP_MAX_NB_REMOTE_STORED_DEVICES; index++)
    {
        /* If this one is free */
        if (app_xml_remote_devices_db[index].in_use == FALSE)
        {
            memset(&app_xml_remote_devices_db[index], 0, sizeof(app_xml_remote_devices_db[0]));
            /* Let's use it */
            app_xml_remote_devices_db[index].in_use = TRUE;
            app_xml_remote_devices_db[index].handle = BSA_HH_INVALID_HANDLE;
            copyBdAddr(app_xml_remote_devices_db[index].bd_addr, bd_addr);

            NN_SDK_LOG("[bluetooth] %s: device count=%d\n", __func__, getDeviceCount());

            return &app_xml_remote_devices_db[index];
        }
    }

    NN_SDK_LOG("[bluetooth] %s: failed to find empty space in table\n", __func__);
    return NULL;
}

/*******************************************************************************
 **
 ** Function        app_xml_find_dev_db
 **
 ** Description     Find the pointer to the device info in the list
 **
 ** Returns         Pointer to the device in the D, NULL if not found
 **
 *******************************************************************************/
tAPP_XML_REM_DEVICE *app_xml_find_dev_db(const BD_ADDR bd_addr)
{
    int index;

    /* First look in Database if this device already exist */
    for (index = 0; index < APP_MAX_NB_REMOTE_STORED_DEVICES; index++)
    {
        if ((app_xml_remote_devices_db[index].in_use) &&
            (isSameBdAddr(app_xml_remote_devices_db[index].bd_addr, bd_addr)))
        {
            return &app_xml_remote_devices_db[index];
        }
    }
    return NULL;
}

/*******************************************************************************
 **
 ** Function        app_xml_update_key_db
 **
 ** Description     Update link key information for a device
 **
 ** Returns         0 if successful, error code otherwise
 **
 *******************************************************************************/
int app_xml_update_key_db(const BD_ADDR bd_addr, const LINK_KEY link_key,
        unsigned char key_type)
{
    tAPP_XML_REM_DEVICE *pDev = app_xml_add_dev_db(bd_addr);
    if (pDev)
    {
        memcpy(pDev->link_key, link_key, sizeof(pDev->link_key));
        pDev->link_key_present = TRUE;
        pDev->key_type = key_type;
        return 0;
    }
    return -1;
}

/*******************************************************************************
 **
 ** Function        app_xml_add_trusted_services_db
 **
 ** Description     Add a trusted service for a device
 **
 ** Returns         0 if successful, error code otherwise
 **
 *******************************************************************************/
int app_xml_add_trusted_services_db(const BD_ADDR bd_addr, tBSA_SERVICE_MASK trusted_services)
{
    tAPP_XML_REM_DEVICE *pDev = app_xml_add_dev_db(bd_addr);
    if (pDev)
    {
        pDev->trusted_services |= trusted_services;
        return 0;
    }
    return -1;
}

/*******************************************************************************
 **
 ** Function        app_xml_update_name_db
 **
 ** Description     Update link key information for a device
 **
 ** Returns         0 if successful, error code otherwise
 **
 *******************************************************************************/
int app_xml_update_name_db(const BD_ADDR bd_addr, const BD_NAME name)
{
    tAPP_XML_REM_DEVICE *pDev = app_xml_add_dev_db(bd_addr);
    if (pDev)
    {
        strncpy((char *) pDev->name, (char *) name, sizeof(pDev->name) - 1);
        pDev->name[sizeof(pDev->name) - 1] = '\0';
        return 0;
    }
    return -1;
}

/*******************************************************************************
 **
 ** Function        app_xml_update_cod_db
 **
 ** Description     Update Class of device information for a device
 **
 ** Returns         0 if successful, error code otherwise
 **
 *******************************************************************************/
int app_xml_update_cod_db(const BD_ADDR bd_addr, DEV_CLASS class_of_device)
{
    tAPP_XML_REM_DEVICE *pDev = app_xml_add_dev_db(bd_addr);
    if (pDev)
    {
        pDev->class_of_device[0] = class_of_device[0];
        pDev->class_of_device[1] = class_of_device[1];
        pDev->class_of_device[2] = class_of_device[2];
        return 0;
    }
    return -1;
}


/*******************************************************************************
 **
 ** Function         app_xml_find_by_handle
 **
 **
 *******************************************************************************/
tAPP_XML_REM_DEVICE *app_xml_find_by_handle(tBSA_HH_HANDLE handle)
{
    int cnt;
    for (cnt = 0; cnt < APP_MAX_NB_REMOTE_STORED_DEVICES; cnt++)
    {
        if (app_xml_remote_devices_db[cnt].handle == handle)
        {
            return &app_xml_remote_devices_db[cnt];
        }
    }
    return NULL;
}


/*******************************************************************************
 **
 ** Function        app_xml_display_devices
 **
 ** Description     Display a list of device stored in database
 **
 ** Returns         0 if successful, error code otherwise
 **
 *******************************************************************************/
int app_xml_display_devices()
{
    int index, deviceNum;

    /* First look in database if this device already exist */
    for (index = 0, deviceNum = 0; index < APP_MAX_NB_REMOTE_STORED_DEVICES; index++)
    {
        if (app_xml_remote_devices_db[index].in_use != FALSE)
        {
            NN_SDK_LOG("[bluetooth] Dev:%d\n", deviceNum);
            NN_SDK_LOG("\t\tBdaddr:%s\n", toString(app_xml_remote_devices_db[index].bd_addr, 6));
            NN_SDK_LOG("\t\tName:%s\n", app_xml_remote_devices_db[index].name);
            NN_SDK_LOG("\t\tClassOfDevice:%s\n", toString(app_xml_remote_devices_db[index].class_of_device, 3));
            NN_SDK_LOG("\t\tTrusted/Available Services:%x %x\n",
                    (int) app_xml_remote_devices_db[index].trusted_services,
                    (int) app_xml_remote_devices_db[index].available_services);

            NN_SDK_LOG("\t\tKeyType/IoCap/Version/DescSize 0x%x 0x%x 0x%x %d\n",
                    app_xml_remote_devices_db[index].key_type,
                    app_xml_remote_devices_db[index].io_cap,
                    app_xml_remote_devices_db[index].version,
                    app_xml_remote_devices_db[index].descriptor_size);

            if (app_xml_remote_devices_db[index].descriptor_size > 0)
                NN_SDK_LOG("\t\tDescriptor %s\n", toString(app_xml_remote_devices_db[index].descriptor,
                    app_xml_remote_devices_db[index].descriptor_size));

            if (app_xml_remote_devices_db[index].link_key_present != FALSE)
                NN_SDK_LOG("\t\tLink Key present:TRUE %s\n", toString(app_xml_remote_devices_db[index].link_key, 16));
            else
                NN_SDK_LOG("\t\tLink Key present:FALSE\n");

            NN_SDK_LOG("\t\tsubcl:%x attrmask:%x infomask:%x handle:%d\n",
                    app_xml_remote_devices_db[index].sub_class,
                    app_xml_remote_devices_db[index].attr_mask,
                    app_xml_remote_devices_db[index].info_mask,
                    app_xml_remote_devices_db[index].handle);

            NN_SDK_LOG("\t\tPid:%x Vid:%x\n", app_xml_remote_devices_db[index].pid,
                    app_xml_remote_devices_db[index].vid);

            NN_SDK_LOG("\t\tDescSize:%d Desc:%s\n",
                    app_xml_remote_devices_db[index].descriptor_size,
                    toString(app_xml_remote_devices_db[index].descriptor, 16));

            NN_SDK_LOG("\t\tBrrSize:%d Brr:%s\n",
                    app_xml_remote_devices_db[index].brr_size,
                    toString(app_xml_remote_devices_db[index].brr, 9));

            NN_SDK_LOG("\t\tMaxLat:%d MinTout:%d SupTout:%d\n",
                    app_xml_remote_devices_db[index].ssr_max_latency,
                    app_xml_remote_devices_db[index].ssr_min_tout,
                    app_xml_remote_devices_db[index].supervision_tout);

            deviceNum++;
        }
    }
    return 0;
}


/*******************************************************************************
 **
 ** Function        app_read_xml_config
 **
 ** Description     This function is used to read the XML Bluetooth configuration file
 **
 ** Parameters      p_xml_config: pointer to the structure to fill
 **
 ** Returns         0 if successful, -1 otherwise
 **
 *******************************************************************************/
int app_read_xml_config(tAPP_XML_CONFIG *p_xml_config)
{
    if (app_xml_config_ready == 0 || p_xml_config == NULL)
    {
        //NN_SDK_LOG("app_xml_config_ready failed: %d", app_xml_config_ready);
        return -1;
    }
    else
    {
        *p_xml_config = app_xml_config;
#if 0
        NN_SDK_LOG("app_read_xml_config:");
        NN_SDK_LOG("\tEnable:%d", p_xml_config->enable);
        NN_SDK_LOG("\tDiscoverable:%d", p_xml_config->discoverable);
        NN_SDK_LOG("\tConnectable:%d", p_xml_config->connectable);
        NN_SDK_LOG("\tName:%s", p_xml_config->name);
        NN_SDK_LOG ("\tBdaddr %02x:%02x:%02x:%02x:%02x:%02x",
                p_xml_config->bd_addr[0], p_xml_config->bd_addr[1],
                p_xml_config->bd_addr[2], p_xml_config->bd_addr[3],
                p_xml_config->bd_addr[4], p_xml_config->bd_addr[5]);
        NN_SDK_LOG("\tClassOfDevice:%02x:%02x:%02x", p_xml_config->class_of_device[0],
                p_xml_config->class_of_device[1], p_xml_config->class_of_device[2]);
        NN_SDK_LOG("\tDefault PIN (%d):%s", p_xml_config->pin_len, p_xml_config->pin_code);
#endif
    }
    return 0;
}

/*******************************************************************************
 **
 ** Function        app_write_xml_config
 **
 ** Description     This function is used to write the XML Bluetooth configuration file
 **
 ** Parameters      p_xml_config: pointer to the structure to write
 **
 ** Returns         0 if successful, -1 otherwise
 **
 *******************************************************************************/
int app_write_xml_config(const tAPP_XML_CONFIG *p_xml_config)
{
    if (p_xml_config == NULL)
    {
        NN_SDK_LOG("[bluetooth] %s: failed: null pointer", __func__);
        return -1;
    }
    else
    {
        app_xml_config = *p_xml_config;
        app_xml_config_ready = 1;
#if 0
        NN_SDK_LOG("\tEnable:%d", p_xml_config->enable);
        NN_SDK_LOG("\tDiscoverable:%d", p_xml_config->discoverable);
        NN_SDK_LOG("\tConnectable:%d", p_xml_config->connectable);
        NN_SDK_LOG("\tName:%s", p_xml_config->name);
        NN_SDK_LOG ("\tBdaddr %02x:%02x:%02x:%02x:%02x:%02x",
                p_xml_config->bd_addr[0], p_xml_config->bd_addr[1],
                p_xml_config->bd_addr[2], p_xml_config->bd_addr[3],
                p_xml_config->bd_addr[4], p_xml_config->bd_addr[5]);
        NN_SDK_LOG("\tClassOfDevice:%02x:%02x:%02x", p_xml_config->class_of_device[0],
                p_xml_config->class_of_device[1], p_xml_config->class_of_device[2]);
        NN_SDK_LOG("\tDefault PIN (%d):%s", p_xml_config->pin_len, p_xml_config->pin_code);
#endif
    }
    return 0;
}


