﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <cstdlib>
#include <cstring>
#include "bluetooth.h"
#include "bluetooth_queue.h"
#include "bluetooth_le_client.h"
#include "circ_buffer.h"
#include <nn/nn_SystemThreadDefinition.h>

/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/
static uint8_t      hidBuf[20000];
static uint8_t      cbBuf[10000];
static uint8_t      bleBuf[10000];
static uint8_t      bleCoreBuf[10000];
static uint8_t      bleHidBuf[10000];

/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/
typedef struct {
    bool                        valid;
    int                         type;
    CircBufferC                 msgQueue;
} HAL_MSG_QUEUE_CONTEXT;

static nn::os::ThreadType          halMsgThread;

static NN_ALIGNAS( 4096 ) char StackOfHidThread[ THREAD_STACK_SIZE ];
static HAL_MSG_QUEUE_CONTEXT HidDataContext;
static HAL_MSG_QUEUE_CONTEXT CbDataContext;
static HAL_MSG_QUEUE_CONTEXT BleDataContext;
static HAL_MSG_QUEUE_CONTEXT BleCoreContext;
static HAL_MSG_QUEUE_CONTEXT BleHidContext;

static struct {
    bool     inProgress;  // flag to prevent unneeded NAND writes
    bool     isNew;       // re-pair of existing device, or new device?
    Btbdaddr bdAddr;
    LINK_KEY linkKey;
    UINT8    keyType;
} bondState = { .inProgress = false };

static BluetoothHhCallbacks                  SfHhCallbacks;
static BluetoothCallbacks                    SfCallbacks;
static BluetoothExtCallbacks                 SfExtCallbacks;
static BluetoothLeCallbacks                  SfBleCallbacks;


static void ProcessCallbackQueue();
static bool ProcessHidQueue();
static void ProcessBleQueue();
static void ProcessBleCoreQueue();
static void ProcessBleHidQueue();


/*******************************************************************************
 **
 ** Function         BtHalCallbackHidRegister
 **
 ** Description      Register all the HID callback's
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalCallbackHidRegister(BluetoothHhCallbacks* callbacks)
{
    if (callbacks)
    {
        NN_SDK_REQUIRES(callbacks->size == sizeof(*callbacks));
        SfHhCallbacks = *callbacks;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: no callbacks registered\n",__func__);
        return;
    }
}

/*******************************************************************************
 **
 ** Function         BtHalCallbackRegisterCore
 **
 ** Description      Register all the main Bluetooh HAL callback's
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalCallbackRegisterCore(BluetoothCallbacks* callbacks)
{
    if (callbacks)
    {
        NN_SDK_REQUIRES(callbacks->size == sizeof(*callbacks));
        SfCallbacks = *callbacks;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: no callbacks registered\n",__func__);
        return;
    }
}

/*******************************************************************************
 **
 ** Function         BtHalCallbackRegisterExtension
 **
 ** Description      Register all the Extension callback's
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalCallbackRegisterExtension(BluetoothExtCallbacks* callbacks)
{
    if (callbacks)
    {
        NN_SDK_REQUIRES(callbacks->size == sizeof(*callbacks));
        SfExtCallbacks = *callbacks;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: no callbacks registered\n",__func__);
        return;
    }
}

/*******************************************************************************
 **
 ** Function         BtHalCallbackRegisterLowEnergy
 **
 ** Description      Register all the low energy callback's
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalCallbackRegisterLowEnergy(BluetoothLeCallbacks* callbacks)
{
    if (callbacks)
    {
        NN_SDK_REQUIRES(callbacks->size == sizeof(*callbacks));
        SfBleCallbacks = *callbacks;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: no callbacks registered\n",__func__);
        return;
    }
}

/*---------------------------------------------------------------------------*
 *
 *    File scope functions/symbols defined for this file
 *    -- Function Prototypes, File Scope Data --
 *
 *---------------------------------------------------------------------------*/

static void _deviceConnected(HAL_PACKET_CONTEXT *pktCtxt)
{
    uint8_t *cbData = pktCtxt->buf;
    tBSA_HH_MSG *p_data = (tBSA_HH_MSG *)cbData;

    Btbdaddr bd_addr;
    memcpy(bd_addr.address, p_data->open.bd_addr,BD_ADDR_LEN);

    bool match = bondState.inProgress && isSameBdAddr(bondState.bdAddr.address, p_data->open.bd_addr);

    if (p_data->open.status != BSA_SUCCESS) {
        if (!match) {
            NN_SDK_LOG("[bluetooth] %s: Failed to connect\n", __func__);
            SfHhCallbacks.ConnectionStateCallback(&bd_addr, BTHH_CONN_STATE_FAILED_GENERIC);
        }
        else
        {
            NN_SDK_LOG("[bluetooth] %s: Failed to connect - bonding failed\n", __func__);
            bondState.inProgress = false;
            SfCallbacks.BondStateChangedCallback(BT_STATUS_AUTH_FAILURE,&bd_addr,BT_BOND_STATE_NONE);
            if (bondState.isNew) {
                app_xml_remove_db(bondState.bdAddr.address);
            }
        }
        return;
    }

    tAPP_XML_REM_DEVICE *p_xml_dev;
    p_xml_dev = app_xml_find_dev_db(p_data->open.bd_addr);
    if (p_xml_dev == NULL) {
        tBSA_HH_CLOSE hh_close_param;
        NN_SDK_LOG("[bluetooth] %s: Failed allocating HH database element, bd_addr=%s\n", __func__, toString(p_data->open.bd_addr,BD_ADDR_LEN));
        BSA_HhCloseInit(&hh_close_param);
        hh_close_param.handle = p_data->open.handle;
        BSA_HhClose(&hh_close_param);
        return;
    }

    // If here, implication is that open.status is BSA_SUCCESS, and p_xml_dev is valid.


    if (!match)
    {
        NN_SDK_LOG("[bluetooth] %s: Connecting bd_addr=%s\n", __func__, toString(p_data->open.bd_addr,BD_ADDR_LEN));

        // Connection was successful, and bonding not in progress.
        BtHalBsaIntHhAddDev(p_xml_dev);
        p_xml_dev->info_mask |= APP_HH_DEV_OPENED;
        SfHhCallbacks.ConnectionStateCallback(&bd_addr, BTHH_CONN_STATE_CONNECTED);
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: Bonding bd_addr=%s\n", __func__, toString(p_data->open.bd_addr,BD_ADDR_LEN));

        // Connection was successful, and bonding is in progress. Fetch the descriptor.
        BtHalBsaGetDscpinfo(p_data->open.handle);

        // Copy the BRR info
        p_xml_dev->brr_size    = sizeof(tBSA_HH_BRR_CFG);
        memcpy(p_xml_dev->brr, &p_data->open.brr_cfg, p_xml_dev->brr_size);
        p_xml_dev->sub_class   = p_data->open.sub_class;
        p_xml_dev->attr_mask   = p_data->open.attr_mask;

        // Save the HH handle and mark as open
        p_xml_dev->handle = p_data->open.handle;
        p_xml_dev->info_mask |= APP_HH_DEV_OPENED;
    }
}


#ifdef BTHAL_CB_DEBUG
static void printBtHalEvents(uint8_t event)
{
#define X(ID) #ID,
    static const char *names[] = { CallbackTypes } ;
#undef X

    if (event==CALLBACK_TYPE_HID_DATA)
        return;

    if (event < sizeof(names) / sizeof(names[0])) {
        NN_SDK_LOG("[bluetooth]: CallbackEvent -- %s\n", names[event]);
    } else {
        NN_SDK_LOG("[bluetooth]: CallbackEvent -- %d is invalid\n", event);
    }
}
#endif

static void printHidInfo(const tBSA_HH_UIPC_REPORT *p_uipc_report)
{
#if 0
    BTHAL_BSA_DEBUG("p_uipc_report=%x",p_uipc_report);

    BTHAL_BSA_DEBUG("Recv Report ID:[x%02X]",p_uipc_report->report.report_data.data[0]);

    BTHAL_BSA_DEBUG("\t BLUETOOTH_HAL_HIDQ: Report Data:[%x] [%x] [%x] [%x] [%x] "
            "                  [%x] [%x] [%x] [%x] [%x]",
            p_uipc_report->report.report_data.data[1], p_uipc_report->report.report_data.data[2],
            p_uipc_report->report.report_data.data[3], p_uipc_report->report.report_data.data[4],
            p_uipc_report->report.report_data.data[5], p_uipc_report->report.report_data.data[6],
            p_uipc_report->report.report_data.data[7], p_uipc_report->report.report_data.data[8],
            p_uipc_report->report.report_data.data[9], p_uipc_report->report.report_data.data[10]);

    BTHAL_BSA_DEBUG("called Handle:%d BdAddr:%02X-%02X-%02X-%02X-%02X-%02X",
            p_uipc_report->report.handle,
            p_uipc_report->report.bd_addr[0], p_uipc_report->report.bd_addr[1],
            p_uipc_report->report.bd_addr[2], p_uipc_report->report.bd_addr[3],
            p_uipc_report->report.bd_addr[4], p_uipc_report->report.bd_addr[5]);
    BTHAL_BSA_DEBUG("\t mode=%d SubClass:0x%x CtryCode:%d len:%d event:%d",
            p_uipc_report->report.mode,
            p_uipc_report->report.sub_class,
            p_uipc_report->report.ctry_code,
            p_uipc_report->report.report_data.length,
            p_uipc_report->hdr.event);
#endif
}

/*******************************************************************************
 **
 ** Function        HalMsgThread
 **
 ** Description     Thread that handles HID messages.
 **
 ** Returns
 **
 *******************************************************************************/
static void HalMsgThread(void* param)
{
    nn::os::MultiWaitType        multiWait;
    nn::os::MultiWaitHolderType  holderForCircBuff;
    nn::os::MultiWaitHolderType  holderForIntEvent;
    nn::os::MultiWaitHolderType  holderForIntEventForHid;
    nn::os::MultiWaitHolderType  holderForIntEventForHidReport;
    nn::os::MultiWaitHolderType  holderForIntEventForBle;
    nn::os::MultiWaitHolderType  holderForIntEventForBleCore;
    nn::os::MultiWaitHolderType  holderForIntEventForBleHid;

    nn::os::Tick exitTimeTick;

    enum {
        EVT_CIRC_BUF,
        EVT_INTERNAL,
        EVT_INTERNAL_HID,
        EVT_INTERNAL_HID_REPORT,
        EVT_INTERNAL_BLE,
        EVT_INTERNAL_BLE_CORE,
        EVT_INTERNAL_BLE_HID,
    };

    nn::os::InitializeEvent(&g_InternalEvent,             false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_InternalEventForHid,       false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_InternalEventForHidReport, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_InternalEventForBle,       false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_InternalEventForBleCore,   false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_InternalEventForBleHid,    false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeMultiWaitHolder(&holderForCircBuff,             &CircBufferC::m_event);
    nn::os::InitializeMultiWaitHolder(&holderForIntEvent,             &g_InternalEvent);
    nn::os::InitializeMultiWaitHolder(&holderForIntEventForHid,       &g_InternalEventForHid);
    nn::os::InitializeMultiWaitHolder(&holderForIntEventForHidReport, &g_InternalEventForHidReport);
    nn::os::InitializeMultiWaitHolder(&holderForIntEventForBle,       &g_InternalEventForBle);
    nn::os::InitializeMultiWaitHolder(&holderForIntEventForBleCore,   &g_InternalEventForBleCore);
    nn::os::InitializeMultiWaitHolder(&holderForIntEventForBleHid,    &g_InternalEventForBleHid);

    nn::os::SetMultiWaitHolderUserData(&holderForCircBuff,             EVT_CIRC_BUF);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEvent,             EVT_INTERNAL);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEventForHid,       EVT_INTERNAL_HID);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEventForHidReport, EVT_INTERNAL_HID_REPORT);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEventForBle,       EVT_INTERNAL_BLE);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEventForBleCore,   EVT_INTERNAL_BLE_CORE);
    nn::os::SetMultiWaitHolderUserData(&holderForIntEventForBleHid,    EVT_INTERNAL_BLE_HID);

    nn::os::InitializeMultiWait(&multiWait);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForCircBuff);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEventForHid);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEventForHidReport);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEventForBle);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEventForBleCore);
    nn::os::LinkMultiWaitHolder(&multiWait, &holderForIntEventForBleHid);

    bool exitThread = false;
    for (;;)
    {
        // Use a timed wait so that exit thread can happen even if g_InternalEvent/Hid/HidReport
        // don't complete for some reason (error condition).
        nn::os::MultiWaitHolderType* holder = nn::os::TimedWaitAny(&multiWait, nn::TimeSpan::FromSeconds(1));
        if (holder != NULL)
        {
            int holderType = nn::os::GetMultiWaitHolderUserData(holder);
            switch (holderType) {
                case EVT_CIRC_BUF:
                    nn::os::ClearEvent(&CircBufferC::m_event);
                    break;
                case EVT_INTERNAL:
                    nn::os::ClearEvent(&g_InternalEvent);
                    g_PendingIntEvent = false;
                    break;
                case EVT_INTERNAL_HID:
                    nn::os::ClearEvent(&g_InternalEventForHid);
                    g_PendingIntEventForHid = false;
                    break;
                case EVT_INTERNAL_HID_REPORT:
                    nn::os::ClearEvent(&g_InternalEventForHidReport);
                    g_PendingIntEventForHidReport = false;
                    break;
                case EVT_INTERNAL_BLE:
                    nn::os::ClearEvent(&g_InternalEventForBle);
                    g_PendingIntEventForBle = false;
                    break;
                case EVT_INTERNAL_BLE_CORE:
                    nn::os::ClearEvent(&g_InternalEventForBleCore);
                    g_PendingIntEventForBleCore = false;
                    break;
                case EVT_INTERNAL_BLE_HID:
                    nn::os::ClearEvent(&g_InternalEventForBleHid);
                    g_PendingIntEventForBleHid = false;
                    break;
                default:
                    NN_SDK_LOG("[bluetooth] %s: invalid holderType %d\n", __FUNCTION__, holderType);
                    break;
            }
        }

        // Only process the queues if not exiting.
        if (exitThread == false)
        {
            if (g_PendingIntEvent == false && g_PendingIntEventForHid == false) {
                ProcessCallbackQueue();
            }

            if (g_PendingIntEventForHidReport == false) {
                exitThread = ProcessHidQueue();
                if (exitThread) {
                    exitTimeTick = nn::os::GetSystemTick();
                }
            }

            if (g_PendingIntEventForBle == false) {
                ProcessBleQueue();
            }

            if (g_PendingIntEventForBleCore == false) {
                ProcessBleCoreQueue();
            }

            if (g_PendingIntEventForBleHid == false) {
                ProcessBleHidQueue();
            }
        }

        // Only exit the thread if there are no pending events, otherwise the impl could
        // try to use a finalized event.
        if (exitThread == true)
        {
            int timeToExitMs =  (nn::os::GetSystemTick() - exitTimeTick).ToTimeSpan().GetMilliSeconds();

            if (g_PendingIntEvent == false && g_PendingIntEventForHid == false &&
                g_PendingIntEventForHidReport == false &&
                g_PendingIntEventForBle == false && g_PendingIntEventForBleCore == false && g_PendingIntEventForBleHid == false) {
                NN_SDK_LOG("[bluetooth] %s: Time to exit thread: %dms\n", __FUNCTION__, timeToExitMs);
                break;
            }
            else if (timeToExitMs > 1000) {
                NN_SDK_LOG("[bluetooth] %s: Timeout waiting for events to clear: %d %d %d %d %d %d\n", __FUNCTION__,
                        g_PendingIntEvent, g_PendingIntEventForHid, g_PendingIntEventForHidReport,
                        g_PendingIntEventForBle, g_PendingIntEventForBleCore, g_PendingIntEventForBleHid);
                break;
            }
        }
    }

    nn::os::UnlinkAllMultiWaitHolder(&multiWait);
    nn::os::FinalizeMultiWaitHolder(&holderForCircBuff);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEvent);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEventForHid);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEventForHidReport);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEventForBle);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEventForBleCore);
    nn::os::FinalizeMultiWaitHolder(&holderForIntEventForBleHid);
    nn::os::FinalizeMultiWait(&multiWait);
    nn::os::FinalizeEvent(&g_InternalEvent);
    nn::os::FinalizeEvent(&g_InternalEventForHid);
    nn::os::FinalizeEvent(&g_InternalEventForHidReport);
    nn::os::FinalizeEvent(&g_InternalEventForBle);
    nn::os::FinalizeEvent(&g_InternalEventForBleCore);
    nn::os::FinalizeEvent(&g_InternalEventForBleHid);
} // NOLINT(impl/function_size)


// Read from the HID queue, possibly calling callbacks.
// Return true if the FINALIZE event occurred, false otherwise.
static bool ProcessHidQueue()
{
    HAL_PACKET_CONTEXT *pktCtxt;

    do {
        pktCtxt = static_cast<HAL_PACKET_CONTEXT*>(HidDataContext.msgQueue.qRead());

        if (pktCtxt == NULL) {
            return false;
        }

        if (pktCtxt->cbType == CALLBACK_TYPE_HID_DATA)
        {
            // Prevent old hid messages for being passed to the upper layer.
            static const int HID_DATA_PACKET_MAX_LIFETIME_MS = 100;
            nn::TimeSpan timespan = (nn::os::GetSystemTick() - pktCtxt->createTime).ToTimeSpan();
            if (timespan > nn::TimeSpan::FromMilliSeconds(HID_DATA_PACKET_MAX_LIFETIME_MS)) {
                HidDataContext.msgQueue.qFree();
            }
            else {
                break; // packet is okay. exit while loop.
            }
        }
        else {
            break; // not a timed packet. exit while loop.
        }

    } while (1);


#ifdef BTHAL_CB_DEBUG
    printBtHalEvents(pktCtxt->cbType);
#endif

    switch(pktCtxt->cbType)
    {
        case CALLBACK_TYPE_HID_DATA:
        {
            tBSA_HH_UIPC_REPORT *p_uipc_report = (tBSA_HH_UIPC_REPORT *)pktCtxt->buf;
            Btbdaddr bd_addr;
            memcpy(bd_addr.address, p_uipc_report->report.bd_addr,BD_ADDR_LEN);
            printHidInfo(p_uipc_report);
            SfHhCallbacks.GetReportCallback(
                    &bd_addr,BTHH_OK,(uint8_t *)p_uipc_report,
                    pktCtxt->dataLen);
            break;
        }
        case CALLBACK_TYPE_SET_REPORT:
        {
            uint8_t *cbData = pktCtxt->buf;
            tBSA_HH_MSG *p_data = (tBSA_HH_MSG *)cbData;
            tBSA_HH_SET_REPORT_MSG *p_set_report = &p_data->set_report;
            //NN_SDK_LOG("%s CALLBACK_TYPE_SET_REPORT status:%d handle:%d\n", __func__, p_set_report->status, p_set_report->handle);
            Btbdaddr bd_addr;
            tAPP_XML_REM_DEVICE *p_xml_dev = app_xml_find_by_handle(p_set_report->handle);
            if (p_xml_dev) {
                memcpy(bd_addr.address, p_xml_dev->bd_addr, sizeof(bd_addr.address));
            } else {
                NN_SDK_LOG("[bluetooth] %s: Failed to find HANDLE for CALLBACK_TYPE_SET_REPORT, %d\n", __func__, p_set_report->handle);
                memset(bd_addr.address, 0, sizeof(bd_addr));
            }
            SfCallbacks.SetReportStatusCallback(&bd_addr, p_set_report->status == BSA_SUCCESS ? BTHH_OK : BTHH_ERR);
            break;
        }
        case CALLBACK_TYPE_GET_REPORT:
        {
            uint8_t *cbData = pktCtxt->buf;
            tBSA_HH_MSG *p_data = (tBSA_HH_MSG *)cbData;
            tBSA_HH_GET_REPORT_MSG *p_get_report = &p_data->get_report;
            //NN_SDK_LOG("%s CALLBACK_TYPE_GET_REPORT status:%d handle:%d len:%d type:0x%x data:%02x:%02x:%02x\n", __func__,
            //    p_get_report->status, p_get_report->handle, p_get_report->report.length,
            //    p_get_report->report.data[0], p_get_report->report.data[1], p_get_report->report.data[2], p_get_report->report.data[3]);
            Btbdaddr bd_addr;
            tAPP_XML_REM_DEVICE *p_xml_dev = app_xml_find_by_handle(p_get_report->handle);
            if (p_xml_dev) {
                memcpy(bd_addr.address, p_xml_dev->bd_addr, sizeof(bd_addr.address));
            } else {
                NN_SDK_LOG("[bluetooth] %s Failed to find HANDLE for CALLBACK_TYPE_GET_REPORT, %d\n", __func__, p_get_report->handle);
                memset(bd_addr.address, 0, sizeof(bd_addr));
            }
            SfCallbacks.GetReportStatusCallback(&bd_addr, p_get_report->status == BSA_SUCCESS ? BTHH_OK : BTHH_ERR,
                    p_get_report->report.length, p_get_report->report.data);
            break;
        }
        case CALLBACK_TYPE_FINALIZE:
        {
            // exit from the callback thread!
            HidDataContext.msgQueue.qFree();
            return true;
        }
        default:
            NN_SDK_LOG("[bluetooth] %s: ERROR: Invalid msg %d in %s line %d\n", __func__, pktCtxt->cbType, __LINE__);
            break;
        }
    HidDataContext.msgQueue.qFree();
    return false;

} // NOLINT(impl/function_size)


static void _deviceFound(HAL_PACKET_CONTEXT *pktCtxt)
{
    uint8_t *cbData = pktCtxt->buf;

    int index;
    tBSA_DISC_MSG *p_data = (tBSA_DISC_MSG *)cbData;
    // check if this device has already been received (update)
    for (index = 0; index < APP_DISC_NB_DEVICES; index++){
        if ((app_discovery_cb.devs[index].in_use == TRUE) && (!bdcmp(app_discovery_cb.devs[index].device.bd_addr, p_data->disc_new.bd_addr))){
            // Update device
            app_discovery_cb.devs[index].device = p_data->disc_new;
            BTHAL_BSA_DEBUG("\tSaved disc device in index %d (update)", index);
            break;
        }
    }
    // If this is a new device
    if (index >= APP_DISC_NB_DEVICES){
    // Look for a free place to store dev info
        for (index = 0; index < APP_DISC_NB_DEVICES; index++){
            if (app_discovery_cb.devs[index].in_use == FALSE){
            app_discovery_cb.devs[index].in_use = TRUE;
            memcpy(&app_discovery_cb.devs[index].device, &p_data->disc_new,sizeof(tBSA_DISC_REMOTE_DEV));
            BTHAL_BSA_DEBUG("\tSaved disc device in index %d (new)", index);
            break;
            }
        }
    }
    // If this is a new device
    if (index >= APP_DISC_NB_DEVICES){
        BTHAL_BSA_DEBUG("No room to save new discovered");
    }

    // If this is a new device
    BTHAL_BSA_DEBUG("\tBdaddr:%02x:%02x:%02x:%02x:%02x:%02x",
        p_data->disc_new.bd_addr[0], p_data->disc_new.bd_addr[1], p_data->disc_new.bd_addr[2],
        p_data->disc_new.bd_addr[3], p_data->disc_new.bd_addr[4], p_data->disc_new.bd_addr[5]);
    BTHAL_BSA_DEBUG("\tName:%s", p_data->disc_new.name);
    BTHAL_BSA_DEBUG("\tClassOfDevice:%02x:%02x:%02x",
        p_data->disc_new.class_of_device[0], p_data->disc_new.class_of_device[1], p_data->disc_new.class_of_device[2]);
    BTHAL_BSA_DEBUG("\tServices:0x%08x",(int) p_data->disc_new.services);
    BTHAL_BSA_DEBUG("\tRssi:%d", p_data->disc_new.rssi);
    if (p_data->disc_new.eir_vid_pid[0].valid)
    {
        BTHAL_BSA_DEBUG("\tVidSrc:%d Vid:0x%04X Pid:0x%04X Version:0x%04X",
            p_data->disc_new.eir_vid_pid[0].vendor_id_source,
            p_data->disc_new.eir_vid_pid[0].vendor,
            p_data->disc_new.eir_vid_pid[0].product,
            p_data->disc_new.eir_vid_pid[0].version);
    }
    if (p_data->disc_new.eir_data[0]) {
        BtHalBsaDiscParseEir(p_data->disc_new.eir_data);
    }

    BluetoothProperty properties;
    int propNum = 1;
    /*
     * FIXME:
     * Still need to return the following
    BluetoothUUID Uuid[1];
    BluetoothServiceRecord SdpRecord;
    BluetoothScanMode ScanMode;
    Btbdaddr BondedDevices[10];
    uint32_t DiscoveryTimeout;
    BluetoothLocalLEfeatures LocalLEfeatures;
     */

    FullDeviceProperties devProp;
    properties.len = sizeof(FullDeviceProperties);
    properties.type = BT_PROPERTY_ALL;
    properties.pVal = &devProp;
    memcpy(&devProp.BdName,             p_data->disc_new.name,               sizeof(BD_NAME));
    memcpy(&devProp.BdAddress,          p_data->disc_new.bd_addr,            BD_ADDR_LEN);
    memcpy(&devProp.RemoteFriendlyName, p_data->remote_name.remote_bd_name,  sizeof(BD_ADDR_LEN));
    memcpy(&devProp.ClassOfDevice,      p_data->disc_new.class_of_device,    sizeof(devProp.ClassOfDevice));
    devProp.TypeOfDevice      = BT_DEVICE_DEVTYPE_BREDR;//p_data->disc_new.device_type; // FIXME: only BR currently supported
    devProp.RemoteRssi        = p_data->disc_new.rssi;
    //devProp.RemoteVersionInfo.SubVer = p_data->dev_info.
    //devProp.RemoteVersionInfo.manufacturer = p_data->dev_info.
    devProp.RemoteVersionInfo.version = p_data->dev_info.version;
    SfCallbacks.DeviceFoundCallback(propNum, &properties);
}

static void _AuthorizeBonding(HAL_PACKET_CONTEXT *pktCtxt)
{
    int      status;
    uint8_t *cbData = pktCtxt->buf;
    tBSA_SEC_AUTH_REPLY authorize_reply;
    tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
    status = BSA_SecAuthorizeReplyInit(&authorize_reply);
    copyBdAddr(authorize_reply.bd_addr, p_data->authorize.bd_addr);
    authorize_reply.trusted_service = p_data->authorize.service;
    authorize_reply.auth = 0; //app_sec_cb.auth;
    switch(authorize_reply.auth)
    {
        case BSA_SEC_NOT_AUTH:
            BTHAL_BSA_DEBUG("Access Refused (%d)", authorize_reply.auth);
            break;
        case BSA_SEC_AUTH_TEMP:
            BTHAL_BSA_DEBUG("Access Temporary Accepted (%d)", authorize_reply.auth);
            break;
        case BSA_SEC_AUTH_PERM:
            BTHAL_BSA_DEBUG("Access Accepted (%d)", authorize_reply.auth);
            break;
        default:
            BTHAL_BSA_DEBUG("Unknown Access response (%d)", authorize_reply.auth);
            break;
    }
    status = BSA_SecAuthorizeReply(&authorize_reply);

    // Add AV service for this devices in XML database
    app_xml_add_trusted_services_db(p_data->authorize.bd_addr,1 << p_data->authorize.service);
    if (strlen((char *)p_data->authorize.bd_name) > 0)
        app_xml_update_name_db(p_data->authorize.bd_addr, p_data->authorize.bd_name);
}

static BluetoothHhStatus _BtHalBondingConnect(Btbdaddr *pBdAddr)
{
    /* Connect HID Device in Standard Report mode*/
    BTHAL_IF_DEBUG("Called");

    tBSA_HH_OPEN hh_open_param;
    BSA_HhOpenInit(&hh_open_param);
    copyBdAddr(hh_open_param.bd_addr, pBdAddr->address);

    hh_open_param.mode = BSA_HH_PROTO_RPT_MODE; // only report mode is currently supported
    hh_open_param.sec_mask = BTA_SEC_AUTHENTICATE;

    tBSA_STATUS status = BSA_HhOpen(&hh_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to connect status:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

static void _AuthenticationFailure(HAL_PACKET_CONTEXT *pktCtxt)
{
    bondState.inProgress = false;
    NN_SDK_LOG("[bluetooth] %s: Bonding failed.\n", __func__);

    uint8_t *cbData = pktCtxt->buf;
    tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;

    Btbdaddr bd_addr;
    memcpy(bd_addr.address,p_data->auth_cmpl.bd_addr,BD_ADDR_LEN);

    SfCallbacks.BondStateChangedCallback(BT_STATUS_AUTH_FAILURE,&bd_addr,BT_BOND_STATE_NONE);
}

static void _AuthenticationSuccess(HAL_PACKET_CONTEXT *pktCtxt)
{
    uint8_t *cbData = pktCtxt->buf;
    tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
    tBSA_DISC_DEV *disc_dev;

    Btbdaddr bd_addr;
    memcpy(bd_addr.address,p_data->auth_cmpl.bd_addr,BD_ADDR_LEN);

    BluetoothHhStatus connectStatus = _BtHalBondingConnect(&bd_addr);
    if (connectStatus != BTHH_OK)
    {
        _AuthenticationFailure(pktCtxt);
        return;
    }

    if (p_data->auth_cmpl.key_present == FALSE)
    {
        NN_SDK_LOG("[bluetooth] %s. expected link key, but it's missing\n", __func__);
        _AuthenticationFailure(pktCtxt);
        return;
    }

    // Set a flag so that xml data will be saved to nand, later when
    // all the info is available, after DSCP info has been received.
    bondState.inProgress = true;
    memcpy(bondState.bdAddr.address,p_data->auth_cmpl.bd_addr,BD_ADDR_LEN);

    bondState.isNew = app_xml_find_dev_db(p_data->auth_cmpl.bd_addr) == NULL;
    app_xml_update_name_db(p_data->auth_cmpl.bd_addr,p_data->auth_cmpl.bd_name);
    memcpy(bondState.linkKey, &p_data->auth_cmpl.key, sizeof(bondState.linkKey));
    bondState.keyType = p_data->auth_cmpl.key_type;

    // Unfortunately, the BSA_SEC_AUTH_CMPL_EVT does not contain COD
    // so update the XML with info found during inquiry
    disc_dev = BtHalBsaDiscFindDevice(p_data->auth_cmpl.bd_addr);
    if (disc_dev != NULL){
        app_xml_update_cod_db(p_data->auth_cmpl.bd_addr,disc_dev->device.class_of_device);
    }
}

/*******************************************************************************
 **
 ** Function        ProcessCallbackQueue
 **
 ** Description     Thread that handles callback messages.
 **
 ** Returns
 **
 *******************************************************************************/
static void ProcessCallbackQueue()
{
    HAL_PACKET_CONTEXT *pktCtxt;
    pktCtxt = static_cast<HAL_PACKET_CONTEXT*>(CbDataContext.msgQueue.qRead());

    if (pktCtxt == NULL)
    {
        return;
    }

    uint8_t *cbData = pktCtxt->buf;
    Btbdaddr bd_addr;
    Btbdname bd_name;
    ClassOfDevice class_of_device;

#ifdef BTHAL_CB_DEBUG
    printBtHalEvents(pktCtxt->cbType);
#endif

    switch(pktCtxt->cbType)
    {
        case CALLBACK_TYPE_DEVICE_FOUND:
            _deviceFound(pktCtxt);
            break;
        case CALLBACK_TYPE_DISCOVER_STATE_CHANGED:
        {
            BluetoothDiscoveryState *btDiscState = (BluetoothDiscoveryState*)cbData;
            SfCallbacks.DiscoveryStateChangedCallback(*btDiscState);
            NN_SDK_LOG("[bluetooth] Discovery %s\n", *btDiscState == BT_DISCOVERY_STARTED ? "Started" : "Stopped");
            break;
        }
        case CALLBACK_TYPE_PIN_REQUEST:
        {
            int      status;
            tAPP_XML_CONFIG xml_local_config;
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            memcpy(&bd_addr.address, &p_data->pin_req.bd_addr,BD_ADDR_LEN);
            // FIXME: Make sure to check that the name length matches between BSA and the HAL
            memcpy(&bd_name.name, p_data->pin_req.bd_name, sizeof(BD_NAME));
            memcpy(class_of_device.cod, p_data->pin_req.class_of_device, DEV_CLASS_LEN);
            SfCallbacks.PinRequestCallback(&bd_addr, &bd_name, &class_of_device);
            status = app_read_xml_config(&xml_local_config);
            if (status < 0){
                NN_SDK_LOG("[bluetooth] %s: app_read_xml_config failed: Unable to Read XML config file\n", __func__);
            }
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_AUTHENTICATION_SUCCESS:
            _AuthenticationSuccess(pktCtxt);
            break;
        case CALLBACK_TYPE_BOND_STATE_SP_CONFIRMED:
        {
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            memcpy(bd_addr.address, p_data->cfm_req.bd_addr,BD_ADDR_LEN);
            memcpy(bd_name.name, p_data->cfm_req.bd_name, sizeof(BD_NAME));
            memcpy(class_of_device.cod, p_data->cfm_req.class_of_device, DEV_CLASS_LEN);
            SfCallbacks.SspRequestCallback(&bd_addr, &bd_name, &class_of_device, BT_SSP_VARIANT_PASSKEY_CONFIRMATION, 0);
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_KEY_NOTIFICATION:
        {
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            uint32_t pass_ke=(uint32_t)p_data->key_notif.passkey;
            memcpy(bd_addr.address, p_data->key_notif.bd_addr,BD_ADDR_LEN);
            // FIXME: Make sure to check that the name length matches between BSA and the HAL
            memcpy(bd_name.name, p_data->key_notif.bd_name, sizeof(BD_NAME));
            memcpy(class_of_device.cod, p_data->key_notif.class_of_device, DEV_CLASS_LEN);
            SfCallbacks.SspRequestCallback(&bd_addr, &bd_name, &class_of_device, BT_SSP_VARIANT_PASSKEY_NOTIFICATION, pass_ke);
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_BONDING:
        {
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            memcpy(bd_addr.address, p_data->cfm_req.bd_addr,BD_ADDR_LEN);
            SfCallbacks.BondStateChangedCallback(BT_STATUS_SUCCESS,&bd_addr,BT_BOND_STATE_BONDING);
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_RESUMED:
        {
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            memcpy(bd_addr.address, p_data->resumed.bd_addr,BD_ADDR_LEN);
            SfCallbacks.BondStateChangedCallback(BT_STATUS_SUCCESS,&bd_addr,BT_BOND_STATE_BONDED);
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_SUSPENDED:
        {
            tBSA_SEC_MSG *p_data = (tBSA_SEC_MSG *)cbData;
            memcpy(bd_addr.address, p_data->suspended.bd_addr,BD_ADDR_LEN);
            SfCallbacks.BondStateChangedCallback(BT_STATUS_NOT_READY,&bd_addr,BT_BOND_STATE_NONE);
            break;
        }
        case CALLBACK_TYPE_BOND_STATE_CANCELLED:
            // FIXME: Need to figure out what BD_ADDR to return
            break;
        case CALLBACK_TYPE_BOND_STATE_AUTHENTICATION_FAILURE:
            _AuthenticationFailure(pktCtxt);
            break;
        case CALLBACK_TYPE_BOND_STATE_AUTHORIZE:
            _AuthorizeBonding(pktCtxt);
            break;

        case CALLBACK_TYPE_CONNECTION_STATE_CONNECTED:
            _deviceConnected(pktCtxt);
            break;
        case CALLBACK_TYPE_CONNECTION_STATE_DISCONNECTED:
        {
            tBSA_HH_CLOSE_MSG *p_data = (tBSA_HH_CLOSE_MSG *)pktCtxt->buf;
            tAPP_XML_REM_DEVICE *p_xml_dev = app_xml_find_by_handle(p_data->handle);
            if (p_xml_dev) {
                p_xml_dev->info_mask &= ~APP_HH_DEV_OPENED;
                memcpy(bd_addr.address, p_xml_dev->bd_addr,BD_ADDR_LEN);
                SfHhCallbacks.ConnectionStateCallback(&bd_addr, BTHH_CONN_STATE_DISCONNECTED);
            }
            else
            {
                NN_SDK_LOG("[bluetooth] %s: Could not find device for handle %d\n", __func__, p_data->handle);
            }
            break;
        }
        case CALLBACK_TYPE_FATAL_ERROR:
        {
            SfCallbacks.FatalErrorCallback();
            break;
        }
        case CALLBACK_TYPE_HID_INFO:
        {
            tAPP_XML_REM_DEVICE *p_xml_dev;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_HH_MSG *p_data = (tBSA_HH_MSG *)cbData;

            BTHAL_BSA_DEBUG("DscpInfo: VID/PID (hex) = %04X/%04X",
                    p_data->dscpinfo.peerinfo.vendor_id,
                    p_data->dscpinfo.peerinfo.product_id);
            BTHAL_BSA_DEBUG("DscpInfo: version = %d", p_data->dscpinfo.peerinfo.version);
            BTHAL_BSA_DEBUG("DscpInfo: ssr_max_latency = %d ssr_min_tout = %d",
                    p_data->dscpinfo.peerinfo.ssr_max_latency,
                    p_data->dscpinfo.peerinfo.ssr_min_tout);
            BTHAL_BSA_DEBUG("DscpInfo: supervision_tout = %d", p_data->dscpinfo.peerinfo.supervision_tout);
            BTHAL_BSA_DEBUG("DscpInfo (descriptor len:%d):", p_data->dscpinfo.dscpdata.length);

            if (p_data->dscpinfo.status == BSA_SUCCESS){
                // Check if the device is already in our internal database
                p_xml_dev = app_xml_find_by_handle(p_data->dscpinfo.handle);
                if (p_xml_dev){
                    //NN_SDK_LOG("Update Remote Device XML file\n");
                    p_xml_dev->vid               = p_data->dscpinfo.peerinfo.vendor_id;
                    p_xml_dev->pid               = p_data->dscpinfo.peerinfo.product_id;
                    p_xml_dev->version           = p_data->dscpinfo.peerinfo.version;
                    p_xml_dev->ssr_max_latency   = p_data->dscpinfo.peerinfo.ssr_max_latency;
                    p_xml_dev->ssr_min_tout      = p_data->dscpinfo.peerinfo.ssr_min_tout;
                    p_xml_dev->supervision_tout  = p_data->dscpinfo.peerinfo.supervision_tout;
                    p_xml_dev->descriptor_size   = p_data->dscpinfo.dscpdata.length;
                    if (p_xml_dev->descriptor_size > sizeof(p_xml_dev->descriptor)){
                        BTHAL_BSA_DEBUG("descriptor too big - %d", p_xml_dev->descriptor_size);
                        p_xml_dev->descriptor_size = 0;
                    }else{
                        memcpy(p_xml_dev->descriptor, p_data->dscpinfo.dscpdata.data, p_xml_dev->descriptor_size);
                    }

                    // Let's add this device to Server HH database to allow it to reconnect
                    BTHAL_BSA_DEBUG("Adding HID Device:%s", p_xml_dev->name);
                    BtHalBsaIntHhAddDev(p_xml_dev);
                    if (bondState.inProgress)
                    {
                        // Bonding has succeeded.
                        app_xml_update_key_db(bondState.bdAddr.address,bondState.linkKey,bondState.keyType);

                        // Send bonding event to upper layer
                        SfCallbacks.BondStateChangedCallback(BT_STATUS_SUCCESS, &bondState.bdAddr, BT_BOND_STATE_BONDED);
                        bondState.inProgress = false;
                    }

                    // Send connection event to upper layer
                    Btbdaddr bd_addr;
                    memcpy(bd_addr.address, p_xml_dev->bd_addr, BD_ADDR_LEN);
                    SfHhCallbacks.ConnectionStateCallback(&bd_addr, BTHH_CONN_STATE_CONNECTED);
                }else{
                    NN_SDK_LOG("[bluetooth] %s %d: Failed to get handle!\n", __func__, __LINE__);
                }
            }
            else
            {
                if (bondState.inProgress)
                {
                    // Bonding has failed.
                    NN_SDK_LOG("[bluetooth] %s: GetDscpInfo error. Bonding failed.\n", __func__);
                    SfCallbacks.BondStateChangedCallback(BT_STATUS_SUCCESS, &bondState.bdAddr, BT_BOND_STATE_NONE);
                    bondState.inProgress = false;
                    if (bondState.isNew) {
                        app_xml_remove_db(bondState.bdAddr.address);
                    }
                }
            }
            break;
        }

        case CALLBACK_TYPE_EXTENSION_SET_TSI:
        {
            BluetoothHhStatus status = BTHH_OK;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_ROBSON_MSG *p_data = (tBSA_ROBSON_MSG*)cbData;
            tBSA_ROBSON_SET_TSI_MSG *p_set_tsi = &p_data->set_tsi;

            if (p_set_tsi->status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s %d failed status: %d\n", __func__, __LINE__, p_set_tsi->status);
                status = BTHH_ERR;
            }
            SfExtCallbacks.SetTsiCallback(pktCtxt->buf, status);
            break;
        }
        case CALLBACK_TYPE_EXTENSION_EXIT_TSI:
        {
            BluetoothHhStatus status = BTHH_OK;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_ROBSON_MSG *p_data = (tBSA_ROBSON_MSG*)cbData;
            tBSA_ROBSON_EXIT_TSI_MSG *p_exit_tsi = &p_data->exit_tsi;

            if (p_exit_tsi->status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s %d failed status: %d\n", __func__, __LINE__, p_exit_tsi->status);
                status = BTHH_ERR;
            }
            SfExtCallbacks.ExitTsiCallback(pktCtxt->buf, status);
            break;
        }
        case CALLBACK_TYPE_EXTENSION_SET_BURST:
        {
            BluetoothHhStatus status = BTHH_OK;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_ROBSON_MSG *p_data = (tBSA_ROBSON_MSG*)cbData;
            tBSA_ROBSON_SET_BURST_MSG *p_set_burst = &p_data->set_burst;

            if (p_set_burst->status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s %d failed status: %d\n", __func__, __LINE__, p_set_burst->status);
                status = BTHH_ERR;
            }
            SfExtCallbacks.SetBurstCallback(pktCtxt->buf, status);
            break;
        }
        case CALLBACK_TYPE_EXTENSION_EXIT_BURST:
        {
            BluetoothHhStatus status = BTHH_OK;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_ROBSON_MSG *p_data = (tBSA_ROBSON_MSG*)cbData;
            tBSA_ROBSON_EXIT_BURST_MSG *p_exit_burst = &p_data->exit_burst;

            if (p_exit_burst->status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s %d failed status: %d\n", __func__, __LINE__, p_exit_burst->status);
                status = BTHH_ERR;
            }
            SfExtCallbacks.ExitBurstCallback(pktCtxt->buf, status);
            break;
        }
        case CALLBACK_TYPE_EXTENSION_SET_ZERO_RETRAN:
        {
            BluetoothHhStatus status = BTHH_OK;
            uint8_t *cbData = pktCtxt->buf;
            tBSA_ROBSON_MSG *p_data = (tBSA_ROBSON_MSG*)cbData;
            tBSA_ROBSON_SET_0RETRAN_MODE_MSG *p_set_0retran_mode = &p_data->set_0retran_mode;

            if (p_set_0retran_mode->status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s %d failed status: %d\n", __func__, __LINE__, p_set_0retran_mode->status);
                status = BTHH_ERR;
            }
            SfExtCallbacks.SetZeroRetranCallback(pktCtxt->buf, status);
            break;
        }
        case CALLBACK_TYPE_EXTENSION_GET_PENDING_CONNECTIONS:
        {
            SfExtCallbacks.GetPendingConnectionsCallback((InfoFromExtensionCallbacks*)pktCtxt->buf);
            break;
        }
        case CALLBACK_TYPE_FINALIZE:
        {
            // exit from the callback thread!
            //CbDataContext.msgQueue.qFree();
            //return;
        }
        default:
            NN_SDK_LOG("[bluetooth] %s: ERROR: Invalid msg %d line %d\n", __func__, pktCtxt->cbType, __LINE__);
            break;
    }
    CbDataContext.msgQueue.qFree();

} // NOLINT(impl/function_size)

/*******************************************************************************
 **
 ** Function        ProcessBleQueue
 **
 ** Description     Thread that handles BLE callback messages.
 **
 ** Returns
 **
 *******************************************************************************/
static void ProcessBleQueue()
{
    HAL_PACKET_CONTEXT *pktCtxt;
    pktCtxt = static_cast<HAL_PACKET_CONTEXT*>(BleDataContext.msgQueue.qRead());

    if (pktCtxt == NULL)
    {
        return;
    }

    switch(pktCtxt->cbType)
    {
        case CALLBACK_TYPE_LE_CLIENT_GATT_OP:
        {
            BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_OP");
            SfBleCallbacks.LeClientGattOperationCallback((InfoFromLeGattOperationCallback*)pktCtxt->buf);
            break;
        }
        case CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED:
        {
            BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED");

            InfoFromLeGattIndicationResponseNeededCallback* pInfo
                = reinterpret_cast<InfoFromLeGattIndicationResponseNeededCallback*>(pktCtxt->buf);

            tBT_UUID serviceUuid;
            tBT_UUID characteristicUuid;

            memcpy(&serviceUuid, &pInfo->serviceId.uuid, sizeof(tBT_UUID));
            memcpy(&characteristicUuid, &pInfo->characteristicId.uuid, sizeof(tBT_UUID));

            BluetoothLeStatus status = BtHalLeClientSendIndicationRsp(
                pInfo->connId,
                serviceUuid, pInfo->serviceId.instanceId, pInfo->isPrimaryService,
                characteristicUuid, pInfo->characteristicId.instanceId
            );

            if (status != BT_OK)
            {
                NN_SDK_LOG("Failed to responsd for GATT indication.");
            }

            NN_UNUSED(status);
            break;
        }
        case CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK:
        {
            BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK");
            SfBleCallbacks.LeServerGattReqCallback((InfoFromLeServerGattReqCallback*)pktCtxt->buf);
            break;
        }
        default:
            NN_SDK_LOG("[bluetooth] %s: ERROR: Invalid msg %d line %d\n", __func__, pktCtxt->cbType, __LINE__);
            break;
    }
    BleDataContext.msgQueue.qFree();
} // NOLINT(impl/function_size)

void ProcessBleCoreQueue()
{
    HAL_PACKET_CONTEXT *pktCtxt;
    pktCtxt = static_cast<HAL_PACKET_CONTEXT*>(BleCoreContext.msgQueue.qRead());

    if (pktCtxt == NULL)
    {
        return;
    }

    switch (pktCtxt->cbType)
    {
    case CALLBACK_TYPE_LE_CLIENT_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_STATE_CHANGED");
        SfBleCallbacks.LeClientStateChangedCallback((InfoFromLeAppStateChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_SERVER_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_STATE_CHANGED");
        SfBleCallbacks.LeServerStateChangedCallback((InfoFromLeAppStateChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_SERVER_PROFILE_CHANGED_CALLBACK:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_PROFILE_CHANGED_CALLBACK");
        SfBleCallbacks.LeServerProfileChangedCallback((InfoFromLeServerProfileChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_SCAN_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SCAN_STATE_CHANGED");
        SfBleCallbacks.LeScanStateChangedCallback((InfoFromLeScanStateChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_SCAN_FILTER_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SCAN_FILTER_STATE_CHANGED");
        SfBleCallbacks.LeScanFilterStateChangedCallback((InfoFromLeScanFilterStateChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_CONN_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_CONN_STATE_CHANGED");

        InfoFromLeConnStateChangedCallback* pInfo = reinterpret_cast<InfoFromLeConnStateChangedCallback*>(pktCtxt->buf);

        if (pInfo->connState == BLE_CONN_STATE_DISCONNECTED)
        {
            NN_ABORT_UNLESS_EQUAL(BtHalLeClientDeregisterNotificationAll(pInfo->clientIf, pInfo->address), BT_OK);
        }

        SfBleCallbacks.LeClientConnStateChangedCallback(pInfo);
        break;
    }
    case CALLBACK_TYPE_LE_SERVER_CONN_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_CONN_STATE_CHANGED");
        SfBleCallbacks.LeServerConnStateChangedCallback((InfoFromLeConnStateChangedCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_GATT_OP:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_OP");
        SfBleCallbacks.LeClientCoreGattOperationCallback((InfoFromLeGattOperationCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED");

        InfoFromLeGattIndicationResponseNeededCallback* pInfo
            = reinterpret_cast<InfoFromLeGattIndicationResponseNeededCallback*>(pktCtxt->buf);

        tBT_UUID serviceUuid;
        tBT_UUID characteristicUuid;

        memcpy(&serviceUuid, &pInfo->serviceId.uuid, sizeof(tBT_UUID));
        memcpy(&characteristicUuid, &pInfo->characteristicId.uuid, sizeof(tBT_UUID));

        BluetoothLeStatus status = BtHalLeClientSendIndicationRsp(
            pInfo->connId,
            serviceUuid, pInfo->serviceId.instanceId, pInfo->isPrimaryService,
            characteristicUuid, pInfo->characteristicId.instanceId
        );

        if (status != BT_OK)
        {
            NN_SDK_LOG("Failed to responsd for GATT indication.");
        }

        NN_UNUSED(status);
        break;
    }
    case CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK");
        SfBleCallbacks.LeServerCoreGattReqCallback((InfoFromLeServerGattReqCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_GATT_SERVICE_DISCOVERY:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_SERVICE_SEARCH");
        SfBleCallbacks.LeClientGattServiceDiscoveryCallback((InfoFromLeClientGattServiceDiscoveryCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_GATT_SERVICE_DISCOVERY_STATE_CHANGED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_SERVICE_SEARCH_STATE_CHANGED");
        SfBleCallbacks.LeClientGattServiceDiscoveryStateChangedCallback((InfoFromLeClientGattServiceDiscoveryCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CONN_PARAM_UPDATE:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CONN_PARAM_UPDATE");
        SfBleCallbacks.LeConnParamUpdateCallback((InfoFromLeConnParamUpdateCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CONN_PARAM_UPDATE_REQ:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CONN_PARAM_UPDATE_REQ");
        SfBleCallbacks.LeConnParamUpdateReqCallback((InfoFromLeConnParamUpdateReqCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_CONFIGURE_MTU:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_CONFIGURE_MTU");
        SfBleCallbacks.LeClientConfigureMtuCallback((InfoFromLeClientMtuConfigurationCallback*)pktCtxt->buf);
        break;
    }
    default:
        NN_SDK_LOG("[bluetooth] %s: ERROR: Invalid msg %d line %d\n", __func__, pktCtxt->cbType, __LINE__);
        break;
    }

    BleCoreContext.msgQueue.qFree();
} // NOLINT(impl/function_size)

void ProcessBleHidQueue()
{
    HAL_PACKET_CONTEXT *pktCtxt;
    pktCtxt = static_cast<HAL_PACKET_CONTEXT*>(BleHidContext.msgQueue.qRead());

    if (pktCtxt == NULL)
    {
        return;
    }

    switch (pktCtxt->cbType)
    {
    case CALLBACK_TYPE_LE_CLIENT_GATT_OP:
    {
//        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_OP");
        SfBleCallbacks.LeClientHidGattOperationCallback((InfoFromLeGattOperationCallback*)pktCtxt->buf);
        break;
    }
    case CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_CLIENT_GATT_INDICATION_RESPONSE_NEEDED");

        InfoFromLeGattIndicationResponseNeededCallback* pInfo
            = reinterpret_cast<InfoFromLeGattIndicationResponseNeededCallback*>(pktCtxt->buf);

        tBT_UUID serviceUuid;
        tBT_UUID characteristicUuid;

        memcpy(&serviceUuid, &pInfo->serviceId.uuid, sizeof(tBT_UUID));
        memcpy(&characteristicUuid, &pInfo->characteristicId.uuid, sizeof(tBT_UUID));

        BluetoothLeStatus status = BtHalLeClientSendIndicationRsp(
            pInfo->connId,
            serviceUuid, pInfo->serviceId.instanceId, pInfo->isPrimaryService,
            characteristicUuid, pInfo->characteristicId.instanceId
        );

        if (status != BT_OK)
        {
            NN_SDK_LOG("Failed to responsd for GATT indication.");
        }

        NN_UNUSED(status);
        break;
    }
    case CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK:
    {
        BTHAL_IF_BLE_DEBUG("CALLBACK_TYPE_LE_SERVER_GATT_REQ_CALLBACK");
        SfBleCallbacks.LeServerHidGattReqCallback((InfoFromLeServerGattReqCallback*)pktCtxt->buf);
        break;
    }
    default:
        NN_SDK_LOG("[bluetooth] %s: ERROR: Invalid msg %d line %d\n", __func__, pktCtxt->cbType, __LINE__);
        break;
    }

    BleHidContext.msgQueue.qFree();
}

/*******************************************************************************
 **
 ** Function       BluetoothHalCreateQueue
 **
 ** Description    Creates message queue's for HID/Callback/BLE data
 **
 ** Parameters     type -
 **
 ** Returns
 **
 *******************************************************************************/
void BluetoothHalCreateQueue(uint8_t type)
{
    BTHAL_BSA_DEBUG("Start");
    if (type == HAL_QUEUE_HID) {
        BTHAL_BSA_DEBUG("initialized HAL_QUEUE_HID");

        HidDataContext.msgQueue.qInitialize(hidBuf, sizeof(hidBuf), "HID");
        HidDataContext.type         = HAL_QUEUE_HID;
        HidDataContext.valid        = true;

        // create a thread that will handle the HID messages
        nn::Result err1 = nn::os::CreateThread(&halMsgThread,
                                              &HalMsgThread,
                                              NULL,
                                              StackOfHidThread,
                                              THREAD_STACK_SIZE,
                                              NN_SYSTEM_THREAD_PRIORITY(bluetooth, HidMessageHandler));
        if (err1.IsFailure()) {
            NN_SDK_LOG("[bluetooth] %s: Error creating hid thread\n",__FUNCTION__);
        }
        nn::os::SetThreadNamePointer(&halMsgThread, NN_SYSTEM_THREAD_NAME(bluetooth, HidMessageHandler));
        nn::os::StartThread( &halMsgThread );
    } else if (type == HAL_QUEUE_CB) {
        BTHAL_BSA_DEBUG("initialized HAL_QUEUE_CB");

        CbDataContext.msgQueue.qInitialize(cbBuf, sizeof(cbBuf), "CBQ");
        CbDataContext.type          = HAL_QUEUE_CB;
        CbDataContext.valid         = true;
    } else if (type == HAL_QUEUE_BLE) {
        BTHAL_BSA_DEBUG("initialized HAL_QUEUE_BLE");

        BleDataContext.msgQueue.qInitialize(bleBuf, sizeof(bleBuf), "BLE");
        BleDataContext.type         = HAL_QUEUE_BLE;
        BleDataContext.valid        = true;
    } else if (type == HAL_QUEUE_BLE_CORE) {
        BTHAL_BSA_DEBUG("initialized HAL_QUEUE_BLE_CORE");

        BleCoreContext.msgQueue.qInitialize(bleCoreBuf, sizeof(bleCoreBuf), "BLE_CORE");
        BleCoreContext.type         = HAL_QUEUE_BLE_CORE;
        BleCoreContext.valid        = true;
    } else if (type == HAL_QUEUE_BLE_HID) {
        BTHAL_BSA_DEBUG("initialized HAL_QUEUE_BLE_HID");

        BleHidContext.msgQueue.qInitialize(bleHidBuf, sizeof(bleHidBuf), "BLE_HID");
        BleHidContext.type = HAL_QUEUE_BLE_HID;
        BleHidContext.valid = true;
    }
}


/*******************************************************************************
 **
 ** Function       BluetoothHalDestroyQueue
 **
 ** Description    Destroy message queue's for HID/Callback/BLE data
 **
 ** Parameters     type -
 **
 ** Returns
 **
 *******************************************************************************/
void BluetoothHalDestroyQueue(uint8_t type)
{
    BTHAL_BSA_DEBUG("Enter");
    HAL_MSG_QUEUE_CONTEXT *pContext;

    if (type == HAL_QUEUE_HID) {
        pContext = &HidDataContext;
    } else if (type == HAL_QUEUE_CB) {
        pContext = &CbDataContext;
    } else if (type == HAL_QUEUE_BLE) {
        pContext = &BleDataContext;
    } else if (type == HAL_QUEUE_BLE_CORE) {
        pContext = &BleCoreContext;
    } else if (type == HAL_QUEUE_BLE_HID) {
        pContext = &BleHidContext;
    } else {
        NN_SDK_LOG("[bluetooth] %s: invalid context. %d\n",__FUNCTION__, type);
        return;
    }

    pContext->valid = false;

    if (type == HAL_QUEUE_HID) {
        pContext->msgQueue.qWrite(CALLBACK_TYPE_FINALIZE, 0, NULL);
        nn::os::WaitThread(&halMsgThread);
        nn::os::DestroyThread(&halMsgThread);
        }

    pContext->msgQueue.qFinalize();

    //std::memset(pContext, 0, sizeof(HAL_MSG_QUEUE_CONTEXT));

    BTHAL_BSA_DEBUG("queue destroyed");
}

/*******************************************************************************
 **
 ** Function    BluetoothHalInsertToQueue
 **
 ** Description Queue callback Data using a Linked List
 **
 ** Parameters  pData     - pointer to report data
 **             dataLen   - data length
 **             type      - message queue target
 **             cbType    - callback type (callback or HID data)
 **
 ** Returns     int
 **
 *******************************************************************************/
int BluetoothHalInsertToQueue(void *pData, int dataLen, uint8_t type, enum CallbackType cbType)
{
    HAL_MSG_QUEUE_CONTEXT *pContext;

    if (type == HAL_QUEUE_HID) {
        pContext = &HidDataContext;

        // if HID DATA and more than 1/2 full, drop it to make sure of room for other events
        if (cbType == CALLBACK_TYPE_HID_DATA && pContext->msgQueue.qWriteableSize() < pContext->msgQueue.qBufferSize() / 2)
        {
           static int hidDataDrops = 0;
           if ((++hidDataDrops % 100) == 0)
           {
               NN_SDK_LOG("[bluetooth] %s: Dropped 100 more HID DATA packets, total=%d\n",__FUNCTION__, hidDataDrops);
           }
           return -1;
        }
    }
    else if (type == HAL_QUEUE_CB) {
        pContext = &CbDataContext;
    }
    else if (type == HAL_QUEUE_BLE) {
        pContext = &BleDataContext;
    }
    else if (type == HAL_QUEUE_BLE_CORE) {
        pContext = &BleCoreContext;
    }
    else if (type == HAL_QUEUE_BLE_HID) {
        pContext = &BleHidContext;
    }
    else {
        NN_SDK_LOG("[bluetooth] %s: invalid context. %d\n",__FUNCTION__, type);
        return -1;
    }

    if (dataLen > BUFFER_SIZE) {
        NN_SDK_LOG("[bluetooth] %s: FAIL - TOO LARGE...data length=%d\n",__FUNCTION__, dataLen);
        return -1;
    }

    if (pContext->valid == false)
    {
        NN_SDK_LOG("[bluetooth] trying to send message to stopped queue: queue:%d cbType:%d\n", type, cbType);
        return -1;
    }

    auto r = pContext->msgQueue.qWrite(cbType, dataLen, pData);
    return r;
    }


