﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "bluetooth_le.h"

/*******************************************************************************
**
** Function:        BtHalLeServerRegister
**
** Description:     Register BLE server application
**
** Parameters:      [in] tBT_UUID uuid  : an identifier of server application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerRegister(tBT_UUID uuid);

/*******************************************************************************
**
** Function:        BtHalLeServerDeregister
**
** Description:     Deregister BLE server application
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
**
** Returns:         status: 0 if success / -1 otherwise
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerDeregister(tBSA_BLE_IF serverIf);

/*******************************************************************************
**
** Function:        BtHalLeServerDeregisterAll
**
** Description:     Deregister all BLE server applications
**                  Internally call BtHalLeServerDeregister for all server applications.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerDeregisterAll();

/*******************************************************************************
**
** Function:        BtHalLeServerCreateService
**
** Description:     Create a GATT service to a server application
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] int attrNum            : Number of attributes which belongs to the service. 2 + Num of char * 2 + Num of descriptor * 1.
** Parameters:      [in] bool isPrimaryService  : flag indicating if the service is primary or not
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerCreateService(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid, int attrNum, bool isPrimaryService);

/*******************************************************************************
**
** Function:        BtHalLeServerAddCharacteristic
**
** Description:     Add character to a service for a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf               : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid               : attribute UUID of the service
** Parameters:      [in] tBT_UUID characterisicUuid         : attribute UUID of the characteristic
** Parameters:      [in] tBSA_BLE_PERM charPermission       : attribute permission of the characteristic
** Parameters:      [in] tBSA_BLE_CHAR_PROP charProperty    : attribute property of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerAddCharacteristic(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid,
                                                 tBT_UUID characterisicUuid, tBSA_BLE_PERM charPermission, tBSA_BLE_CHAR_PROP charProperty);

/*******************************************************************************
**
** Function:        BtHalLeServerAddDescriptor
**
** Description:     Add descriptor to a service of a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf           : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid           : attribute UUID of the service
** Parameters:      [in] tBT_UUID descUuid              : attribute UUID of the descriptor
** Parameters:      [in] tBSA_BLE_PERM descPermission   : attribute permission of the descriptor
**
** Returns:         status: 0 if success / -1 otherwise
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerAddDescriptor(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid,
                                             tBT_UUID descUuid, tBSA_BLE_PERM descPermission);


/*******************************************************************************
**
** Function:        BtHalLeServerStartService
**
** Description:     Start a GATT service of a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerStartService(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid);

/*******************************************************************************
**
** Function:        BtHalLeServerSetResponse
**
** Description:     Set response to client read request
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID uuid          : attribute UUID of the characteristic/descriptor
** Parameters:      [in] uint8_t* pData         : resopnse data
** Parameters:      [in] uint16_t len           : response data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSetResponse(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid, tBT_UUID uuid, const uint8_t* pData, uint16_t len);

/*******************************************************************************
**
** Function:        BtHalLeServerSendNotification
**
** Description:     Send notification to a connected client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the characteristic
** Parameters:      [in] uint8_t* pData         : notification data
** Parameters:      [in] uint16_t len           : notification data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSendNotification(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid, tBT_UUID charUuid, const uint8_t* pData, uint16_t len);

/*******************************************************************************
**
** Function:        BtHalLeServerSendIndication
**
** Description:     Send indication to a connected client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the characteristic
** Parameters:      [in] uint8_t* pData         : notification data
** Parameters:      [in] uint16_t len           : notification data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSendIndication(tBSA_BLE_IF serverIf, tBT_UUID serviceUuid, tBT_UUID charUuid, const uint8_t* pData, uint16_t len);

/*******************************************************************************
**
** Function:        BtHalLeServerOpenConnection
**
** Description:     Connect to a BLE client
**
** Parameters:      [in] tBSA_BLE_IF serverIf       : identifier of server application
** Parameters:      [in] Btbdaddr* pClientBdAddr    : BD Address of the target client
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerOpenConnection(tBSA_BLE_IF serverIf,
                                              const Btbdaddr* pClientBdAddr,
                                              BOOLEAN isDirectConnection);

/*******************************************************************************
**
** Function:        BtHalLeServerCloseConnection
**
** Description:     Disconnect from a connected BLE client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of server application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerCloseConnection(tBSA_BLE_IF server_if);

