﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>
#include "bluetooth_le_server.h"
#include "bluetooth.h"
#include "bluetooth_queue.h"
#include "xml_param.h"      // for toString()
#include "bluetooth_le_interface.h"

/*******************************************************************************
**
** Function:        FindServerFreeSpace
**
** Description:     find free space for BLE server application
**
** Parameters:      None
**
** Returns:         index of an available slot of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindServerFreeSpace()
{
    for(int index = 0; index < BLE_SERVER_MAX ; ++index)
    {
        if(!app_ble_cb.ble_server[index].enabled)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        DisplayServer
**
** Description:     Display BLE servers
**
** Parameters:      None
**
** Returns:         void
**
*******************************************************************************/
static void DisplayServer()
{
    for(int i = 0; i < BLE_SERVER_MAX; ++i)
    {
        if(app_ble_cb.ble_server[i].enabled)
        {
            if(app_ble_cb.ble_server[i].conn_id == BSA_BLE_INVALID_CONN)
            {
                BTHAL_IF_BLE_DEBUG("Server #%d: server_if: %d\n", i, app_ble_cb.ble_server[i].server_if);
            }
            else
            {
                BTHAL_IF_BLE_DEBUG("Server #%d: server_if: %d conn_id: %d\n",
                            i,
                            app_ble_cb.ble_server[i].server_if,
                            app_ble_cb.ble_server[i].conn_id);
            }

            for(int j = 0; j < BSA_BLE_ATTRIBUTE_MAX; ++j)
            {
                if (app_ble_cb.ble_server[i].attr[j].attr_handle != 0x00)
                {
                    BTHAL_IF_BLE_DEBUG("    Attribute #%d\n", j);
                    BTHAL_IF_BLE_DEBUG("        Type: %d (0: Include Service, 1: Characteristic, 2: Descriptor, 3: Service)\n", app_ble_cb.ble_server[i].attr[j].attr_type);
                    BTHAL_IF_BLE_DEBUG("        Service Handle: %d,    Own Handle: %d\n", app_ble_cb.ble_server[i].attr[j].service_handle, app_ble_cb.ble_server[i].attr[j].attr_handle);
                    if (app_ble_cb.ble_server[i].attr[j].attr_UUID.len == 2)
                    {
                        BTHAL_IF_BLE_DEBUG("        UUID: %04X\n", app_ble_cb.ble_server[i].attr[j].attr_UUID.uu.uuid16);
                    }
                    else if (app_ble_cb.ble_server[i].attr[j].attr_UUID.len == 4)
                    {
                        BTHAL_IF_BLE_DEBUG("        UUID: %08X\n", app_ble_cb.ble_server[i].attr[j].attr_UUID.uu.uuid32);
                    }
                    else if (app_ble_cb.ble_server[i].attr[j].attr_UUID.len == 16)
                    {
                        BTHAL_IF_BLE_DEBUG("        UUID: %s\n", toString(app_ble_cb.ble_server[i].attr[j].attr_UUID.uu.uuid128, LEN_UUID_128));
                    }
                    BTHAL_IF_BLE_DEBUG("         Primary Service: %s    Property: %d\n",
                        app_ble_cb.ble_server[i].attr[j].is_pri ? "TRUE" : "FALSE",
                        app_ble_cb.ble_server[i].attr[j].prop);
                }
            }
        }
    }
}

/*******************************************************************************
**
** Function:        FindServerByServerInterface
**
** Description:     find BLE server index by interface
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of a server application
**
** Returns:         index of an server application of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindServerByServerInterface(tBSA_BLE_IF serverIf)
{
    for (int index = 0; index < BLE_SERVER_MAX; index++)
    {
        if (app_ble_cb.ble_server[index].enabled && app_ble_cb.ble_server[index].server_if == serverIf)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        FindServerByConnectionId
**
** Description:     find BLE server index by connection ID
**
** Parameters:      [in] uint16_t connId    : connection ID of a connected BLE client
**
** Returns:         index of an server application of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindServerByConnectionId(uint16_t connId)
{
    for (int index = 0; index < BLE_SERVER_MAX; index++)
    {
        if (app_ble_cb.ble_server[index].enabled && app_ble_cb.ble_server[index].conn_id == connId)
        {
            return index;
        }
    }

    return -1;
}

/*******************************************************************************
**
** Function:        FindAttributeFreeSpace
**
** Description:     find free attr for BLE server application
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of a server application
**
** Returns:         index of an available slot of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindAttributeFreeSpace(tBSA_BLE_IF serverIf)
{
    int serverIdx = FindServerByServerInterface(serverIf);
    if(serverIdx < 0)
    {
        return -1;
    }

    for (int index = 0; index < BSA_BLE_ATTRIBUTE_MAX; index++)
    {
        if (app_ble_cb.ble_server[serverIdx].attr[index].attr_UUID.len == 0)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        FindAttribute
**
** Description:     find attr in BLE server application based on attribute UUID
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of a server application
** Parameters:      [in] tBT_UUID uuid          : attribute UUID of the target attribute
**
** Returns:         index of the attribute of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindAttribute(tBSA_BLE_IF serverIf, tBT_UUID uuid)
{
    int serverIdx = FindServerByServerInterface(serverIf);
    if(serverIdx < 0)
    {
        return -1;
    }

    for (int index = 0; index < BSA_BLE_ATTRIBUTE_MAX; index++)
    {
        if (memcmp(&app_ble_cb.ble_server[serverIdx].attr[index].attr_UUID, &uuid, sizeof(uuid)) == 0)
        {
            return index;
        }
    }
    return -1;
}

/*******************************************************************************
**
** Function:        FindAttribute
**
** Description:     find attr in BLE server application based on attribute handle
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of a server application
** Parameters:      [in] uint16_t handle        : attribute handle of the target attribute
**
** Returns:         index of the attribute of the database if successful, -1 otherwise
**
*******************************************************************************/
static int FindAttribute(tBSA_BLE_IF serverIf, uint16_t handle)
{
    int serverIdx = FindServerByServerInterface(serverIf);
    if (serverIdx < 0)
    {
        return -1;
    }

    for (int index = 0; index < BSA_BLE_ATTRIBUTE_MAX; index++)
    {
        if (app_ble_cb.ble_server[serverIdx].attr[index].attr_handle == handle)
        {
            return index;
        }
    }

    return -1;
}

/*******************************************************************************
**
** Function:        LeServerStateChangedCallback
**
** Description:     Callback for a server application state change.
**
** Parameters:      [in] tBSA_BLE_EVT event : event type
** Parameters:      [in] tBSA_BLE_MSG pData : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerStateChangedCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeAppStateChangedCallback cbData;
    tBSA_BLE_IF serverIf = 0;
    int serverIdx = -1;

    if (event == BSA_BLE_SE_DEREGISTER_EVT)
    {
        serverIf = pData->ser_deregister.server_if;
    }

    serverIdx = FindServerByServerInterface(pData->ser_deregister.server_if);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE server. Server not found.\n", __func__);

        cbData.status       = BT_ERR_SERVER;
        cbData.serverIf     = BSA_BLE_INVALID_IF;
        cbData.isRegistered = false;
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_SE_DEREGISTER_EVT:
            BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_DEREGISTER_EVT");

            app_ble_cb.ble_server[serverIdx].enabled    = false;
            app_ble_cb.ble_server[serverIdx].server_if  = BSA_BLE_INVALID_IF;
            app_ble_cb.ble_server[serverIdx].conn_id    = BSA_BLE_INVALID_CONN;
            for (int i = 0; i < BSA_BLE_ATTRIBUTE_MAX; i++)
            {
                memset(&(app_ble_cb.ble_server[serverIdx].attr[i]), 0, sizeof(tAPP_BLE_ATTRIBUTE));
            }

            NN_SDK_LOG("[bluetooth] %s: BLE server application deleted.\n", __func__);
            DisplayServer();

            cbData.serverIf     = serverIf;
            cbData.status       = LeConvertBsaStatus(pData->ser_deregister.status);
            cbData.isRegistered = false;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeAppStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SERVER_STATE_CHANGED);
}

/*******************************************************************************
**
** Function:        LeServerConnectionStateChangedCallback
**
** Description:     Callback for a server connection state change.
**
** Parameters:      [in] tBSA_BLE_EVT event : event type
** Parameters:      [in] tBSA_BLE_MSG pData : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerConnectionStateChangedCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeConnStateChangedCallback cbData;
    tBSA_BLE_IF serverIf = 0;
    int serverIdx = -1;

    switch (event)
    {
    case BSA_BLE_SE_OPEN_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_OPEN_EVT");
        serverIf = pData->ser_open.server_if;
        break;
    case BSA_BLE_SE_CLOSE_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_CLOSE_EVT");
        serverIf = pData->ser_close.server_if;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    serverIdx = FindServerByServerInterface(serverIf);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Server not found. serverIf: %d\n", __func__, serverIf);

        cbData.status       = BT_ERR_SERVER;
        cbData.connState    = BLE_CONN_STATE_DISCONNECTED;
        cbData.serverIf     = BSA_BLE_INVALID_IF;
        cbData.connId       = BSA_BLE_INVALID_CONN;
        memset(cbData.address.address, 0x00, BD_ADDR_LEN);
        cbData.reason = 0xFF;
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_SE_OPEN_EVT:
            app_ble_cb.ble_server[serverIdx].conn_id = pData->ser_open.conn_id;

            if (pData->ser_open.conn_id != BSA_BLE_INVALID_CONN)
            {
                cbData.connState = BLE_CONN_STATE_CONNECTED;
            }
            else
            {
                cbData.connState = BLE_CONN_STATE_DISCONNECTED;
            }

            cbData.status       = BT_OK;
            cbData.serverIf     = pData->ser_open.server_if;
            cbData.connState    = BLE_CONN_STATE_CONNECTED;
            cbData.connId       = pData->ser_open.conn_id;
            memcpy(cbData.address.address, pData->ser_open.remote_bda, BD_ADDR_LEN);
            cbData.reason       = pData->ser_open.reason;
            break;
        case BSA_BLE_SE_CLOSE_EVT:
            app_ble_cb.ble_client[serverIdx].conn_id = BSA_BLE_INVALID_CONN;
            for (int i = 0; i < BSA_BLE_ATTRIBUTE_MAX; ++i)
            {
                memset(&app_ble_cb.ble_server[serverIdx].attr[i], 0, sizeof(tAPP_BLE_ATTRIBUTE));
            }

            cbData.status       = BT_OK;
            cbData.clientIf     = pData->ser_close.server_if;
            cbData.connState    = BLE_CONN_STATE_DISCONNECTED;
            cbData.connId       = pData->ser_close.conn_id;
            memcpy(cbData.address.address, pData->ser_close.remote_bda, BD_ADDR_LEN);
            cbData.reason       = pData->ser_close.reason;

            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeConnStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SERVER_CONN_STATE_CHANGED);
}

/*******************************************************************************
**
** Function:        LeServerProfileChangedCallback
**
** Description:     Callback for a server profile updates.
**
** Parameters:      [in] tBSA_BLE_EVT event : event type
** Parameters:      [in] tBSA_BLE_MSG pData : event data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerProfileChangedCallback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    InfoFromLeServerProfileChangedCallback cbData;
    uint8_t serverIf;
    int serverIdx = -1;

    switch (event)
    {
    case BSA_BLE_SE_CREATE_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_CREATE_EVT");
        serverIf = pData->ser_create.server_if;
        break;
    case BSA_BLE_SE_ADDCHAR_EVT:
        BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_ADDCHAR_EVT");
        serverIf = pData->ser_addchar.server_if;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    serverIdx = FindServerByServerInterface(serverIf);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Error. Server not found. serverIf: %d\n", __func__, serverIf);

        cbData.status           = BT_ERR_SERVER;
        cbData.serverIf         = BSA_BLE_INVALID_IF;
        memset(&cbData.uuid, 0x00, sizeof(GattAttributeUuid));
        cbData.serviceHandle    = 0x00;
        cbData.attributeHandle  = 0x00;
        cbData.type             = GattAttributeType_Unknown;
        cbData.property         = 0x00;
        cbData.isPrimaryService = false;
    }
    else
    {
        switch (event)
        {
        case BSA_BLE_SE_CREATE_EVT:
        {
            int attrIndex = 0;

            for (attrIndex = 0; attrIndex < BSA_BLE_ATTRIBUTE_MAX; ++attrIndex)
            {
                if (app_ble_cb.ble_server[serverIdx].attr[attrIndex].wait_flag)
                {
                    if (pData->ser_create.status == BSA_SUCCESS)
                    {
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].service_handle = pData->ser_create.service_id;
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_handle    = pData->ser_create.service_id;
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].wait_flag      = false;

                        memcpy(&(cbData.uuid), &(app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_UUID), sizeof(GattAttributeUuid));
                        cbData.serviceHandle    = pData->ser_create.service_id;
                        cbData.attributeHandle  = pData->ser_create.service_id;
                        cbData.type             = (GattAttributeType)app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_type;
                        cbData.isPrimaryService = app_ble_cb.ble_server[serverIdx].attr[attrIndex].is_pri;
                        cbData.property         = 0x00;
                    }
                    else
                    {
                        NN_SDK_LOG("[bluetooth] %s: Failed to create service. status: %d\n", __func__, pData->ser_create.status);
                        memset(&app_ble_cb.ble_server[serverIdx].attr[attrIndex], 0, sizeof(tAPP_BLE_ATTRIBUTE));

                        memcpy(&(cbData.uuid), &(app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_UUID), sizeof(GattAttributeUuid));
                        cbData.serviceHandle    = 0x00;
                        cbData.attributeHandle  = 0x00;
                        cbData.type             = GattAttributeType_Unknown;
                        cbData.isPrimaryService = app_ble_cb.ble_server[serverIdx].attr[attrIndex].is_pri;
                        cbData.property         = 0x00;
                    }

                    cbData.status   = LeConvertBsaStatus(pData->ser_create.status);
                    cbData.serverIf = serverIf;

                    break;
                }
            }

            if (attrIndex >= BSA_BLE_ATTRIBUTE_MAX)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to create service. Attribute not found.\n", __func__);
                cbData.status           = BT_ERR_SERVER;
                cbData.serverIf         = serverIf;
                memset(&cbData.uuid, 0x00, sizeof(GattAttributeUuid));
                cbData.serviceHandle    = 0x00;
                cbData.attributeHandle  = 0x00;
                cbData.type             = GattAttributeType_Unknown;
                cbData.isPrimaryService = false;
                cbData.property         = 0x00;

                break;
            }

            break;
        }
        case BSA_BLE_SE_ADDCHAR_EVT:
        {
            int attrIndex = 0;

            for (attrIndex = 0; attrIndex < BSA_BLE_ATTRIBUTE_MAX; ++attrIndex)
            {
                if (app_ble_cb.ble_server[serverIdx].attr[attrIndex].wait_flag)
                {
                    if (pData->ser_addchar.status == BSA_SUCCESS)
                    {
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].service_handle = pData->ser_addchar.service_id;
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_handle    = pData->ser_addchar.attr_id;
                        app_ble_cb.ble_server[serverIdx].attr[attrIndex].wait_flag      = false;

                        memcpy(&(cbData.uuid), &(app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_UUID), sizeof(GattAttributeUuid));
                        cbData.serviceHandle    = pData->ser_addchar.service_id;
                        cbData.attributeHandle  = pData->ser_addchar.attr_id;

                        if (pData->ser_addchar.is_discr)
                        {
                            cbData.type = GattAttributeType_Descriptor;
                        }
                        else
                        {
                            cbData.type = GattAttributeType_Characteristic;
                        }
                        cbData.isPrimaryService = false;
                        cbData.property = app_ble_cb.ble_server[serverIdx].attr[attrIndex].prop;
                    }
                    else
                    {
                        NN_SDK_LOG("[bluetooth] %s: Failed to add characteristics or descriptor. status: %d\n", __func__, pData->ser_addchar.status);
                        memset(&app_ble_cb.ble_server[serverIdx].attr[attrIndex], 0, sizeof(tAPP_BLE_ATTRIBUTE));

                        memcpy(&(cbData.uuid), &(app_ble_cb.ble_server[serverIdx].attr[attrIndex].attr_UUID), sizeof(GattAttributeUuid));
                        cbData.serviceHandle    = 0x00;
                        cbData.attributeHandle  = 0x00;
                        if (pData->ser_addchar.is_discr)
                        {
                            cbData.type = GattAttributeType_Descriptor;
                        }
                        else
                        {
                            cbData.type = GattAttributeType_Characteristic;
                        }
                        cbData.isPrimaryService = false;
                        cbData.property         = app_ble_cb.ble_server[serverIdx].attr[attrIndex].prop;
                    }

                    cbData.status   = LeConvertBsaStatus(pData->ser_addchar.status);
                    cbData.serverIf = serverIf;

                    break;
                }
            }

            if (attrIndex >= BSA_BLE_ATTRIBUTE_MAX)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to add characteristics or descriptor. Attribute not found.\n", __func__);
                cbData.status           = BT_ERR_SERVER;
                cbData.serverIf         = serverIf;
                memset(&cbData.uuid, 0x00, sizeof(GattAttributeUuid));
                cbData.serviceHandle    = 0x00;
                cbData.attributeHandle  = 0x00;
                cbData.type             = GattAttributeType_Unknown;
                cbData.isPrimaryService = false;
                cbData.property         = 0x00;
            }

            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeServerProfileChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SERVER_PROFILE_CHANGED_CALLBACK);
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        LeServerReadReqCallback
**
** Description:     Callback for read request.
**
** Parameters:      [in] tBSA_BLE_SE_READ_MSG pData  : request data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerReadReqCallback(const tBSA_BLE_SE_READ_MSG* pData)
{
    BTHAL_IF_BLE_DEBUG("called.");
    BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_READ_EVT");

    tBSA_STATUS status = BSA_SUCCESS;
    tBSA_BLE_SE_SENDRSP send_server_resp;
    int serverIdx, attributeIdx;

    InfoFromLeServerGattReqCallback cbData;

    serverIdx = FindServerByConnectionId(pData->conn_id);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. Server not found. Conn Id: %d\n", __func__, pData->conn_id);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleServerError());
    }

    attributeIdx = FindAttribute(app_ble_cb.ble_server[serverIdx].server_if, pData->handle);
    if (attributeIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. Attribute not found. Handle: %d\n", __func__, pData->handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleAttributeNotFound());
    }

    cbData.status           = LeConvertBsaStatus(pData->status);
    cbData.responseStatus   = LeConvertBsaStatus(status);
    cbData.connId           = pData->conn_id;
    cbData.requestType      = BleGattReqType::BLE_GATT_REQ_TYPE_READ;
    cbData.serviceHandle    = app_ble_cb.ble_server[serverIdx].attr[attributeIdx].service_handle;
    cbData.attributeHandle  = app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_handle;
    cbData.attrType         = (GattAttributeType)app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_type;
    memset(cbData.data, 0x00, sizeof(cbData.data));
    cbData.length = 0;
    cbData.offset = 0;

    status = BSA_BleSeSendRspInit(&send_server_resp);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);
        cbData.responseStatus = LeConvertBsaStatus(status);
    }
    else
    {
        send_server_resp.conn_id = pData->conn_id;
        send_server_resp.trans_id = pData->trans_id;
        send_server_resp.status = pData->status;
        send_server_resp.handle = pData->handle;
        send_server_resp.offset = pData->offset;

        if (pData->offset >= app_ble_cb.ble_server[serverIdx].attr[attributeIdx].length)
        {
            send_server_resp.len = 0;
        }
        else
        {
            send_server_resp.len = app_ble_cb.ble_server[serverIdx].attr[attributeIdx].length - pData->offset;
        }

        memcpy(send_server_resp.value, app_ble_cb.ble_server[serverIdx].attr[attributeIdx].value + pData->offset, send_server_resp.len);

        cbData.length = send_server_resp.len;
        memcpy(cbData.data, send_server_resp.value, cbData.length);
        cbData.offset = send_server_resp.offset;

        status = BSA_BleSeSendRsp(&send_server_resp);
        if (status != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);

            cbData.responseStatus = LeConvertBsaStatus(status);
        }
    }

    int serviceIndex = FindAttribute(app_ble_cb.ble_server[serverIdx].server_if, app_ble_cb.ble_server[serverIdx].attr[attributeIdx].service_handle);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. Attribute Service not found. Handle: %d\n", __func__, pData->handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleAttributeNotFound());
    }

    GattAttributeUuid serviceUuid;
    serviceUuid.length = GattAttributeUuidLength(app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len);
    if (serviceUuid.length == GattAttributeUuidLength_16)
    {
        serviceUuid.uu.uuid16 = app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid16;
    }
    else if (serviceUuid.length == GattAttributeUuidLength_32)
    {
        serviceUuid.uu.uuid32 = app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid32;
    }
    else if (serviceUuid.length == GattAttributeUuidLength_128)
    {
        memcpy(serviceUuid.uu.uuid128, app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid128, GattAttributeUuidLength_128);
    }

    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        if (g_GattDataPathFilter[i].inUse && g_GattDataPathFilter[i].uuid == serviceUuid)
        {
            switch (g_GattDataPathFilter[i].type)
            {
            case HAL_QUEUE_BLE:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            case HAL_QUEUE_BLE_CORE:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            case HAL_QUEUE_BLE_HID:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_HID, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    NN_SDK_LOG("[bluetooth] Unhandled BLE GATT operation. Ignored.\n");
    NN_SDK_LOG("[bluetooth] Connection ID: %d   Operation: %d\n", cbData.connId, cbData.requestType);
    if (serviceUuid.length == GattAttributeUuidLength_16)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %04X\n", serviceUuid.uu.uuid16);
    }
    else if (serviceUuid.length == GattAttributeUuidLength_32)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %08X\n", serviceUuid.uu.uuid32);
    }
    else if (serviceUuid.length == GattAttributeUuidLength_128)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %sX\n", toString(serviceUuid.uu.uuid128, GattAttributeUuidLength_128));
    }
    if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_16)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %04X\n", app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid16);
    }
    else if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_32)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %08X\n", app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid32);
    }
    else if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_128)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %sX\n", toString(app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid128, GattAttributeUuidLength_128));
    }

    return;
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        LeServerWriteReqCallback
**
** Description:     Callback for write request.
**
** Parameters:      [in] tBSA_BLE_SE_WRITE_MSG pData  : request data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerWriteReqCallback(const tBSA_BLE_SE_WRITE_MSG* pData)
{
    BTHAL_IF_BLE_DEBUG("called.");
    BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_WRITE_EVT");

    tBSA_STATUS status = BSA_SUCCESS;
    tBSA_BLE_SE_SENDRSP send_server_resp;
    int serverIdx, attributeIdx;

    InfoFromLeServerGattReqCallback cbData;

    serverIdx = FindServerByConnectionId(pData->conn_id);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to receive written data. Server not found. Conn Id: %d\n", __func__, pData->conn_id);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleConnectionError());
    }

    attributeIdx = FindAttribute(app_ble_cb.ble_server[serverIdx].server_if, pData->handle);
    if (attributeIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to receive written data. Attribute not found. Handle: %d\n", __func__, pData->handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleAttributeNotFound());
    }

    cbData.status           = LeConvertBsaStatus(pData->status);
    cbData.responseStatus   = LeConvertBsaStatus(status);
    cbData.connId           = pData->conn_id;
    cbData.requestType      = BleGattReqType::BLE_GATT_REQ_TYPE_WRITE;
    cbData.serviceHandle    = app_ble_cb.ble_server[serverIdx].attr[attributeIdx].service_handle;
    cbData.attributeHandle  = app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_handle;
    cbData.attrType = (GattAttributeType)app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_type;
    memset(cbData.data, 0x00, sizeof(cbData.data));
    cbData.length           = 0;
    cbData.offset           = 0;

    if (pData->need_rsp)
    {
        status = BSA_BleSeSendRspInit(&send_server_resp);
        if (status != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);
        }
        else
        {
            send_server_resp.conn_id    = pData->conn_id;
            send_server_resp.trans_id   = pData->trans_id;
            send_server_resp.status     = pData->status;
            send_server_resp.handle     = pData->handle;
            send_server_resp.len        = pData->len;
            memcpy(send_server_resp.value, pData->value, pData->len);

            status = BSA_BleSeSendRsp(&send_server_resp);
            if (status != BSA_SUCCESS)
            {
                NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);
            }
        }
    }

    cbData.status           = LeConvertBsaStatus(pData->status);
    cbData.responseStatus   = LeConvertBsaStatus(status);
    cbData.length           = pData->len;
    memcpy(cbData.data, pData->value, cbData.length);
    cbData.offset           = pData->offset;

    int serviceIndex = FindAttribute(app_ble_cb.ble_server[serverIdx].server_if, app_ble_cb.ble_server[serverIdx].attr[attributeIdx].service_handle);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. Attribute Service not found. Handle: %d\n", __func__, pData->handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultBleAttributeNotFound());
    }

    GattAttributeUuid serviceUuid;
    serviceUuid.length = GattAttributeUuidLength(app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len);
    if (serviceUuid.length == GattAttributeUuidLength_16)
    {
        serviceUuid.uu.uuid16 = app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid16;
    }
    else if (serviceUuid.length == GattAttributeUuidLength_32)
    {
        serviceUuid.uu.uuid32 = app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid32;
    }
    else if (serviceUuid.length == GattAttributeUuidLength_128)
    {
        memcpy(serviceUuid.uu.uuid128, app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid128, GattAttributeUuidLength_128);
    }

    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        if (g_GattDataPathFilter[i].inUse && g_GattDataPathFilter[i].uuid == serviceUuid)
        {
            switch (g_GattDataPathFilter[i].type)
            {
            case HAL_QUEUE_BLE:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            case HAL_QUEUE_BLE_CORE:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            case HAL_QUEUE_BLE_HID:
                BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeGattOperationCallback), HAL_QUEUE_BLE_HID, CALLBACK_TYPE_LE_CLIENT_GATT_OP);
                return;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    NN_SDK_LOG("[bluetooth] Unhandled BLE GATT operation. Ignored.\n");
    NN_SDK_LOG("[bluetooth] Connection ID: %d   Operation: %d\n", cbData.connId, cbData.requestType);
    if (serviceUuid.length == GattAttributeUuidLength_16)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %04X\n", serviceUuid.uu.uuid16);
    }
    else if (serviceUuid.length == GattAttributeUuidLength_32)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %08X\n", serviceUuid.uu.uuid32);
    }
    else if (serviceUuid.length == GattAttributeUuidLength_128)
    {
        NN_SDK_LOG("[bluetooth] Service UUID: %sX\n", toString(serviceUuid.uu.uuid128, GattAttributeUuidLength_128));
    }
    if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_16)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %04X\n", app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid16);
    }
    else if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_32)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %08X\n", app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid32);
    }
    else if (app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.len == GattAttributeUuidLength_128)
    {
        NN_SDK_LOG("[bluetooth] Attribute UUID: %sX\n", toString(app_ble_cb.ble_server[serverIdx].attr[serviceIndex].attr_UUID.uu.uuid128, GattAttributeUuidLength_128));
    }

    return;
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        LeServerExecWriteCallback
**
** Description:     Callback for exec write request.
**
** Parameters:      [in] tBSA_BLE_SE_EXEC_WRITE_MSG pData  : request data
**
** Returns:         void
**
*******************************************************************************/
static void LeServerExecWriteCallback(const tBSA_BLE_SE_EXEC_WRITE_MSG* pData)
{
    BTHAL_IF_BLE_DEBUG("called.");
    BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_EXEC_WRITE_EVT");

    tBSA_STATUS status;
    tBSA_BLE_SE_SENDRSP send_server_resp;

    status = BSA_BleSeSendRspInit(&send_server_resp);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);
        return;
    }

    send_server_resp.conn_id = pData->conn_id;
    send_server_resp.trans_id = pData->trans_id;
    send_server_resp.status = BSA_SUCCESS;

    status = BSA_BleSeSendRsp(&send_server_resp);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send response. status = %d\n", __func__, status);
        return;
    }

    return;
}

/*******************************************************************************
**
** Function:        LeServerProfileCback
**
** Description:     BLE server general profile callback.
**
** Parameters:      [in] tBSA_BLE_EVT event  : event type
** Parameters:      [in] tBSA_BLE_MSG        : event data
**
** Returns:         void
**
*******************************************************************************/
void LeServerProfileCback(tBSA_BLE_EVT event, tBSA_BLE_MSG *pData)
{
    BTHAL_IF_BLE_DEBUG("called.");

    switch (event)
    {
        case BSA_BLE_SE_DEREGISTER_EVT:
            LeServerStateChangedCallback(event, pData);
            break;
        case BSA_BLE_SE_CREATE_EVT:
        case BSA_BLE_SE_ADDCHAR_EVT:
            LeServerProfileChangedCallback(event, pData);
            break;
        case BSA_BLE_SE_START_EVT:
            BTHAL_IF_BLE_DEBUG("BSA_BLE_SE_START_EVT");
            BTHAL_IF_BLE_DEBUG("Server If: %d    Service Handle: %d", pData->ser_start.server_if, pData->ser_start.service_id);
            break;
        case BSA_BLE_SE_READ_EVT:
            LeServerReadReqCallback(&(pData->ser_read));
            break;
        case BSA_BLE_SE_WRITE_EVT:
            LeServerWriteReqCallback(&(pData->ser_write));
            break;
        case BSA_BLE_SE_EXEC_WRITE_EVT:
            LeServerExecWriteCallback(&(pData->ser_exec_write));
            break;
        case BSA_BLE_SE_OPEN_EVT:
        case BSA_BLE_SE_CLOSE_EVT:
            LeServerConnectionStateChangedCallback(event, pData);
            break;
        default:
            break;
    }
} // NOLINT(impl/function_size)

/*******************************************************************************
**
** Function:        BtHalLeServerRegister
**
** Description:     Register BLE server application
**
** Parameters:      [in] tBT_UUID uuid  : an identifier of server application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerRegister(tBT_UUID uuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_REGISTER ble_register_param;
    InfoFromLeAppStateChangedCallback cbData;

    int serverIdx = FindServerFreeSpace();

    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: BLE server is already registered upto maximum.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeAppRegisterInit(&ble_register_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register server. status = %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_register_param.uuid = uuid;
    ble_register_param.p_cback = LeServerProfileCback;

    status = BSA_BleSeAppRegister(&ble_register_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to register server. status = %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }
    app_ble_cb.ble_server[serverIdx].enabled = TRUE;
    app_ble_cb.ble_server[serverIdx].server_if = ble_register_param.server_if;

    cbData.status       = LeConvertBsaStatus(status);
    cbData.serverIf     = ble_register_param.server_if;
    cbData.isRegistered = true;

    BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeAppStateChangedCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_SERVER_STATE_CHANGED);

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerDeregister
**
** Description:     Deregister BLE server application
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
**
** Returns:         status: 0 if success / -1 otherwise
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerDeregister(tBSA_BLE_IF server_if)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_DEREGISTER ble_deregister_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister BLE server. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeAppDeregisterInit(&ble_deregister_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister server. status = %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_deregister_param.server_if = app_ble_cb.ble_server[serverIdx].server_if;

    status = BSA_BleSeAppDeregister(&ble_deregister_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to deregister server. status = %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerDeregisterAll
**
** Description:     Deregister all BLE server applications
**                  Internally call BtHalLeServerDeregister for all server applications.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerDeregisterAll()
{
    BTHAL_IF_BLE_DEBUG("called.");

    for(int i = 0; i < BLE_SERVER_MAX; ++i)
    {
        if(app_ble_cb.ble_server[i].enabled)
        {
            BluetoothLeStatus status = BtHalLeServerDeregister(app_ble_cb.ble_server[i].server_if);
            if(status != BT_OK)
            {
                return status;
            }
        }
    }

    return BT_OK;
}

/*******************************************************************************
**
** Function:        BtHalLeServerCreateService
**
** Description:     Create a GATT service to a server application
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] int attrNum            : Number of attributes which belongs to the service. 2 + Num of char * 2 + Num of descriptor * 1.
** Parameters:      [in] bool isPrimaryService  : flag indicating if the service is primary or not
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerCreateService(tBSA_BLE_IF server_if, tBT_UUID serviceUuid, int handleNum, bool isPrimaryService)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_CREATE ble_create_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to create service. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int attributeIdx = FindAttributeFreeSpace(server_if);
    if(attributeIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to create service. Service is already registered up to maximum.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeCreateServiceInit(&ble_create_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to create service. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_create_param.service_uuid = serviceUuid;
    ble_create_param.server_if = server_if;
    ble_create_param.num_handle = handleNum;
    ble_create_param.is_primary = isPrimaryService;

    // Should be done first?
    app_ble_cb.ble_server[serverIdx].attr[attributeIdx].wait_flag = true;

    status = BSA_BleSeCreateService(&ble_create_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to create service. status: %d\n", __func__, status);
        app_ble_cb.ble_server[serverIdx].attr[attributeIdx].wait_flag = false;
        return LeConvertBsaStatus(status);
    }

    app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_UUID = serviceUuid;
    app_ble_cb.ble_server[serverIdx].attr[attributeIdx].is_pri = ble_create_param.is_primary;
    app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_type = BSA_GATTC_ATTR_TYPE_SRVC;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerAddCharacteristic
**
** Description:     Add character to a service for a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf               : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid               : attribute UUID of the service
** Parameters:      [in] tBT_UUID characterisicUuid         : attribute UUID of the characteristic
** Parameters:      [in] tBSA_BLE_PERM charPermission       : attribute permission of the characteristic
** Parameters:      [in] tBSA_BLE_CHAR_PROP charProperty    : attribute property of the characteristic
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerAddCharacteristic(tBSA_BLE_IF server_if, tBT_UUID serviceUuid,
                                                 tBT_UUID characterisicUuid, tBSA_BLE_PERM charPermission, tBSA_BLE_CHAR_PROP charProperty)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_ADDCHAR ble_addchar_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add characterisic. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if(serviceIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add characterisic. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int characteristicIdx = FindAttributeFreeSpace(server_if);
    if(characteristicIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add characterisic. Characterisic is already registered up to maximum.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeAddCharInit(&ble_addchar_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add characterisic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_addchar_param.service_id = app_ble_cb.ble_server[serverIdx].attr[serviceIdx].service_handle;
    ble_addchar_param.char_uuid = characterisicUuid;
    ble_addchar_param.is_descr = false;
    ble_addchar_param.perm = charPermission;
    ble_addchar_param.property = charProperty;

    status = BSA_BleSeAddChar(&ble_addchar_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add characterisic. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    /* save all information */
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].attr_UUID = characterisicUuid;
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].prop = charProperty;
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].attr_type = BSA_GATTC_ATTR_TYPE_CHAR;
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].wait_flag = true;

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerAddDescriptor
**
** Description:     Add descriptor to a service of a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf           : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid           : attribute UUID of the service
** Parameters:      [in] tBT_UUID descUuid              : attribute UUID of the descriptor
** Parameters:      [in] tBSA_BLE_PERM descPermission   : attribute permission of the descriptor
**
** Returns:         status: 0 if success / -1 otherwise
**
** Callbacks:       EventFromLeServerProfileChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerAddDescriptor(tBSA_BLE_IF server_if, tBT_UUID serviceUuid,
                                             tBT_UUID descUuid, tBSA_BLE_PERM descPermission)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_ADDCHAR ble_addchar_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add descriptor. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if(serviceIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add descriptor. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int characteristicIdx = FindAttributeFreeSpace(server_if);
    if(characteristicIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add descriptor. Attribute is already registered up to maximum.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeAddCharInit(&ble_addchar_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add descriptor. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_addchar_param.service_id = app_ble_cb.ble_server[serverIdx].attr[serviceIdx].service_handle;
    ble_addchar_param.char_uuid = descUuid;
    ble_addchar_param.is_descr = true;
    ble_addchar_param.perm = descPermission;

    status = BSA_BleSeAddChar(&ble_addchar_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to add descriptor. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    /* save all information */
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].attr_UUID = descUuid;
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].attr_type = BSA_GATTC_ATTR_TYPE_CHAR_DESCR;
    app_ble_cb.ble_server[serverIdx].attr[characteristicIdx].wait_flag = true;

    return LeConvertBsaStatus(status);
}


/*******************************************************************************
**
** Function:        BtHalLeServerStartService
**
** Description:     Start a GATT service of a server
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerStartService(tBSA_BLE_IF server_if, tBT_UUID serviceUuid)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_START ble_start_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start service. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if(serviceIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start service. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeStartServiceInit(&ble_start_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start service. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_start_param.service_id = app_ble_cb.ble_server[serverIdx].attr[serviceIdx].service_handle;
    ble_start_param.sup_transport = BSA_BLE_GATT_TRANSPORT_LE_BR_EDR;

    status = BSA_BleSeStartService(&ble_start_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to start service. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }
    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerSetResponse
**
** Description:     Set response to client read request
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID uuid          : attribute UUID of the characteristic/descriptor
** Parameters:      [in] uint8_t* pData         : resopnse data
** Parameters:      [in] uint16_t len           : response data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSetResponse(tBSA_BLE_IF server_if, tBT_UUID serviceUuid, tBT_UUID uuid, const uint8_t* pData, uint16_t len)
{
    BTHAL_IF_BLE_DEBUG("called.");

    int serverIdx = FindServerByServerInterface(server_if);
    if (serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set response. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if (serviceIdx < 0 || app_ble_cb.ble_server[serverIdx].attr[serviceIdx].attr_type != BSA_GATTC_ATTR_TYPE_SRVC)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set response. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int attributeIdx = FindAttribute(server_if, uuid);
    if (attributeIdx < 0 ||
        (app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_type != BSA_GATTC_ATTR_TYPE_CHAR && app_ble_cb.ble_server[serverIdx].attr[attributeIdx].attr_type != BSA_GATTC_ATTR_TYPE_CHAR_DESCR))
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set response. Attribute not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    if (len > BSA_BLE_MAX_ATTR_LEN)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set response. Too long value. length %d\n", __func__, len);
        return BT_BLE_ERR_INVALID_PARAM;
    }

    app_ble_cb.ble_server[serverIdx].attr[attributeIdx].length = len;
    memcpy(app_ble_cb.ble_server[serverIdx].attr[attributeIdx].value, pData, len);

    return BT_OK;
}

/*******************************************************************************
**
** Function:        BtHalLeServerSendNotification
**
** Description:     Send notification to a connected client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the characteristic
** Parameters:      [in] uint8_t* pData         : notification data
** Parameters:      [in] uint16_t len           : notification data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSendNotification(tBSA_BLE_IF server_if, tBT_UUID serviceUuid, tBT_UUID charUuid, const uint8_t* pData, uint16_t len)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_SENDIND ble_sendind_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if(serviceIdx < 0  || app_ble_cb.ble_server[serverIdx].attr[serviceIdx].attr_type != BSA_GATTC_ATTR_TYPE_SRVC)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int charIdx = FindAttribute(server_if, charUuid);
    if(charIdx < 0  || app_ble_cb.ble_server[serverIdx].attr[charIdx].attr_type != BSA_GATTC_ATTR_TYPE_CHAR)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. Characteristic not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    if((app_ble_cb.ble_server[serverIdx].attr[charIdx].prop & BSA_GATT_CHAR_PROP_BIT_NOTIFY) == 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. Characteristic doesn't have notification property.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeSendIndInit(&ble_sendind_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_sendind_param.conn_id = app_ble_cb.ble_server[serverIdx].conn_id;
    ble_sendind_param.attr_id = app_ble_cb.ble_server[serverIdx].attr[charIdx].attr_handle;

    if(BSA_BLE_MAX_ATTR_LEN < len)
    {
        NN_SDK_LOG("[bluetooth] %s: Indication data too large.", __func__);
        ble_sendind_param.data_len = BSA_BLE_MAX_ATTR_LEN;
    }
    else
    {
        ble_sendind_param.data_len = len;
    }
    memcpy(ble_sendind_param.value, pData, ble_sendind_param.data_len);
    ble_sendind_param.need_confirm = false;

    status = BSA_BleSeSendInd(&ble_sendind_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send notification. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerSendIndication
**
** Description:     Send indication to a connected client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : an identifier of server application
** Parameters:      [in] tBT_UUID serviceUuid   : attribute UUID of the service
** Parameters:      [in] tBT_UUID charUuid      : attribute UUID of the characteristic
** Parameters:      [in] uint8_t* pData         : notification data
** Parameters:      [in] uint16_t len           : notification data length in Byte
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerSendIndication(tBSA_BLE_IF server_if, tBT_UUID serviceUuid, tBT_UUID charUuid, const uint8_t* pData, uint16_t len)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_SENDIND ble_sendind_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int serviceIdx = FindAttribute(server_if, serviceUuid);
    if(serviceIdx < 0  || app_ble_cb.ble_server[serverIdx].attr[serviceIdx].attr_type != BSA_GATTC_ATTR_TYPE_SRVC)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. Service not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    int charIdx = FindAttribute(server_if, charUuid);
    if(charIdx < 0  || app_ble_cb.ble_server[serverIdx].attr[charIdx].attr_type != BSA_GATTC_ATTR_TYPE_CHAR)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. Characteristic not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    if((app_ble_cb.ble_server[serverIdx].attr[charIdx].prop & BSA_GATT_CHAR_PROP_BIT_INDICATE) == 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. Characteristic doesn't have indication property.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeSendIndInit(&ble_sendind_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_sendind_param.conn_id = app_ble_cb.ble_server[serverIdx].conn_id;
    ble_sendind_param.attr_id = app_ble_cb.ble_server[serverIdx].attr[charIdx].attr_handle;

    if(BSA_BLE_MAX_ATTR_LEN < len)
    {
        NN_SDK_LOG("[bluetooth] %s: Indication data too large.", __func__);
        ble_sendind_param.data_len = BSA_BLE_MAX_ATTR_LEN;
    }
    else
    {
        ble_sendind_param.data_len = len;
    }
    memcpy(ble_sendind_param.value, pData, ble_sendind_param.data_len);
    ble_sendind_param.need_confirm = true;

    status = BSA_BleSeSendInd(&ble_sendind_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to send indication. status: %d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerOpenConnection
**
** Description:     Connect to a BLE client
**
** Parameters:      [in] tBSA_BLE_IF serverIf       : identifier of server application
** Parameters:      [in] Btbdaddr* pClientBdAddr    : BD Address of the target client
** Parameters:      [in] bool isDirectConnection    : flag indicating if direct or background connection
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerOpenConnection(tBSA_BLE_IF server_if,
                                              const Btbdaddr* pClientBdAddr,
                                              BOOLEAN isDirectConnection)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_OPEN ble_open_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    if(app_ble_cb.ble_server[serverIdx].conn_id != BSA_BLE_INVALID_CONN)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. Server already has a connection. conn_id: %d\n",
                    __func__,
                    app_ble_cb.ble_server[serverIdx].conn_id);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeConnectInit(&ble_open_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_open_param.server_if = app_ble_cb.ble_server[serverIdx].server_if;
    ble_open_param.is_direct = isDirectConnection;
    copyBdAddr(ble_open_param.bd_addr, pClientBdAddr->address);

    status = BSA_BleSeConnect(&ble_open_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to open connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeServerCloseConnection
**
** Description:     Disconnect from a connected BLE client
**
** Parameters:      [in] tBSA_BLE_IF serverIf   : identifier of server application
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       EventFromLeServerConnStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeServerCloseConnection(tBSA_BLE_IF server_if)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_SE_CLOSE ble_close_param;

    int serverIdx = FindServerByServerInterface(server_if);
    if(serverIdx < 0)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Server not found.\n", __func__);
        return BT_ERR_SERVER;
    }

    if(app_ble_cb.ble_server[serverIdx].conn_id == BSA_BLE_INVALID_CONN)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. Server has no connection.\n", __func__);
        return BT_ERR_SERVER;
    }

    status = BSA_BleSeCloseInit(&ble_close_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    ble_close_param.conn_id = app_ble_cb.ble_server[serverIdx].conn_id;

    status = BSA_BleSeClose(&ble_close_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to close connection. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}


