﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include "bluetooth.h"
#include "bluetooth_queue.h"
#include "bluetooth_le.h"
#include "bluetooth_le_client.h"
#include "bluetooth_le_server.h"
#include "bluetooth_le_disc.h"
#include "bluetooth_hal.h"
#include "xml_param.h"
#include "bluetooth_le_interface.h"

/*
 * Global Variables
 */

tAPP_BLE_CB app_ble_cb;
GattDataPathFilter g_GattDataPathFilter[GATT_DATA_PATH_FILTER_NUM_MAX];

/*******************************************************************************
**
** Function:        BtHalLeInitialize
**
** Description:     Clear BLE client/server application database.
**
** Parameters:      None
**
** Returns:         void
**
*******************************************************************************/
void BtHalLeInitialize()
{
    BTHAL_IF_BLE_DEBUG("called.");

    memset(&app_ble_cb, 0, sizeof(app_ble_cb));

    for(int i = 0; i < BLE_CLIENT_MAX; ++i)
    {
        app_ble_cb.ble_client[i].enabled = false;
        app_ble_cb.ble_client[i].client_if = BSA_BLE_INVALID_IF;
        app_ble_cb.ble_client[i].bsa_conn_id = BSA_BLE_INVALID_CONN;
        app_ble_cb.ble_client[i].conn_id = BleInvalidConnectionHandle;
    }

    for(int i = 0; i < BLE_SERVER_MAX; ++i)
    {
        app_ble_cb.ble_server[i].enabled = false;
        app_ble_cb.ble_server[i].server_if = BSA_BLE_INVALID_IF;
        app_ble_cb.ble_server[i].conn_id = BSA_BLE_INVALID_CONN;

        for (int j = 0; j < BSA_BLE_ATTRIBUTE_MAX; ++j)
        {
            memset(&(app_ble_cb.ble_server[i].attr[j]), 0x00, sizeof(tAPP_BLE_ATTRIBUTE));
        }
    }

    for (int i = 0; i < GATT_DATA_PATH_FILTER_NUM_MAX; ++i)
    {
        g_GattDataPathFilter[i].inUse = false;
    }
}

/*******************************************************************************
**
** Function:        BtHalLeEnable
**
** Description:     Enable BLE features. Deregister all the client/server application.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeEnable()
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS  status;
    tBSA_BLE_ENABLE enable_param;

    BluetoothHalCreateQueue(HAL_QUEUE_BLE);
    BluetoothHalCreateQueue(HAL_QUEUE_BLE_CORE);
    BluetoothHalCreateQueue(HAL_QUEUE_BLE_HID);

    status = BSA_BleEnableInit(&enable_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to enable BLE. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    status = BSA_BleEnable(&enable_param);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to enable BLE. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    BtHalLeClientDeregisterAll();
    BtHalLeServerDeregisterAll();

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeDisable
**
** Description:     Disable BLE features. Deregister all the client/server application.
**
** Parameters:      None
**
** Returns:         BluetoothLeStatus
**
** Callbacks:       InfoFromLeAppStateChangedCallback
**
*******************************************************************************/
BluetoothLeStatus BtHalLeDisable()
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_STATUS status;
    tBSA_BLE_DISABLE param;

    /* Deregister all applications */
    BtHalLeClientDeregisterAll();
    BtHalLeServerDeregisterAll();

    status = BSA_BleDisableInit(&param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to disable BLE status:%d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    status = BSA_BleDisable(&param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to disable BLE status:%d", __func__, status);
        return LeConvertBsaStatus(status);
    }

    BluetoothHalDestroyQueue(HAL_QUEUE_BLE);
    BluetoothHalDestroyQueue(HAL_QUEUE_BLE_CORE);
    BluetoothHalDestroyQueue(HAL_QUEUE_BLE_HID);

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetVisibility
**
** Description:     Start/Stop BLE advertising.
**
** Parameters:      [in] bool discoverable : Start/Stop BLE advertising as discoverable device
** Parameters:      [in] bool connectable : Start/Stop BLE advertising as connectable device
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetVisibility(bool discoverable, bool connectable)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE visivility. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    /* Set visibility configuration */
    bt_config.config_mask = BSA_DM_CONFIG_BLE_VISIBILITY_MASK;
    bt_config.ble_connectable = connectable;
    bt_config.ble_discoverable = discoverable;

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE visibility. status: %d\n", __func__,  status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetConnectionParameter
**
** Description:     Start BLE connection parameter update for a connected device with prefferred parameters.
**
** Parameters:      [in] tBSA_DM_BLE_CONN_PARAM* pParam : prefferred connection parameters.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetConnectionParameter(const tBSA_DM_BLE_CONN_PARAM *pParam)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE connection paramter. status: %d\n", __func__,  status);

        return LeConvertBsaStatus(status);
    }

    bt_config.enable = true;

    bt_config.config_mask = BSA_DM_CONFIG_BLE_CONN_PARAM_MASK;
    copyBdAddr(bt_config.ble_conn_param.bd_addr, pParam->bd_addr);
    bt_config.ble_conn_param.max_conn_int = pParam->max_conn_int;
    bt_config.ble_conn_param.min_conn_int = pParam->min_conn_int;
    bt_config.ble_conn_param.slave_latency = pParam->slave_latency;
    bt_config.ble_conn_param.supervision_tout = pParam->supervision_tout;
    bt_config.ble_conn_param.min_ce_len = pParam->min_ce_len;
    bt_config.ble_conn_param.max_ce_len = pParam->max_ce_len;

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE connection paramter. status: %d\n", __func__,  status);

        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetDefaultConnectionParameter
**
** Description:     Set default BLE connection parameter for incoming connection. Actual update will start when connection established.
**
** Parameters:      [in] tBSA_DM_BLE_CONN_PARAM* pParam : prefferred connection parameters.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetDefaultConnectionParameter(const tBSA_DM_BLE_CONN_PARAM *pParam)
{
    BTHAL_IF_BLE_DEBUG("called.");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE connection paramter. status: %d\n", __func__,  status);

        return LeConvertBsaStatus(status);
    }

    bt_config.enable = true;

    bt_config.config_mask = BSA_DM_CONFIG_BLE_DEFAULT_CONN_PARAM_MASK;
    bt_config.ble_conn_param.max_conn_int = pParam->max_conn_int;
    bt_config.ble_conn_param.min_conn_int = pParam->min_conn_int;
    bt_config.ble_conn_param.slave_latency = pParam->slave_latency;
    bt_config.ble_conn_param.supervision_tout = pParam->supervision_tout;
    bt_config.ble_conn_param.min_ce_len = pParam->min_ce_len;
    bt_config.ble_conn_param.max_ce_len = pParam->max_ce_len;

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE connection paramter. status: %d\n", __func__, status);

        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetAdvertiseData
**
** Description:     Set BLE advertising packet data.
**
** Parameters:      [in] tBSA_DM_BLE_ADV_CONFIG* pConfig : contens of advertising data.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetAdvertiseData(const tBSA_DM_BLE_ADV_CONFIG *pConfig)
{
    BTHAL_IF_BLE_DEBUG("called");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE advertise data. status: %d\n", __func__,  status);

        return LeConvertBsaStatus(status);
    }

    bt_config.enable = true;
    bt_config.config_mask = BSA_DM_CONFIG_BLE_ADV_CONFIG_MASK;
    bt_config.adv_config.len = pConfig->len;
    bt_config.adv_config.flag = pConfig->flag;
    bt_config.adv_config.adv_data_mask = pConfig->adv_data_mask;
    bt_config.adv_config.appearance_data = pConfig->appearance_data;
    bt_config.adv_config.num_service = pConfig->num_service;
    bt_config.adv_config.is_scan_rsp = pConfig->is_scan_rsp;

    memcpy(bt_config.adv_config.uuid_val, pConfig->uuid_val,pConfig->num_service * sizeof(uint16_t));
    memcpy(bt_config.adv_config.p_val,pConfig->p_val,pConfig->len);

    bt_config.adv_config.service_data_len = pConfig->service_data_len;
    bt_config.adv_config.service_data_uuid.len = pConfig->service_data_uuid.len;
    bt_config.adv_config.service_data_uuid.uu.uuid16 = pConfig->service_data_uuid.uu.uuid16;
    memcpy(bt_config.adv_config.service_data_val,pConfig->service_data_val,pConfig->service_data_len);

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE advertise data. status: %d\n", __func__,  status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetAdvertiseParam
**
** Description:     Set BLE advertising parameters.
**
** Parameters:      [in] tBSA_DM_BLE_ADV_PARAM* pParam : parameters for BLE advertising.
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetAdvertiseParam(const tBSA_DM_BLE_ADV_PARAM* pParam)
{
    BTHAL_IF_BLE_DEBUG("called");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE advertise paramter. status: %d\n", __func__,  status);
        return LeConvertBsaStatus(status);
    }

    bt_config.enable = true;
    bt_config.config_mask = BSA_DM_CONFIG_BLE_ADV_PARAM_MASK;
    bdcpy(bt_config.ble_adv_param.dir_bda.bd_addr, pParam->dir_bda.bd_addr);
    bt_config.ble_adv_param.adv_int_min = pParam->adv_int_min;
    bt_config.ble_adv_param.adv_int_max = pParam->adv_int_max;

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE advertise paramter. status: %d\n", __func__,  status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}

/*******************************************************************************
**
** Function:        BtHalLeSetScanParameter
**
** Description:     Set BLE scan parameters.
**
** Parameters:      [in] uint16_t scanInterval : Scan interval. Actual interval would be (scanInterval * 0.625) msec
** Parameters:      [in] uint16_t scanWindow : Scan Window. Actual window would be (scanWindow * 0.625) msec
**
** Conditions:      scanWindow <= scanInterval
**
** Returns:         BluetoothLeStatus
**
*******************************************************************************/
BluetoothLeStatus BtHalLeSetScanParameter(uint16_t scanInterval, uint16_t scanWindow)
{
    BTHAL_IF_BLE_DEBUG("called");

    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS status;

    if (scanInterval < scanWindow)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE scan parameter. Scan interval(%d) must be bigger than scan window(%d).\n",
            __func__,
            scanInterval, scanWindow);
        return BT_ERR_BAD_PARAM;
    }

    status = BSA_DmSetConfigInit(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE scan paramter. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    bt_config.enable = true;

    bt_config.config_mask = BSA_DM_CONFIG_BLE_SCAN_PARAM_MASK;
    bt_config.ble_scan_interval = scanInterval;
    bt_config.ble_scan_window = scanWindow;

    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Failed to set LE scan paramter. status: %d\n", __func__, status);
        return LeConvertBsaStatus(status);
    }

    return LeConvertBsaStatus(status);
}


