﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/* default inclusion */
#include "bluetooth_hal.h"
#include "bsa_robson_api.h"
/* for tracing functions */
#include "bluetooth_hal_utils.h"
#include "bluetooth_hal_hh.h"
#include "bluetooth_queue.h"
#include "bluetooth.h"
#include <nn/bluetooth/bluetooth_TypesHal.h>
#include "bluetooth_hid.h"
#include "bluetooth_le.h"
#include "bluetooth_le_interface.h"

#include <nn/nn_SdkLog.h>
/*---------------------------------------------------------------------------*
 *
 *    Globals/Externs
 *    -- Variables/Functions --
 *
 *---------------------------------------------------------------------------*/
tAPP_ROBSON_CB app_robson_cb;
tAPP_SECURITY_CB app_sec_cb;
tAPP_DISCOVERY_CB app_discovery_cb;

bool bonding_connect = false;

static nn::os::EventType g_SetConfigEvent;
static nn::os::MutexType g_SetConfigMutex;

static tBSA_ROBSON_SET_CONFIG_MSG g_SetConfigMsg;

/* Temporary, need to ask NCL where to get this parameter from */
static uint8_t featureSet = 0x68;
/*---------------------------------------------------------------------------*
 *
 *    Constants defined for this file
 *    -- #Defines's --
 *
 *---------------------------------------------------------------------------*/
#define CONNECT_AFTER_PAIRING

/* Default BdAddr */
#define APP_DEFAULT_BD_ADDR             {0x43, 0x56, 0x00, 0x00, 0x00, 0x01}

/* Default local Name */
#define APP_DEFAULT_BT_NAME             "Nintendo Switch"

/* Default COD SetTopBox (Major Service = none) (MajorDevclass = Audio/Video) (Minor=Display&Speaker) */
#define APP_DEFAULT_CLASS_OF_DEVICE     {0x00, 0x04, 0x3C}

#define APP_DEFAULT_ROOT_PATH           "./"

#define APP_DEFAULT_PIN_CODE            "0000"

/*
 * Simple Pairing Input/Output Capabilities:
 * BSA_SEC_IO_CAP_OUT  =>  DisplayOnly
 * BSA_SEC_IO_CAP_IO   =>  DisplayYesNo
 * BSA_SEC_IO_CAP_NONE =>  NoInputNoOutput
 */
#define APP_DEFAULT_IO_CAPABILITIES BSA_SEC_IO_CAP_NONE

/*---------------------------------------------------------------------------*
 *
 *    Data types defined for this file
 *    -- Struct's, Typedef's, Enum's --
 *
 *---------------------------------------------------------------------------*/
enum {APP_DEFAULT_PIN_LEN = 4};

enum {HCI_EIR_DEVICE_ID_TYPE = 0x10};  /* Device Id EIR Tag (not yet official) */

/* For signaling g_SetConfigEvent*/
static tBSA_ROBSON_CONFIG_MASK expectedSetConfigMask = 0;

/*---------------------------------------------------------------------------*
 *
 *    File scope functions/symbols defined for this file
 *    -- Function Prototypes, File Scope Data --
 *
 *---------------------------------------------------------------------------*/
static void BtHalCallbackDiscovery(tBSA_DISC_EVT event, tBSA_DISC_MSG *p_data);

static tBSA_STATUS BtHalSetConfig(tBSA_ROBSON_SET_CONFIG set_config, int timeOutMs);

/*******************************************************************************
**
** Function:        FindClientByServerAddress
**
** Description:     find BLE client index by connected GATT server's bluetooth address
**
** Parameters:      [in] BluetoothAddress:  connected GATT server's bluetooth address
**
** Returns:         index of client application database if successful, -1 otherwise
**
*******************************************************************************/
static int FindClientByServerAddress(const BluetoothAddress& address)
{
    for (int index = 0; index < BLE_CLIENT_MAX; index++)
    {
        if (memcmp(app_ble_cb.ble_client[index].server_addr, address.address, BD_ADDR_LEN) == 0)
        {
            return index;
        }
    }
    return -1;
}

// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// MOVE ////////////////////////////////////////// //
/*******************************************************************************
 **
 ** Function        app_robson_init_xml
 **
 ** Description     Create the XML configuration file if it does not exist
 **
 ** Parameters      None
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int app_robson_init_xml()
{
    int status;
    BD_ADDR local_bd_addr = APP_DEFAULT_BD_ADDR;
    DEV_CLASS local_class_of_device = APP_DEFAULT_CLASS_OF_DEVICE;

    status = app_read_xml_config(&app_robson_cb.xml_config);
    if (status < 0)
    {
        //NNC_SDK_LOG("[bluetooth] Creating default XML config file");
        app_robson_cb.xml_config.enable = TRUE;
        app_robson_cb.xml_config.discoverable = FALSE;
        app_robson_cb.xml_config.connectable = FALSE;
        strncpy((char *)app_robson_cb.xml_config.name, APP_DEFAULT_BT_NAME, sizeof(app_robson_cb.xml_config.name) - 1);
        app_robson_cb.xml_config.name[sizeof(app_robson_cb.xml_config.name) - 1] = '\0';

        copyBdAddr(app_robson_cb.xml_config.bd_addr, local_bd_addr);

        memcpy(app_robson_cb.xml_config.class_of_device, local_class_of_device, sizeof(DEV_CLASS));

        strncpy((char *)app_robson_cb.xml_config.pin_code, APP_DEFAULT_PIN_CODE, APP_DEFAULT_PIN_LEN);
        app_robson_cb.xml_config.pin_len = APP_DEFAULT_PIN_LEN;

        app_robson_cb.xml_config.io_cap = APP_DEFAULT_IO_CAPABILITIES;

        status = app_write_xml_config(&app_robson_cb.xml_config);
        if (status < 0)
        {
            NN_SDK_LOG("[bluetooth] Failed creating default XML config file\n");
            return status;
        }
    }
    return 0;
}

/*******************************************************************************
 **
 ** Function        app_robson_apply_xml
 **
 ** Description     Configure the BSA server with XML config
 **
 ** Parameters      None
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int app_robson_apply_xml()
{
    BOOLEAN enable = TRUE;
    /* Apply the XML config */
    //app_dm_set_local_bt_config(TRUE);
    int status;
    tBSA_DM_SET_CONFIG bt_config;
    tAPP_XML_CONFIG xml_local_config;

    status = app_read_xml_config(&xml_local_config);
    if (status < 0)
    {
        NN_SDK_LOG("[bluetooth] Unable to Read XML config file\n");
        return status;
    }

    /* Init config parameter */
    status = BSA_DmSetConfigInit(&bt_config);

    /* Set Bluetooth configuration (from XML contents) */
    bt_config.enable = enable;
    bt_config.discoverable = xml_local_config.discoverable;
    bt_config.connectable = xml_local_config.connectable;
    copyBdAddr(bt_config.bd_addr, xml_local_config.bd_addr);
    strncpy((char *)bt_config.name, (char *)xml_local_config.name, sizeof(bt_config.name));
    bt_config.name[sizeof(bt_config.name) - 1] = '\0';
    memcpy(bt_config.class_of_device, xml_local_config.class_of_device, sizeof(DEV_CLASS));

    BTHAL_IF_DEBUG("Set Local Bluetooth Config:");
    BTHAL_IF_DEBUG("\tEnable:%d", bt_config.enable);
    BTHAL_IF_DEBUG("\tDiscoverable:%d", bt_config.discoverable);
    BTHAL_IF_DEBUG("\tConnectable:%d", bt_config.connectable);
    BTHAL_IF_DEBUG("\tName:%s", bt_config.name);
    BTHAL_IF_DEBUG("\tBdaddr %02x:%02x:%02x:%02x:%02x:%02x",
            bt_config.bd_addr[0], bt_config.bd_addr[1],
            bt_config.bd_addr[2], bt_config.bd_addr[3],
            bt_config.bd_addr[4], bt_config.bd_addr[5]);
    BTHAL_IF_DEBUG("\tClassOfDevice:%02x:%02x:%02x", bt_config.class_of_device[0],
            bt_config.class_of_device[1], bt_config.class_of_device[2]);

    /* Apply BT config */
    status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Unable to set BT config to server status:%d\n", status);
        return(-1);
    }
    return 0;
}

/*******************************************************************************
 **
 ** Function        app_robson_set_config
 **
 ** Description     Configure the BSA server with ROBSON specific parameters
 **
 ** Parameters      None
 **
 ** Returns         status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int app_robson_set_config()
{
    tBSA_DM_SET_CONFIG bt_config;
    tBSA_STATUS bsa_status;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    /* Indicate the specific entries to configure */
    bt_config.config_mask = 0;

    /* Obviously */
    bt_config.enable = TRUE;

    /* Configure the Page Scan parameters */
    bt_config.config_mask |= BSA_DM_CONFIG_PAGE_SCAN_PARAM_MASK;
    bt_config.page_scan_interval = 2048; //1280 ms
    bt_config.page_scan_window = 18;//11.25 ms

    /* Configure the Inquiry Scan parameters */
    bt_config.config_mask |= BSA_DM_CONFIG_INQ_SCAN_PARAM_MASK;
    bt_config.inquiry_scan_interval = 4096; //2560 ms
    bt_config.inquiry_scan_window = 18; //11.25ms

    bsa_status = BSA_DmSetConfig(&bt_config);
    if (bsa_status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_DmSetConfig failed status:%d\n", bsa_status);
        return(-1);
    }

#ifdef RSSI_STATS
    NN_SDK_LOG("[bluetooth] RSSI Monitoring Is On\n");
    BSA_DmSetConfigInit(&bt_config);
    bt_config.enable = TRUE;
    bt_config.config_mask = BSA_DM_CONFIG_MONITOR_RSSI;
    bt_config.monitor_rssi_param.enable = 1;
    bt_config.monitor_rssi_param.period = 1;
    bsa_status = BSA_DmSetConfig(&bt_config);
    if (bsa_status != BSA_SUCCESS)
    {
        NN_SDK_LOG("BSA_DmSetConfig failed status:%d ", bsa_status);
    }
#endif

    return 0;
}

// ////////////////////////////////////////// REMOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// REMOVE ////////////////////////////////////////// //
// ////////////////////////////////////////// REMOVE ////////////////////////////////////////// //


/*******************************************************************************
 **
 ** Function         BtHalBsaDiscFindDevice
 **
 ** Description      Find a device in the list of discovered devices
 **
 ** Parameters       bda: BD address of the searched device
 **
 ** Returns          Pointer to the discovered device, NULL if not found
 **
 *******************************************************************************/
tBSA_DISC_DEV *BtHalBsaDiscFindDevice(BD_ADDR bda)
{
    int index;

    /* Unfortunately, the BSA_SEC_AUTH_CMPL_EVT does not contain COD */
    /* let's look in the Discovery database */
    for (index = 0; index < APP_DISC_NB_DEVICES; index++)
    {
        if ((app_discovery_cb.devs[index].in_use != FALSE) &&
            (bdcmp(app_discovery_cb.devs[index].device.bd_addr, bda) == 0))
        {
            return &app_discovery_cb.devs[index];
        }
    }
    return NULL;
}


// ////////////////////////////////////////// HAL CORE INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL CORE INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL CORE INTERFACE ////////////////////////////////////////// //


/*******************************************************************************
 **
 ** Function         BtHalCallbackSecurity
 **
 ** Description      ROBSON Security callback
 **
 ** Parameters
 **     tBSA_SEC_EVT event: security event
 **     tBSA_SEC_MSG *p_data: data
 **
 ** Returns         BOOLEAN: True if the generic callback must return (do not handle the event)
 **                          False if the generic callback must handle the event
 **
 *******************************************************************************/
BOOLEAN BtHalCallbackSecurity(tBSA_SEC_EVT event, tBSA_SEC_MSG *p_data)
{
    switch(event)
    {
        case BSA_SEC_LINK_UP_EVT:       /* A device is physically connected (for info) */
            BTHAL_IF_DEBUG("BSA_SEC_LINK_UP_EVT bd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->link_up.bd_addr[0], p_data->link_up.bd_addr[1],
                    p_data->link_up.bd_addr[2], p_data->link_up.bd_addr[3],
                    p_data->link_up.bd_addr[4], p_data->link_up.bd_addr[5]);
            BTHAL_IF_DEBUG("ClassOfDevice:%02x:%02x:%02x",
                    p_data->link_up.class_of_device[0],
                    p_data->link_up.class_of_device[1],
                    p_data->link_up.class_of_device[2]);
            break;

        case BSA_SEC_LINK_DOWN_EVT:     /* A device is physically disconnected (for info) */
            BTHAL_IF_DEBUG("BSA_SEC_LINK_DOWN_EVT bd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->link_down.bd_addr[0], p_data->link_down.bd_addr[1],
                    p_data->link_down.bd_addr[2], p_data->link_down.bd_addr[3],
                    p_data->link_down.bd_addr[4], p_data->link_down.bd_addr[5]);
            BTHAL_IF_DEBUG("Reason: %d (0x%x)", p_data->link_down.status, p_data->link_down.status);
            break;

        case BSA_SEC_PIN_REQ_EVT:
            BTHAL_IF_DEBUG("BSA_SEC_PIN_REQ_EVT (Pin Code Request) received from:");
            BTHAL_IF_DEBUG("\tbd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->pin_req.bd_addr[0], p_data->pin_req.bd_addr[1],
                    p_data->pin_req.bd_addr[2], p_data->pin_req.bd_addr[3],
                    p_data->pin_req.bd_addr[4],p_data->pin_req.bd_addr[5]);

            /* For keyboards, etc */
            BluetoothHalInsertToQueue(&p_data->pin_req,
                                      sizeof(p_data->pin_req),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_PIN_REQUEST);
            break;

        case BSA_SEC_AUTH_CMPL_EVT:
            NN_SDK_LOG("[bluetooth] BSA_SEC_AUTH_CMPL_EVT received\n");
            NN_SDK_LOG("[bluetooth] \tName:%s\n", p_data->auth_cmpl.bd_name);

            if (p_data->auth_cmpl.success == 0)
            {
                NN_SDK_LOG("[bluetooth] \t: Success:%d => FAIL failure_reason:%d\n",
                        p_data->auth_cmpl.success, p_data->auth_cmpl.fail_reason);
                BluetoothHalInsertToQueue(&p_data->auth_cmpl,
                                          sizeof(p_data->auth_cmpl),
                                          HAL_QUEUE_CB,
                                          CALLBACK_TYPE_BOND_STATE_AUTHENTICATION_FAILURE);
            }
            else
            {
                BTHAL_IF_DEBUG("\tSuccess:%d => OK", p_data->auth_cmpl.success);

#ifndef CONNECT_AFTER_PAIRING
                bonding_connect = true;
#endif

                BluetoothHalInsertToQueue(&p_data->auth_cmpl,
                                          sizeof(p_data->auth_cmpl),
                                          HAL_QUEUE_CB,
                                          CALLBACK_TYPE_BOND_STATE_AUTHENTICATION_SUCCESS);
            }

            NN_SDK_LOG("[bluetooth]\tbd_addr:%02x:%02x:%02x:%02x:%02x:%02x\n",
                    p_data->auth_cmpl.bd_addr[0], p_data->auth_cmpl.bd_addr[1],
                    p_data->auth_cmpl.bd_addr[2], p_data->auth_cmpl.bd_addr[3],
                    p_data->auth_cmpl.bd_addr[4], p_data->auth_cmpl.bd_addr[5]);
            if (p_data->auth_cmpl.key_present != FALSE)
            {
                NN_SDK_LOG("[bluetooth]\tLinkKey:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
                        p_data->auth_cmpl.key[0], p_data->auth_cmpl.key[1],
                        p_data->auth_cmpl.key[2], p_data->auth_cmpl.key[3],
                        p_data->auth_cmpl.key[4], p_data->auth_cmpl.key[5],
                        p_data->auth_cmpl.key[6], p_data->auth_cmpl.key[7],
                        p_data->auth_cmpl.key[8], p_data->auth_cmpl.key[9],
                        p_data->auth_cmpl.key[10], p_data->auth_cmpl.key[11],
                        p_data->auth_cmpl.key[12], p_data->auth_cmpl.key[13],
                        p_data->auth_cmpl.key[14], p_data->auth_cmpl.key[15]);
            }
            break;

        case BSA_SEC_BOND_CANCEL_CMPL_EVT:
            BTHAL_IF_DEBUG("BSA_SEC_BOND_CANCEL_CMPL_EVT status=%d",
                       p_data->bond_cancel.status);

            BluetoothHalInsertToQueue(&p_data->bond_cancel,
                                      sizeof(p_data->bond_cancel),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_CANCELLED);
            break;

        case BSA_SEC_AUTHORIZE_EVT:  /* Authorization request */
            BTHAL_IF_DEBUG("BSA_SEC_AUTHORIZE_EVT received");
            BTHAL_IF_DEBUG("\tRemote device:%s ", p_data->authorize.bd_name);
            BTHAL_IF_DEBUG("bd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->authorize.bd_addr[0], p_data->authorize.bd_addr[1],
                    p_data->authorize.bd_addr[2], p_data->authorize.bd_addr[3],
                    p_data->authorize.bd_addr[4],p_data->authorize.bd_addr[5]);
            BTHAL_IF_DEBUG("\tRequest access to service:%x",(int)p_data->authorize.service);


            break;

        case BSA_SEC_SP_CFM_REQ_EVT:     /* Simple Pairing confirm request */
            BTHAL_IF_DEBUG("BSA_SEC_SP_CFM_REQ_EVT received");
            BluetoothHalInsertToQueue(&p_data->cfm_req,
                                      sizeof(p_data->cfm_req),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_BONDING);

            BluetoothHalInsertToQueue(&p_data->cfm_req,
                                      sizeof(p_data->cfm_req),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_SP_CONFIRMED);

            copyBdAddr(app_sec_cb.sec_bd_addr, p_data->cfm_req.bd_addr);
            break;

        case BSA_SEC_SP_KEY_NOTIF_EVT: /* Simple Pairing Passkey Notification */
            BTHAL_IF_DEBUG("BSA_SEC_SP_KEY_NOTIF_EVT received");
            BTHAL_IF_DEBUG("\tbd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->key_notif.bd_addr[0], p_data->key_notif.bd_addr[1],
                    p_data->key_notif.bd_addr[2], p_data->key_notif.bd_addr[3],
                    p_data->key_notif.bd_addr[4], p_data->key_notif.bd_addr[5]);
            BTHAL_IF_DEBUG("\tClassOfDevice:%02x:%02x:%02x", p_data->key_notif.class_of_device[0],
                    p_data->key_notif.class_of_device[1], p_data->key_notif.class_of_device[2]);
            BTHAL_IF_DEBUG("\tNumeric Value:%d", p_data->key_notif.passkey);

            BluetoothHalInsertToQueue(&p_data->key_notif,
                                      sizeof(p_data->key_notif),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_KEY_NOTIFICATION);
            break;

        case BSA_SEC_SP_KEYPRESS_EVT: /* Simple Pairing Key press notification event. */
            BTHAL_IF_DEBUG("BSA_SEC_SP_KEYPRESS_EVT received type:%d", p_data->key_press.notif_type);
            BTHAL_IF_DEBUG("\tbd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->key_press.bd_addr[0], p_data->key_press.bd_addr[1],
                    p_data->key_press.bd_addr[2], p_data->key_press.bd_addr[3],
                    p_data->key_press.bd_addr[4], p_data->key_press.bd_addr[5]);
            break;

        case BSA_SEC_SP_RMT_OOB_EVT: /* Simple Pairing Remote OOB Data request. */
            BTHAL_IF_DEBUG("BSA_SEC_SP_RMT_OOB_EVT received");
            BTHAL_IF_DEBUG("BLUETOOTH_HAL: Not Yet Implemented");
            break;

        case BSA_SEC_SUSPENDED_EVT: /* Connection Suspended */
            BTHAL_IF_DEBUG("BSA_SEC_SUSPENDED_EVT bd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->suspended.bd_addr[0], p_data->suspended.bd_addr[1],
                    p_data->suspended.bd_addr[2], p_data->suspended.bd_addr[3],
                    p_data->suspended.bd_addr[4], p_data->suspended.bd_addr[5]);
            BluetoothHalInsertToQueue(&p_data->suspended,
                                      sizeof(p_data->suspended),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_SUSPENDED);
            break;

        case BSA_SEC_RESUMED_EVT: /* Connection Resumed */
            BTHAL_IF_DEBUG("BSA_SEC_RESUMED_EVT bd_addr: %02x:%02x:%02x:%02x:%02x:%02x",
                    p_data->resumed.bd_addr[0], p_data->resumed.bd_addr[1],
                    p_data->resumed.bd_addr[2], p_data->resumed.bd_addr[3],
                    p_data->resumed.bd_addr[4], p_data->resumed.bd_addr[5]);

            BluetoothHalInsertToQueue(&p_data->resumed,
                                      sizeof(p_data->resumed),
                                      HAL_QUEUE_CB,
                                      CALLBACK_TYPE_BOND_STATE_RESUMED);
            break;

#ifdef RSSI_STATS
       case BSA_SEC_RSSI_EVT: /* RSSI report received */
            NN_SDK_LOG("[bluetooth] RSSI: %d dBm\n", p_data->sig_strength.rssi_value );
            break;
#endif
       case BSA_SEC_BLE_CONN_PARAM_UPDATE_EVT:
       {
            BTHAL_IF_BLE_DEBUG("BSA_SEC_BLE_CONN_PARAM_UPDATE_EVT");
            BTHAL_IF_BLE_DEBUG("Address: %02X:%02X:%02X:%02X:%02X:%02X",
                                p_data->conn_param.bd_addr[0], p_data->conn_param.bd_addr[1], p_data->conn_param.bd_addr[2],
                                p_data->conn_param.bd_addr[3], p_data->conn_param.bd_addr[4], p_data->conn_param.bd_addr[5]);
            BTHAL_IF_BLE_DEBUG("Conn Interval: %d, Slave Latency: %d, Supervision Timeout: %d", p_data->conn_param.conn_interval, p_data->conn_param.conn_latency, p_data->conn_param.conn_timeout);

            InfoFromLeConnParamUpdateCallback cbData;

            BluetoothAddress address;
            memcpy(address.address, p_data->conn_param.bd_addr, BD_ADDR_LEN);

            int clientIndex = FindClientByServerAddress(address);

            if (clientIndex < 0)
            {
                NN_SDK_LOG("[bluetooth] %s: Error. Client not found.\n", NN_CURRENT_FUNCTION_NAME);

                cbData.status               = BT_ERR_CLIENT;
                cbData.connId               = nn::bluetooth::BleInvalidConnectionHandle;
                cbData.interval             = 0;
                cbData.slaveLatency         = 0;
                cbData.supervisionTimeout   = 0;
            }
            else
            {
                cbData.status                = LeConvertBsaStatus(p_data->conn_param.status);
                cbData.connId                = app_ble_cb.ble_client[clientIndex].conn_id;
                cbData.interval              = p_data->conn_param.conn_interval;
                cbData.slaveLatency          = p_data->conn_param.conn_latency;
                cbData.supervisionTimeout    = p_data->conn_param.conn_timeout;
            }

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeConnParamUpdateCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CONN_PARAM_UPDATE);

            break;
       }
       case BSA_SEC_BLE_CONN_PARAM_UPDATE_REQ_EVT:
       {
            BTHAL_IF_BLE_DEBUG("BSA_SEC_BLE_CONN_PARAM_UPDATE_REQ_EVT");
            BTHAL_IF_BLE_DEBUG("Address: %02X:%02X:%02X:%02X:%02X:%02X",
                                p_data->conn_param_upd_req.bd_addr[0], p_data->conn_param_upd_req.bd_addr[1], p_data->conn_param_upd_req.bd_addr[2],
                                p_data->conn_param_upd_req.bd_addr[3], p_data->conn_param_upd_req.bd_addr[4], p_data->conn_param_upd_req.bd_addr[5]);
            BTHAL_IF_BLE_DEBUG("Conn Interval Min: %d    Conn Interval Max: %d    Slave Latency: %d    Supervision Timeout: %d",
                                p_data->conn_param_upd_req.min_interval, p_data->conn_param_upd_req.max_interval, p_data->conn_param_upd_req.latency, p_data->conn_param_upd_req.timeout);

            InfoFromLeConnParamUpdateReqCallback cbData;

            BluetoothAddress address;
            memcpy(address.address, p_data->conn_param_upd_req.bd_addr, BD_ADDR_LEN);

            int clientIndex = FindClientByServerAddress(address);

            if (clientIndex < 0)
            {
                NN_SDK_LOG("[bluetooth] %s: Error. Client not found\n", NN_CURRENT_FUNCTION_NAME);

                cbData.connId               = BluetoothLeStatus::BT_ERR_CLIENT;
                cbData.intervalMin          = 0;
                cbData.intervalMax          = 0;
                cbData.slaveLatency         = 0;
                cbData.supervisionTimeout   = 0;
            }
            else
            {
                cbData.connId               = app_ble_cb.ble_client[clientIndex].conn_id;
                cbData.intervalMin          = p_data->conn_param_upd_req.min_interval;
                cbData.intervalMax          = p_data->conn_param_upd_req.max_interval;
                cbData.slaveLatency         = p_data->conn_param_upd_req.latency;
                cbData.supervisionTimeout   = p_data->conn_param_upd_req.timeout;
            }

            BluetoothHalInsertToQueue(&cbData, sizeof(InfoFromLeConnParamUpdateReqCallback), HAL_QUEUE_BLE_CORE, CALLBACK_TYPE_LE_CONN_PARAM_UPDATE_REQ);

            break;
       }
        default:
            NN_SDK_LOG("[bluetooth] %s: unknown event:%d\n", __func__, event);
            break;
    }

    /* Let generic Security callback handle the event */
    return FALSE;
} //NOLINT(impl/function_size)

/*******************************************************************************
 **
 ** Function        BtHalGetRemoteServices
 **
 ** Description     Start SDP to get remote services
 **                 The Bluetooth SDP provides a means by which service applications
 **                 running on different Bluetooth enabled devices may discover each
 **                 other's existence, and exchange information to determine their
 **                 characteristics.
 **
 ** Parameters      pRemoteAddr   - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalGetRemoteServices(Btbdaddr *pRemoteAddr)
{
    tBSA_DISC_START disc_start_param;

    BTHAL_IF_DEBUG("Called");

    BSA_DiscStartInit(&disc_start_param);

    disc_start_param.cback = BtHalCallbackDiscovery;
    disc_start_param.nb_devices = 0;
    disc_start_param.duration = 4;

    disc_start_param.services = BSA_ALL_SERVICE_MASK;
    memset(app_discovery_cb.devs, 0, sizeof(app_discovery_cb.devs));

    BTHAL_IF_DEBUG("Start Services filtered Discovery");
    //BTHAL_IF_DEBUG("Look for (%s) services", app_service_mask_to_string(services));
    tBSA_STATUS status = BSA_DiscStart(&disc_start_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: BSA_DiscStart failed status:%d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalBsaDiscParseEir
 **
 ** Description      This function is used to parse EIR data
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalBsaDiscParseEir(UINT8 *p_eir)
{
    UINT8 *p = p_eir;
    UINT8 eir_length;
    UINT8 eir_tag;

    while(1)
    {
        /* Read Tag's length */
        STREAM_TO_UINT8(eir_length, p);
        if (eir_length == 0)
        {
            break;    /* Last Tag */
        }
        eir_length--;

        /* Read Tag Id */
        STREAM_TO_UINT8(eir_tag, p);

        switch(eir_tag)
        {
        case HCI_EIR_FLAGS_TYPE:
            scru_dump_hex(p, "FLAGS_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_MORE_16BITS_UUID_TYPE:
            scru_dump_hex(p, "MORE_16BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_COMPLETE_16BITS_UUID_TYPE:
            scru_dump_hex(p, "COMPLETE_16BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_MORE_32BITS_UUID_TYPE:
            scru_dump_hex(p, "MORE_32BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_COMPLETE_32BITS_UUID_TYPE:
            scru_dump_hex(p, "COMPLETE_32BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_MORE_128BITS_UUID_TYPE:
            scru_dump_hex(p, "MORE_128BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_COMPLETE_128BITS_UUID_TYPE:
            scru_dump_hex(p, "COMPLETE_128BITS_UUID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_SHORTENED_LOCAL_NAME_TYPE:
            scru_dump_hex(p, "SHORTENED_LOCAL_NAME_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_COMPLETE_LOCAL_NAME_TYPE:
            scru_dump_hex(p, "COMPLETE_LOCAL_NAME_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_TX_POWER_LEVEL_TYPE:
            scru_dump_hex(p, "TX_POWER_LEVEL_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_3D_SYNC_TYPE:
            scru_dump_hex(p, "3D_SYNC_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_MANUFACTURER_SPECIFIC_TYPE:
            scru_dump_hex(p, "MANUFACTURER_SPECIFIC_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case HCI_EIR_DEVICE_ID_TYPE:
            scru_dump_hex(p, "DEVICE_ID_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        case BTM_BLE_AD_TYPE_SERVICE_DATA:
            scru_dump_hex(p, "SERVICE_DATA_TYPE", eir_length, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        default:
            scru_dump_hex(p - 1, "Unknown EIR (Tag and data)", eir_length + 1, TRACE_LAYER_NONE, TRACE_TYPE_DEBUG);
            break;
        }
        p += eir_length;
    }
}

/*******************************************************************************
 **
 ** Function         BtHalCallbackDiscovery
 **
 ** Description      Discovery callback. Discovery events are handled in the
 **                  Bluetooth HAL callback thread.
 **
 ** Returns          void
 **
 *******************************************************************************/
static void BtHalCallbackDiscovery(tBSA_DISC_EVT event, tBSA_DISC_MSG *p_data)
{
    switch (event)
    {
    /* a New Device has been discovered */
    case BSA_DISC_NEW_EVT:
    {
        BluetoothHalInsertToQueue(&p_data->disc_new,
                                  sizeof(p_data->disc_new),
                                  HAL_QUEUE_CB,
                                  CALLBACK_TYPE_DEVICE_FOUND);
        break;
    }

    case BSA_DISC_CMPL_EVT: /* Discovery complete. */
    {
        BTHAL_IF_DEBUG("BSA_DISC_CMPL_EVT");
        BluetoothDiscoveryState state = BT_DISCOVERY_STOPPED;
        BluetoothHalInsertToQueue(&state,
                                  sizeof(state),
                                  HAL_QUEUE_CB,
                                  CALLBACK_TYPE_DISCOVER_STATE_CHANGED);
        break;
    }

    case BSA_DISC_DEV_INFO_EVT: /* Discovery Device Info */
        BTHAL_IF_DEBUG("Discover Device Info status:%d", p_data->dev_info.status);
        BTHAL_IF_DEBUG("\tBdaddr:%02x:%02x:%02x:%02x:%02x:%02x",
                 p_data->dev_info.bd_addr[0],
                 p_data->dev_info.bd_addr[1],
                 p_data->dev_info.bd_addr[2],
                 p_data->dev_info.bd_addr[3],
                 p_data->dev_info.bd_addr[4],
                 p_data->dev_info.bd_addr[5]);
        break;
    case BSA_DISC_REMOTE_NAME_EVT:
        BTHAL_IF_DEBUG("BSA_DISC_REMOTE_NAME_EVT (status=%d)", p_data->remote_name.status);
        if (p_data->dev_info.status == BSA_SUCCESS)
        {
            BTHAL_IF_DEBUG("Read device name for %02x:%02x:%02x:%02x:%02x:%02x",
                p_data->remote_name.bd_addr[0], p_data->remote_name.bd_addr[1],
                p_data->remote_name.bd_addr[2], p_data->remote_name.bd_addr[3],
                p_data->remote_name.bd_addr[4], p_data->remote_name.bd_addr[5]);
            BTHAL_IF_DEBUG("Name:%s", p_data->remote_name.remote_bd_name);
        }
        break;

    default:
        NN_SDK_LOG("[bluetooth] %s: unknown event:%d\n", __func__, event);
        break;
    }
}

/*******************************************************************************
 **
 ** Function         BtHalStartDiscovery
 **
 ** Description      Start the discovery process
 **
 ** Parameters
 **
 ** Returns          void
 **
 *******************************************************************************/
void BtHalStartDiscovery()
{
    BTHAL_IF_DEBUG("Called");

    int status;
    tBSA_DISC_START disc_start_param;
    BSA_DiscStartInit(&disc_start_param);

    disc_start_param.cback = BtHalCallbackDiscovery ;
    disc_start_param.nb_devices = 0;
    disc_start_param.duration = 4;
    disc_start_param.mode = BSA_DM_GENERAL_INQUIRY;

    // FOR LONGER DISCOVERY WITH NOTIFICATION AS SOON AS NAME IS AVAILABLE
    disc_start_param.update = BSA_DISC_UPDATE_NAME;
    disc_start_param.duration = 8;  // optional: change the duration

    memset(app_discovery_cb.devs, 0, sizeof(app_discovery_cb.devs));

    status = BSA_DiscStart(&disc_start_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] BSA_DiscStart failed status:%d\n", status);
    }
    else
    {
        BluetoothDiscoveryState state = BT_DISCOVERY_STARTED;
        BluetoothHalInsertToQueue(&state,
                                  sizeof(state),
                                  HAL_QUEUE_CB,
                                  CALLBACK_TYPE_DISCOVER_STATE_CHANGED);
    }
}

/*******************************************************************************
 **
 ** Function        BtHalCancelDiscovery
 **
 ** Description     Cancel the discovery process
 **
 ** Parameters      None
 **
 ** Returns         void
 **
 *******************************************************************************/
void BtHalCancelDiscovery()
{
    tBSA_DISC_ABORT disc_abort_param;
    BTHAL_IF_DEBUG("Called");

    BSA_DiscAbortInit(&disc_abort_param);
    tBSA_STATUS status = BSA_DiscAbort(&disc_abort_param);
    if (status != BSA_SUCCESS && status != BSA_ERROR_SRV_DISC_ABORT_ERROR)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to abort discovery:%d\n", __func__, status);
    }
    return;
}

/*******************************************************************************
 **
 ** Function        BtHalCreateBond
 **
 ** Description     Create a bond with the remote device
 **
 ** Parameters      pBdAddr   - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalCreateBond(const Btbdaddr *pBdAddr)
{
    tBSA_SEC_BOND sec_bond;

    BTHAL_IF_DEBUG("Called");

    BSA_SecBondInit(&sec_bond);
    copyBdAddr(sec_bond.bd_addr, pBdAddr->address);

    tBSA_STATUS status = BSA_SecBond(&sec_bond);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to create bond:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalRemoveBond
 **
 ** Description     Remove bond with the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalRemoveBond(const Btbdaddr *pBdAddr)
{
    tBSA_STATUS status = BSA_SUCCESS;

    BTHAL_IF_DEBUG("Called");

    if (!app_xml_find_dev_db(pBdAddr->address))
    {
        NN_SDK_LOG("[bluetooth] %s: Invalid BD_ADDR\n", __func__);
        return BTHH_ERR_BAD_PARAM;
    }

    {
        /* Remove this device from BSA's HH database */
        tBSA_HH_REMOVE_DEV hh_remove;
        BSA_HhRemoveDevInit(&hh_remove);
        copyBdAddr(hh_remove.bd_addr, pBdAddr->address);

        status = BSA_HhRemoveDev(&hh_remove);
        if (status != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] %s: Unable to remove device from HH:%d\n", __func__, status);
        }

        /* Remove the device from Security database (BSA Server) */
        tBSA_SEC_REMOVE_DEV sec_remove;
        BSA_SecRemoveDeviceInit(&sec_remove);
        copyBdAddr(sec_remove.bd_addr, pBdAddr->address);
        status = BSA_SecRemoveDevice(&sec_remove);
        if (status != BSA_SUCCESS)
        {
            NN_SDK_LOG("[bluetooth] %s: Unable to remove bond:%d\n", __func__, status);
        }

        /* delete the device from database */
        app_xml_remove_db(pBdAddr->address);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalCancelBond
 **
 ** Description     Cancel bond with the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalCancelBond(const Btbdaddr *pBdAddr)
{
    tBSA_SEC_BOND_CANCEL sec_bond_cancel;

    BTHAL_IF_DEBUG("Called");

    BTHAL_IF_DEBUG("\tBdaddr:%02x:%02x:%02x:%02x:%02x:%02x",
                                pBdAddr->address[0],
                                pBdAddr->address[1],
                                pBdAddr->address[2],
                                pBdAddr->address[3],
                                pBdAddr->address[4],
                                pBdAddr->address[5]);

    BSA_SecBondCancelInit(&sec_bond_cancel);
    copyBdAddr(sec_bond_cancel.bd_addr, pBdAddr->address);
    tBSA_STATUS status = BSA_SecBondCancel(&sec_bond_cancel);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to cancel bond:%d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalWakeController
 **
 ** Description     Send wake command to the remote device
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalWakeController(const Btbdaddr *pBdAddr)
{
    tBSA_HH_OPEN hh_open_param;

    BTHAL_IF_DEBUG("Called");

    /* Check if device is known before attempting to connect */
    tAPP_XML_REM_DEVICE *p_xml_dev;
    p_xml_dev = app_xml_find_dev_db(pBdAddr->address);
    if (p_xml_dev == NULL){
        NN_SDK_LOG("[bluetooth] Unable to wake controller, invalid address\n");
        return BTHH_ERR_BAD_PARAM;
    }

    BSA_HhOpenInit(&hh_open_param);
    copyBdAddr(hh_open_param.bd_addr, pBdAddr->address);
    hh_open_param.mode = BSA_HH_PROTO_RPT_MODE;
    hh_open_param.sec_mask = BSA_SEC_AUTHENTICATION;
    hh_open_param.brcm_mask = BSA_HH_BRCM_TBFC_PAGE;

    tBSA_STATUS status = BSA_HhOpen(&hh_open_param);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] Unable to wake controller, status:%d\n", status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalPinReply
 **
 ** Description     Function to reply to pin request
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **                 accept  - 1 to accept, 0 to reject
 **                 pin_len - length of the pin code
 **                 pPinCode- pin code
 **
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalPinReply(const Btbdaddr *pBdAddr, uint8_t accept,uint8_t pin_len, BluetoothPinCode *pPinCode)
{
    tBSA_SEC_PIN_CODE_REPLY pin_code_reply;

    BTHAL_IF_DEBUG("Reply to PIN Pairing");

    BSA_SecPinCodeReplyInit(&pin_code_reply);
    copyBdAddr(pin_code_reply.bd_addr, pBdAddr->address);
    pin_code_reply.pin_len = BT_DEFAULT_PIN_LEN;
    strncpy((char *)pin_code_reply.pin_code, (char *)BT_DEFAULT_PIN_CODE,BT_DEFAULT_PIN_LEN);
    /* note that this code will not work if pin_len = 16 */
    pin_code_reply.pin_code[PIN_CODE_LEN - 1] = '\0';
    tBSA_STATUS status = BSA_SecPinCodeReply(&pin_code_reply);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to Reply to pin Req status:%d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function        BtHalSspReply
 **
 ** Description     Function used to accept/refuse Simple Pairing
 **
 ** Parameters      pBdAddr - Bluetooth device address of remote device
 **                 variant - Just works,Numeric comparison or passkey entry
 **                 accept  - 1 to accept, 0 to reject
 **                 passkey - pass key provided
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSspReply(const Btbdaddr *pBdAddr, BluetoothSspVariant variant,uint8_t accept, uint32_t passkey)
{
    tBSA_SEC_SP_CFM_REPLY   app_sec_sp_cfm_reply;

    BTHAL_IF_DEBUG("Reply to Simple Pairing");

    BSA_SecSpCfmReplyInit(&app_sec_sp_cfm_reply);
    app_sec_sp_cfm_reply.accept = accept;
    copyBdAddr(app_sec_sp_cfm_reply.bd_addr, pBdAddr->address);
    tBSA_STATUS status = BSA_SecSpCfmReply(&app_sec_sp_cfm_reply);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to Reply to SP Req status:%d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}


// ////////////////////////////////////////// HAL EXTENSIONS INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL EXTENSIONS INTERFACE ////////////////////////////////////////// //
// ////////////////////////////////////////// HAL EXTENSIONS INTERFACE ////////////////////////////////////////// //
/*******************************************************************************
 **
 ** Function        BtHalExtensionCallback
 **
 ** Description     Extension callback
 **
 ** Parameters      event   - TSI event
 **                 p_data  - connect data (e.g. bd_addr)
 **
 ** Returns         void
 **
 *******************************************************************************/
void BtHalCallbackExtension(tBSA_ROBSON_EVT event, tBSA_ROBSON_MSG *p_data)
{
    enum CallbackType evt = CALLBACK_TYPE_END;  // invalid
    bool callback=true;

    switch (event)
    {
    case BSA_ROBSON_SET_TSI_EVT:
        evt = CALLBACK_TYPE_EXTENSION_SET_TSI;
        BTHAL_IF_DEBUG("BSA_ROBSON_SET_TSI_EVT");
#ifdef HID_INPUT_STATS
        {
            tBSA_ROBSON_TSI_MODE tsiMode = ((tBSA_ROBSON_SET_TSI_MSG*)p_data)->curr_tsi;
            Btbdaddr bdAddr;

            memcpy(bdAddr.address, ((tBSA_ROBSON_SET_TSI_MSG*)p_data)->peer_addr, sizeof(BD_ADDR));

            BtUpdateTsiForStats(tsiMode, bdAddr);
        }
#endif
        break;
    case BSA_ROBSON_EXIT_TSI_EVT:
        evt = CALLBACK_TYPE_EXTENSION_EXIT_TSI;
        BTHAL_IF_DEBUG("BSA_ROBSON_EXIT_TSI_EVT");
        break;
    case BSA_ROBSON_SET_BURST_EVT:
        evt = CALLBACK_TYPE_EXTENSION_SET_BURST;
        BTHAL_IF_DEBUG("BSA_ROBSON_SET_BURST_EVT");
        break;
    case BSA_ROBSON_EXIT_BURST_EVT:
        evt = CALLBACK_TYPE_EXTENSION_EXIT_BURST;
        BTHAL_IF_DEBUG("BSA_ROBSON_EXIT_BURST_EVT");
        break;
    case BSA_ROBSON_SET_CONFIG_EVT:
        BTHAL_IF_DEBUG("BSA_ROBSON_SET_CONFIG_EVT");
        if (p_data->set_config.status != BSA_SUCCESS) {
            NN_SDK_LOG("[bluetooth] BSA_ROBSON_SET_CONFIG_EVT had error status: config_mask=0x%x status=%d\n",
                p_data->set_config.config_mask, p_data->set_config.status);
        }
        callback=false;

        if (p_data->set_config.config_mask == BSA_ROBSON_CONFIG_SET_MC_MASK) {
            const char* mc2StateNames[] = { "MC2_DISABLE", "MC2_ENABLE", "MC2_LATCHING", "MC2_LATCHED" };
            const char* pMc2StateName = "MC2_UNKNOWN";
            if ((int)p_data->set_config.mc2_state < sizeof(mc2StateNames) / sizeof(mc2StateNames[0])) {
                pMc2StateName = mc2StateNames[p_data->set_config.mc2_state];
            }
            NN_SDK_LOG("[bluetooth] %s MC2 state: %s\n", __func__, pMc2StateName);
        }

        /* Only signal for callbacks we wait for */
        if (p_data->set_config.config_mask == expectedSetConfigMask)
        {
            /* Store the set config msg and signal that the callback occured */
            g_SetConfigMsg = p_data->set_config;
            nn::os::SignalEvent(&g_SetConfigEvent);
        }
        else if (p_data->set_config.config_mask != BSA_ROBSON_CONFIG_SET_MC_MASK)
        {
            NN_SDK_LOG("[bluetooth] %s UNEXPECTED MASK %x %x\n", __func__,
                    p_data->set_config.config_mask, expectedSetConfigMask);
        }

        break;
    case BSA_ROBSON_SET_0RETRAN_MODE_EVT:
        evt = CALLBACK_TYPE_EXTENSION_SET_ZERO_RETRAN;
        BTHAL_IF_DEBUG("BSA_ROBSON_SET_0RETRAN_MODE_EVT");
        break;
    default:
        NN_SDK_LOG("[bluetooth] %s unknown event = %d \n", __func__, event);
        callback=false;
        break;
    }

    if (callback){
        BluetoothHalInsertToQueue(p_data,
                                  sizeof(tBSA_ROBSON_MSG),
                                  HAL_QUEUE_CB,
                                  evt);
    }
}

/*******************************************************************************
 **
 ** Function        BtHalSetTsi
 **
 ** Description     Set TSI
 **
 ** Parameters      pBdAddr  - Bluetooth device address of remote device
 **
 **                 tsi_mode - TSI mode
 **
 ** Returns         BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetTsi(Btbdaddr *pBdAddr,uint8_t tsi_mode)
{
    tBSA_ROBSON_SET_TSI tsi_config;

    if (bonding_connect)
    {
        NN_SDK_LOG("[bluetooth] %s: ERROR cannot set TSI while bonding\n",__func__);
        return BTHH_HS_HID_NOT_READY;
    }

    /* Set TSI mode configuration */
    BSA_RobsonSetTsiModeInit(&tsi_config);

    memcpy(tsi_config.peer_addr, pBdAddr->address, sizeof(BD_ADDR));
    tsi_config.tsi_mode = static_cast<tBSA_ROBSON_TSI_MODE>(tsi_mode);

    tBSA_STATUS status = BSA_RobsonSetTsiMode(&tsi_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set tsi mode, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalExitTsi
 **
 ** Description      This function exits tsi mode
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalExitTsi(Btbdaddr *pBdAddr)
{
    tBSA_ROBSON_EXIT_TSI tsi_config;

    if (bonding_connect)
    {
        NN_SDK_LOG("[bluetooth] %s: ERROR cannot exit TSI while bonding\n",__func__);
        return BTHH_HS_HID_NOT_READY;
    }

    /* Exit TSI mode configuration */
    BSA_RobsonExitTsiModeInit(&tsi_config);

    memcpy(tsi_config.peer_addr, pBdAddr->address, sizeof(BD_ADDR));

    tBSA_STATUS status = BSA_RobsonExitTsiMode(&tsi_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to exit tsi mode, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetBurstMode
 **
 ** Description      This function sets burst mode
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetBurstMode(Btbdaddr *pBdAddr)
{
    tBSA_ROBSON_SET_BURST burst_config;

    if (bonding_connect)
    {
        NN_SDK_LOG("[bluetooth] %s ERROR cannot set burst mode while bonding\n",__func__);
        return BTHH_HS_HID_NOT_READY;
    }

    /* Set Burst mode configuration */
    BSA_RobsonSetBurstModeInit(&burst_config);

    memcpy(burst_config.peer_addr, pBdAddr->address, sizeof(BD_ADDR));

    tBSA_STATUS status = BSA_RobsonSetBurstMode(&burst_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set burst mode, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalExitBurstMode
 **
 ** Description      This function exits burst mode
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalExitBurstMode(Btbdaddr *pBdAddr)
{
    tBSA_ROBSON_EXIT_BURST burst_config;

    if (bonding_connect)
    {
        NN_SDK_LOG("[bluetooth] %s: ERROR cannot exit burst mode while bonding\n",__func__);
        return BTHH_HS_HID_NOT_READY;
    }

    /* Exit Burst mode configuration */
    BSA_RobsonExitBurstModeInit(&burst_config);

    memcpy(burst_config.peer_addr, pBdAddr->address, sizeof(BD_ADDR));

    tBSA_STATUS status = BSA_RobsonExitBurstMode(&burst_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to exit burst mode, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetAdapterProperties
 **
 ** Description      This function gets the adapter properties
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalGetAdapterProperties(AdapterProperty *pAdapterProperty)
{
    tBSA_DM_GET_CONFIG  bt_config;

    /* Get Bluetooth configuration */
    BSA_DmGetConfigInit(&bt_config);
    tBSA_STATUS status = BSA_DmGetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get bt_config:%d\n", __func__, status);
    }
    else
    {
        int len = 0;

        /* Get the length of the received name */
        while(bt_config.name[len] && (len < sizeof(BD_NAME)))
        {
            len++;
        }

        /* Copy the properties we care about */
        memcpy(pAdapterProperty->bluetoothAddress.address, bt_config.bd_addr, sizeof(BluetoothAddress));
        memcpy(pAdapterProperty->bluetoothName.name, bt_config.name, len);
        memcpy(pAdapterProperty->classOfDevice.cod, bt_config.class_of_device, sizeof(ClassOfDevice));

        /* NCL wants this but it isn't a BSA configuration, hard coded for now */
        pAdapterProperty->featureSet = 0x68;
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetHostBdAddr
 **
 ** Description      This function gets the host BD_ADDR
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalGetHostBdAddr(Btbdaddr *pBdAddr)
{
    tBSA_DM_GET_CONFIG  bt_config;

    /* Get Bluetooth configuration */
    BSA_DmGetConfigInit(&bt_config);
    tBSA_STATUS status = BSA_DmGetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get bt_config:%d\n", __func__, status);
    }
    else
    {
        memcpy(pBdAddr->address, bt_config.bd_addr, sizeof(Btbdaddr));
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetHostBdAddr
 **
 ** Description      This function sets the host BD_ADDR
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetHostBdAddr(const Btbdaddr *pBdAddr)
{
    tBSA_DM_SET_CONFIG  bt_config;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    bt_config.config_mask = BSA_DM_CONFIG_BDADDR_MASK;
    memcpy(bt_config.bd_addr, pBdAddr->address, sizeof(BD_ADDR));

    tBSA_STATUS status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to change BD_ADDR:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetHostBdName
 **
 ** Description      This function gets the host BD_NAME
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalGetHostBdName(Btbdname *pBdName, uint8_t *length)
{
    tBSA_DM_GET_CONFIG  bt_config;

    /* Get Bluetooth configuration */
    BSA_DmGetConfigInit(&bt_config);
    tBSA_STATUS status = BSA_DmGetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get bt_config:%d\n", __func__, status);
    }
    else
    {
        int len = 0;

        /* Get the length of the received name */
        while(bt_config.name[len] && (len < sizeof(BD_NAME)))
        {
            len++;
        }

        memcpy(pBdName->name, bt_config.name, len);

        *length = len;
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetHostBdName
 **
 ** Description      This function sets the host BD_NAME
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetHostBdName(const Btbdname *pBdName, uint8_t length)
{
    tBSA_DM_SET_CONFIG  bt_config;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    /* BSA_DmSetConfigInit sets a default name, clear it */
    memset(&bt_config.name, 0, sizeof(BD_NAME));
    bt_config.config_mask = BSA_DM_CONFIG_NAME_MASK;
    memcpy(bt_config.name, pBdName->name, length);

    tBSA_STATUS status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to change BD_NAME:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetHostClassOfDevice
 **
 ** Description      This function gets the host DEV_CLASS
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalGetHostClassOfDevice(ClassOfDevice *pBdClass)
{
    tBSA_DM_GET_CONFIG  bt_config;

    /* Get Bluetooth configuration */
    BSA_DmGetConfigInit(&bt_config);
    tBSA_STATUS status = BSA_DmGetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get bt_config:%d\n", __func__, status);
    }
    else
    {
        memcpy(pBdClass->cod, bt_config.class_of_device, sizeof(DEV_CLASS));
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetHostClassOfDevice
 **
 ** Description      This function sets the host DEV_CLASS
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetHostClassOfDevice(const ClassOfDevice *pBdClass)
{
    tBSA_DM_SET_CONFIG  bt_config;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    bt_config.config_mask = BSA_DM_CONFIG_DEV_CLASS_MASK;
    memcpy(bt_config.class_of_device, pBdClass->cod, sizeof(DEV_CLASS));

    tBSA_STATUS status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to change DEV_CLASS:%d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetHostFeatureSet
 **
 ** Description      This function gets the host feature set
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalGetHostFeatureSet(uint8_t *pfeatureSet)
{
    *pfeatureSet = featureSet;

    return BTHH_OK;
}

/*******************************************************************************
 **
 ** Function         BtHalSetHostFeatureSet
 **
 ** Description      This function sets the host feature set
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetHostFeatureSet(const uint8_t *pfeatureSet)
{
    featureSet = *pfeatureSet;

    return BTHH_OK;
}
/*******************************************************************************
 **
 ** Function         BtHalSetHostSuperBdAddr
 **
 ** Description      This function sets the host BD_ADDR
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetHostSuperBdAddr(const Btbdaddr *pBdAddr)
{
    tBSA_ROBSON_SET_CONFIG set_config;

    /* Set the super piconet bd address */
    BSA_RobsonSetConfigInit(&set_config);

    set_config.config_mask = BSA_ROBSON_CONFIG_SET_SECONDARY_BD_ADDR_MASK;
    memcpy(set_config.second_addr, pBdAddr->address, sizeof(BD_ADDR));

    tBSA_STATUS status = BtHalSetConfig(set_config, 500);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set super piconet: %d\n", __func__, status);
    }

    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalGetNumPendingConnections
 **
 *******************************************************************************/
BluetoothHhStatus BtHalGetNumPendingConnections()
{
    tBSA_DM_GET_CONFIG  bt_config;
    InfoFromExtensionCallbacks cbData;

    /* Get Bluetooth configuration */
    tBSA_STATUS status = BSA_DmGetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to get bt_config:%d\n", __func__, status);
        bt_config.num_pending_connection = 0;
    }

    /* Insert a callback into the HAL queue to synchronize with
       possible connection events */
    std::memset(cbData.bluetoothAddress.address, 0, BD_ADDR_LEN);
    cbData.eventType = EventFromGetPendingConnectionsCallback;
    cbData.status = BtHalBsaConvertResult(status);
    cbData.connections.numPending = bt_config.num_pending_connection;

    BluetoothHalInsertToQueue(&cbData,
                              sizeof(InfoFromExtensionCallbacks),
                              HAL_QUEUE_CB,
                              CALLBACK_TYPE_EXTENSION_GET_PENDING_CONNECTIONS);

    return BtHalBsaConvertResult(status);
}


/*******************************************************************************
 **
 ** Function         BtHalSetSfh
 **
 ** Description      This function configure the static AFH channels to controller for MC2 use case
 **
 **                        MC2 use case :
 **                        The Robson#1 should receive the VSE_WL_STROBE_SYNC_ALIGNED_EVENT VSE to align with WiFi strobe and also receive the
 **                        VSE about 16 good channels to configure static AFH to hop.
 **                        Robson#1 should dispatch the information(channels information and also offset) to Robson#2-#4 thru WiFi, then
 **                        host can configure Static AFH with these information to local controller.
 **                        Also Robson#2-4 also needs to align with WiFi strobe first to get this mechanism to work.
 **                        Frequency offset: (0/4/8/12)
 **
 **
 ** Returns          status
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetSfh(uint8_t sfh_on, uint8_t *pChannels,uint32_t length, uint32_t sfh_offset)
{
    tBSA_ROBSON_SET_CONFIG set_config;

#if 0
    if (!app_robson_cb.wl_aligned)
    {
        int ans=0;
        NN_SDK_LOG("[bluetooth] Bluetooth is not in alignment with WiFi strobe ! Enabling static AFH may not work with other Robsons!!\n");
        // FIXME:
    }
#endif

    if (sfh_on)
    {
        if (!(  (sfh_offset == 0)|| (sfh_offset == 4) || (sfh_offset == 8) || (sfh_offset == 12) ))
        {
            NN_SDK_LOG("[bluetooth] Wrong frequency offset (must be 0/4/8/12), exit...\n");
            return BTHH_ERR;
        }
#if 0
        if (pChannels[9]> 0x7F)
        {
            NN_SDK_LOG("[bluetooth] Wrong input channel information, exit...\n");
            return BTHH_ERR;
        }
#endif
    }

    /* Set static AFH configuration */
    BSA_RobsonSetConfigInit(&set_config);
    set_config.config_mask = BSA_ROBSON_CONFIG_STATIC_AFH_MASK;
    set_config.sfh_enable = (sfh_on != 0);
    set_config.sfh_offset = sfh_offset;
    memcpy(set_config.sfh_channels, pChannels, BTA_AFH_CHNL_MAP_SIZE);

    tBSA_STATUS status = BtHalSetConfig(set_config, 250);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set static AFH, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function    BtHalStartLlrMode
 **
 ** Description Start LLR mode by setting the GPIO and calling the BSA function.
 **
 ** Parameters
 **
 ** Returns     BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalStartLlrMode()
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);

    uint8_t num_llr_scan = 0;
    for (int i = 0; i < sizeof(app_xml_remote_devices_db) / sizeof(app_xml_remote_devices_db[0]); i++)
    {
        if ((app_xml_remote_devices_db[i].in_use != FALSE) &&
                (app_xml_remote_devices_db[i].link_key_present))
        {
            memcpy(set_config.llr_scan_list_bd_addr[num_llr_scan],  app_xml_remote_devices_db[i].bd_addr, BD_ADDR_LEN);
            memcpy(set_config.llr_scan_list_link_key[num_llr_scan], app_xml_remote_devices_db[i].link_key, LINK_KEY_LEN);
            num_llr_scan++;
        }
    }

    set_config.num_llr_scan = num_llr_scan;
    set_config.config_mask = BSA_ROBSON_CONFIG_LLR_START_MASK;
    tBSA_STATUS status = BtHalSetConfig(set_config, 2000);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to start llr scan, status: %d\n", __func__, status);
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: LLR scan started for %d devices\n", __func__, num_llr_scan);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function    BtHalExitLlrMode
 **
 ** Description Exit LLR mode by setting the GPIO and calling the BSA function.
 **
 ** Parameters
 **
 ** Returns     BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalExitLlrMode()
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);

    int8_t num_llr_scan = -1;

    set_config.num_llr_scan = num_llr_scan;
    set_config.config_mask = BSA_ROBSON_CONFIG_LLR_START_MASK;
    tBSA_STATUS status = BtHalSetConfig(set_config, 250);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to stop llr scan, status: %d\n", __func__, status);
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: LLR scan stopped\n", __func__);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetMcMode
 **
 ** Description      This function sets the MC mode
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetMcMode(bool isMcModeEnabled)
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);
    set_config.config_mask = BSA_ROBSON_CONFIG_SET_MC_MASK;
    set_config.mc_enable = isMcModeEnabled;

    tBSA_STATUS status = BtHalSetConfig(set_config, 500);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set mc mode: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}
/*******************************************************************************
 **
 ** Function         BtHalSetZeroRetran
 **
 ** Description      This function sets 0-re-transmission
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus BtHalSetZeroRetran(Btbdaddr *pBdAddr, uint8_t numIds, const uint8_t *pIds)
{
    tBSA_ROBSON_SET_0RETRAN_MODE zero_retran_mode;

    BSA_RobsonSet0RetranModeInit(&zero_retran_mode);

    if (numIds > MAX_NUM_ZERORETRAN_REPORTID)
    {
        NN_SDK_LOG("[bluetooth] %s: error - too many IDs (%d), truncating\n", __func__, numIds);
        numIds = MAX_NUM_ZERORETRAN_REPORTID;
    }

    if (numIds > 0)
    {
        zero_retran_mode.enable = 1;
        zero_retran_mode.num = numIds;
        memcpy(zero_retran_mode.peer_addr, pBdAddr->address, sizeof(BD_ADDR));
        memcpy(zero_retran_mode.reportID, pIds, numIds);
    }
    else
    {
        zero_retran_mode.enable = 0;
        memcpy(zero_retran_mode.peer_addr, pBdAddr->address, sizeof(BD_ADDR));
    }

    tBSA_STATUS status = BSA_RobsonSet0RetranMode(&zero_retran_mode);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set zero re-tran mode, status: %d\n", __func__, status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetRadio
 **
 ** Description      This function will turn the radio on/off (airplane mode if off)
 **
 ** Parameters       enable: FALSE for off, TRUE for on
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetRadio(bool enable)
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);
    set_config.config_mask = BSA_ROBSON_CONFIG_SET_RADIO_MASK;
    set_config.radio_on = enable;

    tBSA_STATUS status = BtHalSetConfig(set_config, 2000);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to Turn Radio to %s, status: %d\n", __func__, (enable ? "ON" : "OFF"), status);
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: Turn Radio %s Successfully\n", __func__, (enable ? "ON" : "OFF"));
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetVisibility
 **
 ** Description      Set the Device Visibility parameters (InquiryScan & PageScan)
 **
 ** Parameters       discoverable: FALSE if not discoverable
 **                  connectable: FALSE if not connectable
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetVisibility(bool discoverable, bool connectable)
{
    tBSA_DM_SET_CONFIG bt_config;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    bt_config.enable = TRUE;

    /* Set visibility configuration */
    bt_config.config_mask = BSA_DM_CONFIG_VISIBILITY_MASK;
    bt_config.connectable = connectable;
    bt_config.discoverable = discoverable;

    tBSA_STATUS status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: BSA_DmSetConfig failed status: %d\n", __func__,  status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSetTbfc
 **
 ** Description      Set TBFC scan enabled/disabled
 **
 ** Parameters       enable
 **
 ** Returns          BluetoothHhStatus
 **
 *******************************************************************************/
BluetoothHhStatus  BtHalSetTbfcScan(bool enable)
{
    tBSA_DM_SET_CONFIG bt_config;

    /* Set Bluetooth configuration */
    BSA_DmSetConfigInit(&bt_config);

    bt_config.enable = TRUE;

    /* Set visibility configuration */
    bt_config.config_mask = BSA_DM_CONFIG_TBFC_SCAN_MASK;
    bt_config.tbfc_scan_enable = enable;

    tBSA_STATUS status = BSA_DmSetConfig(&bt_config);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: BSA_DmSetConfig failed status: %d\n", __func__,  status);
    }
    return BtHalBsaConvertResult(status);
}

/*******************************************************************************
 **
 ** Function         BtHalSet10DbBoost
 **
 ** Description      Set the power level to 10dB
 **
 ** Parameters       none
 **
 ** Returns          status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int BtHalSet10DbBoost()
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);
    set_config.config_mask = BSA_ROBSON_CONFIG_SET_10DB_MASK;

    tBSA_STATUS status = BtHalSetConfig(set_config, 3000);
    if (status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to set 10dB boost, status: %d\n", __func__, status);
        return -1;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: Set 10dB boost Successfully\n", __func__);
    }

    return 0;
}

/*******************************************************************************
 **
 ** Function         BtHalDisableAfh
 **
 ** Description      Disable Bluetooth AFH
 **
 ** Paramters        None
 **
 ** Returns          status: 0 if success / -1 otherwise
 **
 *******************************************************************************/
int BtHalDisableAfh()
{
    tBSA_ROBSON_SET_CONFIG set_config;
    BSA_RobsonSetConfigInit(&set_config);
    set_config.config_mask = BSA_ROBSON_CONFIG_SET_AFH_MASK;
    set_config.afh_enable = false;

    tBSA_STATUS status = BtHalSetConfig(set_config, 250);
    if(status != BSA_SUCCESS)
    {
        NN_SDK_LOG("[bluetooth] %s: Unable to disable bluetooth AFH\n", __func__);
        return -1;
    }
    else
    {
        NN_SDK_LOG("[bluetooth] %s: Disable bluetooth AFH Successfully\n", __func__);
    }

    return 0;
}

/*******************************************************************************
 **
 ** Function         BtHalSetConfig
 **
 ** Description      Sets robson config, waits for cb event, clears event & mutex
 **
 ** Parameters       set_config: configuration to be set
 **                  timeOutMs: wait timeout in milliseconds
 **
 ** Returns          tBSA_STATUS
 **
 *******************************************************************************/
static tBSA_STATUS BtHalSetConfig(tBSA_ROBSON_SET_CONFIG set_config, int timeOutMs)
{
    nn::os::LockMutex(&g_SetConfigMutex);

    expectedSetConfigMask = set_config.config_mask;

    tBSA_STATUS status = BSA_RobsonSetConfig(&set_config);
    if (status == BSA_SUCCESS)
    {
        auto timeout = nn::os::TimedWaitEvent(&g_SetConfigEvent, nn::TimeSpan::FromMilliSeconds(timeOutMs));
        if(timeout == FALSE)
        {
            NN_SDK_LOG("[bluetooth] BSA_RobsonSetConfig callback event timed out\n");
            status = BSA_ERROR_CLI_BAD_MSG;
        }
        else
        {
            // Got the callback, check if it matches
            if(g_SetConfigMsg.config_mask != set_config.config_mask)
            {
                NN_SDK_LOG("[bluetooth] BSA_RobsonSetConfig error. mask %d did not match callback %d\n", set_config.config_mask, g_SetConfigMsg.config_mask);
            }

            // Return status received from callback
            status = g_SetConfigMsg.status;
        }
    }
    else
    {
        NN_SDK_LOG("[bluetooth] BSA_RobsonSetConfig failed, status: %d\n", status);
    }

    expectedSetConfigMask = 0;

    nn::os::UnlockMutex(&g_SetConfigMutex);

    return status;
}

/*******************************************************************************
 **
 ** Function         BtHalInitialize
 **
 ** Description      Initializes an event and mutex for BtHalSetConfig
 **
 ** Parameters       none
 **
 ** Returns          nothing
 **
 *******************************************************************************/
void BtHalInitialize()
{
    nn::os::InitializeEvent(&g_SetConfigEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeMutex(&g_SetConfigMutex, false, 0);
}

/*******************************************************************************
 **
 ** Function         BtHalFinalize
 **
 ** Description      Finalizes an event and mutex for BtHalSetConfig
 **
 ** Parameters       none
 **
 ** Returns          nothing
 **
 *******************************************************************************/
void BtHalFinalize()
{
    nn::os::FinalizeMutex(&g_SetConfigMutex);
    nn::os::FinalizeEvent(&g_SetConfigEvent);
}
