﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <cstdio>
#include <cstring>
#include "bluetooth.h"
#include "bluetooth_queue.h"
#include <nn/psc.h>

//Includes for the IPC
#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/bluetooth/bluetooth_IBluetoothDriver.sfdl.h>
#include <nn/bluetooth/bluetooth_Driver.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_ServiceName.h>
#include "../../Libraries/bluetooth/bluetooth_DriverImpl.h"

//Functions for starting IPC server.
namespace {
struct MyServerManagerOption
{
    static const size_t PointerTransferBufferSize = 4096;
};
//-----------------------------------------------------------------------------
// nn::sf::HipcSimpleAllInOneServerManager は HIPC のサーバのポートと
// セッションを一元管理する。引数は最大セッション数と最大ポート数。
//
class BluetoothServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 2, MyServerManagerOption>
{
};

// 繰返しのサーバ起動／終了が可能となるように placement new で初期化を行う。
// 繰返しの起動と終了が必要ない場合には BluetoothServerManager は直接配置してもよい。
std::aligned_storage<sizeof(BluetoothServerManager), NN_ALIGNOF(BluetoothServerManager)>::type g_BluetoothServerManagerStorage;

//-----------------------------------------------------------------------------
// サーバ実装用のサービスオブジェクトへの共有ポインタ
BluetoothServerManager*  g_pBluetoothServerManager;

nn::psc::PmModule g_PmModule; //pscからの通知を受け取るオブジェクト

//-----------------------------------------------------------------------------
//  Bluetoothサーバーを初期化する
//
void InitializeBluetoothDriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pBluetoothServerManager);

    // BluetoothServerManager オブジェクトのコンストラクト（placement new）
    g_pBluetoothServerManager = new (&g_BluetoothServerManagerStorage) BluetoothServerManager;

    // サービス名の登録とポートの初期化
    auto sessionCountMax = 30;
    typedef nn::sf::ObjectFactory<nn::sf::impl::StaticOneAllocationPolicy>  Factory;
    g_pBluetoothServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::bluetooth::IBluetoothDriver, nn::bluetooth::BluetoothDriverImpl>(), sessionCountMax, nn::bluetooth::BluetoothDriverServiceName);
    g_pBluetoothServerManager->RegisterObjectForPort(Factory::CreateSharedEmplaced<nn::bluetooth::IBluetoothUser, nn::bluetooth::BluetoothUserImpl>(), sessionCountMax, nn::bluetooth::BluetoothUserServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pBluetoothServerManager->Start();

    //pscからの通知を受けとるオブジェクトの初期化。pscへの登録
    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Uart, nn::psc::PmModuleId_Gpio};
    auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Bluetooth,
            dependencies,
            sizeof(dependencies) / sizeof(dependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}


//-----------------------------------------------------------------------------
//  Bluetoothサーバーを終了する
//
void FinalizeBluetoothDriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBluetoothServerManager);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pBluetoothServerManager->~BluetoothServerManager();
    g_pBluetoothServerManager = nullptr;
}


//-----------------------------------------------------------------------------
//  Bluetoothのサーバメッセージ処理をループで行なう
//
void LoopBluetoothDriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBluetoothServerManager);
    //g_pBluetoothServerManager->LoopAuto();

    nn::os::MultiWaitHolderType pmModuleEventHolder;

    //g_PmModuleのMultiWaitHolderを作成し、SFでのWait対象に追加
    nn::os::InitializeMultiWaitHolder(
             &pmModuleEventHolder,
             g_PmModule.GetEventPointer()->GetBase());

    // use module PM Id as a tag for event processing
    nn::os::SetMultiWaitHolderUserData(
             &pmModuleEventHolder,
             nn::psc::PmModuleId_Bluetooth);

    g_pBluetoothServerManager->AddUserWaitHolder(&pmModuleEventHolder);

    //LoopAutoの内容を実装
    while (NN_STATIC_CONDITION(true))
    {
         nn::os::MultiWaitHolderType* p = g_pBluetoothServerManager->Wait();

         switch (nn::os::GetMultiWaitHolderUserData(p))
         {
         // SF processing
         case BluetoothServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
         case BluetoothServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
              g_pBluetoothServerManager->ProcessAuto(p);
              break;

         // PM event processing for FGM module.
         // Similarly handle events for other modules from this thread, if needed.
         case nn::psc::PmModuleId_Bluetooth:
         {
             bool tryWaitResult = g_PmModule.GetEventPointer()->TryWait();
             NN_SDK_ASSERT(tryWaitResult);
             NN_UNUSED(tryWaitResult);
             g_PmModule.GetEventPointer()->Clear();
             g_pBluetoothServerManager->AddUserWaitHolder(&pmModuleEventHolder);

             nn::psc::PmState    state;
             nn::psc::PmFlagSet  flags;
             auto result = g_PmModule.GetRequest(&state, &flags);
             NN_ABORT_UNLESS_RESULT_SUCCESS(result);

             switch (state)
             {
             // PM modules are expected to handle at least
             // PmState_FullAwake
             // PmState_MinimumAwake
             // PmState_SleepReady
             case nn::psc::PmState_FullAwake:
                 NN_SDK_LOG("[bluetooth] Got FullAwake Notification\n");
                 //BtHalExitLlrMode();
                 break;
             case nn::psc::PmState_MinimumAwake:
                 NN_SDK_LOG("[bluetooth] Got MinimumAwake Notification\n");
                 break;
             case nn::psc::PmState_SleepReady:
                 NN_SDK_LOG("[bluetooth] Got SleepReady Notification\n");
                 //BtHalStartLlrMode();
                 break;
             default:
                 NN_SDK_LOG("[bluetooth] Got Other Notification from psc\n");
                 break;
             }
             g_PmModule.Acknowledge(state, nn::ResultSuccess());
             break;
         }
        default:
             NN_UNEXPECTED_DEFAULT;
        }
    }
}


//-----------------------------------------------------------------------------
//  Bluetoothのサーバメッセージ処理ループを停止する要求を送出する
//
void RequestStopBluetoothDriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pBluetoothServerManager);
    g_pBluetoothServerManager->RequestStop();
}

} // namespace

extern "C" void nninitStartup()
{
    // no malloc!
}

extern "C" int64_t timeSinceBootInMs()
{
    return nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
}

extern "C" int64_t timeSinceBootInNs()
{
    return nn::os::GetSystemTick().ToTimeSpan().GetNanoSeconds();
}

extern "C" int64_t timeSinceBootInSec()
{
    return nn::os::GetSystemTick().ToTimeSpan().GetSeconds();
}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    //NN_SDK_LOG("bluetooth nnMain() SF\n");
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(bluetooth, Main));

    NN_UNUSED( RequestStopBluetoothDriverServer );
    // Start Bluetooth service
    InitializeBluetoothDriverServer();

    {
        NN_SDK_LOG("[bluetooth] Server process is running.\n");
        // This function continues until invoking StopDevice() method.
        LoopBluetoothDriverServer();
        NN_SDK_LOG("[bluetooth] Server process is finished.\n");
    }

    // Stop Bluetooth service
    FinalizeBluetoothDriverServer();
}
