﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/lbl/lbl.h>
#include <nn/psm/psm.h>
#include <nn/apm/apm_Lib.h>
#include <nn/apm/apm_SystemLib.h>
#include <nn/btm/btm_Api.h>
#include <nn/tc/tc_Lib.h>
#include <nn/tcap/tcap_Lib.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <nn/cec/cec_Api.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/hdcp/hdcp.h>
#include <nn/omm/detail/omm_PortName.h>
#include <nn/omm/srv/omm_OperationModeManager.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "am_OperationModeManagerMain.h"

namespace nn { namespace am {

namespace {

    os::SdkMutex g_StartupLogoMutex;
    bool g_IsStartupLogoShown;

    class OperationModeManagerServerManager : public nn::sf::HipcSimpleAllInOneServerManager<10, 1>{};

    sf::UnmanagedServiceObject<omm::detail::IOperationModeManager, omm::srv::OperationModeManager> g_OperationModeManager;

    OperationModeManagerServerManager g_ServerManager;
    NN_OS_ALIGNAS_THREAD_STACK char g_ServerManagerThreadStack[16 * 1024];
    os::ThreadType g_ServerManagerThread;

    void ServerManagerThreadFunction(void*)
    {
        psm::Initialize();
        NN_UTIL_SCOPE_EXIT{ psm::Finalize(); };

        apm::Initialize();
        NN_UTIL_SCOPE_EXIT{ apm::Finalize(); };

        apm::InitializeForSystem();
        NN_UTIL_SCOPE_EXIT{ apm::FinalizeForSystem(); };

        tc::Initialize();
        NN_UTIL_SCOPE_EXIT{ tc::Finalize(); };

        tcap::Initialize();
        NN_UTIL_SCOPE_EXIT{ tcap::Finalize(); };

        spsm::Initialize();
        NN_UTIL_SCOPE_EXIT{ spsm::Finalize(); };

        btm::InitializeBtmInterface();
        NN_UTIL_SCOPE_EXIT{ btm::FinalizeBtmInterface(); };

        g_OperationModeManager.GetImpl().Initialize();
        g_ServerManager.LoopAuto();
    }
}

void InitializeOperationModeManagerForShowStartupLogo() NN_NOEXCEPT
{
    lbl::Initialize();
    usb::pd::Initialize();
    usb::pd::InitializeCradle();

    // hdcp 初期化 (cec の serviceobject が必要)
    nn::sf::SharedPointer<nn::hdcp::detail::IHdcpController> pHdcpService;
    cec::GetHdcpServiceObject(&pHdcpService);
    hdcp::Initialize(pHdcpService);

    g_OperationModeManager.GetImpl().InitializeForShowStartupLogo();
}

void RegisterOperationModeManager() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_OperationModeManager.GetShared(), 10, nn::omm::detail::PortNameForOperationModeManager));
}

void StartOperationModeManagerMain() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_ServerManagerThread, ServerManagerThreadFunction, nullptr, g_ServerManagerThreadStack, sizeof(g_ServerManagerThreadStack), NN_SYSTEM_THREAD_PRIORITY(omm, IpcServer)));
    nn::os::SetThreadNamePointer(&g_ServerManagerThread, NN_SYSTEM_THREAD_NAME(omm, IpcServer));
    nn::os::StartThread(&g_ServerManagerThread);
}

void ShowStartupLogoOfOperationModeManager() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_StartupLogoMutex)> lock(g_StartupLogoMutex);
    NN_SDK_ASSERT(!g_IsStartupLogoShown);
    g_IsStartupLogoShown = true;

    // ディスプレイをシステムモードに切替
    am::service::ChangeDisplayLayerControlModeToSystem();
    // NX ロゴのフェードイン
    g_OperationModeManager.GetImpl().ShowStartupLogo();
}

void HideStartupLogoOfOperationModeManager() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_StartupLogoMutex)> lock(g_StartupLogoMutex);
    if (!g_IsStartupLogoShown)
    {
        return;
    }
    g_IsStartupLogoShown = false;

    // NX ロゴのフェードアウト
    g_OperationModeManager.GetImpl().HideStartupLogo();
    // ディスプレイをアプレットモードに切替
    am::service::ChangeDisplayLayerControlModeToApplet();
}

}}
