﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_AccountDaemon.h"
#include "account_AccountServer.h"
#include "account_DauthServer.h"

#include <algorithm>
#include <new>
#include <curl/curl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/bgtc/bgtc_Api.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/dauth/dauth_ApiPrivate.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/spl/spl_Api.h>
#include <nn/time/time_Api.h>
#include <nn/util/util_ScopeExit.h>

// 例外関連コードのリンク抑制のためのオーバーライド
// 参考：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-34162
void* operator new(std::size_t size) { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT { free(ptr); }

void* operator new[](std::size_t size) { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT{ return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT{ free(ptr); }

namespace
{
#define HEAP_INITIALIZER_INITIALIZER {false}
struct HeapInitializer
{
    bool _initialized;
    nn::lmem::HeapHandle _handle;
    char _buffer[128 * 1024];

    void Initialize() NN_NOEXCEPT
    {
        _handle = nn::lmem::CreateExpHeap(_buffer, sizeof(_buffer), nn::lmem::CreationOption_NoOption);
        _initialized = true;
    }
    nn::lmem::HeapHandle GetHandle() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(_initialized);
        return _handle;
    }
} g_HeapInitializer = HEAP_INITIALIZER_INITIALIZER;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapInitializer.GetHandle(), size);
}
void Deallocate(void* ptr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapInitializer.GetHandle(), ptr);
}

typedef std::aligned_storage<sizeof(nn::DauthResourceType), NN_ALIGNOF(nn::DauthResourceType)>::type DauthResourceStorageType;
DauthResourceStorageType g_DauthResourceStorage;

typedef std::aligned_storage<sizeof(nn::AccountDaemon), NN_ALIGNOF(nn::AccountDaemon)>::type AccountDaemonStorageType;
AccountDaemonStorageType g_AccountDaemonStorage;
typedef std::aligned_storage<sizeof(nn::AccountResourceType), NN_ALIGNOF(nn::AccountResourceType)>::type AccountResourceStorageType;
AccountResourceStorageType g_AccountResourceStorage;

// libcurl が使用する malloc/free 用のバッファ
NN_ALIGNAS(4096) char g_BufferForStandardAllocator[1 * 1024 * 1024];

// Socket が要求するメモリプール
const int SocketCount = 3;
const int SocketConcurrencyCount = SocketCount;
nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> g_SocketConfigWithMemory(SocketConcurrencyCount);

// AccountDaemon 用のスレッド資源
const size_t AccoutDaemonThreadStackSize = 4 * 4096;
NN_ALIGNAS(4096) char g_AccountDaemonThreadStack[AccoutDaemonThreadStackSize];
nn::os::ThreadType g_AccountDaemonThread;

void MainLoop(nn::AccountDaemon& daemon) NN_NOEXCEPT
{
    const nn::psc::PmModuleId PmDependencies[] = {nn::psc::PmModuleId_Bgtc};
    nn::psc::PmModule module;
    NN_ABORT_UNLESS_RESULT_SUCCESS(module.Initialize(
        nn::psc::PmModuleId_Account,
        PmDependencies, static_cast<uint32_t>(std::extent<decltype(PmDependencies)>::value),
        nn::os::EventClearMode_ManualClear));
    auto& ePmRequest = *module.GetEventPointer();

    auto& eBgtcTriggered = nn::bgtc::GetTriggerEvent();

    nn::psc::PmState currentPmState = nn::psc::PmState_FullAwake;
    while (NN_STATIC_CONDITION(true))
    {
        // bgtc による半起床通知は、 psc による状態遷移の拡張であるため同時待ちする
        auto const Index = nn::os::WaitAny(
            ePmRequest.GetBase(),
            eBgtcTriggered.GetBase());
        // イベントには優先度があるため index は使用しない
        NN_UNUSED(Index);

        // 半起床の検出 (最終的に psc に従うので先に処理する)
        if (eBgtcTriggered.TryWait())
        {
            eBgtcTriggered.Clear();
            if (nn::bgtc::IsInHalfAwake())
            {
                daemon.NotifyEnteredHalfAwake();
            }
        }

        // psc のリクエスト処理
        if (ePmRequest.TryWait())
        {
            ePmRequest.Clear();

            nn::psc::PmFlagSet flags;
            nn::psc::PmState request;
            NN_ABORT_UNLESS_RESULT_SUCCESS(module.GetRequest(&request, &flags));
            switch (request)
            {
            case nn::psc::PmState_FullAwake:
                daemon.OnEnteringFullAwake(currentPmState);
                break;
            case nn::psc::PmState_MinimumAwake:
                daemon.OnEnteringMinimumAwake(currentPmState);
                break;
            case nn::psc::PmState_SleepReady:
                daemon.OnEnteringSleepReady(currentPmState);
                if (currentPmState == nn::psc::PmState_MinimumAwake)
                {
                    // BGTC への半起床要求用
                    auto nextWakeup = daemon.GetNextWakeupRequestInUptime();
                    auto interval = std::max(
                        (nextWakeup - nn::os::GetSystemTick().ToTimeSpan()).GetSeconds(),
                        static_cast<int64_t>(0));
                    nn::bgtc::ScheduleTask(interval);
                    NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                        "Requested next wakeup after %02dh%02dm%02ds\n",
                        interval / 60 / 60, (interval / 60) % 60, interval % 60);
                }
                break;
            case nn::psc::PmState_ShutdownReady:
                daemon.OnEnteringShutdownReady(currentPmState);
                break;
            default:
                // nop
                break;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(module.Acknowledge(request, nn::ResultSuccess()));

            // postprocess
            currentPmState = request;
        }
    }
}
} // ~namespace <anonymous>

extern "C" void nninitStartup()
{
    // malloc, free の初期化
    nn::init::InitializeAllocator(g_BufferForStandardAllocator, sizeof(g_BufferForStandardAllocator));
}
extern "C" void nndiagStartup()
{
}
extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(account, Main));

    // 環境の初期化 ---------------------------------------------------------------------------------
    // fs サービスを複数セッションで初期化
    nn::fs::InitializeWithMultiSessionForSystem();

    // ヒープ
    g_HeapInitializer.Initialize();
    nn::fs::SetAllocator(Allocate, Deallocate);

    // 依存するサービスの初期化
    nn::spl::InitializeForCrypto();
    nn::es::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

    // サービスの初期化 ----------------------------------------------------------------------------
    auto pDauth = new(&g_DauthResourceStorage) nn::DauthResourceType;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::StartDauthServer(*pDauth));

    auto pSrc = new(&g_AccountResourceStorage) nn::AccountResourceType;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::StartAccountServer(*pSrc));

    // バックグラウンドデーモンの初期化 -------------------------------------------------------------
    auto pDaemon = new(&g_AccountDaemonStorage) nn::AccountDaemon;
    pSrc->RegisterBackgroundDaemon(nn::AccountDaemon::ControlRequestHandler, pDaemon);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &g_AccountDaemonThread,
        nn::AccountDaemon::ThreadFunction, pDaemon,
        g_AccountDaemonThreadStack, sizeof(g_AccountDaemonThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(account, EventHandler)));
    nn::os::SetThreadNamePointer(&g_AccountDaemonThread, NN_SYSTEM_THREAD_NAME(account, EventHandler));

    // デーモンの開始
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bgtc::Initialize());
    nn::dauth::InitializeWith(pDauth->GetSharedService());
    nn::account::InitializeWith(pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
    nn::os::StartThread(&g_AccountDaemonThread);

    // ループ  -----------------------------------------------------------------------------------
    MainLoop(*pDaemon);

    // 終了処理 -----------------------------------------------------------------------------------
#if 0
    module.Finalize();

    pDaemon->RequestStop();
    nn::os::WaitThread(&g_AccountDaemonThread);
    nn::os::DestroyThread(&g_AccountDaemonThread);
    nn::account::Finalize();

    pDaemon->~AccountDaemon();

    nn::StopAccountServer();
    pSrc->~ServiceResource();

    nn::StopDauthServer();
    pDauth->~DauthResourceType();

    curl_global_cleanup();
    nn::socket::Finalize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    nn::es::Finalize();
#endif
}
