﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_DauthServer.h"

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn {
namespace {
/* サービススレッドの資源 ----------------------------------------------------------------------------*/
NN_ALIGNAS(4096) char g_DauthServerThreadStack[DauthServerThreadStackSize];
os::ThreadType g_DauthServerThread;

DauthServer* g_pDauthServer;
std::aligned_storage<sizeof(DauthServer), NN_ALIGNOF(DauthServer)>::type g_ServerStroage;
void DauthServerThreadFunction(void* p) NN_NOEXCEPT
{
    auto& server = *reinterpret_cast<DauthServer*>(p);
    server.LoopAuto();
}

/* Executorスレッドの資源 ----------------------------------------------------------------------------*/
NN_ALIGNAS(4096) char g_DauthExecThreadStack[DauthExecThreadStackSize];
os::ThreadType g_DauthExecThread;
os::Event g_DauthExecThreadTerminator(os::EventClearMode_ManualClear);

struct ExecutionContext
{
    dauth::Executor* pExecutor;
    dauth::ExecutionResource resource;
    os::Event* pTerminator;
} g_ExecutionContext = {};

void ExecutionThreadFunction(void* ptr) NN_NOEXCEPT
{
    auto& context = *reinterpret_cast<ExecutionContext*>(ptr);
    auto& executor = *context.pExecutor;
    auto& resource = context.resource;
    auto& terminator = *context.pTerminator;

    resource.curlHandle = curl_easy_init();
    NN_UTIL_SCOPE_EXIT {
        curl_easy_cleanup(resource.curlHandle);
    };
    os::TimerEvent curlTimer(os::EventClearMode_ManualClear);

    bool isAlive = true;
    while (isAlive)
    {
        switch (os::WaitAny(
            terminator.GetBase(),
            curlTimer.GetBase(),
            executor))
        {
        case 0: // terminator
            isAlive = false;
            break;

        case 1: // curlTimer
            curlTimer.Clear();
            curl_easy_cleanup(resource.curlHandle);
            resource.curlHandle = curl_easy_init();
            NN_ABORT_UNLESS_NOT_NULL(resource.curlHandle);
            break;

        case 2: // executor
            executor.ClearSignal();
            executor.Execute(resource);
            // curlHandle のキャッシュを、連続するリクエストの 10 秒後に破棄する。
            curlTimer.StartOneShot(TimeSpan::FromSeconds(10));
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/* 各スレッドの初期化 --------------------------------------------------------------------------------*/
Result InitializeServiceThread(DauthResourceType& resource) NN_NOEXCEPT
{
    g_pDauthServer = new(&g_ServerStroage) DauthServer();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pDauthServer->RegisterObjectForPort(
        resource.GetSharedService(), DauthSessionCountMax, dauth::ServiceName));

    NN_RESULT_DO(os::CreateThread(
        &g_DauthServerThread,
        DauthServerThreadFunction, g_pDauthServer,
        g_DauthServerThreadStack, sizeof(g_DauthServerThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(dauth, IpcServer)));
    os::SetThreadNamePointer(
        &g_DauthServerThread,
        NN_SYSTEM_THREAD_NAME(dauth, IpcServer));

    NN_RESULT_SUCCESS;
}

Result InitializeExecutorThread(DauthResourceType& resource) NN_NOEXCEPT
{
    g_ExecutionContext.pExecutor = &resource.GetExecutorRef();
    g_ExecutionContext.pTerminator = &g_DauthExecThreadTerminator;

    NN_RESULT_DO(os::CreateThread(
        &g_DauthExecThread,
        ExecutionThreadFunction, &g_ExecutionContext,
        g_DauthExecThreadStack, sizeof(g_DauthExecThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(dauth, Executor)));
    os::SetThreadNamePointer(
        &g_DauthExecThread,
        NN_SYSTEM_THREAD_NAME(dauth, Executor));

    NN_RESULT_SUCCESS;
}
} // ~namespace <anonymous>

Result StartDauthServer(DauthResourceType& resource) NN_NOEXCEPT
{
    // サービススレッド, Executor スレッドの初期化
    NN_RESULT_DO(InitializeServiceThread(resource));
    NN_RESULT_DO(InitializeExecutorThread(resource));

    // 開始
    os::StartThread(&g_DauthExecThread);
    os::StartThread(&g_DauthServerThread);
    NN_RESULT_SUCCESS;
}

void StopDauthServer() NN_NOEXCEPT
{
    // 全スレッドに終了要求
    g_pDauthServer->RequestStop();
    g_DauthExecThreadTerminator.Signal();

    // 終了待ち, 破棄
    os::WaitThread(&g_DauthServerThread);
    os::WaitThread(&g_DauthExecThread);
    os::DestroyThread(&g_DauthServerThread);
    os::DestroyThread(&g_DauthExecThread);

    // リソース解放
    g_pDauthServer->~DauthServer();
}
} // ~namespace nn
