﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_AccountDaemon.h"

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/detail/account_Log.h>
#include <nn/bgtc/bgtc_Api.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/os/os_MultipleWaitUtility.h>

namespace nn {
namespace {
class ScopedBgtcActivation
{
    NN_DISALLOW_COPY(ScopedBgtcActivation);
private:
    bgtc::Task m_Task;
    bool m_FinishNotificationRequied;
public:
    ScopedBgtcActivation() NN_NOEXCEPT
        : m_FinishNotificationRequied(true)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Task.Initialize());

        auto r = m_Task.NotifyStarting();
        if (!r.IsSuccess())
        {
            NN_ACCOUNT_WARN_WITH_TIMESTAMP(
                "bgtc::Task::NotifyStarting() failed with %03d-%04d\n",
                r.GetModule(), r.GetDescription());
            m_FinishNotificationRequied = false;
        }
    }
    ~ScopedBgtcActivation() NN_NOEXCEPT
    {
        if (m_FinishNotificationRequied)
        {
            m_Task.NotifyFinished();
        }
        m_Task.Finalize();
    }
};
} // ~namespace nn::<anonymous>

AccountDaemon::RunnableState::RunnableState() NN_NOEXCEPT
    : m_Notifier(os::EventClearMode_ManualClear)
{
}
void AccountDaemon::RunnableState::Update(bool value) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    auto currentValue = m_IsRunnable;
    m_IsRunnable = value;

    // エッジに対する挙動
    if (!currentValue && value)
    {
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("Runnability: gets TRUE\n");
        m_Notifier.Signal();
    }
    else if (currentValue && !value)
    {
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("Runnability: gets FALSE\n");
    }
}
bool AccountDaemon::RunnableState::Test() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_IsRunnable;
}

AccountDaemon::AccountDaemon() NN_NOEXCEPT
    : m_Terminator(os::EventClearMode_ManualClear)
    , m_SuspendRequest(os::EventClearMode_ManualClear)
    , m_SuspendResponse(os::EventClearMode_ManualClear)
    , m_Settings(LoadAccountDaemonSettings())
    , m_Scheduler(m_Settings.scheduler)
{
}
void AccountDaemon::ThreadFunctionImpl() NN_NOEXCEPT
{
    account::Notifier profileUpdateNotifier;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetProfileUpdateNotifier(&profileUpdateNotifier));
    os::SystemEvent eProfileUpdate;
    NN_ABORT_UNLESS_RESULT_SUCCESS(profileUpdateNotifier.GetSystemEvent(&eProfileUpdate));

    auto const pTerminator = m_Terminator.GetBase();
    auto const pRunnableStateChanged = m_RunnableState.GetEventBase();
    auto const pSuspendRequest = m_SuspendRequest.GetBase();
    auto const pProfileUpdate = eProfileUpdate.GetBase();
    auto const pScheduler = m_Scheduler.GetEventBase();
    while (NN_STATIC_CONDITION(true))
    {
        auto const Index = (!IsRunning())
            ? os::WaitAny(pTerminator, pRunnableStateChanged, pSuspendRequest)
            : os::WaitAny(pTerminator, pRunnableStateChanged, pSuspendRequest, pProfileUpdate, pScheduler);
        // イベントには優先度があるため index は使用しない
        NN_UNUSED(Index);

        // 終了要求の検出
        if (m_Terminator.TryWait())
        {
            break;
        }

        // 実行可能性の変化
        m_RunnableState.Clear();

        // 中断要求への対応
        if (m_SuspendRequest.TryWait())
        {
            m_SuspendRequest.Clear();
            ReplyToControlRequest();
        }

        // 中断と実行可能性の検証
        if (IsRunning())
        {
            // bgtc へのスリープ抑制要求
            ScopedBgtcActivation bgtcActivation;

            // キャンセル処理を受け付ける
            CancelPoint::ScopedActivation c(m_CancelPoint);

            NN_ACCOUNT_INFO_WITH_TIMESTAMP("Daemon running\n");

            // プロフィールの更新時には即時タスクを実行する
            if (eProfileUpdate.TryWait())
            {
                eProfileUpdate.Clear();
                m_Scheduler.RescheduleImmediately();
            }

            // タスク実行
            if (m_Scheduler.TryWait())
            {
                m_Scheduler.Clear();
                m_Scheduler.ExecuteScheduledTasks(m_CancelPoint);
            }
        }
#if !defined(NN_SDK_BUILD_RELEASE)
        else
        {
            NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                "Daemon state: [Runnable=%s, Suspended=%s]\n",
                m_RunnableState.Test() ? "true" : "false",
                m_IsSuspended ? "true" : "false");
        }
#endif
    }
}
void AccountDaemon::ControlRequestHandlerImpl(account::BackgroundDaemonControlRequest request) NN_NOEXCEPT
{
    if (request == account::BackgroundDaemonControlRequest::Resume)
    {
        m_Scheduler.RescheduleImmediately();
    }

    NN_SDK_ASSERT(!m_SuspendRequest.TryWait());
    m_SuspendRequestBody = request;
    m_SuspendRequest.Signal();

    if (request == account::BackgroundDaemonControlRequest::Suspend)
    {
        m_CancelPoint.Signal();
    }

    // 実行中の BG タスクがあれば、完了までサスペンドは成功しない
    m_SuspendResponse.Wait();
    m_SuspendResponse.Clear();
}
void AccountDaemon::ReplyToControlRequest() NN_NOEXCEPT
{
    switch (m_SuspendRequestBody)
    {
    case account::BackgroundDaemonControlRequest::Resume:
        m_IsSuspended = false;
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("Daemon resumed\n");
        break;
    case account::BackgroundDaemonControlRequest::Suspend:
        m_IsSuspended = true;
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("Daemon suspended\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_SuspendResponse.Signal();
}

TimeSpan AccountDaemon::GetNextWakeupRequestInUptime() const NN_NOEXCEPT
{
    return m_Scheduler.GetNextExecutionInUptime();
}

void AccountDaemon::NotifyEnteredHalfAwake() NN_NOEXCEPT
{
    NN_ACCOUNT_INFO_WITH_TIMESTAMP("Entered HalfAwake\n");
    m_RunnableState.Update(true);
}
void AccountDaemon::OnEnteringFullAwake(nn::psc::PmState currentState) NN_NOEXCEPT
{
    NN_UNUSED(currentState);
    NN_ACCOUNT_INFO_WITH_TIMESTAMP("Entered FullAwake\n");
    // FullAwake 用にスケジュール間隔を標準に戻す
    m_Scheduler.ResetInterval();
    m_RunnableState.Update(true);
}
void AccountDaemon::OnEnteringMinimumAwake(nn::psc::PmState currentState) NN_NOEXCEPT
{
    switch (currentState)
    {
    case psc::PmState_EssentialServicesAwake:
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("MinimumAwake <- EssentialServicesAwake\n");
        break;
    default:
        break;
    }
}
void AccountDaemon::OnEnteringSleepReady(nn::psc::PmState currentState) NN_NOEXCEPT
{
    switch (currentState)
    {
    case psc::PmState_MinimumAwake:
        NN_ACCOUNT_INFO_WITH_TIMESTAMP("MinimumAwake -> SleepReady\n");
        m_RunnableState.Update(false);
        m_CancelPoint.Signal();
        // HalfAwake 用にスケジュール間隔を半起床用の値に変更する
        m_Scheduler.UpdateInterval(TimeSpan::FromSeconds(m_Settings.backgroundAwakingPeriodicity));
        break;
    default:
        break;
    }
}
void AccountDaemon::OnEnteringShutdownReady(nn::psc::PmState currentState) NN_NOEXCEPT
{
    NN_UNUSED(currentState);
    NN_ACCOUNT_INFO_WITH_TIMESTAMP("Shutting down\n");
    m_CancelPoint.Signal();
}
void AccountDaemon::RequestStop() NN_NOEXCEPT
{
    m_CancelPoint.Signal();
    m_Terminator.Signal();
}

} // ~namespace nn
