﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/osdbg.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/util/util_FormatString.h>

#include "dmnt_ArraySize.h"
#include "dmnt_Rsp.h"
#include "dmnt_DebugMonitor.h"
#include "dmnt_OsDebug.h"

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cinttypes>
#include "gdbserver_log.h"

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
#define PSTATE cpsr
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
#define PSTATE pstate
#endif

namespace
{
enum
{
    SDK_BKPT                    = 0xE7FFFFFF,
    SDK_BKPT_MASK               = 0xFFFFFFFF,

    ARM_BKPT                    = 0xE7FFDEFE,   // Invalid instruction in all three modes
    ARM_BKPT_MASK               = 0xFFFFFFFF,

    A64_BRK                     = 0xD4200000,   // 1101 0100 001  Im16    0 0000
    A64_BRK_MASK                = 0xFFE0001F,

    A64_HLT                     = 0xD4400000,   // 1101 0100 010  Im16    0 0000
    A64_HLT_MASK                = 0xFFE0001F,

    A32_BKPT                    = 0xE1200070,   // 1110 0001 0010 Im12 0111  Im4
    A32_BKPT_MASK               = 0xFFF000F0,

    T16_BKPT                    = 0x0000BE00,   // 1011 1110  Im8
    T16_BKPT_MASK               = 0x0000FF00,

    SDK_T16_BKPT               = 0x0000B68E,   //
    SDK_T16_BKPT_MASK          = 0x0000FFFF,
};
char const *g_PacketError_BreakParseZ             = "E01";
char const *g_PacketError_BreakParsez             = "E02";
char const *g_PacketError_SetBreak                = "E03";
char const *g_PacketError_SetBreakHardware        = "E04";
char const *g_PacketError_SetWatch                = "E05";
char const *g_PacketError_SetWatchRead            = "E06";
char const *g_PacketError_SetWatchWrite           = "E07";
char const *g_PacketError_ClearBreak              = "E08";
char const *g_PacketError_ClearBreakHardware      = "E09";
char const *g_PacketError_ClearWatch              = "E10";
char const *g_PacketError_ThreadList              = "E11";
char const *g_PacketError_vCont                   = "E12";
char const *g_PacketError_LibReadParse            = "E13";
char const *g_PacketError_LibOffset               = "E14";
char const *g_PacketError_FeaturesParse           = "E15";
char const *g_PacketError_ThreadReadReg           = "E16";
char const *g_PacketError_ThreadContinue          = "E17";
char const *g_PacketError_SetThreadContext        = "E18";
char const *g_PacketError_WriteRegisters          = "E19";
char const *g_PacketError_ReadRegisters           = "E20";
//char const *g_PacketError_ReadMemory              = "E21";
char const *g_PacketError_ReadMemorySize          = "E22";
char const *g_PacketError_WriteMemory             = "E23";
char const *g_PacketError_WriteMemorySize         = "E24";
//char const *g_PacketError_Continue                = "E25";
char const *g_PacketError_ContinueThread          = "E26";
//char const *g_PacketError_Step                    = "E27";
char const *g_PacketError_StepThread              = "E28";
char const *g_PacketError_ThreadDead              = "E29";
char const *g_PacketError_ReadRegisterN           = "E30";
char const *g_PacketError_WriteRegisterN          = "E31";
char const *g_PacketError_WriteRegisterGetContext = "E32";

const uint32_t HW_PSR_THUMB_STATE = 0x20; // From kern_RegisterDefinition.h
const char* hexchars = "0123456789abcdef";
}

namespace nn { namespace dmnt { namespace rsp {
int PacketParser::HexToInteger(char ch)
{
    if (ch >= 'a' && ch <= 'f')
    {
        return ch-'a'+10;
    }
    if (ch >= '0' && ch <= '9')
    {
        return ch-'0';
    }
    if (ch >= 'A' && ch <= 'F')
    {
        return ch-'A'+10;
    }
    return -1;
}

int PacketParser::HexToInteger(char *p)
{
    if (p == nullptr)
    {
        return -1;
    }
    int x = 0;
    bool isNegative = false;
    if (*p == '-')
    {
        isNegative=true;
        p++;
    }

    while (*p)
    {
        int n = HexToInteger(*p++);
        if (n == -1)
        {
            return -1;
        }
        x <<= 4;
        x |= (n & 0xF);
    }

    return isNegative?-x:x;
}

void PacketParser::Mem2Hex(char* buf, const void* mem, size_t size)
{
    uint8_t ch;
    const uint8_t* p = static_cast<const uint8_t*>(mem);

    while (size-- > 0)
    {
        ch = *p++;
        *buf++ = hexchars[ch >> 4];
        *buf++ = hexchars[ch & 0xf];
    }
    *buf = '\0';
}

void PacketParser::Hex2Mem(void *mem, const char* buf, size_t size)
{
    uint8_t* p = static_cast<uint8_t*>(mem);

    for (size_t i = 0; i < size; i++)
    {
        uint8_t ch;
        ch  = HexToInteger(*buf++) << 4;
        ch |= HexToInteger(*buf++);
        *p++ = ch;
    }
}

void SetReplyStringOK(char* pReplyPacket) NN_NOEXCEPT
{
    std::strcpy(pReplyPacket, "OK");
}

void SetReplyError(char* pReplyPacket, char const *errorString) NN_NOEXCEPT
{
    GDB_TRACE_E("%s:%d Reply: Error: '%s'\n", __func__,__LINE__, errorString);
    std::strcpy(pReplyPacket, errorString);
}

void SetReplyString(char* pReplyPacket, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list vaList;

    va_start(vaList, pFormat);
    nn::util::VSNPrintf(pReplyPacket, BUFFER_SIZE, pFormat, vaList);
    va_end(vaList);
}

void AppendReplyString(char* pReplyPacket, const char* pFormat, ...) NN_NOEXCEPT
{
    std::va_list vaList;
    int len = strlen(pReplyPacket);
    char *p = pReplyPacket + len;

    va_start(vaList, pFormat);
    nn::util::VSNPrintf(p, BUFFER_SIZE - len, pFormat, vaList);
    va_end(vaList);
}

void PacketParser::Z()
{
    // Z2,7c41703c,4
    int breakType = 0;
    uint64_t addr = 0;
    uint64_t len = 0;

    int n = std::sscanf(m_pReceivedPacket,"Z%d,%" SCNx64 " ,%" SCNx64, &breakType, &addr, &len);
    if (n != 3)
    {
        GDB_TRACE_E("%s:%d Break: n=%d, '%s' ERROR\n",__func__,__LINE__, n, m_pReceivedPacket);
        SetReplyError(m_pReplyPacket, g_PacketError_BreakParseZ);
        return;
    }

    switch (breakType)
    {
    case 0: // Software breakpoint
        {
            if (len == 2 || len == 4)
            {
                if (m_pDebugProcess->SetBreakPoint(addr, len, false).IsSuccess())
                {
                    SetReplyStringOK(m_pReplyPacket);
                }
                else
                {
                    SetReplyError(m_pReplyPacket, g_PacketError_SetBreak);
                }
            }
            // On unsupported length, return empty string.
        }
        break;
    case 1: // Hardware breakpoint
        {
            if (len == 2 || len == 4)
            {
                if (m_pDebugProcess->SetHardwareBreakPoint(addr, len, false).IsSuccess())
                {
                    SetReplyStringOK(m_pReplyPacket);
                }
                else
                {
                    SetReplyError(m_pReplyPacket, g_PacketError_SetBreakHardware);
                }
            }
            // On unsupported length, return empty string.
        }
        break;
    case 2: // Write watch point
        if (DebugProcess::IsValidWatch(addr, len))
        {
            if (m_pDebugProcess->SetWatchPoint(addr, len, false, true).IsSuccess())
            {
                SetReplyStringOK(m_pReplyPacket);
            }
            else
            {
                SetReplyError(m_pReplyPacket, g_PacketError_SetWatch);
            }
        }
        // On unsupported address/length, return empty string.
        break;
    case 3: // Read watch point
        if (DebugProcess::IsValidWatch(addr, len))
        {
            if (m_pDebugProcess->SetWatchPoint(addr, len, true, false).IsSuccess())
            {
                SetReplyStringOK(m_pReplyPacket);
            }
            else
            {
                SetReplyError(m_pReplyPacket, g_PacketError_SetWatchRead);
            }
        }
        // On unsupported address/length, return empty string.
        break;
    case 4: // Access watch point
        if (DebugProcess::IsValidWatch(addr, len))
        {
            if (m_pDebugProcess->SetWatchPoint(addr, len, true, true).IsSuccess())
            {
                SetReplyStringOK(m_pReplyPacket);
            }
            else
            {
                SetReplyError(m_pReplyPacket, g_PacketError_SetWatchWrite);
            }
        }
        // On unsupported address/length, return empty string.
        break;
    default:
        // Unsupported
        break;
    }
}

void PacketParser::z()
{
    // Z2,7c41703c,4
    int breakType = 0;
    uint64_t addr = 0;
    uint64_t len = 0;

    int n = std::sscanf(m_pReceivedPacket,"z%d,%" SCNx64 " ,%" SCNx64, &breakType, &addr, &len);
    if (n != 3)
    {
        GDB_TRACE_E("%s:%d Break: n=%d, '%s' ERROR\n",__func__,__LINE__, n, m_pReceivedPacket);
        SetReplyError(m_pReplyPacket, g_PacketError_BreakParsez);
        return;
    }

    switch (breakType)
    {
    case 0: // Software breakpoint
        {
            if (m_pDebugProcess->ClearBreakPoint(addr, len).IsSuccess())
            {
                SetReplyStringOK(m_pReplyPacket);
            }
            else
            {
                SetReplyError(m_pReplyPacket, g_PacketError_ClearBreak);
            }
        }
        break;
    case 1: // Hardware breakpoint
        {
            if (m_pDebugProcess->ClearHardwareBreakPoint(addr, len).IsSuccess())
            {
                SetReplyStringOK(m_pReplyPacket);
            }
            else
            {
                SetReplyError(m_pReplyPacket, g_PacketError_ClearBreakHardware);
            }
        }
        break;
    case 2: // Write watch point
    case 3: // Read watch point
    case 4: // Access watch point
        if (m_pDebugProcess->ClearWatchPoint(addr, len).IsSuccess())
        {
            SetReplyStringOK(m_pReplyPacket);
        }
        else
        {
            SetReplyError(m_pReplyPacket, g_PacketError_ClearWatch);
        }
        break; // Unsupported
    default:
        // Unsupported
        break;
    }
}

int GetThreadIdIndex(uint32_t *threadIds, int32_t numThreads, uint32_t threadId)
{
    for (int i = 0; i < numThreads; i++)
    {
        if (threadIds[i] == threadId)
        {
            return i;
        }
    }
    return 0;
}

Result PacketParser::ParseVCont(char * const token, uint32_t *threadIds, uint8_t *threadActions, int32_t numThreads, nn::dmnt::ContinueMode &defaultAction)
{
    Result result = ResultSuccess();
    int tid = -1; // -1==AllThreads, 0==AnyThread
    int sig = -1;
    int threadIndex = -1;
    if (token[0] && token[1]) // Looking for: /[cCsStr]:/ from the string "vCont;s:11e;c"
    {
        char *threadId = strchr(token, ':');
        if (threadId)
        {
            *threadId = 0;
            tid = HexToInteger(threadId + 1);
            threadIndex = GetThreadIdIndex(threadIds, numThreads, tid);
        }
    }

    if (tid == -1)
    {
        if (defaultAction != STOPPED)
        {
            GDB_TRACE_E("vCont:%d Error: Too many default actions - GDB Error\n",__LINE__);
        }
    }
    switch (token[0])
    {
    case 'c':
        if (tid > 0)
        {
            GDB_TRACE_P("vCont:%d Continue tid:%d\n",__LINE__, tid);
            threadActions[threadIndex] = CONTINUE;
        }
        else
        {
            defaultAction = CONTINUE;
        }
        break;
    case 'C':
        if (token[1])
        {
            sig = HexToInteger(token + 1);
        }
        GDB_TRACE_W("vCont:%d Warning: Ignoring vCont;C with signal %d\n", __LINE__, sig);
        if (tid > 0)
        {
            GDB_TRACE_P("vCont:%d Continue tid:%d, sig:%d\n",__LINE__, tid, sig);
            threadActions[threadIndex] = CONTINUE;
        }
        else
        {
            defaultAction = CONTINUE;
        }
        break;
    case 's':
        if (tid > 0)
        {
            GDB_TRACE_P("vCont:%d Step tid:%d\n",__LINE__, tid);
            threadActions[threadIndex] = STEP;
        }
        else
        {
            defaultAction = STEP;
        }
        break;
    case 'S':
        if (token[1])
        {
            sig = HexToInteger(token + 1);
        }
        GDB_TRACE_W("vCont:%d Warning: Ignoring vCont;S with signal %d\n", __LINE__, sig);
        if (tid > 0)
        {
            GDB_TRACE_P("vCont:%d Step tid:%d, sig:%d\n",__LINE__, tid, sig);
            threadActions[threadIndex] = STEP;
        }
        else
        {
            defaultAction = STEP;
        }
        break;
    case 't':
        GDB_TRACE_W("vCont:%d Warning: Ignoring vCont;t \n", __LINE__);
        break;
    case 'r':
        GDB_TRACE_W("vCont:%d Warning: Ignoring vCont;r \n", __LINE__);
        break;
    default:
        GDB_TRACE_W("vCont:%d Warning: Ignoring vCont;%c \n", __LINE__, token[0]);
        break;
    }
    return result;
}

void PacketParser::v()
{
    //  vCont;s:fc;c
    //  vCont;C04:fe;c
    //  vCont[;action[:thread-id]]...
    //  Actions:
    //      s step
    //      S Step with signal
    //      c continue
    //      C continue with signal
    //      t stop
    //      r range step
    //  If an action doesn't have a ThreadId, apply that action to all threads.
    //  If there are no default actions, any threads not specified should remain stopped.
    // ThreadId:
    //      -1: All threads
    //       0: Any thread
    //      positive hex string: A ThreadId
    //
    // Horizon doesn't support detailed per-thread control.
    // As soon as a thread is continued, it starts running and can generate more debugging events.
    // Once that happens, the debugger can't do anything with the other threads.
    //
    if (std::strcmp(m_pReceivedPacket, "vCont?") == 0)
    {
        SetReplyString(m_pReplyPacket, "vCont;c;C;s;S;");
        return;
    }

    char *save;
    char *token = strtok_r(m_pReceivedPacket, ";", &save);

    if (token == nullptr || std::strcmp(token, "vCont") != 0)
    {
        return;
    }

    uint32_t threadIds[DebugProcess::NUM_THREAD] = {};
    uint8_t threadActions[DebugProcess::NUM_THREAD] = {};

    int32_t numThreads;
    if (!m_pDebugProcess->GetThreadList(&numThreads, threadIds, ARRAYSIZE(threadIds)).IsSuccess())
    {
        GDB_TRACE_E("vCont:%d Error: Can't get thread list\n",__LINE__);
        SetReplyError(m_pReplyPacket, g_PacketError_ThreadList);
        return;
    }

    Result result = ResultSuccess();
    token = strtok_r(NULL, ";", &save);
    nn::dmnt::ContinueMode defaultAction = STOPPED;

    while (token != nullptr && result.IsSuccess())
    {
        result = ParseVCont(token, threadIds, threadActions, numThreads, defaultAction);
        token = strtok_r(NULL, ";", &save);
    }

    GDB_TRACE_P("vCont:%d ThreadCount:%d, Default Thread action: %d, \n",__LINE__, numThreads, defaultAction);

    int tid = -1; // -1==AllThreads, 0==AnyThread
    for (int i = 0; i < numThreads; i++)
    {
        if (threadActions[i] == STEP || (threadActions[i] == STOPPED && defaultAction == STEP))
        {
            tid = threadIds[i];
            result = m_pDebugProcess->Step(threadIds[i]);
        }
    }
    if (tid == GetLastThreadId())
    {
        result = m_pDebugProcess->Continue(tid);
    }
    else
    {
        // If we try to continue a thread different than the one that the debug event
        // was for, then Horizon will likely end up terminating the inferior.
        // So, in this case just continue all threads.
        result = m_pDebugProcess->Continue();
    }

    if (result.IsSuccess())
    {
        SetReplyStringOK(m_pReplyPacket);
    }
    else
    {
        GDB_TRACE_E("vCont:%d Result:x%x\n", __LINE__, (int)result.GetInnerValueForDebug());
        SetReplyError(m_pReplyPacket, g_PacketError_vCont);
    }
}

void PacketParser::Q()
{
    if (std::strncmp(m_pReceivedPacket, "QStartNoAckMode", std::strlen("QStartNoAckMode")) == 0)
    {
        m_pReceivedPacket += std::strlen("QStartNoAckMode");
        qStartNoAckMode();
    }
    else
    {
        GDB_TRACE_P("%s:%d Unknown packet: \n",__func__,__LINE__, m_pReceivedPacket);
    }
}

void PacketParser::q()
{
    if (std::strcmp(m_pReceivedPacket, "qC") == 0) // Get current Thread Id
    {
        SetReplyString(m_pReplyPacket, "QC%x", GetLastThreadId());
    }
    else if (std::strcmp(m_pReceivedPacket, "qfThreadInfo") == 0) // Get list of all thread IDs
    {
        m_pReceivedPacket += std::strlen("qfThreadInfo");
        qfThreadInfo();
    }
    else if (std::strcmp(m_pReceivedPacket, "qsThreadInfo") == 0) // Continue Get list of all thread IDs
    {
        m_pReceivedPacket += std::strlen("qsThreadInfo");
        qsThreadInfo();
    }
    else if (std::strncmp(m_pReceivedPacket, "qSupported:", std::strlen("qSupported:")) == 0)
    {
        m_pReceivedPacket += std::strlen("qSupported:");
        qSupported();
    }
    else if (std::strncmp(m_pReceivedPacket, "qAttached", std::strlen("qAttached")) == 0)
    {
        m_pReceivedPacket += std::strlen("qAttached");
        qAttached();
    }
    else if (std::strncmp(m_pReceivedPacket, "qXfer:features:read:", std::strlen("qXfer:features:read:")) == 0)
    {
        m_pReceivedPacket += std::strlen("qXfer:features:read:");
        qXferFeaturesRead();
    }
    else if (std::strncmp(m_pReceivedPacket, "qXfer:libraries:read:", std::strlen("qXfer:libraries:read:")) == 0)
    {
        m_pReceivedPacket += std::strlen("qXfer:libraries:read:");
        qXferLibrariesRead();
    }
    else if (std::strncmp(m_pReceivedPacket, "qTStatus", std::strlen("qTStatus")) == 0)
    {
        m_pReceivedPacket += std::strlen("qTStatus");
        qTStatus();
    }
    else if (std::strncmp(m_pReceivedPacket, "qOffsets", std::strlen("qOffsets")) == 0)
    {
        m_pReceivedPacket += std::strlen("qOffsets");
        qOffsets();
    }
}

void PacketParser::qTStatus()
{
}

void PacketParser::qSupported()
{
    // Sample qSupported string received from GDB 7.10.50
    // qSupported:multiprocess+;swbreak+;hwbreak+;qRelocInsn+;fork-events+;vfork-events+;exec-events+;vContSupported+
    //
    // multiprocess+
    // swbreak+
    // hwbreak+
    // qRelocInsn+
    // fork-events+
    // vfork-events+
    // exec-events+
    // vContSupported+

    // TODO: Check that the client supports swbreak and hwbreak
    //       and store this information somewhere we can keep it around.
    SetReplyString(m_pReplyPacket, "PacketSize=%x", BUFFER_SIZE - 1);
    AppendReplyString(m_pReplyPacket, ";qXfer:features:read+");
    AppendReplyString(m_pReplyPacket, ";qXfer:libraries:read+");
    AppendReplyString(m_pReplyPacket, ";multiprocess-");
    AppendReplyString(m_pReplyPacket, ";QStartNoAckMode+");
    AppendReplyString(m_pReplyPacket, ";swbreak+");
    AppendReplyString(m_pReplyPacket, ";hwbreak+");
}

void PacketParser::qAttached()
{
    SetReplyString(m_pReplyPacket, "1");
}

bool PacketParser::IsSetNoAck() const
{
    return m_NoAckIsSet;
}

void PacketParser::qStartNoAckMode()
{
    SetReplyStringOK(m_pReplyPacket);
    m_NoAckIsSet = true;
}

void PacketParser::qOffsets()
{
    size_t numModules = m_pDebugProcess->GetNumModule();
    size_t moduleIndex = m_pDebugProcess->GetMainModuleIndex();
    if (moduleIndex < numModules)
    {
        SetReplyString(m_pReplyPacket, "TextSeg=%x", m_pDebugProcess->GetBaseAddress(moduleIndex ));
        GDB_TRACE_T("QOFFSET TextSeg=%x\n", m_pDebugProcess->GetBaseAddress(moduleIndex ));
        return;
    }

    GDB_TRACE_T("%s:%d qOffset error: NumModules:%d, moduleIndex:%d\n", __func__,__LINE__, (int)m_pDebugProcess->GetNumModule(), (int)moduleIndex );
    return;
}

void PacketParser::qXferLibrariesRead()
{
    if (std::strncmp(m_pReceivedPacket, ":", std::strlen(":")) == 0)
    {
        m_pReceivedPacket += std::strlen(":");
        uint32_t offset;
        uint32_t length;
        int n = std::sscanf(m_pReceivedPacket,"%x,%x",&offset, &length);
        if (n != 2)
        {
            SetReplyError(m_pReplyPacket, g_PacketError_LibReadParse);
            return;
        }
        if (offset != 0)
        {
            SetReplyError(m_pReplyPacket, g_PacketError_LibOffset);
            return;
        }

        SetReplyString(m_pReplyPacket, "l<?xml version=\"1.0\"?>\n");
        AppendReplyString(m_pReplyPacket, "<library-list>\n");

        size_t numModules = m_pDebugProcess->GetNumModule();
        for (size_t moduleIndex = 0; moduleIndex < numModules; ++moduleIndex)
        {
            AppendReplyString(m_pReplyPacket,
                    "<library name=\"%s\">\n"
                    "   <segment address=\"0x%p\"/>\n"
                    "</library>\n",
                    m_pDebugProcess->GetModuleName(moduleIndex),
                    m_pDebugProcess->GetBaseAddress(moduleIndex));
        }
        AppendReplyString(m_pReplyPacket, "</library-list>\n");
    }
}   // NOLINT [readability/fn_size]


void PacketParser::qXferFeaturesRead()
{
    if (std::strncmp(m_pReceivedPacket, "target.xml:", std::strlen("target.xml:")) == 0)
    {
        m_pReceivedPacket += std::strlen("target.xml:");
        uint32_t offset;
        uint32_t length;
        int n = std::sscanf(m_pReceivedPacket,"%x,%x",&offset, &length);
        if (n != 2)
        {
            SetReplyError(m_pReplyPacket, g_PacketError_FeaturesParse);
            return;
        }
        const char aarch32[] =
            "l<?xml version=\"1.0\"?>\n"
            "<target>"
            "<feature name=\"org.gnu.gdb.arm.core\">\n"
            "   <reg name=\"r0\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r1\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r2\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r3\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r4\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r5\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r6\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r7\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r8\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r9\"  bitsize=\"32\"/>\n"
            "   <reg name=\"r10\" bitsize=\"32\"/>\n"
            "   <reg name=\"r11\" bitsize=\"32\"/>\n"
            "   <reg name=\"r12\" bitsize=\"32\"/>\n"
            "   <reg name=\"sp\"  bitsize=\"32\" type=\"data_ptr\"/>\n"
            "   <reg name=\"lr\"  bitsize=\"32\"/>\n"
            "   <reg name=\"pc\"  bitsize=\"32\" type=\"code_ptr\"/>\n"
            "   <reg name=\"cpsr\" bitsize=\"32\" regnum=\"25\"/>\n"
            "</feature>\n"
            "<feature name=\"org.gnu.gdb.arm.vfp\">\n"
            "   <reg name=\"d0\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d1\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d2\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d3\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d4\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d5\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d6\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d7\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d8\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d9\"  bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d10\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d11\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d12\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d13\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d14\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d15\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d16\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d17\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d18\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d19\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d20\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d21\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d22\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d23\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d24\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d25\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d26\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d27\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d28\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d29\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d30\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"d31\" bitsize=\"64\" type=\"ieee_double\"/>\n"
            "   <reg name=\"fpscr\" bitsize=\"32\" type=\"int\" group=\"float\"/>\n"
            "</feature>\n"
            "</target>\n";
        const char aarch64[] =
            "l<?xml version=\"1.0\"?>\n"
            "<target>\n"
            "<architecture>aarch64</architecture>\n"
            "<feature name=\"org.gnu.gdb.aarch64.core\">\n"
            "<reg name=\"x0\" bitsize=\"64\"/>\n"
            "<reg name=\"x1\" bitsize=\"64\"/>\n"
            "<reg name=\"x2\" bitsize=\"64\"/>\n"
            "<reg name=\"x3\" bitsize=\"64\"/>\n"
            "<reg name=\"x4\" bitsize=\"64\"/>\n"
            "<reg name=\"x5\" bitsize=\"64\"/>\n"
            "<reg name=\"x6\" bitsize=\"64\"/>\n"
            "<reg name=\"x7\" bitsize=\"64\"/>\n"
            "<reg name=\"x8\" bitsize=\"64\"/>\n"
            "<reg name=\"x9\" bitsize=\"64\"/>\n"
            "<reg name=\"x10\" bitsize=\"64\"/>\n"
            "<reg name=\"x11\" bitsize=\"64\"/>\n"
            "<reg name=\"x12\" bitsize=\"64\"/>\n"
            "<reg name=\"x13\" bitsize=\"64\"/>\n"
            "<reg name=\"x14\" bitsize=\"64\"/>\n"
            "<reg name=\"x15\" bitsize=\"64\"/>\n"
            "<reg name=\"x16\" bitsize=\"64\"/>\n"
            "<reg name=\"x17\" bitsize=\"64\"/>\n"
            "<reg name=\"x18\" bitsize=\"64\"/>\n"
            "<reg name=\"x19\" bitsize=\"64\"/>\n"
            "<reg name=\"x20\" bitsize=\"64\"/>\n"
            "<reg name=\"x21\" bitsize=\"64\"/>\n"
            "<reg name=\"x22\" bitsize=\"64\"/>\n"
            "<reg name=\"x23\" bitsize=\"64\"/>\n"
            "<reg name=\"x24\" bitsize=\"64\"/>\n"
            "<reg name=\"x25\" bitsize=\"64\"/>\n"
            "<reg name=\"x26\" bitsize=\"64\"/>\n"
            "<reg name=\"x27\" bitsize=\"64\"/>\n"
            "<reg name=\"x28\" bitsize=\"64\"/>\n"
            "<reg name=\"x29\" bitsize=\"64\"/>\n"
            "<reg name=\"x30\" bitsize=\"64\"/>\n"
            "<reg name=\"sp\" bitsize=\"64\" type=\"data_ptr\"/>\n"
            "<reg name=\"pc\" bitsize=\"64\" type=\"code_ptr\"/>\n"
            "<reg name=\"cpsr\" bitsize=\"64\"/>\n"
            "</feature>\n"
            "<feature name=\"org.gnu.gdb.aarch64.fpu\">\n"
            "<vector id=\"v2d\" type=\"ieee_double\" count=\"2\"/>\n"
            "<vector id=\"v2u\" type=\"uint64\" count=\"2\"/>\n"
            "<vector id=\"v2i\" type=\"int64\" count=\"2\"/>\n"
            "<vector id=\"v4f\" type=\"ieee_single\" count=\"4\"/>\n"
            "<vector id=\"v4u\" type=\"uint32\" count=\"4\"/>\n"
            "<vector id=\"v4i\" type=\"int32\" count=\"4\"/>\n"
            "<vector id=\"v8u\" type=\"uint16\" count=\"8\"/>\n"
            "<vector id=\"v8i\" type=\"int16\" count=\"8\"/>\n"
            "<vector id=\"v16u\" type=\"uint8\" count=\"16\"/>\n"
            "<vector id=\"v16i\" type=\"int8\" count=\"16\"/>\n"
            "<vector id=\"v1u\" type=\"uint128\" count=\"1\"/>\n"
            "<vector id=\"v1i\" type=\"int128\" count=\"1\"/>\n"
            "<union id=\"vnd\">\n"
            "<field name=\"f\" type=\"v2d\"/>\n"
            "<field name=\"u\" type=\"v2u\"/>\n"
            "<field name=\"s\" type=\"v2i\"/>\n"
            "</union>\n"
            "<union id=\"vns\">\n"
            "<field name=\"f\" type=\"v4f\"/>\n"
            "<field name=\"u\" type=\"v4u\"/>\n"
            "<field name=\"s\" type=\"v4i\"/>\n"
            "</union>\n"
            "<union id=\"vnh\">\n"
            "<field name=\"u\" type=\"v8u\"/>\n"
            "<field name=\"s\" type=\"v8i\"/>\n"
            "</union>\n"
            "<union id=\"vnb\">\n"
            "<field name=\"u\" type=\"v16u\"/>\n"
            "<field name=\"s\" type=\"v16i\"/>\n"
            "</union>\n"
            "<union id=\"vnq\">\n"
            "<field name=\"u\" type=\"v1u\"/>\n"
            "<field name=\"s\" type=\"v1i\"/>\n"
            "</union>\n"
            "<union id=\"aarch64v\">\n"
            "<field name=\"d\" type=\"vnd\"/>\n"
            "<field name=\"s\" type=\"vns\"/>\n"
            "<field name=\"h\" type=\"vnh\"/>\n"
            "<field name=\"b\" type=\"vnb\"/>\n"
            "<field name=\"q\" type=\"vnq\"/>\n"
            "</union>\n"
            "<reg name=\"v0\" bitsize=\"128\" type=\"aarch64v\" regnum=\"34\"/>\n"
            "<reg name=\"v1\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v2\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v3\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v4\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v5\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v6\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v7\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v8\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v9\" bitsize=\"128\" type=\"aarch64v\" />\n"
            "<reg name=\"v10\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v11\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v12\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v13\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v14\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v15\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v16\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v17\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v18\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v19\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v20\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v21\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v22\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v23\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v24\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v25\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v26\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v27\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v28\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v29\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v30\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"v31\" bitsize=\"128\" type=\"aarch64v\"/>\n"
            "<reg name=\"fpsr\" bitsize=\"32\"/>\n"
            "<reg name=\"fpcr\" bitsize=\"32\"/>\n"
            "</feature>\n"
            "</target>\n";
        static_assert(sizeof(aarch32) + 8 < BUFFER_SIZE,"");
        static_assert(sizeof(aarch64) + 8 < BUFFER_SIZE,"");
        const char* xml = m_pDebugProcess->Is64Bit()? aarch64: aarch32;
        std::strncpy(m_pReplyPacket, xml + offset, length);
        m_pReplyPacket[length] = '\0';
        m_pReplyPacket += std::strlen(m_pReplyPacket);
    }
}   // NOLINT [readability/fn_size]

// アクティブなスレッドIDの取得
// スレッドが多かった時のためにサブシーケンスが準備されている。
// qfThreadInfo() は最初のパケット
// qsThreadInfo() は次以降のパケット
//  Reply
//      m thread-id,thread-id...,l
//      →最後は l で終える

void PacketParser::qfThreadInfo()
{
    uint32_t threadIds[DebugProcess::NUM_THREAD];
    int32_t numThreads;
    if (m_pDebugProcess->GetThreadList(&numThreads, threadIds, ARRAYSIZE(threadIds)).IsSuccess())
    {
        m_pReplyPacket[0] = 0;
        for (int i = 0; i < numThreads; i++)
        {
            AppendReplyString(m_pReplyPacket, "%s%x",(i ? ",": "m"), threadIds[i]);
        }

        // スレッド一覧取得の最初のパケットで、ログへスレッド情報も出力
        const int   arrayMax = DebugProcess::NUM_THREAD;
        int32_t     numThreads;
        nn::osdbg::ThreadInfo*  threadInfoArray[arrayMax];
        Result result = m_pDebugProcess->GetThreadInfoList(&numThreads, threadInfoArray, arrayMax);
        if (result.IsSuccess())
        {
            nn::dmnt::osdbg::PrintAllThreadInfo(threadInfoArray, numThreads);
        }
    }
    else
    {
        SetReplyString(m_pReplyPacket, "l");
    }
}

void PacketParser::qsThreadInfo()
{
    SetReplyString(m_pReplyPacket, "l");
}

void PacketParser::LastSignal()
{
    if (GetLastThreadId() == 0)
    {
        SetReplyString(m_pReplyPacket, "X01");
    }
    else
    {
        SetReplyString(m_pReplyPacket, "T%02xthread:%x;",GetLastSignal(), GetLastThreadId());
    }
}


void PacketParser::Hg()
{
    m_pReceivedPacket++;
    bool isSuccess = false;
    int32_t tid;
    if (std::sscanf(m_pReceivedPacket, "%x", &tid) == 1)
    {
        uint32_t threadIds[DebugProcess::NUM_THREAD];
        int32_t numThreads;
        if (m_pDebugProcess->GetThreadList(&numThreads, threadIds, ARRAYSIZE(threadIds)).IsSuccess())
        {
            if (tid == 0)
            {
                tid = threadIds[0];
            }
            for (int i = 0; i < numThreads; i++)
            {
                if (tid == -1 || threadIds[i] == tid)
                {
                    nn::svc::ThreadContext context;
                    if (m_pDebugProcess->GetThreadContext(&context, threadIds[i], nn::svc::ContextFlag_Control).IsSuccess())
                    {
                        isSuccess = true;
                        if (tid != -1)
                        {
                            // The next "g", "G", etc, command will use this threadId instead of the current thread
                            m_pDebugProcess->SetTidOverride(threadIds[i]);
                        }
                    }
                }
            }
        }
    }
    if (isSuccess)
    {
        SetReplyStringOK(m_pReplyPacket);
    }
    else
    {
        SetReplyError(m_pReplyPacket, g_PacketError_ThreadReadReg);
    }
}

void PacketParser::HG()
{
    // 切り替えだけ?
    Hg();
}

void PacketParser::Hc()
{
    m_pReceivedPacket++;
    bool isSuccess = false;
    int32_t tid;

    tid = HexToInteger(m_pReceivedPacket);
    {
        uint32_t threadIds[DebugProcess::NUM_THREAD];
        int32_t numThreads;
        if (m_pDebugProcess->GetThreadList(&numThreads, threadIds, ARRAYSIZE(threadIds)).IsSuccess())
        {
            if (tid == 0)
            {
                tid = threadIds[0];
            }
            for (int i = 0; i < numThreads; i++)
            {
                if (tid == -1 || threadIds[i] == tid)
                {
                    nn::svc::ThreadContext context;
                    if (m_pDebugProcess->GetThreadContext(&context, threadIds[i], nn::svc::ContextFlag_Control).IsSuccess())
                    {
                        isSuccess = true;
                        if (tid != -1)
                        {
                            // The next "c", "C", "s", "S", command will use this threadId instead of the current thread
                            m_pDebugProcess->SetTidOverrideStepContinue(threadIds[i]);
                        }
                    }
                }
            }
        }
    }
    if (isSuccess)
    {
        SetReplyStringOK(m_pReplyPacket);
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_ThreadContinue);
    }
}

void PacketParser::G()
{
    nn::svc::ThreadContext context;
    m_pReceivedPacket++;
    const uint32_t flags = nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl;

    uint32_t tid = m_pDebugProcess->GetTidOverride();
    if (tid == 0 || tid == -1) // 0==any, -1==all
    {
        tid = GetLastThreadId();
    }
    if (m_pDebugProcess->GetThreadContext(&context, tid, flags).IsSuccess())
    {
        UpdateGdbRegister(&context, m_pReceivedPacket);
        if (m_pDebugProcess->SetThreadContext(&context, tid, flags).IsSuccess())
        {
            SetReplyStringOK(m_pReplyPacket);
        }
        else
        {
            SetReplyString(m_pReplyPacket, g_PacketError_SetThreadContext);
        }
    }
    else
    {
        SetReplyError(m_pReplyPacket, g_PacketError_WriteRegisters);
    }
}

void PacketParser::g()
{
    const uint32_t flags = nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl;
    nn::svc::ThreadContext context;

    uint32_t tid = m_pDebugProcess->GetTidOverride();
    if (tid == 0 || tid == -1) // 0==any, -1==all
    {
        tid = GetLastThreadId();
    }
    Result result = m_pDebugProcess->GetThreadContext(&context, tid, flags);
    if (result.IsSuccess())
    {
        ToGdbRegister(m_pReplyPacket, &context);
    }
    else
    {
        GDB_TRACE_W("%s:%d GetThreadContext failed Failed:0x%04x\n", __func__, __LINE__, (int)result.GetInnerValueForDebug());
        SetReplyString(m_pReplyPacket, g_PacketError_ReadRegisters);
    }
}

void PacketParser::H()
{
    m_pReceivedPacket++;
    switch (m_pReceivedPacket[0])
    {
    case 'g':
        {
            Hg();
        }
        break;
    case 'G':
        {
            HG();
        }
        break;
    case 'c':
        {
            Hc();
        }
        break;
    default:
        break;
    }
}
void PacketParser::m()
{
    m_pReceivedPacket++;
    uint32_t addr;
    uint32_t len;
    char *save;

    addr = HexToInteger(strtok_r(m_pReceivedPacket, ",", &save));
    len = HexToInteger(strtok_r(NULL, ",", &save));

    if (len < sizeof(m_Buffer))
    {
        memset(m_Buffer, -1, sizeof(m_Buffer)); // Default value is -1
        Result result = m_pDebugProcess->ReadMemory(m_Buffer, addr, len);
        if (result.IsFailure())
        {
            // GDB may read an invalid area, or read past the end of valid memory.
            // Return success even on failure because otherwise GDB doesn't deal with it well.
            // Ideally, we should change the length to represent the number of valid bytes
            // but here we don't know - we just get all success or all fail, even when
            // some of the data is OK.
            //SetReplyString(m_pReplyPacket, g_PacketError_ReadMemory);
            Mem2Hex(m_pReplyPacket, m_Buffer, len);
        }
        else
        {
            Mem2Hex(m_pReplyPacket, m_Buffer, len);
        }
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_ReadMemorySize);
    }
}

void PacketParser::M()
{
    m_pReceivedPacket++;
    uint32_t addr;
    uint32_t len;
    char *tok;
    char *save;

    addr = HexToInteger(strtok_r(m_pReceivedPacket, ",:", &save));
    len = HexToInteger(strtok_r(NULL, ",:", &save));
    tok = strtok_r(NULL, ",:", &save);
    GDB_TRACE_P("%s:%d addr=%p len=%p %s\n",__func__,__LINE__,addr,len, tok);

    if (len < sizeof(m_Buffer))
    {
        Hex2Mem(m_Buffer, tok, len);
        Result result = m_pDebugProcess->WriteMemory(m_Buffer, addr, len);
        if (result.IsFailure())
        {
            GDB_TRACE_E("%s:%d WriteMemory addr=%p len=%p Failed:%d\n", __func__, __LINE__, addr, len, result.GetInnerValueForDebug());
            SetReplyString(m_pReplyPacket, g_PacketError_WriteMemory);
        }
        else
        {
            SetReplyStringOK(m_pReplyPacket);
        }
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_WriteMemorySize);
    }
}

void PacketParser::Continue()
{
    GDB_TRACE_P("%s:%d\n",__func__,__LINE__);

    uint32_t tid = m_pDebugProcess->GetTidOverrideStepContinue();
    if (tid == 0 || tid == -1) // 0==any, -1==all. Not sure what step-all should really do.
    {
        tid = GetLastThreadId();
    }

    return Continue(tid);
}

void PacketParser::Continue(uint32_t tid)
{
    GDB_TRACE_P("%s:%d tid=%d\n",__func__,__LINE__,tid);
    Result result;
    if (tid == GetLastThreadId())
    {
        result = m_pDebugProcess->Continue(tid);
    }
    else
    {
        result = m_pDebugProcess->Continue();
    }

    if (result.IsSuccess())
    {
        SetReplyStringOK(m_pReplyPacket);
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_ContinueThread );
    }
}

void PacketParser::Step()
{
    uint32_t tid = m_pDebugProcess->GetTidOverrideStepContinue();
    if (tid == 0 || tid == -1) // 0==any, -1==all. Not sure what step-all should really do.
    {
        tid = GetLastThreadId();
    }
    Step(tid);
}

void PacketParser::Step(uint32_t tid)
{
    if (m_pDebugProcess->Step(tid).IsSuccess())
    {
        Continue(tid);
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_StepThread);
    }
}
void PacketParser::c()
{
    Continue();
}

void PacketParser::k()
{
    m_pDebugProcess->Terminate();
}

void PacketParser::D()
{
    m_pDebugProcess->Detach();
    SetReplyStringOK(m_pReplyPacket);
}


uint32_t PacketParser::GetLastThreadId()
{
    return m_pDebugProcess->GetLastThreadId();
}

uint32_t PacketParser::GetLastSignal()
{
    return m_pDebugProcess->GetLastSignal();
}

void PacketParser::SetLastSignal(uint32_t sig)
{
    m_pDebugProcess->SetLastSignal(sig);
}

void PacketParser::T()
{
    m_pReceivedPacket++;
    uint32_t tid = HexToInteger(m_pReceivedPacket);
    nn::svc::ThreadContext context;
    // 必要?
    if (m_pDebugProcess->GetThreadContext(&context, tid, nn::svc::ContextFlag_Control).IsSuccess())
    {
        SetReplyStringOK(m_pReplyPacket);
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_ThreadDead);
    }
}

void PacketParser::p()
{
    m_pReceivedPacket++;
    int32_t regNum = HexToInteger(m_pReceivedPacket);
    nn::svc::ThreadContext context;
    uint32_t flags = RegNumToContextFlag(regNum);

    uint32_t tid = m_pDebugProcess->GetTidOverride();
    if (tid == 0 || tid == -1) // 0==any, -1==all
    {
        tid = GetLastThreadId();
    }
    if (m_pDebugProcess->GetThreadContext(&context, tid, flags).IsSuccess())
    {
        ToGdbRegister(m_pReplyPacket, &context, regNum);
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_ReadRegisterN);
    }
}

void PacketParser::P()
{
    char* save;
    m_pReceivedPacket++;
    int32_t regNum = HexToInteger(strtok_r(m_pReceivedPacket, "=", &save));
    char *regString = strtok_r(NULL, "", &save);
    nn::svc::ThreadContext context;

    uint32_t flags = RegNumToContextFlag(regNum);

    uint32_t tid = m_pDebugProcess->GetTidOverride();
    if (tid == 0 || tid == -1) // 0==any, -1==all
    {
        tid = GetLastThreadId();
    }
    if (m_pDebugProcess->GetThreadContext(&context, tid, flags).IsSuccess())
    {
        UpdateGdbRegister(&context, regString, regNum);
        if (m_pDebugProcess->SetThreadContext(&context, tid, flags).IsSuccess())
        {
            SetReplyStringOK(m_pReplyPacket);
        }
        else
        {
            SetReplyString(m_pReplyPacket, g_PacketError_WriteRegisterN);
        }
    }
    else
    {
        SetReplyString(m_pReplyPacket, g_PacketError_WriteRegisterGetContext);
    }
}


void PacketParser::Process()
{
    GDB_TRACE_P("%s:%d Command: %s\n", __FILE__, __LINE__, m_pReceivedPacket);
    m_pReplyPacket[0] = '\0';
    switch (m_pReceivedPacket[0])
    {
    case 'v':
        {
            v();
        }
        break;
    case '?':
        {
            LastSignal();
        }
        break;
    case 'q':
        {
            q();
        }
        break;
    case 'p':
        {
            p();
        }
        break;
    case 'P':
        {
            P();
        }
        break;
    case 'g':
        {
            g();
        }
        break;
    case 'G':
        {
            G();
        }
        break;
    case 'Q':
        {
            Q();
        }
        break;
    case 'H':
        {
            H();
        }
        break;
    case 'm':
        {
            m();
        }
        break;
    case 'M':
        {
            M();
        }
        break;
    case 'T':
        {
            T();
        }
        break;
    case 'c':
        {
            c();
        }
        break;
    case 'Z':
        {
            Z();
        }
        break;
    case 'z':
        {
            z();
        }
        break;
    case 'k':
        {
            k();
        }
        break;
    case 'D':
        {
            D();
        }
        break;
    case '!':
        {
            SetReplyStringOK(m_pReplyPacket);
        }
        break;
    default:
        {
            GDB_TRACE_P("%s:%d Not implemented. %s\n",__func__,__LINE__,m_pReceivedPacket);
        }
        break;
    }
} // NOLINT(impl/function_size)


nn::Result ProcessDebugEvent(bool* pReply, char* pReplyBuffer, DebugProcess *pDebugProcess)
{
    *pReply = false;
    nn::svc::DebugEventInfo eventInfo;
    nn::Result result = pDebugProcess->GetProcessDebugEvent(&eventInfo);
    if (result.IsFailure())
    {
        return result;
    }
    uint32_t tid = eventInfo.threadId;
    uint32_t info;
    pDebugProcess->ClearStep();
    switch (eventInfo.event)
    {
    case nn::svc::DebugEvent_Exception:
        {
            switch (eventInfo.info.exception.exceptionCode)
            {
            case nn::svc::DebugException_AttachBreak:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_ATTACH_BREAK tid=%d\n",__func__,__LINE__, tid);
                    info = GDB_SIGNAL_TRAP;
                }
                break;
            case nn::svc::DebugException_BreakPoint:
                {
                    // ON data breakpoint, "exceptionAddress" == the data location.
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_BREAKPOINT tid=%d, exceptionAddress:0x%llx, Type:%s\n",__func__,__LINE__, tid,
                                eventInfo.info.exception.exceptionAddress,
                                eventInfo.info.exception.detail.breakPoint.type == nn::svc::BreakPointType_HardwareInstruction ? "Instr" : "Data");
                    info = GDB_SIGNAL_TRAP;
                    if (eventInfo.info.exception.detail.breakPoint.type == nn::svc::BreakPointType_HardwareInstruction)
                    {
                        SetReplyString(pReplyBuffer, "T%02xthread:%x;hwbreak:;", info, tid);
                    }
                    else
                    {
                        bool readAccess = false;
                        bool writeAccess = false;
                        char const *watchType = "watch"; // Write access
                        if (pDebugProcess->GetWatchPointInfo(eventInfo.info.exception.exceptionAddress, readAccess, writeAccess).IsSuccess())
                        {
                            if (readAccess && writeAccess)
                            {
                                watchType = "awatch"; // read or write access
                            }
                            else if (readAccess)
                            {
                                watchType = "rwatch"; // read access
                            }
                        }
                        else
                        {
                            GDB_TRACE_E("%s:%d nn::svc::DEBUG_EXCEPTION_BREAKPOINT tid=%d, exceptionAddress:0x%llx, Type:%s. Could not find watch point\n",__func__,__LINE__, tid,
                                eventInfo.info.exception.exceptionAddress,
                                eventInfo.info.exception.detail.breakPoint.type == nn::svc::BreakPointType_HardwareInstruction ? "Instr" : "Data");
                        }
                        SetReplyString(pReplyBuffer, "T%02xthread:%x;%s:%x;", info, tid, watchType, eventInfo.info.exception.exceptionAddress);
                    }
                    *pReply = true;
                }
                break;
            case nn::svc::DebugException_UserBreak:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_USER_BREAK tid=%d, ExCode:0x%x, Address:0x%llx, reason:%d, data:0x%llx, size:0x%llx\n",__func__,__LINE__,
                                tid,
                                eventInfo.info.exception.exceptionCode,
                                eventInfo.info.exception.exceptionAddress,
                                eventInfo.info.exception.detail.userBreak.reason,
                                eventInfo.info.exception.detail.userBreak.data,
                                eventInfo.info.exception.detail.userBreak.size);

                    nn::svc::ThreadContext context;
                    if (pDebugProcess->GetThreadContext(&context, tid, nn::svc::ContextFlag_Control).IsSuccess())
                    {
                        uint32_t newCode = 0;

                        if (pDebugProcess->ReadMemory(&newCode, context.pc, 4).IsSuccess())
                        {
                            // swi #NN_SVC_ID_BREAK 命令かどうかをチェックする。
                            // PCを書き換えられたあとにブレークイベントが来た場合は、Continueする。
                            if (newCode != (pDebugProcess->Is64Bit()?(0xd4000001 | (NN_SVC_ID_BREAK << 5)):(0xef000000 | NN_SVC_ID_BREAK)))
                            {
                                pDebugProcess->Continue();
                                GDB_TRACE_E("%s:%d ProcessDebugEvent return = 0x%04x\n",__func__,__LINE__, (int)result.GetInnerValueForDebug());
                                return result;
                            }
                        }
                    }

                    info = GDB_SIGNAL_TRAP;
                }
                break;
            case nn::svc::DebugException_DebuggerBreak:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_DEBUGGER_BREAK tid=%d, lastTid=%d\n",__func__,__LINE__, tid, pDebugProcess->GetLastThreadId());
                    info = GDB_SIGNAL_INT;
                    tid = pDebugProcess->GetLastThreadId(); // revert back to the last TID
                    pDebugProcess->SetLastThreadId(tid);
                }
                break;
            case nn::svc::DebugException_UndefinedInstruction:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_UNDEFINED_INSTRUCTION tid=%d, PC:0x%llx, Instruction:0x%08x\n",__func__,__LINE__, tid, eventInfo.info.exception.exceptionAddress, eventInfo.info.exception.detail.undefinedInstruction.code);
                    info = GDB_SIGNAL_ILL;
                    nn::svc::ThreadContext context;
                    uint32_t newCode = 0;
                    uintptr_t addr = eventInfo.info.exception.exceptionAddress;
                    uint32_t const code = eventInfo.info.exception.detail.undefinedInstruction.code;
                    if (pDebugProcess->GetThreadContext(&context, tid, nn::svc::ContextFlag_Control).IsSuccess())
                    {
                        bool isChanged = false;
                        // ブレークポイントが書き戻された後に例外イベントが来ることがある。
                        // PCが変更されていた場合はリトライする。
                        if (context.PSTATE & HW_PSR_THUMB_STATE)
                        {
                            addr &= ~1;
                            if (pDebugProcess->ReadMemory(&newCode, addr, 2).IsSuccess())
                            {
                                GDB_TRACE_P("%s:%d Thumb tid=%d code=0x%08x\n",__func__,__LINE__,tid, code);
                                switch ((newCode >> 11) & 0x1F)
                                {
                                case 0x1D:
                                    NN_FALL_THROUGH;
                                case 0x1E:
                                    NN_FALL_THROUGH;
                                case 0x1F:
                                    {
                                        if (pDebugProcess->ReadMemory(reinterpret_cast<char*>(&newCode) + 2, addr + 2, 2).IsSuccess())
                                        {
                                            isChanged = (newCode != code);
                                        }
                                    }
                                    break;
                                default:
                                    {
                                        isChanged = (newCode != code);
                                    }
                                    break;
                                }
                            }
                            if ((code & T16_BKPT_MASK ) == T16_BKPT)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                            else if ((code & SDK_T16_BKPT_MASK ) == SDK_T16_BKPT)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                        }
                        else
                        {
                            if (pDebugProcess->ReadMemory(&newCode, addr, 4).IsSuccess())
                            {
                                isChanged = (newCode != code);
                            }
                            if (code == ARM_BKPT || code == SDK_BKPT)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                            else if ((code & A64_BRK_MASK  ) == A64_BRK)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                            else if ((code & A64_HLT_MASK  ) == A64_HLT)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                            else if ((code & A32_BKPT_MASK ) == A32_BKPT)
                            {
                                info = GDB_SIGNAL_TRAP;
                            }
                        }
                        if (isChanged)
                        {
                            GDB_TRACE_P("%s:%d Code is changed tid=%d code=%08x newCode=%08x\n",__func__,__LINE__,tid, code, newCode);
                        }
                    }
                    if (info == GDB_SIGNAL_ILL)
                    {
                        GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_UNDEFINED_INSTRUCTION tid=%d\n",__func__,__LINE__, tid);
                        GDB_TRACE_P("%s:%d exceptionAddress=%p\n",__func__,__LINE__, addr);
                        GDB_TRACE_P("%s:%d code=%p\n",__func__,__LINE__, code);
                    }
                    else if (info == GDB_SIGNAL_TRAP && code != SDK_BKPT)
                    {
                        GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_UNDEFINED_INSTRUCTION tid=%d -- non-SDK Breakpoint instruction\n",__func__,__LINE__, tid);
                        GDB_TRACE_P("%s:%d exceptionAddress=%p\n",__func__,__LINE__, addr);
                        GDB_TRACE_P("%s:%d code=%p\n",__func__,__LINE__, code);
                    }
                    if (info == GDB_SIGNAL_TRAP)
                    {
                        SetReplyString(pReplyBuffer, "T%02xthread:%x;swbreak:;", info, tid);
                        *pReply = true;
                    }
                    pDebugProcess->ClearStep();
                }
                break;
            case nn::svc::DebugException_AccessViolationInstruction:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_ACCESS_VIOLATION_INSTRUCT tid=%d\n",__func__,__LINE__, tid);
                    info = GDB_SIGNAL_SEGV;
                }
                break;
            case nn::svc::DebugException_AccessViolationData:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_ACCESS_VIOLATION_DATA tid=%d\n",__func__,__LINE__, tid);
                    info = GDB_SIGNAL_SEGV;
                }
                break;
            case nn::svc::DebugException_DataTypeMissaligned:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_DATATYPE_MISSALIGNED tid=%d\n",__func__,__LINE__, tid);
                    info = GDB_SIGNAL_BUS;
                }
                break;
            case nn::svc::DebugException_UndefinedSystemCall:
                {
                    GDB_TRACE_P("%s:%d nn::svc::DEBUG_EXCEPTION_UNDEFINED_SYSTEMCALL tid=%d\n",__func__,__LINE__, tid);
                    info = GDB_SIGNAL_SYS;
                }
                break;
            default:
                {
                    GDB_TRACE_W("%s:%d unknown exceptionCode(%d) tid=%d\n",__func__,__LINE__, eventInfo.info.exception.exceptionCode, tid);
                    info = GDB_SIGNAL_SEGV;
                }
                break;
            }
            if (!*pReply)
            {
                SetReplyString(pReplyBuffer, "T%02xthread:%x;", info, tid);
                *pReply = true;
            }
            pDebugProcess->SetLastThreadId(tid);
            pDebugProcess->SetLastSignal(info);
        }
        break;
    case nn::svc::DebugEvent_CreateThread:
        GDB_TRACE_E("%s: CreateThread%d\n",__func__,__LINE__);
        {
            if (!pDebugProcess->IsValid())
            {
                SetReplyString(pReplyBuffer, "W00");
                *pReply = true;
            }
            else
            {
                // TODO: Correct response to GDB indicating thread exit
                GDB_TRACE_P("%s: CreateThread%d\n",__func__,__LINE__);
                pDebugProcess->Continue();
            }
        }
        break;
    case nn::svc::DebugEvent_ExitThread:
        GDB_TRACE_E("%s: ExitThread%d\n",__func__,__LINE__);
        {
            if (!pDebugProcess->IsValid())
            {
                SetReplyString(pReplyBuffer, "W00");
                *pReply = true;
            }
            else
            {
                // TODO: Correct response to GDB indicating thread exit
                GDB_TRACE_P("%s: ExitThread%d\n",__func__,__LINE__);
                pDebugProcess->Continue();
            }
        }
        break;
    case nn::svc::DebugEvent_ExitProcess:
        {
            GDB_TRACE_P("%s:%d ExitProcess\n",__func__,__LINE__);
            if (eventInfo.info.exitProcess.exitReason == nn::svc::ProcessExitReason_ExitProcess)
            {
                SetReplyString(pReplyBuffer, "W00");
            }
            else
            {
                SetReplyString(pReplyBuffer, "X%02x", GDB_SIGNAL_KILL);
            }
            pDebugProcess->Detach();
            *pReply = true;
        }
        break;
    default:
        {
            GDB_TRACE_W("%s:%d Unknown DebugEvent tid=%d %d\n",__func__,__LINE__, tid, eventInfo.event);
            pDebugProcess->Continue();
        }
        break;
    }
    return result;
}   // NOLINT [readability/fn_size]

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
uint32_t PacketParser::RegNumToContextFlag(int32_t regNum)
{
    uint32_t flags = 0;
    if (0 <= regNum && regNum <= 12)
    {
        flags = nn::svc::ContextFlag_General;
    }
    else if (13 <= regNum && regNum <= 15)
    {
        flags = nn::svc::ContextFlag_Control;
    }
    else if (regNum == 25)
    {
        flags = nn::svc::ContextFlag_Control;
    }
    else if (26 <= regNum && regNum <= 57)
    {
        flags = nn::svc::ContextFlag_Fpu;
    }
    else if (regNum == 58)
    {
        flags = nn::svc::ContextFlag_FpuControl;
    }

    return flags;
}

void PacketParser::ToGdbRegister(char* pBuffer, nn::svc::ThreadContext* pContext)
{
    for (int i = 0; i < 17; i++)
    {
        union
        {
            uint32_t w;
            uint8_t b[4];
        } u;
        if (0 <= i && i <= 12)
        {
            u.w = pContext->r[i];
        }
        else if (i == 13)
        {
            u.w = pContext->sp;
        }
        else if (i == 14)
        {
            u.w = pContext->lr;
        }
        else if (i == 15)
        {
            u.w = pContext->pc;
        }
        else if (i == 16)
        {
            u.w = pContext->cpsr;
        }
        nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
        pBuffer += std::strlen(pBuffer);
    }

    for (int i = 0; i < 32; i++)
    {
        union
        {
            uint64_t dw;
            uint8_t b[8];
        } u;
        u.dw = pContext->fpuRegisters[i];
        nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x%02x%02x%02x%02x",
                u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7]);
        pBuffer += std::strlen(pBuffer);
    }
    {
        union
        {
            uint32_t w;
            uint8_t b[4];
        } u;
        u.w = pContext->fpscr;
        nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
        pBuffer += std::strlen(pBuffer);
    }
}

void PacketParser::ToGdbRegister(char* pBuffer, nn::svc::ThreadContext* pContext, int32_t regNum)
{
    size_t regSize = 0;
    union
    {
        uint32_t w;
        uint64_t dw;
        uint8_t b[8];
    } u;

    if (0 <= regNum && regNum <= 12)
    {
        u.w = pContext->r[regNum];
        regSize = 4;
    }
    else if (regNum == 13)
    {
        u.w = pContext->sp;
        regSize = 4;
    }
    else if (regNum == 14)
    {
        u.w = pContext->lr;
        regSize = 4;
    }
    else if (regNum == 15)
    {
        u.w = pContext->pc;
        regSize = 4;
    }
    else if (regNum == 25)
    {
        u.w = pContext->cpsr;
        regSize = 4;
    }
    else if (26 <= regNum && regNum <= 57)
    {
        u.dw = pContext->fpuRegisters[regNum - 26];
        regSize = 8;
    }
    else if (regNum == 58)
    {
        u.w = pContext->fpscr;
        regSize = 4;
    }
    switch (regSize)
    {
    case 4:
        {
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
        }
        break;
    case 8:
        {
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7]);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void PacketParser::UpdateGdbRegister(nn::svc::ThreadContext* pContext, char* pString)
{
    for (int i = 0; i < 17; i++)
    {
        union
        {
            uint32_t w;
            uint8_t b[4];
        } u;
        for (int j = 0; j < 4; j++)
        {
            int h = HexToInteger(*pString++);
            if (h < 0)
            {
                return;
            }
            int l = HexToInteger(*pString++);
            if (l < 0)
            {
                return;
            }
            u.b[j] = (h << 4) | l;
        }
        if (0 <= i && i < 13)
        {
            pContext->r[i] = u.w;
        }
        else if (i == 13)
        {
            pContext->sp = u.w;
        }
        else if (i == 14)
        {
            pContext->lr = u.w;
        }
        else if (i == 15)
        {
            pContext->pc = u.w;
        }
        else if (i == 16)
        {
            pContext->cpsr = u.w;
        }
    }

    for (int i = 0; i < 32; i++)
    {
        union
        {
            uint64_t dw;
            uint8_t b[8];
        } u;
        for (int j = 0; j < 8; j++)
        {
            int h = HexToInteger(*pString++);
            if (h < 0)
            {
                return;
            }
            int l = HexToInteger(*pString++);
            if (l < 0)
            {
                return;
            }
            u.b[j] = (h << 4) | l;
        }
        pContext->fpuRegisters[i] = u.dw;
    }
    {
        union
        {
            uint32_t w;
            uint8_t b[4];
        } u;
        for (int j = 0; j < 4; j++)
        {
            int h = HexToInteger(*pString++);
            if (h < 0)
            {
                return;
            }
            int l = HexToInteger(*pString++);
            if (l < 0)
            {
                return;
            }
            u.b[j] = (h << 4) | l;
        }
        pContext->fpscr = u.w;
    }
}

void PacketParser::UpdateGdbRegister(nn::svc::ThreadContext* pContext, char* pString, int32_t regNum)
{
    size_t regSize = 0;
    void* regPtr = NULL;

    if (0 <= regNum && regNum <= 12)
    {
        regPtr = &pContext->r[regNum];
        regSize = 4;
    }
    else if (regNum == 13)
    {
        regPtr = &pContext->sp;
        regSize = 4;
    }
    else if (regNum == 14)
    {
        regPtr = &pContext->lr;
        regSize = 4;
    }
    else if (regNum == 15)
    {
        regPtr = &pContext->pc;
        regSize = 4;
    }
    else if (regNum == 25)
    {
        regPtr = &pContext->cpsr;
        regSize = 4;
    }
    else if (26 <= regNum && regNum <= 57)
    {
        regPtr = &pContext->fpuRegisters[regNum - 26];
        regSize = 8;
    }
    else if (regNum == 58)
    {
        regPtr = &pContext->fpscr;
        regSize = 4;
    }
    switch (regSize)
    {
    case 4:
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            *reinterpret_cast<uint32_t*>(regPtr) = u.w;
        }
        break;
    case 8:
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            for (int j = 0; j < 8; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            *(reinterpret_cast<uint32_t*>(regPtr) + 0) = (u.dw & 0xFFFFFFFF);
            *(reinterpret_cast<uint32_t*>(regPtr) + 1) = (u.dw >> 32);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#elif defined(NN_BUILD_CONFIG_CPU_ARM_V8A)

namespace {
int32_t AArch32ToAArch64(int32_t regNum)
{
    if (0 <= regNum && regNum <= 14)
    {
        return regNum;
    }
    else if (regNum == 15)
    {
        return 32;
    }
    else if (regNum == 25)
    {
        return 33;
    }
    else if (26 <= regNum && regNum <= 57)
    {
        return regNum + (34 - 26);
    }
    else if (regNum == 58)
    {
        return 66;
    }
    NN_ABORT("Unknown reg num %d\n", regNum);
}
int32_t AArch64ToAArch32(int32_t regNum)
{
    if (0 <= regNum && regNum <= 14)
    {
        return regNum;
    }
    else if (regNum == 32)
    {
        return 15;
    }
    else if (regNum == 33)
    {
        return 25;
    }
    else if (34 <= regNum && regNum <= 65)
    {
        return regNum + (26 - 34);
    }
    else if (regNum == 66)
    {
        return 58;
    }
    NN_ABORT("Unknown reg num %d\n", regNum);
}
}
uint32_t PacketParser::RegNumToContextFlag(int32_t regNum)
{
    uint32_t flags = 0;

    if (!m_pDebugProcess->Is64Bit())
    {
        regNum = AArch32ToAArch64(regNum);
    }

    if (0 <= regNum && regNum < 29)
    {
        flags = nn::svc::ContextFlag_General;
    }
    else if (29 <= regNum && regNum <= 33)
    {
        flags = nn::svc::ContextFlag_Control;
    }
    else if (34 <= regNum && regNum <= 65)
    {
        flags = nn::svc::ContextFlag_Fpu;
    }
    else if (regNum == 66)
    {
        flags = nn::svc::ContextFlag_FpuControl;
    }

    return flags;
}

void PacketParser::ToGdbRegister(char* pBuffer, nn::svc::ThreadContext* pContext)
{
    if (m_pDebugProcess->Is64Bit())
    {
        for (int i = 0; i < 34; i++)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            if (0 <= i && i < 29)
            {
                u.dw = pContext->r[i];
            }
            else if (i == 29)
            {
                u.dw = pContext->fp;
            }
            else if (i == 30)
            {
                u.dw = pContext->lr;
            }
            else if (i == 31)
            {
                u.dw = pContext->sp;
            }
            else if (i == 32)
            {
                u.dw = pContext->pc;
            }
            else if (i == 33)
            {
                u.dw = pContext->pstate;
            }
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7]);
            pBuffer += std::strlen(pBuffer);
        }

        for (int i = 0; i < 32; i++)
        {
            union
            {
                Bit128 qw;
                uint8_t b[16];
            } u;
            u.qw = pContext->v[i];
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,
                    "%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7],
                    u.b[8], u.b[9], u.b[10], u.b[11], u.b[12], u.b[13], u.b[14], u.b[15]);
            pBuffer += std::strlen(pBuffer);
        }
        for (int i = 0; i < 2; i++)
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            if (i == 0)
            {
                u.w = pContext->fpsr;
            }
            else if (i == 1)
            {
                u.w = pContext->fpcr;
            }
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
            pBuffer += std::strlen(pBuffer);
        }
    }
    else
    {
        for (int i = 0; i < 17; i++)
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            if (0 <= i && i <= 14)
            {
                u.w = pContext->r[i];
            }
            else if (i == 15)
            {
                u.w = pContext->pc;
            }
            else if (i == 16)
            {
                u.w = pContext->pstate;
            }
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
            pBuffer += std::strlen(pBuffer);
        }

        for (int i = 0; i < 32; i++)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            union
            {
                uint64_t dw[2];
                Bit128 qw;
            } q;

            q.qw = pContext->v[i / 2];
            u.dw = q.dw[i % 2];
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7]);
            pBuffer += std::strlen(pBuffer);
        }
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            uint32_t fpscr = (pContext->fpsr & 0xF80000FF) | (pContext->fpcr & 0x07FFFF00);
            u.w = fpscr;
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
            pBuffer += std::strlen(pBuffer);
        }
    }
}   // NOLINT [readability/fn_size]

void PacketParser::ToGdbRegister(char* pBuffer, nn::svc::ThreadContext* pContext, int32_t regNum)
{
    size_t regSize = 0;
    union
    {
        uint32_t w;
        uint64_t dw;
        Bit128 qw;
        uint8_t b[16];
    } u;

    if (m_pDebugProcess->Is64Bit())
    {
        if (0 <= regNum && regNum < 29)
        {
            u.dw = pContext->r[regNum];
            regSize = 8;
        }
        else if (regNum == 29)
        {
            u.dw = pContext->fp;
            regSize = 8;
        }
        else if (regNum == 30)
        {
            u.dw = pContext->lr;
            regSize = 8;
        }
        else if (regNum == 31)
        {
            u.dw = pContext->sp;
            regSize = 8;
        }
        else if (regNum == 32)
        {
            u.dw = pContext->pc;
            regSize = 8;
        }
        else if (regNum == 33)
        {
            u.dw = pContext->pstate;
            regSize = 8;
        }
        else if (34 <= regNum && regNum < 66)
        {
            u.qw = pContext->v[regNum - 34];
            regSize = 16;
        }
        else if (66 <= regNum && regNum < 68)
        {
            regSize = 4;
        }
    }
    else
    {
        if (0 <= regNum && regNum <= 14)
        {
            u.w = pContext->r[regNum];
            regSize = 4;
        }
        else if (regNum == 15)
        {
            u.w = pContext->pc;
            regSize = 4;
        }
        else if (regNum == 25)
        {
            u.w = pContext->pstate;
            regSize = 4;
        }
        else if (26 <= regNum && regNum <= 57)
        {
            union
            {
                uint64_t dw[2];
                Bit128 qw;
            } q;

            q.qw = pContext->v[(regNum - 26) / 2];
            u.dw = q.dw[(regNum - 26) % 2];
            regSize = 8;
        }
        else if (regNum == 58)
        {
            uint32_t fpscr = (pContext->fpsr & 0xF80000FF) | (pContext->fpcr & 0x07FFFF00);
            u.w = fpscr;
            regSize = 4;
        }
    }

    switch (regSize)
    {
    case 4:
        {
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x", u.b[0], u.b[1], u.b[2], u.b[3]);
        }
        break;
    case 8:
        {
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,"%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7]);
        }
        break;
    case 16:
        {
            nn::util::SNPrintf(pBuffer, BUFFER_SIZE,
                    "%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
                    u.b[0], u.b[1], u.b[2], u.b[3], u.b[4], u.b[5], u.b[6], u.b[7],
                    u.b[8], u.b[9], u.b[10], u.b[11], u.b[12], u.b[13], u.b[14], u.b[15]);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}   // NOLINT [readability/fn_size]

void PacketParser::UpdateGdbRegister(nn::svc::ThreadContext* pContext, char* pString)
{
    if (m_pDebugProcess->Is64Bit())
    {
        for (int i = 0; i < 34; i++)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            for (int j = 0; j < 8; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (0 <= i && i < 29)
            {
                pContext->r[i] = u.dw;
            }
            else if (i == 29)
            {
                pContext->fp = u.dw;
            }
            else if (i == 30)
            {
                pContext->lr = u.dw;
            }
            else if (i == 31)
            {
                pContext->sp = u.dw;
            }
            else if (i == 32)
            {
                pContext->pc = u.dw;
            }
            else if (i == 33)
            {
                pContext->pstate = u.dw;
            }
        }
        for (int i = 0; i < 32; i++)
        {
            union
            {
                Bit128 qw;
                uint8_t b[16];
            } u;
            for (int j = 0; j < 16; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            pContext->v[i] = u.qw;
        }
        for (int i = 0; i < 2; i++)
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (i == 0)
            {
                pContext->fpsr = u.w;
            }
            else if (i == 1)
            {
                pContext->fpcr = u.w;
            }
        }
    }
    else
    {
        for (int i = 0; i < 17; i++)
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (0 <= i && i <= 14)
            {
                pContext->r[i] = u.w;
            }
            else if (i == 15)
            {
                pContext->pc = u.w;
            }
            else if (i == 16)
            {
                pContext->pstate = u.w;
            }
        }

        for (int i = 0; i < 32; i++)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            for (int j = 0; j < 8; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            union
            {
                uint64_t dw[2];
                Bit128 qw;
            } q;

            q.qw = pContext->v[i / 2];
            q.dw[i % 2] = u.dw;
            pContext->v[i / 2] = q.qw;
        }
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }

            pContext->fpsr = u.w & 0xF80000FF;
            pContext->fpcr = u.w & 0x07FFFF00;
        }
    }
}   // NOLINT [readability/fn_size]

void PacketParser::UpdateGdbRegister(nn::svc::ThreadContext* pContext, char* pString, int32_t regNum)
{
    if (m_pDebugProcess->Is64Bit())
    {
        if (0 <= regNum && regNum < 34)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            for (int j = 0; j < 8; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (0 <= regNum && regNum < 29)
            {
                pContext->r[regNum] = u.dw;
            }
            else if (regNum == 29)
            {
                pContext->fp = u.dw;
            }
            else if (regNum == 30)
            {
                pContext->lr = u.dw;
            }
            else if (regNum == 31)
            {
                pContext->sp = u.dw;
            }
            else if (regNum == 32)
            {
                pContext->pc = u.dw;
            }
            else if (regNum == 33)
            {
                pContext->pstate = u.dw;
            }
        }
        else if (34 <= regNum && regNum < 66)
        {
            union
            {
                Bit128 qw;
                uint8_t b[16];
            } u;
            for (int j = 0; j < 16; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            pContext->v[regNum - 34] = u.qw;
        }
        else if (66 <= regNum && regNum < 68)
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (regNum == 66)
            {
                pContext->fpsr = u.w;
            }
            else if (regNum == 67)
            {
                pContext->fpcr = u.w;
            }
        }
    }
    else
    {
        if ((0 <= regNum && regNum < 16) || (regNum == 25) || (regNum == 58))
        {
            union
            {
                uint32_t w;
                uint8_t b[4];
            } u;
            for (int j = 0; j < 4; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            if (0 <= regNum && regNum <= 14)
            {
                pContext->r[regNum] = u.w;
            }
            else if (regNum == 15)
            {
                pContext->pc = u.w;
            }
            else if (regNum == 25)
            {
                pContext->pstate = u.w;
            }
            else if (regNum == 58)
            {
                pContext->fpsr = u.w & 0xF80000FF;
                pContext->fpcr = u.w & 0x07FFFF00;
            }
        }
        else if (26 <= regNum && regNum < 58)
        {
            union
            {
                uint64_t dw;
                uint8_t b[8];
            } u;
            for (int j = 0; j < 8; j++)
            {
                int h = HexToInteger(*pString++);
                if (h < 0)
                {
                    return;
                }
                int l = HexToInteger(*pString++);
                if (l < 0)
                {
                    return;
                }
                u.b[j] = (h << 4) | l;
            }
            union
            {
                uint64_t dw[2];
                Bit128 qw;
            } q;

            q.qw = pContext->v[(regNum - 26) / 2];
            q.dw[(regNum - 26) % 2] = u.dw;
            pContext->v[(regNum - 26) / 2] = q.qw;
        }
    }
} // NOLINT [readability/fn_size]

#endif
}}}
