﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include "dmnt_GdbServer.h"
#include "dmnt_Rsp.h"
#include "dmnt_DebugMonitor.h"
#include "dmnt_DebugIO.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Synchronization.h>
#include "gdbserver_log.h"

namespace nn { namespace dmnt { namespace gdbserver {

namespace
{
const int GDB_SESSION_NUM = 4;
template <size_t align, typename T>
    constexpr T RoundUp(T x)
    {
        return static_cast<T>((x + (align - 1)) & ~(align - 1));
    }


nn::os::MessageQueueType g_WorkerQueue;
nn::os::MessageQueueType g_WorkerThread1Queue;
nn::os::MessageQueueType g_WorkerThread2Queue;
nn::os::ThreadType g_WorkerThread1[GDB_SESSION_NUM];
nn::os::ThreadType g_WorkerThread2[GDB_SESSION_NUM];

void WorkerThread1(void *arg);
void WorkerThread2(void *arg);


class Worker
{
private:
    class InternalConditionVariable;
    class InternalMutex
    {
        friend InternalConditionVariable;
    public:
        void Initialize()
        {
            nn::os::InitializeMutex(&m_Mutex, false, 0);
        }
        void Finalize()
        {
            nn::os::FinalizeMutex(&m_Mutex);
        }
        void lock()
        {
            nn::os::LockMutex(&m_Mutex);
        }
        void unlock()
        {
            nn::os::UnlockMutex(&m_Mutex);
        }
    private:
        nn::os::MutexType m_Mutex;
    };
    class InternalConditionVariable
    {
    public:
        void Initialize()
        {
            nn::os::InitializeConditionVariable(&m_ConditionVariable);
        }
        void Finalize()
        {
            nn::os::FinalizeConditionVariable(&m_ConditionVariable);
        }
        void Signal()
        {
            nn::os::SignalConditionVariable(&m_ConditionVariable);
        }
        void Broadcast()
        {
            nn::os::BroadcastConditionVariable(&m_ConditionVariable);
        }
        void Wait(InternalMutex* lock)
        {
            nn::os::WaitConditionVariable(&m_ConditionVariable, &lock->m_Mutex);
        }
    private:
        nn::os::ConditionVariableType m_ConditionVariable;
    };

    bool                        m_IsValid;
    bool                        m_ProcessDebugEnd;
    io::DebugSession*           m_pSession;

    InternalMutex               m_Mutex;
    InternalConditionVariable   m_ConditionVariable;
    io::DebugServerSession      m_ServerSession;
    rsp::RspPacket              m_RspPacket;
    dmnt::DebugProcess          m_DebugProcess;
    os::SystemEventType         m_EndEvent;

private:
    void SendPacket(bool* pBreak, const char* pSendBuffer);
    char* ReceivePacket(bool* pBreak, char* pBuffer, size_t size);
    bool IsValid();
    void Start();
public:
    void Initialize(nn::svc::Handle* pOut, int port);
    void Finalize();
    Result Attach(Bit64 processId);
    Result Attach(nn::svc::Handle handle, const char* name);
    void ProcessDebugEvent();
    void ProcessSession();
};


void Worker::Initialize(nn::svc::Handle* pOut, int port)
{
    m_IsValid = true;
    m_ProcessDebugEnd = false;
    m_pSession = NULL;

    os::CreateSystemEvent(&m_EndEvent, os::EventClearMode_ManualClear, true);
    os::NativeHandle handle = os::GetReadableHandleOfSystemEvent(&m_EndEvent);
    *pOut = svc::Handle(handle);

    m_Mutex.Initialize();
    m_ConditionVariable.Initialize();
    m_ServerSession.Initialize(port);
    m_RspPacket.Initialize();
    m_DebugProcess.Initialize();
}

Result Worker::Attach(Bit64 processId)
{
    Result result = m_DebugProcess.Attach(processId);
    if (result.IsSuccess())
    {
        Start();
    }
    else
    {
        GDB_TRACE_E("%s: Attach error pid=%d result=%x\n", __func__,processId, *reinterpret_cast<uint32_t*>(&result));
    }
    return result;
}
Result Worker::Attach(nn::svc::Handle handle, const char* name)
{
     Result result = m_DebugProcess.Attach(handle, name);
    if (result.IsSuccess())
    {
        Start();
    }
    return result;
}


void Worker::Start()
{
    SendMessageQueue(&g_WorkerThread1Queue, reinterpret_cast<uintptr_t>(this));
}

bool Worker::IsValid()
{
    return m_DebugProcess.IsValid() && (m_pSession? m_pSession->IsValid(): true);
}

void Worker::SendPacket(bool* pBreak, const char* pSendBuffer)
{
    if (m_pSession)
    {
        m_RspPacket.SendPacket(pBreak, pSendBuffer, m_pSession);
        m_ConditionVariable.Broadcast();
    }
}

char* Worker::ReceivePacket(bool* pBreak, char* pBuffer, size_t size)
{
    char* packet = NULL;
    if (m_pSession)
    {
        m_Mutex.unlock();
        packet = m_RspPacket.ReceivePacket(pBreak, pBuffer, size, m_pSession);
        m_Mutex.lock();
    }
    return packet;
}

void Worker::ProcessSession()
{
    io::DebugSession session(m_ServerSession.GetListenFd());
    // セッション確立直後にclose
    m_ServerSession.Finalize();

    m_Mutex.lock();
    m_pSession = &session;

    SendMessageQueue(&g_WorkerThread2Queue, reinterpret_cast<uintptr_t>(this));

    while (IsValid())
    {
        bool breaked = false;
        char recvBuffer[nn::dmnt::rsp::BUFFER_SIZE];
        char* packet = ReceivePacket(&breaked, recvBuffer, sizeof(recvBuffer));

        if (!breaked && packet)
        {
            char sendBuffer[nn::dmnt::rsp::BUFFER_SIZE];
            nn::dmnt::rsp::PacketParser parser(packet, sendBuffer, &m_DebugProcess);
            parser.Process();
            SendPacket(&breaked, sendBuffer);
            if (parser.IsSetNoAck())
            {
                GDB_TRACE_N("%s:%d Setting NoAckMode\n",__func__,__LINE__);
                m_RspPacket.SetNoAck();
            }
        }
        if (breaked)
        {
            m_DebugProcess.Break();
        }
    }
    m_DebugProcess.Terminate(); // Kill the debugged process in case remote debugger detaches because there is no way to reattach later.
    m_ConditionVariable.Broadcast();
    m_pSession = NULL;
    m_Mutex.unlock();
}

void Worker::ProcessDebugEvent()
{
    m_Mutex.lock();

    while (IsValid())
    {
        while (IsValid())
        {
            if (m_DebugProcess.GetStatus() != RUN)
            {
                m_ConditionVariable.Wait(&m_Mutex);
            }
            else
            {
                m_Mutex.unlock();
                int32_t index;
                nn::svc::Handle handle = m_DebugProcess.GetHandle();
                nn::svc::WaitSynchronization(&index, &handle, 1, nn::svc::WAIT_INFINITE);
                m_Mutex.lock();
                break;
            }
        }

        bool reply;
        char sendBuffer[nn::dmnt::rsp::BUFFER_SIZE];
        while (IsValid() && nn::dmnt::rsp::ProcessDebugEvent(&reply, sendBuffer, &m_DebugProcess).IsSuccess())
        {
            if (reply)
            {
                bool breaked;
                SendPacket(&breaked, sendBuffer);
                if (breaked)
                {
                    m_DebugProcess.Break();
                }
            }
        }
    }
    m_ProcessDebugEnd = true;
    m_ConditionVariable.Broadcast();
    m_Mutex.unlock();
}


void Worker::Finalize()
{
    m_Mutex.lock();
    m_DebugProcess.Detach();
    while (!m_ProcessDebugEnd)
    {
        m_ConditionVariable.Wait(&m_Mutex);
    }
    m_Mutex.unlock();

    m_Mutex.Finalize();
    m_ConditionVariable.Finalize();

    os::SignalSystemEvent(&m_EndEvent);
    os::DestroySystemEvent(&m_EndEvent);
}


void WorkerThread1(void *arg)
{
    for (;;)
    {
        uintptr_t p;
        ReceiveMessageQueue(&p, &g_WorkerThread1Queue);
        if (p == 0)
        {
            break;
        }
        reinterpret_cast<Worker*>(p)->ProcessSession();
        reinterpret_cast<Worker*>(p)->Finalize();
        SendMessageQueue(&g_WorkerQueue, p);
    }
}

void WorkerThread2(void *arg)
{
    for (;;)
    {
        uintptr_t p;
        ReceiveMessageQueue(&p, &g_WorkerThread2Queue);
        if (p == 0)
        {
            break;
        }
        reinterpret_cast<Worker*>(p)->ProcessDebugEvent();
    }
}
}

void Initialize()
{
    const size_t WorkerStackSize = RoundUp<nn::os::StackRegionAlignment>(nn::dmnt::rsp::BUFFER_SIZE * 4 + nn::os::StackRegionAlignment);

    static Worker s_Worker[GDB_SESSION_NUM];
    static uintptr_t s_WorkerQueueBuffer[GDB_SESSION_NUM];
    static uintptr_t s_WorkerThread1QueueBuffer[GDB_SESSION_NUM];
    static uintptr_t s_WorkerThread2QueueBuffer[GDB_SESSION_NUM];

    NN_ALIGNAS(nn::os::StackRegionAlignment) static char s_WorkerThread1Stack[GDB_SESSION_NUM][WorkerStackSize];
    NN_ALIGNAS(nn::os::StackRegionAlignment) static char s_WorkerThread2Stack[GDB_SESSION_NUM][WorkerStackSize];

    nn::os::InitializeMessageQueue(&g_WorkerQueue, s_WorkerQueueBuffer, sizeof(s_WorkerQueueBuffer) / sizeof(*s_WorkerQueueBuffer));
    nn::os::InitializeMessageQueue(&g_WorkerThread1Queue, s_WorkerThread1QueueBuffer, sizeof(s_WorkerThread1QueueBuffer) / sizeof(*s_WorkerThread1QueueBuffer));
    nn::os::InitializeMessageQueue(&g_WorkerThread2Queue, s_WorkerThread2QueueBuffer, sizeof(s_WorkerThread2QueueBuffer) / sizeof(*s_WorkerThread2QueueBuffer));

    for (int i = 0; i < sizeof(s_Worker) / sizeof(*s_Worker); i++)
    {
        SendMessageQueue(&g_WorkerQueue, reinterpret_cast<uintptr_t>(&s_Worker[i]));
    }

    for (int i = 0; i < sizeof(g_WorkerThread1) / sizeof(*g_WorkerThread1); i++)
    {
        nn::os::CreateThread(&g_WorkerThread1[i], &WorkerThread1, 0, reinterpret_cast<void*>(s_WorkerThread1Stack[i]), WorkerStackSize, nn::os::HighestThreadPriority - 1);
        nn::os::StartThread(&g_WorkerThread1[i]);
    }
    for (int i = 0; i < sizeof(g_WorkerThread2) / sizeof(*g_WorkerThread2); i++)
    {
        nn::os::CreateThread(&g_WorkerThread2[i], &WorkerThread2, 0, reinterpret_cast<void*>(s_WorkerThread2Stack[i]), WorkerStackSize, nn::os::HighestThreadPriority - 1);
        nn::os::StartThread(&g_WorkerThread2[i]);
    }
}

void AttachProcess(nn::svc::Handle* pOut, Bit64 processId, int port)
{
    uintptr_t p;
    ReceiveMessageQueue(&p, &g_WorkerQueue);
    reinterpret_cast<Worker *>(p)->Initialize(pOut, port);
    reinterpret_cast<Worker *>(p)->Attach(processId);
}

void AttachProcess(nn::svc::Handle* pOut, nn::svc::Handle handle, int port, const char* name)
{
    uintptr_t p;
    ReceiveMessageQueue(&p, &g_WorkerQueue);
    reinterpret_cast<Worker *>(p)->Initialize(pOut, port);
    reinterpret_cast<Worker *>(p)->Attach(handle, name);
}


void Finalize()
{
    for (int i = 0; i < sizeof(g_WorkerThread1) / sizeof(*g_WorkerThread1); i++)
    {
        SendMessageQueue(&g_WorkerThread1Queue, 0);
    }
    for (int i = 0; i < sizeof(g_WorkerThread2) / sizeof(*g_WorkerThread2); i++)
    {
        SendMessageQueue(&g_WorkerThread2Queue, 0);
    }
    for (int i = 0; i < sizeof(g_WorkerThread1) / sizeof(*g_WorkerThread1); i++)
    {
        nn::os::WaitThread(&g_WorkerThread1[i]);
        nn::os::DestroyThread(&g_WorkerThread1[i]);
    }
    for (int i = 0; i < sizeof(g_WorkerThread2) / sizeof(*g_WorkerThread2); i++)
    {
        nn::os::WaitThread(&g_WorkerThread2[i]);
        nn::os::DestroyThread(&g_WorkerThread2[i]);
    }
    nn::os::FinalizeMessageQueue(&g_WorkerQueue);
    nn::os::FinalizeMessageQueue(&g_WorkerThread1Queue);
    nn::os::FinalizeMessageQueue(&g_WorkerThread2Queue);
}


}}}

