﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/os.h>
#include <nn/htcs.h>
#include <nn/util/util_TFormatString.h>

#include "dmnt_DebugIO.h"
#include "dmnt_Allocator.h"
#include "gdbserver_log.h"

namespace {
    void HtcsReceiveThread(void *pDebugSession)
    {
        reinterpret_cast<nn::dmnt::io::DebugSession*>(pDebugSession)->HtcsRecvLoop();
    }
}

namespace nn { namespace dmnt { namespace io {

void DebugServerSession::Initialize(int port)
{
    m_ListenFd = nn::htcs::Socket();
    NN_ABORT_UNLESS(m_ListenFd >= 0);

    nn::htcs::SockAddrHtcs server;
    server.family = nn::htcs::HTCS_AF_HTCS;
    server.peerName = nn::htcs::GetPeerNameAny();
    nn::util::TSNPrintf(server.portName.name, sizeof(server.portName.name), "DebugMonitor0_%d", port);

    NN_ABORT_UNLESS(nn::htcs::Bind(m_ListenFd, &server) == 0);
    NN_ABORT_UNLESS(nn::htcs::Listen(m_ListenFd, 1) == 0);
}

void DebugServerSession::Finalize()
{
    nn::htcs::Close(m_ListenFd);
}

DebugSession::DebugSession(int fd): m_Valid(false)
{
    m_ClientFd = nn::htcs::Accept(fd, nullptr);
    if (m_ClientFd >= 0)
    {
        GDB_TRACE_S("[dmnt0] Htcs debug session accept OK.\n");

        m_Valid = true;

        // htcs 受信スレッド開始
        const size_t StackSize = 4096;
        m_pReceiveThreadStack = nn::dmnt::alloc::AllocateStack(StackSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_ReceiveThread, HtcsReceiveThread, this,
            m_pReceiveThreadStack, StackSize, nn::os::HighestThreadPriority - 1));
        nn::os::StartThread(&m_ReceiveThread);
    }
    else
    {
        GDB_TRACE_E("[dmnt0] Htcs debug session accept error. (%d)\n", nn::htcs::GetLastError());
    }
}

DebugSession::~DebugSession()
{
    if (m_ClientFd != -1)
    {
        GDB_TRACE_S("[dmnt0] Htcs debug session close.\n");

        m_HtcsReceiveBuffer.Invalidate();
        // htcs 受信スレッド終了
        nn::htcs::Shutdown(m_ClientFd, 2);
        nn::os::WaitThread(&m_ReceiveThread);
        nn::os::DestroyThread(&m_ReceiveThread);
        nn::htcs::Close(m_ClientFd);
        nn::dmnt::alloc::Deallocate(m_pReceiveThreadStack);
    }
}

int DebugSession::RecvPoll(int timeout)
{
    if (timeout == -1)
    {
        return m_HtcsReceiveBuffer.WaitToBeReadable();
    }
    else
    {
        return m_HtcsReceiveBuffer.WaitToBeReadable(nn::TimeSpan::FromMilliSeconds(timeout));
    }
}

int DebugSession::GetDebugChar()
{
    char c;
    m_HtcsReceiveBuffer.WaitToBeReadable();
    int result = m_HtcsReceiveBuffer.Read(&c, sizeof(uint8_t));
    if (result > 0)
    {
        return c;
    }
    else
    {
        return -1;
    }
}

int DebugSession::PutDebugChar(int c)
{
    char b = c;
    int result = nn::htcs::Send(m_ClientFd, &b, 1, 0);
    if (result < 0)
    {
        m_Valid = false;
    }
    return result;
}

int DebugSession::PutDebugString(const char* pString)
{
    size_t len = std::strlen(pString);
    size_t left = len;
    while (left > 0)
    {
        int result = nn::htcs::Send(m_ClientFd, pString, left, 0);
        if (result >= 0)
        {
            left -= result;
            pString += result;
        }
        else
        {
            m_Valid = false;
            return result;
        }
    }
    return len;
}

// htcs 経由でデータを受信して、HtcsReceiveBuffer に格納していく
void DebugSession::HtcsRecvLoop()
{
    uint8_t buffer [HtcsReceiveBufferSize];

    for (;;)
    {
        int result = nn::htcs::Recv(m_ClientFd, buffer, sizeof(buffer), 0);
        if (result > 0)
        {
            m_HtcsReceiveBuffer.WaitToBeWritable();
            m_HtcsReceiveBuffer.Write(buffer, result);
        }
        else
        {
            if (nn::htcs::GetLastError() != nn::htcs::HTCS_EAGAIN)
            {
                m_Valid = false;
                break;
            }
        }
    }
    m_HtcsReceiveBuffer.Invalidate();
}

}}}
