﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_Vibration.h>
#include "xcd_VibrationAmFmEncoder.h"
#include "xcd_VibrationAmFmPack.h"
#include "xcd_VibrationOnConnect.h"

namespace nn { namespace xcd {

//!< コントローラに送信する AMFM 符号の個数を決定するクラス
class VibrationCountDecider final
{
    NN_DISALLOW_COPY(VibrationCountDecider);
    NN_DISALLOW_MOVE(VibrationCountDecider);

public:
    VibrationCountDecider() NN_NOEXCEPT : m_PrevCount(0) {}
    ~VibrationCountDecider() NN_NOEXCEPT {}
    void Reset() NN_NOEXCEPT;                       //!< 初期化します
    int DecideNextCount(const VibratorBufferStatus& bufferStatus) NN_NOEXCEPT;  //!< 次に送るべき個数を決定します
    void SetPrevCount(int prevCount) NN_NOEXCEPT;   //!< 実際に送った個数をセットします

public:
    static const int BufferLength = 4;  //!< 過去何回分の時間間隔を参照するか

private:
    //!< 個数決定時に用いる設定値
    struct Config
    {
        int defaultCount;   //!< デフォルトの個数
        int dataLevelMin;   //!< remainingDataLevel がこの値を下回ったら個数を増やそうとします
        int dataLevelMax;   //!< remainingDataLevel がこの値を上回ったら個数を減らそうとします
    };

    static const int CountMin = 0;      //!< 個数の最小値
    static const int CountMax = 3;      //!< 個数の最大値
    static const Config s_Config5ms;    //!< 通信間隔 = 5ms の際に用いる設定値
    static const Config s_Config10ms;   //!< 通信間隔 = 10ms の際に用いる設定値
    static const Config s_Config15ms;   //!< 通信間隔 = 15ms の際に用いる設定値
    const Config& GetConfig() NN_NOEXCEPT;  //!< 現在の通信間隔に合わせて適切な設定値を取得します

    int m_BufferPosition;                       //!< リングバッファ内の現在の位置
    nn::os::Tick m_TickBuffer[BufferLength];    //!< 過去のチック値を保持するリングバッファ
    int m_PrevCount;    //!< 前回実際に送った個数
};


//!< 上位ライブラリから送られた振動値を一時格納するリング状のキュー
class VibrationQueue final
{
    NN_DISALLOW_COPY(VibrationQueue);
    NN_DISALLOW_MOVE(VibrationQueue);

public:
    static const int QueueLength = 6;   //!< キューのサイズ

    VibrationQueue() NN_NOEXCEPT : m_EnqueueIdx(0), m_DequeueIdx(0) {}
    ~VibrationQueue() NN_NOEXCEPT {}
    void Clear() NN_NOEXCEPT;                               //!< キューをクリアします
    bool IsFull() const NN_NOEXCEPT;                        //!< キューが満杯か取得します
    bool IsEmpty() const NN_NOEXCEPT;                       //!< キューが空か取得します
    int GetCount() const NN_NOEXCEPT;                       //!< キューに格納されている振動値の個数を取得します
    void Enqueue(const VibrationValue& value) NN_NOEXCEPT;  //!< キュー末尾に振動値を追加します。キューが満杯の場合は末尾のキューを置き換えます。
    const VibrationValue& Dequeue() NN_NOEXCEPT;            //!< キュー先頭の振動値を取り出します。キューが空の場合は前回と同じ値を返します。

private:
    VibrationValue m_Queue[QueueLength];    //!< キューの実体
    int m_EnqueueIdx;                       //!< キュー末尾のインデックス
    int m_DequeueIdx;                       //!< キュー先頭のインデックス
};


//!< 個々の振動子デバイスの内部状態を扱うクラス
class VibratorAgent final
{
public:
    static const float BaseFrequencyLow;    //!< 低帯域のベース周波数 (単位は Hz)
    static const float BaseFrequencyHigh;   //!< 高帯域のベース周波数 (単位は Hz)

public:
    VibratorAgent() NN_NOEXCEPT :
        m_IsActivated(false),
        m_IsAmplitudeLimitEnabled(true),
        m_IsEqualizerEnabled(true),
        m_IsAmFm7bitCodeAvailable(true),
        m_Mutex(true),
        m_pVibrationOnConnect(nullptr)
    {
        Reset();
    }
    ~VibratorAgent() NN_NOEXCEPT {}

    void Activate(DeviceType type) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;

    void ParseInputReport(uint8_t report) NN_NOEXCEPT;
    int GetOutputReport(uint8_t* pOutValue) NN_NOEXCEPT;

    void GetBufferStatus(VibratorBufferStatus* pOutValue) NN_NOEXCEPT;
    void GetRequiredVibrationValueCount(int* pOutValue) NN_NOEXCEPT;
    Result SendVibrationValue(const VibrationValue& value) NN_NOEXCEPT;
    void GetActualVibrationValue(VibrationValue* pOutValue) NN_NOEXCEPT;
    void SetCalibrationValue(const VibratorCalibrationValue& value) NN_NOEXCEPT;
    void GetCalibrationValue(VibratorCalibrationValue* pOutValue) NN_NOEXCEPT;
    void SetAmplitudeLimitEnabled(bool isEnabled) NN_NOEXCEPT;
    bool IsAmplitudeLimitEnabled() NN_NOEXCEPT;
    void SetEqualizerEnabled(bool isEnabled) NN_NOEXCEPT;
    bool IsEqualizerEnabled() NN_NOEXCEPT;
    void SetAmFm7bitCodeAvailable(bool isAvailable) NN_NOEXCEPT;
    bool IsAmFm7bitCodeAvailable() NN_NOEXCEPT;

    void SetVibrationOnConnect(VibrationOnConnect* pVibrationOnConnect) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_pVibrationOnConnect = pVibrationOnConnect;
    }

private:
    static const float AmplitudeLimitCoeff; //!< 振幅に制限をかける際の定数
    static const uint8_t BufferUnderrunMask = static_cast<uint8_t>(0x08);       //!< bufferUnderrun ビットを取り出すためのマスク
    static const uint8_t RemainingDataLevelMask = static_cast<uint8_t>(0x07);   //!< remainingDataLevel 値を取り出すためのマスク

    //!< 内部キューから AMFM 符号ペアを取り出して、次に送信する 32 ビットのデータ (PackedAmFmCodes) を生成します
    VibrationAmFmPack GeneratePackedAmFmCodes(int codePairCount) NN_NOEXCEPT;

    //!< 振幅値に制限をかけます
    void LimitAmplitude(float* pAmplitudeLow, float* pAmplitudeHigh) NN_NOEXCEPT;

    //!< 周波数に応じた調整係数を取得します
    float GetEqualizerCoeff(float frequency) NN_NOEXCEPT;

    //!< 周波数に応じた調整係数を振幅値に適用します
    void ApplyEqualizer(float* pAmplitudeLow, float* pAmplitudeHigh, float frequencyLow, float frequencyHigh) NN_NOEXCEPT;

    //!< 周波数に応じた調整係数を振幅値に適用する以前の状態に戻します
    void RevertEqualizer(float* pAmplitudeLow, float* pAmplitudeHigh, float frequencyLow, float frequencyHigh) NN_NOEXCEPT;

    //!< マスターボリュームを振幅値に適用します
    void ApplyMasterVolume(float* pAmplitudeLow, float* pAmplitudeHigh, float masterVolume) NN_NOEXCEPT;

    bool m_IsActivated;
    bool m_IsAmplitudeLimitEnabled;
    bool m_IsEqualizerEnabled;
    bool m_IsAmFm7bitCodeAvailable;
    VibrationCountDecider m_CountDecider;
    VibrationQueue m_Queue;
    VibrationAmFmEncoderContext m_EncoderContextLow;    //!< 低帯域用のエンコーダー
    VibrationAmFmEncoderContext m_EncoderContextHigh;   //!< 高帯域用のエンコーダー
    VibratorBufferStatus m_BufferStatus;                //!< バッファ状態の保存先
    nn::os::Mutex m_Mutex;

    DeviceType m_Type;
    VibrationOnConnect* m_pVibrationOnConnect;
};

}} // namespace nn::xcd
