﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include "xcd_VibrationOnConnect.h"
#include "xcd_VibratorBase.h"

namespace nn { namespace xcd {

namespace {

    //!< 振動値
    const VibrationValue ZeroVibration = { 0.0f, 160.0f, 0.0f, 320.0f };
    const VibrationValue FirstVibration = { 1.0f, 160.0f, 0.0f, 320.0f };
    const VibrationValue SecondVibration = { 0.0f, 160.0f, 0.8f, 320.0f };

    //!< 振動の時間間隔 (単位は MilliSecond)
    const int SamplingInterval = 5;
    const int IntervalBeforeFirst = 30;
    const int FirstDuration = 5;
    const int IntervalBetweenFirstAndSecond = 120;
    const int SecondDuration = 5;
    const int IntervalAfterSecond = 840;

    //!< 振動値を切り替えるタイミング
    const int FirstStartPosition = IntervalBeforeFirst / SamplingInterval;
    const int FirstEndPosition = (IntervalBeforeFirst + FirstDuration) / SamplingInterval;
    const int SecondStartPosition = (IntervalBeforeFirst + FirstDuration + IntervalBetweenFirstAndSecond) / SamplingInterval;
    const int SecondEndPosition = (IntervalBeforeFirst + FirstDuration + IntervalBetweenFirstAndSecond + SecondDuration) / SamplingInterval;
    const int TotalEndPosition = (IntervalBeforeFirst + FirstDuration + IntervalBetweenFirstAndSecond + SecondDuration + IntervalAfterSecond) / SamplingInterval;
}

void VibrationOnConnect::StartPlaying(VibrationPatternOnConnect pattern) NN_NOEXCEPT
{
    m_Pattern = pattern;
    m_Position = 0;
    m_Length = (pattern == VibrationPatternOnConnect_None) ? 0 : TotalEndPosition;
}

VibrationValue VibrationOnConnect::GetNextVibration() NN_NOEXCEPT
{
    VibrationValue v = ZeroVibration;

    if (m_IsDevicePowerOn && (m_Position < m_Length))
    {
        if (m_Pattern == VibrationPatternOnConnect_JoyCon || m_Pattern == VibrationPatternOnConnect_FullConLeft)
        {
            // FirstVibration を必要に応じて生成
            if (FirstStartPosition <= m_Position && m_Position < FirstEndPosition)
            {
                v = FirstVibration;
            }
        }

        if (m_Pattern == VibrationPatternOnConnect_JoyCon || m_Pattern == VibrationPatternOnConnect_FullConRight)
        {
            // SecondVibration を必要に応じて生成
            if (SecondStartPosition <= m_Position && m_Position < SecondEndPosition)
            {
                v = SecondVibration;
            }
        }

        m_Position++;

        // 再生終了時には VibratorBase に通知
        if (m_Position == m_Length)
        {
            m_pVibratorBase->NotifyVibrationOnConnectFinished();
        }
    }

    return v;
}

bool VibrationOnConnect::IsPlaying() NN_NOEXCEPT
{
    return (m_Position < m_Length);
}

void VibrationOnConnect::SetDevicePowerStatus(bool isDevicePowerOn) NN_NOEXCEPT
{
    m_IsDevicePowerOn = isDevicePowerOn;
}

}} // namespace nn::xcd
