﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_Vibration.h>

#include "detail/xcd_LockableMutexType.h"

namespace {

//!< 振動のマスターボリュームの初期値
const float DefaultMasterVolume = 1.0f;

//!< 振動のマスターボリュームの最小値
const float MasterVolumeMin = 0.0f;

//!< 振動のマスターボリュームの最大値
const float MasterVolumeMax = 1.0f;

//!< 振動のマスターボリューム操作時に利用するミューテックス
::nn::xcd::detail::LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

//!< 振動のマスターボリュームを格納する変数
float s_MasterVolume = DefaultMasterVolume;

}

namespace nn { namespace xcd {

Result SetVibrationMasterVolume(float masterVolume) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        MasterVolumeMin <= masterVolume && masterVolume <= MasterVolumeMax,
        ::nn::xcd::ResultVibrationMasterVolumeOutOfRange());

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);
    s_MasterVolume = masterVolume;

    NN_RESULT_SUCCESS;
}

Result SetAndSaveVibrationMasterVolume(float masterVolume) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        MasterVolumeMin <= masterVolume && masterVolume <= MasterVolumeMax,
        ::nn::xcd::ResultVibrationMasterVolumeOutOfRange());

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);
    s_MasterVolume = masterVolume;
    ::nn::settings::system::SetVibrationMasterVolume(masterVolume);

    NN_RESULT_SUCCESS;
}

Result GetVibrationMasterVolume(float* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);
    *pOutValue = s_MasterVolume;

    NN_RESULT_SUCCESS;
}


Result LoadAndGetVibrationMasterVolume(float* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);
    float masterVolume = ::nn::settings::system::GetVibrationMasterVolume();

    NN_RESULT_THROW_UNLESS(
        MasterVolumeMin <= masterVolume && masterVolume <= MasterVolumeMax,
        ::nn::xcd::ResultVibrationMasterVolumeOutOfRange());
    s_MasterVolume = masterVolume;
    *pOutValue = masterVolume;

    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
