﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_Result.h>
#include "xcd_CommandHandler.h"
#include "detail/xcd_TeraCommon.h"
#include "detail/xcd_TimeCounter.h"

namespace nn { namespace xcd {

/**
 * @brief   Tera MCU の状態を管理するクラス
 */
class TeraStateMachine final : public ICommandListener
{
    NN_DISALLOW_COPY(TeraStateMachine);
    NN_DISALLOW_MOVE(TeraStateMachine);

public:
    /**
     * @brief   ステート遷移時に呼び出すコールバック関数の型
     */
    typedef void(*StateTransitionCallbackType)(detail::InternalMcuState, ICommandListener*);

    /**
     * @brief   MCU リセットのタイムアウト時に呼び出すコールバック関数の型
     */
    typedef void(*ResetTimeoutCallbackType)(ICommandListener*);

public:
    TeraStateMachine() NN_NOEXCEPT
        : m_Mutex(true)
        , m_Type(DeviceType_Unknown)
        , m_pCommandHandler(nullptr)
        , m_WaitStateTransitTimer()
        , m_IsStateSetRequested(false)
        , m_IsTransitionWithoutMcuInput(false)
        , m_IsMcuWriteSent(false)
        , m_McuPowerStatus(McuPowerStatus::Disabled)
        , m_CurrentMcuState(detail::InternalMcuState_Standby)
        , m_DestinationMcuState(detail::InternalMcuState_Nop)
        , m_StateTransitionCallbackFunction(nullptr)
        , m_pStateTransitionCallbackListener(nullptr)
        , m_ResetTimeoutCallbackFunction(nullptr)
        , m_pResetTimeoutCallbackListener(nullptr)
    {}

    /**
     * @brief   MCU ステートの管理を開始
     */
    void Activate(
        DeviceType type,
        CommandHandler* pCommandHandler,
        StateTransitionCallbackType callback,
        ICommandListener* pCallbackListener) NN_NOEXCEPT;

    /**
     * @brief   MCU ステートの管理を終了
     */
    void Deactivate() NN_NOEXCEPT;

    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   次に遷移する MCU ステートを設定
     */
    nn::Result SetDestinationState(McuState state) NN_NOEXCEPT;

    /**
     * @brief   次に遷移する MCU ステートを設定。ステート遷移処理判定を MCU Input でなく MCU Read で行う
     */
    nn::Result SetDestinationStateWithoutMcuInput(McuState state) NN_NOEXCEPT;

    /**
     * @brief   FW アップデート向けのステート遷移指定
     */
    nn::Result SetDestinationStateForFirmwareUpdate(McuState state) NN_NOEXCEPT;

    /**
     * @brief   MCU ステート遷移を中断
     */
    void AbortStateTransition() NN_NOEXCEPT;

    /**
     * @brief   MCU をリセットする
     */
    nn::Result Reset(ResetTimeoutCallbackType callback, ICommandListener* pCallbackListener) NN_NOEXCEPT;

    /**
     * @brief   現在の MCU ステートを取得
     */
    McuState GetCurrentState() NN_NOEXCEPT;

    /**
     * @brief   MCU ステートが適切か判定
     */
    nn::Result CheckMcuState(McuState expectedState) NN_NOEXCEPT;

    /**
     * @brief   MCU ステート情報の更新 (MCU Input)
     */
    void UpdateState(detail::InternalMcuState currentState) NN_NOEXCEPT;

    /**
     * @brief   MCU ステート情報の更新 (MCU Read)
     */
    void UpdateStateByMcuRead(const uint8_t* pBuffer, size_t size, detail::InternalMcuState currentState) NN_NOEXCEPT;

    /**
     * @brief   MCU が初期化中か
     */
    bool IsMcuInitializing() const NN_NOEXCEPT;

    /**
     * @brief   MCU がコマンド受付可能か
     */
    bool IsMcuReady() const NN_NOEXCEPT;

    /**
     * @brief   システムブート状態か判定
     */
    bool IsSystemBoot() const NN_NOEXCEPT;

    /**
     * @brief   MCU Input を使用せずにステート遷移判定を行うか
     */
    bool IsTransitionWithoutMcuInput() const NN_NOEXCEPT;

    /**
     * @brief   ステート遷移処理中か判定
     */
    bool IsProcessing() const NN_NOEXCEPT;

    /**
     * @brief   MCU Write を送信済みか
     */
    bool IsMcuWriteSent() const NN_NOEXCEPT;

    /**
     * @brief   内部状態をクリアします
     */
    void ClearInternalState() NN_NOEXCEPT;

private:
    /**
     * @brief   Tera の起動状態
     */
    enum class McuPowerStatus
    {
        Disabled,               //!< 停止
        DisableRequested,       //!< 停止要求発行済み
        Enabled,                //!< 通常起動
        EnableRequested,        //!< 起動要求発行中
        SystemBooted,           //!< システムブート (FW アップデート用)
        SystemBootRequested     //!< システムブート要求発行中
    };

private:
    /**
     * @brief   Tera MCU の起動/停止通知の確認
     */
    void CheckMcuResumeNotify() NN_NOEXCEPT;

    /**
     * @brief   MCU Resume 受信に伴う起動状態の更新
     */
    void UpdateMcuPowerStatusOnMcuResume(bool* pOutIsCallbackNeeded) NN_NOEXCEPT;

    /**
     * @brief   MCU ステート遷移コマンドを発行
     */
    void SendStateSetCommand(detail::InternalMcuState state) NN_NOEXCEPT;

    /**
     * @brief   MCU の ON/OFF 要求を発行
     */
    void RequestMcuResume(McuResumeValueType resumeType) NN_NOEXCEPT;

    /**
     * @brief   コールバック関数を呼び出してステート遷移を通知する
     */
    void NotifyStateTransition() NN_NOEXCEPT;

    /**
     * @brief   MCU の内部状態を設定
     */
    void UpdateCurrentMcuState(detail::InternalMcuState state) NN_NOEXCEPT;

    /**
     * @brief   ステート遷移完了判定
     */
    void CheckStateTransitionComplete(detail::InternalMcuState currentState) NN_NOEXCEPT;

    /**
     * @brief   リセットのタイムアウト判定
     */
    void CheckResetTimeout() NN_NOEXCEPT;

    /**
     * @brief   ステート遷移のタイムアウト判定
     */
    void CheckStateTransitionTimeout() NN_NOEXCEPT;

    /**
     * @brief   ステート遷移タイムアウト判定を停止
     */
    void StopStateTransitionTimeout() NN_NOEXCEPT;

    /**
     * @brief   ステート遷移処理の実行
     */
    void ProcessStateTransition() NN_NOEXCEPT;

    /**
     * @brief   遷移先ステート設定の実処理
     */
    nn::Result SetDestinationStateImpl(McuState state) NN_NOEXCEPT;

private:
    mutable nn::os::Mutex       m_Mutex;                            //!< メンバへのアクセスを排他するための Mutex

    DeviceType                  m_Type;                             //!< デバイスの種類
    CommandHandler*             m_pCommandHandler;                  //!< HID コマンド制御のためのハンドラ
    detail::TimeCounter         m_WaitStateTransitTimer;            //!< ステート遷移のタイムアウトを検出するためのタイマー
    bool                        m_IsStateSetRequested;              //!< ステート遷移要求済みフラグ
    bool                        m_IsTransitionWithoutMcuInput;      //!< ステート遷移判定を MCU Input ではなく HID コマンドで行う
    bool                        m_IsMcuWriteSent;                   //!< 直近の処理で MCU Write を送信したかどうか (HID コマンドの過剰送信抑止に使用)
    McuPowerStatus              m_McuPowerStatus;                   //!< TeraMCU の現在の電源状態
    detail::InternalMcuState    m_CurrentMcuState;                  //!< 前回の通信タイミングで TeraMCU から取得したモード
    detail::InternalMcuState    m_DestinationMcuState;              //!< 次回の通信タイミングで TeraMCU に設定するモード
    StateTransitionCallbackType m_StateTransitionCallbackFunction;  //!< ステート遷移コールバック
    ICommandListener*           m_pStateTransitionCallbackListener; //!< ステート遷移コールバック関数に渡される ICommandListener
    ResetTimeoutCallbackType    m_ResetTimeoutCallbackFunction;     //!< リセットタイムアウト通知コールバック
    ICommandListener*           m_pResetTimeoutCallbackListener;    //!< リセットタイムアウト通知コールバック関数に渡される ICommandListener
};

}} // namespace nn::xcd
