﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Tick.h>

#include "xcd_CommandHandler.h"
#include "xcd_ICommandListener.h"

namespace nn { namespace xcd {

enum SerialFlashAccessState
{
    SerialFlashAccessState_Idle,         //!< シリアルフラッシュへアクセスを行っていない状態
    SerialFlashAccessState_OnRead,       //!< Read を行っている状態
    SerialFlashAccessState_OnWrite,      //!< Write を行っている状態
    SerialFlashAccessState_Completed,    //!< 操作が完了し、結果取得待ちをしている状態
};

//!< シリアルフラッシュへのアクセスを行うための
class SerialFlashAccessor final : public ICommandListener
{

private:
    //!< CommandHandler
    CommandHandler* m_pCommand;

    //!< Serial Flash のアクセス状態
    SerialFlashAccessState m_State;

    //!< Serial Flash への操作が完了した際に通知するイベント
    nn::os::SystemEventType* m_pSystemEvent;

    //!< アプリからアクセス要求アドレス
    int m_Address;

    //!< アプリからアクセス要求サイズ
    int m_Size;

    //!< 現在制御中のアドレスオフセット
    int m_AddressOffset;

    //!< 現在制御中のサイズ
    int m_SizeUnit;

    //!< アプリから受け取った読み込みバッファ
    uint8_t* m_pReadBuffer;

    //!< アプリから受け取った書き込みバッファ
    const uint8_t* m_pWriteBuffer;

    //!< 操作に対する結果
    Result m_Result;

public:
    SerialFlashAccessor() NN_NOEXCEPT;
    virtual ~SerialFlashAccessor() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスが接続されたときに呼ばれる関数
    void Activate() NN_NOEXCEPT;

    //!< デバイスが切断された時に呼ばれる関数
    void Deactivate() NN_NOEXCEPT;

    //!< コマンドハンドラーをセットする
    void SetCommandHandler(CommandHandler* pCommand) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスから読み込みを行います
    Result Read(
        const uint32_t address,
        uint8_t* pOutBuffer,
        int size,
        nn::os::SystemEventType* m_pEvent) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスへの書き込みを行います
    Result Write(
        const uint32_t address,
        const uint8_t* pBuffer,
        int size,
        nn::os::SystemEventType* m_pEvent) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作結果を取得します
    Result GetResult() NN_NOEXCEPT;

    //!< HidCommandに対して任意の SerialFlash 領域の読み出しが完了した際にに通知を受けるための関数です
    virtual void NotifySerialFlashRead(const uint8_t* pBuffer, int size) NN_NOEXCEPT NN_OVERRIDE;

    //!< Ack の受領を受け取るための通知関数
    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT NN_OVERRIDE;

private:
    void ProceedRead() NN_NOEXCEPT;

    void ProceedWrite() NN_NOEXCEPT;

    void OperationComplete(Result result) NN_NOEXCEPT;
};

}} // namespace nn::xcd
