﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/xcd/xcd_BleTypes.h>
#include "xcd_BleReportTypes.h"
#include "xcd_IBleCommandListener.h"
#include "detail/xcd_BleGattClientTypes.h"
#include "detail/xcd_IBleNhogClient.h"

namespace nn { namespace xcd {

//!< Palma フォーマットの Command Handler
class PalmaCommandHandler final : public IBleCommandListener
{
    NN_DISALLOW_COPY(PalmaCommandHandler);
    NN_DISALLOW_MOVE(PalmaCommandHandler);

private:
    //!< コマンド送信のためのバッファ
    uint8_t m_Buffer[detail::GattOperationPayloadMaxLength];

    //!< 分割送信のためのバッファ
    uint8_t m_BufferForFragmentation[detail::GattOperationPayloadMaxLength];

    //!< Nintendo Hid Over Gatt クライアント
    detail::IBleNhogClient* m_pClient;

    //!< コマンドリスナー
    IBleCommandListener* m_pListener;

    //!< Ack ハンドリングのためのオペレーション種別
    BleDeviceOperationType m_OperationTypeForAck;

    //!< OutputCommand のデスクリプタ
    BleOutputCommandDescriptor m_OutputCommandDescriptor;

    //!< 対象の Database Entry の情報
    BleDeviceDatabaseEntryConfig m_DatabaseEntryConfig;

    //!< Database Entry の書き込み中かどうか
    bool m_IsWritingDatabaseEntry;

    //!< コマンドの送受信中でビジーかどうか
    bool m_IsBusy;

public:
    PalmaCommandHandler() NN_NOEXCEPT;
    virtual ~PalmaCommandHandler() NN_NOEXCEPT;

    //!< Characteristic Write が完了した際の通知を受け取るための関数です
    virtual void CharacteristicWriteComplete(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT NN_OVERRIDE;

    //!< サポートするフォーマットバージョンかどうか
    bool IsSupportedVersion(BleOutputCommandFormatVersion commandVer, BleCommandResponseFormatVersion responseVer) NN_NOEXCEPT;

    //!< コマンドのやり取りを行う NHOG サービスのクライアントをセットする
    void SetGattClient(detail::IBleNhogClient* pClient) NN_NOEXCEPT;

    //!< コマンドのリスナーをセットする
    void SetListener(IBleCommandListener* pListener) NN_NOEXCEPT;

    //!< コマンドハンドラを終了する
    void Close() NN_NOEXCEPT;

    //!< コマンドの応答をハンドリングします
    Result HandleResponseReceivedEvent(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< Activity を再生する
    Result PlayActivity(uint16_t slotNumber) NN_NOEXCEPT;

    //!< ProductType を取得するコマンドを発行する
    Result GetProductType() NN_NOEXCEPT;

    //!< ModelInformation を取得するコマンドを発行する
    Result GetModelInformation() NN_NOEXCEPT;

    //!< Sensor Cal を取得するコマンドを発行する
    Result GetSensorCalibrationValue() NN_NOEXCEPT;

    //!< AnalogStick Cal を取得するコマンドを発行する
    Result GetAnalogStickCalibrationValue() NN_NOEXCEPT;

    //!< UniqueCode を取得するコマンドを発行する
    Result GetContentUniqueCode() NN_NOEXCEPT;

    //!< データベース識別バージョンを設定するコマンドを発行する
    Result SetDatabaseIdentificationVersion(int32_t version) NN_NOEXCEPT;

    //!< データベース識別バージョンを取得するコマンドを発行する
    Result GetDatabaseIdentificationVersion() NN_NOEXCEPT;

    //!< Database Entry の書き込みを行う
    Result WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< FrModeType を設定するコマンドを発行する
    Result SetFrModeType(uint32_t frModeType) NN_NOEXCEPT;

    //!< StepCount を0で初期化するコマンドを発行する
    Result ResetStepCount() NN_NOEXCEPT;

    //!< StepCount を取得するコマンドを発行する
    Result GetStepCount() NN_NOEXCEPT;

    //!< Step 機能を設定するコマンドを発行する
    Result EnableStepCounter(bool isEnabled) NN_NOEXCEPT;

    //!< Application セクションの読み出しを行う
    Result ReadApplicationSection(int32_t address, int32_t readSize) NN_NOEXCEPT;

    //!< Application セクションへ書き込みを行う
    Result WriteApplicationSection(int32_t address, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< ContentUniqueCode の有効/無効設定を取得する
    Result GetContentState() NN_NOEXCEPT;

    //!< ContentUniqueCode の有効/無効設定を行う
    Result SetContentState(bool isEnabled) NN_NOEXCEPT;

    //!< FeatureSet の機能の一時停止を行う
    Result SuspendFeature(uint32_t featureFlagSet) NN_NOEXCEPT;

    //!< PlayLog を取得するコマンドを発行する
    Result ReadPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< PlayLog を0で初期化するコマンドを発行する
    Result ResetPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< Response のタイムアウトをハンドリングします
    void HandleResponseTimeout() NN_NOEXCEPT;

    //!< Database Entry の書き込みを中断する
    void CancelWriteDatabaseEntry() NN_NOEXCEPT;

private:
    //!< Ack をパースします
    Result ParseAck(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespFactorySetting をパースします
    Result ParseRespFactorySetting(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespSystemSaveData をパースします
    Result ParseRespSystemSaveData(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespRetransmission をパースします
    Result ParseRespRetransmission(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespReadApplicationSaveData をパースします
    Result ParseRespReadApplicationSaveData(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespSuspendFeatureSet をパースします
    Result ParseRespSuspendFeatureSet(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< RespReadPlayLog をパースします
    Result ParseRespPlayLog(const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< EntryId を除いたデータから ProductType をパースします
    void ParseProductType(BleDeviceProductType* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< EntryId を除いたデータから ModelInformation をパースします
    void ParseModelInformation(BleDeviceModelInformation* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< EntryId を除いたデータから Sensor Cal をパースします
    void ParseSensorCalibration(SensorCalibrationValue* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< EntryId を除いたデータから AnalogStick Cal をパースします
    void ParseAnalogStickCalibration(AnalogStickValidRange* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT;

    //!< 指定したシーケンス番号から再送処理を行います
    void HandleRetransmission(uint8_t sequenceNumber) NN_NOEXCEPT;

    //!< 初めに送信するフラグメントを取得します
    size_t GetDatabaseEntryFirstFragment(uint8_t* pOutFragment, BleOutputCommandDescriptor& descriptor, const BleDeviceDatabaseEntryConfig& config) NN_NOEXCEPT;

    //!< パケットの分割を行い、次に送信するフラグメントを取得します
    size_t GetDatabaseEntryFragment(uint8_t* pOutFragment, BleOutputCommandDescriptor& descriptor) NN_NOEXCEPT;

    //!< コマンド操作の完了通知を行います
    void HandleCommandCompleted(uint8_t commandId, uint8_t ackResult) NN_NOEXCEPT;

    //!< コマンドを Gatt サーバに送信します
    Result SendCommand(uint8_t id, const void* buffer, size_t size) NN_NOEXCEPT;
};

}} // namespace nn::xcd
