﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Macro.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/xcd/xcd_Pairing.h>

#include "xcd_BluetoothTypes.h"
#include "xcd_IEventTask.h"
#include "xcd_MultiWaitEvent.h"
#include "xcd_TaskManager.h"

namespace nn { namespace xcd {

//!< ペアリング処理を管理するクラス
class PairingManager : public IEventTask
{
private:
    //!< デバイスが登録された際に通知するイベント
    ::nn::os::SystemEventType* m_pEvent;

    //!< ペアリングされているデバイスのリスト
    RegisteredDeviceInfo m_RegisteredDeviceList[RegisteredDeviceCountMax];

    //!< 登録されているデバイスの数
    int m_RegisteredDeviceCount;

    //!< 新規にデバイスが登録されるたびにインクリメントするカウンター
    int64_t m_RegistrationCount;

    //!< デバイスリスト排他のための mutex
    ::nn::os::Mutex m_Mutex;

    //!< コントローラーのデバイスへの登録待ちリスト
    RegisteredDeviceInfo m_RegistrationPendingQueue[RegisteredDeviceCountMax];

    //!< デバイスへの登録待ちリスト内の数
    int m_PendingDeviceCount;

    //!< Bluetooth ペアリング情報更新待ちを待ち受けるイベント
    ::nn::os::SystemEventType m_PairingUpdateEvent;

    //!< ペアリング情報が更新された場合に通知するためのイベント
    ::nn::os::LightEventType* m_pDatabaseClearedEvent;

public:
    PairingManager() NN_NOEXCEPT;
    virtual ~PairingManager() NN_NOEXCEPT NN_OVERRIDE;

    //!< ペアリングマネージャーをアクティベートします
    void Activate() NN_NOEXCEPT;

    //!< Eventが通知された際に呼ばれる関数
    virtual void EventFunction(const ::nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT NN_OVERRIDE;

    //!< 定常的な処理を行う関数
    virtual void PeriodicEventFunction() NN_NOEXCEPT NN_OVERRIDE;

    //!< 新たにペアリングでデバイスが登録された際にに通知されるイベントを登録します。
    void BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< 新たにペアリングでデバイスが登録された際にに通知されるイベントを解除します。
    void UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

    //!< ペアリングで登録されたデバイスを列挙します
    size_t GetRegisteredDevices(RegisteredDeviceInfo* pOutVales, size_t length) NN_NOEXCEPT;

    //!< デバイスを新規にデータベースに登録します
    void RegisterDevice(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT;

    //!< アドレスから登録情報を取得します
    bool GetRegisteredDeviceInfo(RegisteredDeviceInfo* pOutControllerInfo, ::nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< デバイスがBluetoothペアリング済みかどうか判定します
    bool IsDevicePaired(::nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< デバイスが有線ペアリングする必要があるか判定します
    bool IsWiredPairingRequired(InterfaceType interfaceForPairing, DeviceType deviceType) NN_NOEXCEPT;

    //!< ペアリング情報が全削除された際に通知するためのイベントをセットする
    void SetDatabaseClearedEvent(::nn::os::LightEventType* pEvent) NN_NOEXCEPT;

private:
    //!< ペアリングデータベースとコントローラーデーターベースを同期する
    void SyncWithPairingDatabaseOnBoot() NN_NOEXCEPT;

    //!< ペアリングデータベースとコントローラーデーターベースを同期する
    void SyncWithPairingDatabase() NN_NOEXCEPT;

    //!< 有線ペアリングされた Bluetooth ペアリング情報を登録する
    void RegisteredWiredPairingInfo(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT;

    //!< 指定したデバイスについて、ペアリング情報を確認の上登録可能であれば登録する
    bool TryAppendNxControllerInfo(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT;

    //!< ペンディングキューに登録されているデバイスを Nx Controller のデータベースに登録する
    bool TryAddPendingDeviceToNxControllerInfo() NN_NOEXCEPT;

    //!< Nx Controller の settings の更新を行う
    void UpdateNxControllerDatabaseSettings() NN_NOEXCEPT;

    void AddToPendingQueue(const RegisteredDeviceInfo& deviceInfo) NN_NOEXCEPT;

    //!< 登録待ちリストから削除する
    void RemoveFromPendingQueue(int index) NN_NOEXCEPT;
};

}} // namespace nn::xcd
