﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_IrsensorTypes.h>
#include "xcd_Peripheral.h"
#include "xcd_IrsensorMomentProcessor.h"
#include "xcd_IrsensorClusteringProcessor.h"
#include "xcd_IrsensorImageTransferProcessor.h"
#include "xcd_IrsensorTeraPluginProcessor.h"
#include "xcd_IrsensorDpdProcessor.h"

//#define VERBOSE_IRSENSOR_LOG
/*
 * VERBOSE モード時のみ、引数に指定された文字列を、Irsensor のログとして出力します。
 */
#ifdef VERBOSE_IRSENSOR_LOG

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_XCD_IRSENSOR_LOG(...)             NN_SDK_LOG("[xcd: Irsensor] " ##__VA_ARGS__)
#else
#define NN_XCD_IRSENSOR_LOG(format, ...)     NN_SDK_LOG("[xcd: Irsensor] " format, ##__VA_ARGS__)
#endif

#else // VERBOSE_IRSENSOR_LOG
#define NN_XCD_IRSENSOR_LOG(...)
#endif // VERBOSE_IRSENSOR_LOG

namespace nn { namespace xcd {

const size_t IRSENSOR_PACKET_PAYLOAD_SIZE    = 300; //!< 1パケット内で IR センサが使用可能なデータ領域のサイズ
const size_t IRSENSOR_PACKET_PROTOCOL_OFFSET = 0;   //!< 受信パケット内のMisoProtocol領域のオフセット
const size_t IRSENSOR_PACKET_RESULT_OFFSET   = 1;   //!< 受信パケット内のリザルト領域のオフセット

/**
 * @brief   IR センサへのパケットのステート
 */
enum class IrsensorPacketType : uint8_t
{
    DataRead       = 0x00, //!< データ取得用
    ModeSet        = 0x01, //!< モード設定用
    ModeGet        = 0x02, //!< モード取得用
    RegisterRead   = 0x03, //!< レジスタ読み込み用
    RegisterWrite  = 0x04, //!< レジスタ書き込み用
    ModeSetEx      = 0x05, //!< 高速モード設定用
    RegisterWriteEx = 0x06, //!< レジスタ高速書き込み用
    None,                  //!< パケットは送らない
};

/**
 * @brief   IR センサへのパケットのリザルト
 */
enum class IrsensorMcuPacketResult : uint8_t
{
    Success       = 0x00,  //!< 正常状態です。
    DataNotReady  = 0x64,  //!< ImageTransfer モードの初期化中です。
    HardwareError = 0x65,  //!< IR センサーのハードウェアが故障しています。
};

/**
 * @brief   DataRead コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorDataReadOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    uint8_t resendFlag;
    uint8_t resendId;
    uint8_t imageId;
    uint8_t receivedFrameFlag[32]; //!< どのフレームidを受け取ったかのフラグ
    uint8_t crc8;
};

/**
 * @brief   DataRead 内の Ack データ用の構造体
 */
struct IrsensorAckDataPack
{
    typedef ::nn::util::BitPack8::Field<0, 4, uint8_t> WriteRegAckCount; //!< レジスタ Writeの Ack 番号
    typedef ::nn::util::BitPack8::Field<4, 4, uint8_t> Reserved;         //!< 予約
};

/**
 * @brief   DataRead コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(2) IrsensorDataReadInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t dataType;
    uint8_t imageId;
    uint8_t averageIntensityOn;
    uint8_t averageIntensityOff;
    uint16_t lightPixelOn;
    uint16_t lightPixelOff;
    uint8_t payload[IRSENSOR_PACKET_PAYLOAD_SIZE];
    uint8_t timeStamp;
    ::nn::util::BitPack8 ackData;
    uint8_t crc8;
};

/**
 * @brief   ModeSet コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorModeSetOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    uint8_t mode;
    uint8_t transferPacketCount;
    uint8_t majorVersionHigher;
    uint8_t majorVersionLower;
    uint8_t minorVersionHigher;
    uint8_t minorVersionLower;
    uint8_t imageTranserProtocol;
    ::nn::util::BitPack8 pluginSettings;
    uint8_t pluginParameter[16];
    uint8_t reserved[11];
    uint8_t crc8;
};

/**
 * @brief   ModeSet コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorModeSetInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t reserved[310];
    uint8_t crc8;
};

/**
 * @brief   ModeGet コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorModeGetOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    uint8_t reserved[35];
    uint8_t crc8;
};

/**
 * @brief   ModeGet コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorModeGetInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t mode;
    uint8_t majorVersionHigher;
    uint8_t majorVersionLower;
    uint8_t minorVersionHigher;
    uint8_t minorVersionLower;
    uint8_t chipVersionPidLower;
    uint8_t chipVersionPidHigher;
    uint8_t chipVersionCid;
    uint8_t chipVersionReserved;
    uint8_t reserved[301];
    uint8_t crc8;
};

/**
 * @brief   RegisterRead コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorRegisterReadOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    uint8_t setFlag;
    uint8_t bank;
    uint8_t startAddr;
    uint8_t endAddr;
    uint8_t reserved[31];
    uint8_t crc8;
};

/**
 * @brief   RegisterRead コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorRegisterReadInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t bank;
    uint8_t startAddr;
    uint8_t endAddr;
    uint8_t value[256];
    uint8_t reserved[51];
    uint8_t crc8;
};

/**
 * @brief   RegisterWrite コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) WriteRegisterBlock
{
    uint8_t bankId;
    uint8_t address;
    uint8_t value;
};

struct NN_ALIGNAS(1) IrsensorRegisterWriteOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    uint8_t regCount;
    WriteRegisterBlock regBlock[IrWriteRegisterCountMax];
    uint8_t reserved[7];
    uint8_t crc8;
};

/**
 * @brief   RegisterWrite コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorRegisterWriteInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t reserved[310];
    uint8_t crc8;
};

/**
 * @brief   WriteEx 用のアドレス構造体
 */
struct IrsensorWriteExAddressPack
{
    typedef ::nn::util::BitPack8::Field<0, 7, uint8_t> Address;     //!< レジスタアドレス
    typedef ::nn::util::BitPack8::Field<7, 1, uint8_t> Bank;        //!< バンクアドレス(0 or 1)
};

/**
 * @brief   WriteEx 用の設定領域用構造体
 */
struct IrsensorWriteExConfigPack
{
    typedef ::nn::util::BitPack8::Field<0, 7, uint8_t> RegisterNum;  //!< レジスタ数
    typedef ::nn::util::BitPack8::Field<7, 1, uint8_t> FastModeFlag; //!< FastModeFlag(0 or 1)
};

/**
 * @brief   ModeSet/ ModeSetEx のPlugin設定用構造体
 */
struct IrsensorModeSetPluginSettingPack
{
    typedef ::nn::util::BitPack8::Field<0, 5, uint8_t> ParameterSize;    //!< PluginParameter のパラメータサイズ
    typedef ::nn::util::BitPack8::Field<5, 2, uint8_t> Reserved;         // 予約
    typedef ::nn::util::BitPack8::Field<7, 1, uint8_t> ParameterEnable;  //!< PluginParameter の有効無効
};

/**
 * @brief   RegisterWriteEx コマンドの送信 (Console->Mcu) パケットです。
 */
struct NN_ALIGNAS(1) WriteRegisterExBlock
{
    ::nn::util::BitPack8 address;
    uint8_t value;
};

struct NN_ALIGNAS(1) IrsensorRegisterWriteExOutPacket
{
    ::nn::util::BitPack8 header; //!< 送信パケットヘッダ
    uint8_t command;
    ::nn::util::BitPack8 config; //!< 設定関連情報
    WriteRegisterExBlock regBlock[IrWriteRegisterExCountMax];
    uint8_t crc8;
};

/**
 * @brief   RegisterWriteEx コマンドの受信 (Mcu->Console) パケットです。
 */
struct NN_ALIGNAS(1) IrsensorRegisterWriteExInPacket
{
    ::nn::util::BitPack8 misoProtocol;
    uint8_t resultCode;
    uint8_t reserved[310];
    uint8_t crc8;
};


//!< Irsensorの実装のためのベースとなるクラス
class IrsensorBase final : public Peripheral, public ICommandListener
{
public:
    IrsensorBase() NN_NOEXCEPT :
        m_Mutex(true),
        m_pIrSamplingEventType(nullptr),
        m_pIrCommandCompletionEventType(nullptr),
        m_CurrentIrProcessorType(IrProcessorType::Reset),
        m_NextPacketType(IrsensorPacketType::None),
        m_ReadRegSetting(),
        m_ReadRegisterState(),
        m_RegisterReadRetryCount(0),
        m_WriteRegisterState(),
        m_IsSamplingEnabled(false),
        m_CurrentMcuCompatibleVersion(),
        m_ModeGetResult(IrsensorMcuPacketResult::Success),
        m_RegisterWriteId(0),
        m_RegisterWriteTrialCount(0){}
    virtual ~IrsensorBase() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual void ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT NN_OVERRIDE;
    virtual size_t GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual void NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

public:
    //!< IR センサー制御用の通知イベントを設定します。
    Result SetIrControlEvent(nn::os::SystemEventType* pIrSamplingEventType, nn::os::SystemEventType* pIrCommandCompletionEventType) NN_NOEXCEPT;
    //!< IR センサー制御用の通知イベントを解放します。
    Result ReleaseIrControlEvent() NN_NOEXCEPT;
    //!< IR センサーのプロセッサモードを設定します。
    Result SetIrProcessorType(IrProcessorType type, int modeOffset, IrTeraPluginParameter param, IrMcuVersion requiredVersion, IrCommandType commandType) NN_NOEXCEPT;
    //!< IR センサーのプロセッサモードを取得します。
    Result GetIrProcessorType(IrProcessorType* pType, IrMcuVersion* pCompatibleVersion) NN_NOEXCEPT;
    //!< IR センサーのサンプリングを開始します。
    Result StartSampling() NN_NOEXCEPT;
    //!< IR センサーのサンプリングを停止します。停止するまで返りません。
    Result StopSampling() NN_NOEXCEPT;
    //!< IR センサーのプロセッサに渡したワークバッファの解放を行います。
    Result TeardownProcessor() NN_NOEXCEPT;
    //!< IR センサーの通信時に Polling モードを有効化します。
    Result EnablePollingMode() NN_NOEXCEPT;
    //!< IR センサーの通信時に Polling モードを無効化します。
    Result DisablePollingMode() NN_NOEXCEPT;
    //!< Momentモードの初期設定を行います。
    Result SetupMomentProcessor(IrCommonData* pIrCommonWorkBuffer, IrMomentProcessorState* pMomentProcessorWorkBuffer) NN_NOEXCEPT;
    //!< Momentモードの結果を取得します。
    Result GetMomentStates(IrCommonData* pOutIrCommonData, IrMomentProcessorState* pOutMomentProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT;
    //!< Clusteringモードの初期設定を行います。
    Result SetupClusteringProcessor(IrCommonData* pIrCommonWorkBuffer, IrClusteringProcessorState* pClusteringProcessorWorkBuffer) NN_NOEXCEPT;
    //!< Clusteringモードの結果を取得します。
    Result GetClusteringStates(IrCommonData* pOutIrCommonData, IrClusteringProcessorState* pOutClusteringProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT;
    //!< ImageTransferモードの初期設定を行います。
    Result SetupImageTransferProcessor(IrCommonData* pIrCommonWorkBuffer, IrImageTransferProcessorState* pImageTransferProcessorWorkBuffer, IrImageTransferProcessorFormat size) NN_NOEXCEPT;
    //!< ImageTransferモードの結果を取得します。
    Result GetImageTransferState(IrCommonData* pOutIrCommonData, IrImageTransferProcessorState* pOutImageTransferProcessorState) NN_NOEXCEPT;
    //!< TeraPluginモードの初期設定を行います。
    Result SetupTeraPluginProcessor(IrCommonData* pIrCommonWorkBuffer, IrTeraPluginProcessorState* pTeraPluginProcessorWorkBuffer) NN_NOEXCEPT;
    //!< TeraPluginモードの結果を取得します。
    Result GetTeraPluginState(IrCommonData* pOutIrCommonData, IrTeraPluginProcessorState* pOutTeraPluginProcessorState) NN_NOEXCEPT;
    //!< Dpd モードの初期設定を行います。
    Result SetupDpdProcessor(IrCommonData* pIrCommonWorkBuffer, IrDpdProcessorState* pClusteringProcessorWorkBuffer) NN_NOEXCEPT;
    //!< Dpd モードの結果を取得します。
    Result GetDpdStates(IrCommonData* pOutIrCommonData, IrDpdProcessorState* pOutClusteringProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT;
    //!< レジスタ読み込みを開始します。
    Result StartReadRegister(const IrReadRegisterSetting& irReadRegisterSetting) NN_NOEXCEPT;
    //!< レジスタ書き込みを開始します。
    Result StartWriteRegister(const IrWriteRegisterSetting& irWriteRegisterSetting) NN_NOEXCEPT;
    //!< レジスタEx書き込みを開始します。
    Result StartWriteRegisterEx(const IrWriteRegisterSettingEx& irWriteRegisterSetting) NN_NOEXCEPT;
    //!< レジスタ読み込みの結果を取得します。
    Result GetReadRegisterState(IrReadRegisterState* pOutIrReadRegisterState) NN_NOEXCEPT;
    //!< レジスタ書き込みの結果を取得します。
    Result GetWriteRegisterState(IrWriteRegisterState* pOutIrWriteRegisterState) NN_NOEXCEPT;

private:
    void CreateDataReadPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize,
        uint8_t resendFlag, uint8_t resendId, uint8_t imageId, ReceivedFrameFlag receivedFrameFlag) NN_NOEXCEPT;
    void CreateModeSetPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize, IrProcessorType type,
        int modeOffset, IrTeraPluginParameter param, IrMcuVersion requiredVersion, uint8_t spiImagePacketCountMax, IrCommandType commandType, IrImageTransferProtocol protocol)NN_NOEXCEPT;
    void CreateModeGetPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize) NN_NOEXCEPT;
    void CreateRegisterReadPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize, uint8_t setFlag, IrReadRegisterSetting setting) NN_NOEXCEPT;
    void CreateRegisterWritePacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize, IrWriteRegisterSetting setting) NN_NOEXCEPT;
    void CreateRegisterWriteExPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize, IrWriteRegisterSettingEx setting) NN_NOEXCEPT;
    Result ParseProcessorInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;
    int ParseRegisterReadInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT;

private:
    static const size_t InputReportSize = 313; //!< 読み込みデータのレポートサイズ
    static const size_t OutputReportSize = 38; //!< 書き込みデータのレポートサイズ
    static const int VerifyRegisterReadRetryCountMax = 3; //!< レジスタ読み込みの内容確認シーケンスのリトライ最大数
    static const int RegisterWriteCountTimeout = 100; //!< レジスタ書き込みが完了しない場合のタイムアウト (1.5秒)
    static const int RegisterWriteCountDiffMax = 3; //!< レジスタ書き込み番号がパケ落ちなどによって本体とズレるが、それを許容する最大数
    static const int RegisterWriteCountMax
        = 1 << (IrsensorAckDataPack::WriteRegAckCount::Next - IrsensorAckDataPack::WriteRegAckCount::Pos); //!< レジスタ書き込み番号のカウント最大数

private:
    nn::os::Mutex m_Mutex;                           //!< IRセンサプロセッサの各モード間の排他制御用
    nn::os::SystemEventType* m_pIrSamplingEventType; //!< データ転送割り込み通知用のイベント
    nn::os::SystemEventType* m_pIrCommandCompletionEventType; //!< コマンド送信完了通知用のイベント
    IrProcessorType m_CurrentIrProcessorType;               //!< 現在のIRプロセッサの状態
    IrsensorPacketType m_NextPacketType;           //!< 次に IR センサに送るパケットのタイプ
    IrsensorMomentProcessor m_MomentProcessor; //!< Momentモードのプロセッサ
    IrsensorClusteringProcessor m_ClusteringProcessor; //!< Clusteringモードのプロセッサ
    IrsensorImageTransferProcessor m_ImageTransferProcessor; //!< ImageTransfer モードのプロセッサ
    IrsensorTeraPluginProcessor m_TeraPluginProcessor; //!< TeraPlugin モードのプロセッサ
    IrsensorDpdProcessor m_DpdProcessor; //!< Dpd モードのプロセッサ
    IrReadRegisterSetting m_ReadRegSetting; //!< 読み込むレジスタの情報
    IrReadRegisterState m_ReadRegisterState; //!< レジスタReadの結果
    int m_RegisterReadRetryCount;              //!< レジスタ Read の Ack パケットチェックのリトライ回数
    IrWriteRegisterState m_WriteRegisterState; //!< レジスタ Write の結果
    bool m_IsSamplingEnabled;                  //!< サンプリングの有効無効を表すフラグ
    IrMcuVersion m_CurrentMcuCompatibleVersion; //!< ライブラリから要求されたバージョンに合わせて、現在 MCU が互換動作をしているバージョン
    IrsensorMcuPacketResult m_ModeGetResult;    //!< ModeGet コマンドで取得したリザルト
    int m_RegisterWriteId;                      //!< レジスタ Write のコマンド番号（Mcuからの Ack 判定用）
    int m_RegisterWriteTrialCount;              //!< レジスタ Write のコマンド番号の連続チェック回数
};

}} // namespace nn::xcd
