﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_IrsensorTypes.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include "xcd_IrsensorBase.h"
#include "detail/xcd_TeraCommon.h"

namespace nn { namespace xcd {

IrsensorBase::~IrsensorBase() NN_NOEXCEPT
{
    // 何もしない
}

void IrsensorBase::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(type);
    // TeraMCU のモードが IR モードになった直後に呼ばれる
    m_Activated = true;
    // IR センサーの初期化が完了しているはずなので、Ready に設定する
    m_CurrentIrProcessorType = IrProcessorType::Ready;

    // Tera の起動時に内部状態の Ackカウンタがクリア状態になっているので、それにあわせる
    m_RegisterWriteId = 0;
    m_RegisterWriteTrialCount = 0;

    m_FirmwareVersion = firmwareVersion;
}

void IrsensorBase::Deactivate() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // TeraMCU のモードが Standby モードになった直後に呼ばれる
    //
    if (m_Activated)
    {
        // Tera の停止時に内部状態の Ackカウンタがクリアされるので、それにあわせてクリア
        m_RegisterWriteId = 0;
        m_RegisterWriteTrialCount = 0;
        m_CurrentIrProcessorType = IrProcessorType::Reset;
        m_Activated = false;
    }
}


void IrsensorBase::ParseInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pBuffer);

    if (!m_Activated)
    {
        // 既に IrsensorBase が終了処理済みの場合は、データを捨てる
        return;
    }

    // Protocol のチェック
    nn::xcd::detail::ReceiveProtocolType prevProtocolType;
    prevProtocolType.SetValue(pBuffer[IRSENSOR_PACKET_PROTOCOL_OFFSET]);
    auto prevPacketType = static_cast<IrsensorPacketType>(prevProtocolType.GetCommandId());

    //
    // InputReport受信時の処理を実装
    //
    switch(prevPacketType)
    {
    case IrsensorPacketType::ModeSet:
    case IrsensorPacketType::ModeSetEx:
    case IrsensorPacketType::RegisterWrite:
    case IrsensorPacketType::RegisterWriteEx:
        {
            // McuRead を受け取った直後のパケットで、それぞれのデータが残っていることがあるため、
            // そのデータは捨てるようにする。 (WorkAround)
        }
        break;
    case IrsensorPacketType::ModeGet:
        {
            // ModeGet パケットにキャスト
            auto pModeGetInPacket = reinterpret_cast<const IrsensorModeGetInPacket*>(pBuffer);
            // Result を取得
            m_ModeGetResult = static_cast<IrsensorMcuPacketResult>(pModeGetInPacket->resultCode);
            // 受信したデータによって、現在のプロセッサタイプを設定
            if (pModeGetInPacket->mode >= static_cast<int>(IrProcessorType::TeraPlugin))
            {
                m_CurrentIrProcessorType = IrProcessorType::TeraPlugin;
            }
            else
            {
                m_CurrentIrProcessorType = static_cast<IrProcessorType>(pModeGetInPacket->mode);
            }
            // Mcu の互換動作モード
            m_CurrentMcuCompatibleVersion.major = (pModeGetInPacket->majorVersionHigher << 8) | pModeGetInPacket->majorVersionLower;
            m_CurrentMcuCompatibleVersion.minor = (pModeGetInPacket->minorVersionHigher << 8) | pModeGetInPacket->minorVersionLower;
            NN_XCD_IRSENSOR_LOG("ModeGet:%d CompatibleVersion:%d.%d ChipId:%X ChipVer:%d samplingNum:%d\n",
                m_CurrentIrProcessorType,
                m_CurrentMcuCompatibleVersion.major,
                m_CurrentMcuCompatibleVersion.minor,
                ((pModeGetInPacket->chipVersionPidHigher << 8) | pModeGetInPacket->chipVersionPidLower), // chipId
                pModeGetInPacket->chipVersionCid, // chipVer
                sampleNumber);
        }
        break;
    case IrsensorPacketType::RegisterRead:
        {
            // リザルトが OK でないときは何もしない。
            if (pBuffer[IRSENSOR_PACKET_RESULT_OFFSET] != detail::InternalMcuResult_Ok)
            {
                return;
            }

            int result = ParseRegisterReadInputReport(pBuffer, size, sampleNumber);
            if ((result == 0) || (m_RegisterReadRetryCount > VerifyRegisterReadRetryCountMax))
            {
                // コマンドが成功した場合、もしくはリトライを最大数繰り返しても失敗が続いた場合は、
                // CommandCompletionEvent をシグナルする
                m_NextPacketType = IrsensorPacketType::None;
                NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
                nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
            }
            else
            {
                // 失敗した場合は、リトライ用に次フレームのコマンドをここで設定
                m_NextPacketType = IrsensorPacketType::RegisterRead;
                m_RegisterReadRetryCount++;
            }
        }
        break;
    case IrsensorPacketType::DataRead:
        {
            // リザルトが OK でないときは何もしない。
            if (pBuffer[IRSENSOR_PACKET_RESULT_OFFSET] != detail::InternalMcuResult_Ok)
            {
                return;
            }

            // プロセッサからの処理データをパースする
            Result result = ParseProcessorInputReport(pBuffer, size, sampleNumber);
            if (result.IsSuccess())
            {
                // SamplingEvent をシグナルする
                NN_SDK_ASSERT_NOT_NULL(m_pIrSamplingEventType);
                nn::os::SignalSystemEvent(m_pIrSamplingEventType);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

}

Result IrsensorBase::ParseProcessorInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(sampleNumber);
    NN_SDK_ASSERT_NOT_NULL(pBuffer);

    Result result = nn::ResultSuccess();
    if (!m_IsSamplingEnabled)
    {
        // サンプリング停止中はバッファを操作しない。
        return result;
    }

    switch(m_CurrentIrProcessorType)
    {
    case xcd::IrProcessorType::Ready:
        {
        }
        break;
    case xcd::IrProcessorType::Moment:
        {
            result = m_MomentProcessor.ParseMomentData(pBuffer, size, sampleNumber);
        }
        break;
    case xcd::IrProcessorType::Clustering:
        {
            result = m_ClusteringProcessor.ParseClusteringData(pBuffer, size, sampleNumber);
        }
        break;
    case xcd::IrProcessorType::ImageTransfer:
        {
            result = m_ImageTransferProcessor.ParseImageTransferData(pBuffer, size, sampleNumber);
        }
        break;
    case xcd::IrProcessorType::TeraPlugin:
        {
            result = m_TeraPluginProcessor.ParseTeraPluginData(pBuffer, size, sampleNumber);
        }
        break;
    case xcd::IrProcessorType::Dpd:
        {
            result = m_DpdProcessor.ParseDpdData(pBuffer, size, sampleNumber);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return result;
}

/*
 * IR の RegisterRead Responseのパケットをパース
 */
int IrsensorBase::ParseRegisterReadInputReport(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);
    NN_SDK_ASSERT_NOT_NULL(pBuffer);

    // RegisterRead パケットのポインタにキャスト
    auto pRegisterReadInPacket = reinterpret_cast<const IrsensorRegisterReadInPacket*>(pBuffer);

    if ((m_ReadRegSetting.bankId != pRegisterReadInPacket->bank)
    || (m_ReadRegSetting.startAddress != pRegisterReadInPacket->startAddr)
    || (m_ReadRegSetting.size != static_cast<size_t>(pRegisterReadInPacket->endAddr - pRegisterReadInPacket->startAddr + 1)))
    {
        // 結果が送信したコマンドに対応しない場合はエラーとする。
        // パケ落ちの暫定対策として、Ackの機能を果たす。
        // 同じ Read コマンドを連続して投げた場合に厳密に区別することはできないので注意する。
        return -1;
    }

    // RegisterRead の結果をそのままコピー
    m_ReadRegisterState.regSetting.bankId = pRegisterReadInPacket->bank;
    m_ReadRegisterState.regSetting.startAddress = pRegisterReadInPacket->startAddr;
    m_ReadRegisterState.regSetting.size = static_cast<size_t>(pRegisterReadInPacket->endAddr - pRegisterReadInPacket->startAddr + 1);
    std::memcpy(&m_ReadRegisterState.bankData[0], &(pRegisterReadInPacket->value), IrReadRegisterCountMax);

    return 0;
}

size_t IrsensorBase::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    nn::Result result = ResultSuccess();;
    size_t packetSize = 0;

    switch(m_NextPacketType)
    {
    case IrsensorPacketType::None:
        {
            //Do nothing
            NN_XCD_IRSENSOR_LOG("Next:%d, CurrentIrProcType=%d\n", m_NextPacketType, m_CurrentIrProcessorType);
            return packetSize;
        }
        break;
    case IrsensorPacketType::ModeGet:
        {
            CreateModeGetPacket(&packetSize, (char*)pOutValue, size);
        }
        break;
    case IrsensorPacketType::RegisterRead:
        {
            // 新しいデータを読む場合は 1 に設定する
            // ここでは、パケット落ちをケアするために常に設定を送る仕様とする。
            uint8_t setFlag = 1;
            CreateRegisterReadPacket(&packetSize, (char*)pOutValue, size, setFlag, m_ReadRegSetting);
        }
        break;
    case IrsensorPacketType::DataRead:
        {
            uint8_t resendFlag = 0;
            uint8_t resendId = 0;
            uint8_t imageId = 0;
            ReceivedFrameFlag receivedFrameFlag;
            receivedFrameFlag.Reset(); // ImageTransferのFastModeでは無いときはこれまでどおり ALL 0 を送る
            if(m_CurrentIrProcessorType == IrProcessorType::ImageTransfer)
            {
                result = m_ImageTransferProcessor.SetDataReadCommand(resendFlag, resendId, imageId, receivedFrameFlag);
            }
            CreateDataReadPacket(&packetSize, (char*)pOutValue, size, resendFlag, resendId, imageId, receivedFrameFlag);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_XCD_IRSENSOR_LOG("Next:%d, CurrentIrProcType=%d\n", m_NextPacketType, m_CurrentIrProcessorType);
    m_NextPacketType = IrsensorPacketType::None;

    // CommandCompletionEvent をシグナルする
    NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
    nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);

    // ImageTransfer モードかつ、画像取得が完了済みの場合
    if (m_CurrentIrProcessorType == IrProcessorType::ImageTransfer && nn::xcd::ResultIrImageTransferCompleted::Includes(result))
    {
        // 完了コマンドを連続で送信すると、画像フレーム中にキャプチャされる可能性があり、
        // 画像が乱れるため、 McuWrite で発行して、McuOut は送信しない。
        m_pCommand->McuWriteWithoutDuplication(pOutValue, size, this);
        packetSize = 0;
    }
    return packetSize;
}

void IrsensorBase::NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pBuffer);
    NN_UNUSED(size);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // MCU Write で送信したコマンドに対する応答は次の通信時に返ってくる
    // (= MCU Read には、送信したコマンドの結果は入っていない) ので、
    // 今のところ中身のパースしない。
    NN_XCD_IRSENSOR_LOG("Sent McuWrite command to MCU.\n");
}

Result IrsensorBase::SetIrControlEvent(nn::os::SystemEventType* pIrSamplingEventType,
                                       nn::os::SystemEventType* pIrCommandCompletionEventType ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_REQUIRES_NOT_NULL(pIrSamplingEventType);
    NN_SDK_REQUIRES_NOT_NULL(pIrCommandCompletionEventType);

    m_pIrSamplingEventType = pIrSamplingEventType;
    m_pIrCommandCompletionEventType = pIrCommandCompletionEventType;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::ReleaseIrControlEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // イベントのクリア
    nn::os::ClearSystemEvent(m_pIrSamplingEventType);
    nn::os::ClearSystemEvent(m_pIrCommandCompletionEventType);

    m_pIrSamplingEventType = nullptr;
    m_pIrCommandCompletionEventType = nullptr;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::SetIrProcessorType(IrProcessorType type, int modeOffset, IrTeraPluginParameter param, IrMcuVersion requiredVersion, IrCommandType commandType) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_ModeGetResult = IrsensorMcuPacketResult::Success;

    // 現在と異なるモードに設定する場合のみコマンドを発行する。
    if (m_CurrentIrProcessorType != type)
    {
        NN_XCD_IRSENSOR_LOG("Next:%d, NextIrProcType=%d(offset:%d)\n", m_NextPacketType, type, modeOffset);

        // Mcu Writeコマンドの生成
        uint8_t commandBuffer[McuWriteSize_Data] = {0};
        uint8_t packetCountMax = 0;
        auto imageTransferProtocol = IrImageTransferProtocol::Normal;
        if (type == IrProcessorType::ImageTransfer)
        {
            auto transferSize = nn::xcd::IrImageTransferProcessorFormat::ImageSize_320x240;
            m_ImageTransferProcessor.GetTransferFormat(&transferSize);
            packetCountMax = static_cast<uint8_t>(m_ImageTransferProcessor.GetTransferPacketCount(transferSize));

            if (requiredVersion >= nn::xcd::FirmwareVersionTera_0410)
            {
                // 0410 以降のファームでは fast mode を有効化する
                // requiredVersion はこの時点で適切に設定されている
                imageTransferProtocol = IrImageTransferProtocol::FastMode;
            }
        }
        // プロトコルをプロセッサに設定
        m_ImageTransferProcessor.SetTransferProtocol(imageTransferProtocol);

        size_t packetSize = 0;
        CreateModeSetPacket(&packetSize, (char*)commandBuffer, McuWriteSize_Data, type, modeOffset, param, requiredVersion, packetCountMax, commandType, imageTransferProtocol);

        // McuWriteコマンドで発行
        m_pCommand->McuWriteWithoutDuplication(commandBuffer, packetSize, this);
        NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
        nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
    }
    else
    {
        // 既に指定するモードになっている場合は何もせずイベントをシグナルしておく。
        NN_XCD_IRSENSOR_LOG("IrProcType is already %d(offset:%d).\n", type, modeOffset);
        NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
        nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
    }
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetIrProcessorType(IrProcessorType* pType, IrMcuVersion* pCompatibleVersion) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // 最新の処理方式を取得する
    m_NextPacketType = IrsensorPacketType::ModeGet;

    // Mcu から HW エラーのリザルトが返る場合は上位にエラー通知する。
    NN_RESULT_THROW_UNLESS(
        m_ModeGetResult != IrsensorMcuPacketResult::HardwareError,
        ::nn::xcd::ResultIrHardwareError());

    // 現状のデータをコピーする
    *pType = m_CurrentIrProcessorType;
    *pCompatibleVersion = m_CurrentMcuCompatibleVersion;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::StartSampling() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_NextPacketType = IrsensorPacketType::DataRead;

    m_IsSamplingEnabled = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::StopSampling() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // サンプリングの停止
    if(m_NextPacketType != IrsensorPacketType::None)
    {
        NN_XCD_IRSENSOR_LOG("StopSampling event: %d\n", m_NextPacketType);
        // コマンドの予約中に停止した場合、イベントは解放しておく
        NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
        nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
    }
    m_ModeGetResult = IrsensorMcuPacketResult::Success;
    m_NextPacketType = IrsensorPacketType::None;
    m_IsSamplingEnabled = false;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::TeardownProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // サンプリングが停止している必要があります。
    NN_SDK_REQUIRES(!m_IsSamplingEnabled);

    m_MomentProcessor.StopProcessor();
    m_ClusteringProcessor.StopProcessor();
    m_ImageTransferProcessor.StopProcessor();
    m_DpdProcessor.StopProcessor();
    m_TeraPluginProcessor.StopProcessor();

    m_ModeGetResult = IrsensorMcuPacketResult::Success;

    NN_RESULT_SUCCESS;
}


Result IrsensorBase::EnablePollingMode() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_0381)
    {
        // モーメント、クラスタリング、DPDモード時は、McuPollingモードを設定する
        if (m_CurrentIrProcessorType == IrProcessorType::Moment
            || m_CurrentIrProcessorType == IrProcessorType::Clustering
            || m_CurrentIrProcessorType == IrProcessorType::ImageTransfer
            || m_CurrentIrProcessorType == IrProcessorType::Dpd
            || m_CurrentIrProcessorType == IrProcessorType::TeraPlugin)
        {
            uint8_t resendFlag = 0;
            uint8_t resendId = 0;
            uint8_t imageId = 0;
            ReceivedFrameFlag receivedFrameFlag;
            receivedFrameFlag.Set(); // ALL FF を送るようにする
            uint8_t pollingCommandBuffer[McuWriteSize_Data] = { 0 };
            size_t pollingPacketSize = 0;
            CreateDataReadPacket(&pollingPacketSize, (char*)pollingCommandBuffer, McuWriteSize_Data, resendFlag, resendId, imageId, receivedFrameFlag);
            NN_XCD_IRSENSOR_LOG("Enable Polling Mode\n");
            m_pCommand->McuPollingEnable(pollingCommandBuffer, pollingPacketSize, this);
        }
    }
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::DisablePollingMode() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_0381)
    {
        NN_XCD_IRSENSOR_LOG("Disable Polling Mode\n");
        m_pCommand->McuPollingDisable(this);
    }
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::SetupMomentProcessor(IrCommonData* pIrCommonWorkBuffer, IrMomentProcessorState* pMomentProcessorWorkBuffer) NN_NOEXCEPT
{
    m_MomentProcessor.SetupProcessor(pIrCommonWorkBuffer, pMomentProcessorWorkBuffer);
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetMomentStates(IrCommonData* pOutIrCommonData, IrMomentProcessorState* pOutMomentProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT
{
    int writeAckCount = 0;
    nn::Result result = m_MomentProcessor.GetMomentStates(pOutIrCommonData, pOutMomentProcessorStates, &writeAckCount, pOutCount, countMax);
    if (m_CurrentMcuCompatibleVersion >= FirmwareVersionTera_0412)
    {
        m_RegisterWriteTrialCount++;
        //Ackカウンタが一致しない状態が続いた場合は、エラーを上げてリトライする
        if (m_RegisterWriteTrialCount >= RegisterWriteCountTimeout)
        {
            m_RegisterWriteId = 0;
            m_RegisterWriteTrialCount = 0;
            NN_RESULT_THROW(nn::xcd::ResultIrWriteRegisterCountTimeout());
        }
        int distance = (writeAckCount - m_RegisterWriteId + RegisterWriteCountMax) % RegisterWriteCountMax;
        NN_RESULT_THROW_UNLESS(distance < RegisterWriteCountDiffMax , nn::xcd::ResultIrProcessorNotReady());
        // 成功した場合は、ID の値を最新にあわせておく (基本的には一致する)
        m_RegisterWriteId = writeAckCount;
        m_RegisterWriteTrialCount = 0;
    }
    NN_RESULT_THROW(result);
}

Result IrsensorBase::SetupClusteringProcessor(IrCommonData* pIrCommonWorkBuffer, IrClusteringProcessorState* pClusteringProcessorWorkBuffer) NN_NOEXCEPT
{
    m_ClusteringProcessor.SetupProcessor(pIrCommonWorkBuffer, pClusteringProcessorWorkBuffer);
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetClusteringStates(IrCommonData* pOutIrCommonData, IrClusteringProcessorState* pOutClusteringProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT
{
    int writeAckCount = 0;
    nn::Result result = m_ClusteringProcessor.GetClusteringStates(pOutIrCommonData, pOutClusteringProcessorStates, &writeAckCount, pOutCount, countMax);
    if (m_CurrentMcuCompatibleVersion >= FirmwareVersionTera_0412)
    {
        m_RegisterWriteTrialCount++;
        //Ackカウンタが一致しない状態が続いた場合は、エラーを上げてリトライする
        if (m_RegisterWriteTrialCount >= RegisterWriteCountTimeout)
        {
            m_RegisterWriteId = 0;
            m_RegisterWriteTrialCount = 0;
            NN_RESULT_THROW(nn::xcd::ResultIrWriteRegisterCountTimeout());
        }
        int distance = (writeAckCount - m_RegisterWriteId + RegisterWriteCountMax) % RegisterWriteCountMax;
        NN_RESULT_THROW_UNLESS(distance < RegisterWriteCountDiffMax , nn::xcd::ResultIrProcessorNotReady());
        // 成功した場合は、ID の値を最新にあわせておく (基本的には一致する)
        m_RegisterWriteId = writeAckCount;
        m_RegisterWriteTrialCount = 0;
    }
    NN_RESULT_THROW(result);
}

Result IrsensorBase::SetupImageTransferProcessor(IrCommonData* pIrCommonWorkBuffer, IrImageTransferProcessorState* pImageTransferProcessorWorkBuffer, IrImageTransferProcessorFormat size) NN_NOEXCEPT
{
    m_ImageTransferProcessor.SetupProcessor(pIrCommonWorkBuffer, pImageTransferProcessorWorkBuffer, size);
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetImageTransferState(IrCommonData* pOutIrCommonData, IrImageTransferProcessorState* pOutImageTransferProcessorState) NN_NOEXCEPT
{
    int writeAckCount = 0;
    nn::Result result = m_ImageTransferProcessor.GetImageTransferState(pOutIrCommonData, pOutImageTransferProcessorState, &writeAckCount);
    if (m_CurrentMcuCompatibleVersion >= FirmwareVersionTera_0412)
    {
        m_RegisterWriteTrialCount++;
        //Ackカウンタが一致しない状態が続いた場合は、エラーを上げてリトライする
        if (m_RegisterWriteTrialCount >= RegisterWriteCountTimeout)
        {
            m_RegisterWriteId = 0;
            m_RegisterWriteTrialCount = 0;
            NN_RESULT_THROW(nn::xcd::ResultIrWriteRegisterCountTimeout());
        }
        int distance = (writeAckCount - m_RegisterWriteId + RegisterWriteCountMax) % RegisterWriteCountMax;
        NN_RESULT_THROW_UNLESS(distance < RegisterWriteCountDiffMax , nn::xcd::ResultIrProcessorNotReady());
        // 成功した場合は、ID の値を最新にあわせておく (基本的には一致する)
        m_RegisterWriteId = writeAckCount;
        m_RegisterWriteTrialCount = 0;
    }
    NN_RESULT_THROW(result);
}

Result IrsensorBase::SetupTeraPluginProcessor(IrCommonData* pIrCommonWorkBuffer, IrTeraPluginProcessorState* pTeraPluginProcessorWorkBuffer) NN_NOEXCEPT
{
    m_TeraPluginProcessor.SetupProcessor(pIrCommonWorkBuffer, pTeraPluginProcessorWorkBuffer);
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetTeraPluginState(IrCommonData* pOutIrCommonData, IrTeraPluginProcessorState* pOutTeraPluginProcessorState) NN_NOEXCEPT
{
    return m_TeraPluginProcessor.GetTeraPluginState(pOutIrCommonData, pOutTeraPluginProcessorState);
}

Result IrsensorBase::SetupDpdProcessor(IrCommonData* pIrCommonWorkBuffer, IrDpdProcessorState* pDpdProcessorWorkBuffer) NN_NOEXCEPT
{
    m_DpdProcessor.SetupProcessor(pIrCommonWorkBuffer, pDpdProcessorWorkBuffer);
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetDpdStates(IrCommonData* pOutIrCommonData, IrDpdProcessorState* pOutDpdProcessorStates, int* pOutCount, int countMax) NN_NOEXCEPT
{
    return m_DpdProcessor.GetDpdStates(pOutIrCommonData, pOutDpdProcessorStates, pOutCount, countMax);
}

Result IrsensorBase::StartReadRegister(const IrReadRegisterSetting& irReadRegisterSetting) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_NextPacketType = IrsensorPacketType::RegisterRead;
    m_ReadRegSetting = irReadRegisterSetting;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::StartWriteRegister(const IrWriteRegisterSetting& irWriteRegisterSetting) NN_NOEXCEPT
{
    // McuWriteコマンドの生成
    uint8_t commandBuffer[McuWriteSize_Data] = {0};
    // 新しいデータを読む場合は 1 に設定する
    size_t packetSize = 0;
    CreateRegisterWritePacket(&packetSize, (char*)commandBuffer, McuWriteSize_Data, irWriteRegisterSetting);
    // McuWriteコマンドで発行
    m_pCommand->McuWriteWithoutDuplication(commandBuffer, packetSize, this);
    NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
    nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
    // WriteReg のコマンド番号をインクリメント (連続で大量に発行されることがないため、4bit で十分)
    m_RegisterWriteId = (m_RegisterWriteId + 1) % RegisterWriteCountMax;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::StartWriteRegisterEx(const IrWriteRegisterSettingEx& irWriteRegisterSetting) NN_NOEXCEPT
{
    // McuWriteコマンドの生成
    uint8_t commandBuffer[McuWriteSize_Data] = {0};
    // 新しいデータを読む場合は 1 に設定する
    size_t packetSize = 0;
    CreateRegisterWriteExPacket(&packetSize, (char*)commandBuffer, McuWriteSize_Data, irWriteRegisterSetting);
    // McuWriteコマンドで発行
    m_pCommand->McuWriteWithoutDuplication(commandBuffer, packetSize, this);
    NN_SDK_ASSERT_NOT_NULL(m_pIrCommandCompletionEventType);
    nn::os::SignalSystemEvent(m_pIrCommandCompletionEventType);
    // WriteReg のコマンド番号をインクリメント (連続で大量に発行されることがないため、4bit で十分)
    m_RegisterWriteId = (m_RegisterWriteId + 1) % RegisterWriteCountMax;
    NN_RESULT_SUCCESS;
}

Result IrsensorBase::GetReadRegisterState(IrReadRegisterState* pOutIrReadRegisterState) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // リトライ回数上限を超えていた場合は、APIリトライ要求エラーを返す
    if (m_RegisterReadRetryCount > VerifyRegisterReadRetryCountMax)
    {
        m_RegisterReadRetryCount = 0;
        return nn::xcd::ResultIrCommandFailed();
    }
    else
    {
        *pOutIrReadRegisterState = m_ReadRegisterState;
        m_RegisterReadRetryCount = 0;
        NN_RESULT_SUCCESS;
    }
}

Result IrsensorBase::GetWriteRegisterState(IrWriteRegisterState* pOutIrWriteRegisterState) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    *pOutIrWriteRegisterState = m_WriteRegisterState;
    NN_RESULT_SUCCESS;
}

/*
 * 次に送信する IR コマンドパケットを生成
 */
void IrsensorBase::CreateDataReadPacket(
    size_t* pOutPacketSize,
    char* pOutPacketData,
    size_t maxSize,
    uint8_t resendFlag,
    uint8_t resendId,
    uint8_t imageId,
    ReceivedFrameFlag receivedFrameFlag) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorDataReadOutPacket dataReadOutPacket = {};
    dataReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    dataReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_Ir));
    dataReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    dataReadOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::DataRead);
    dataReadOutPacket.resendFlag = resendFlag;
    dataReadOutPacket.resendId = resendId;
    dataReadOutPacket.imageId = imageId;
    std::memcpy(dataReadOutPacket.receivedFrameFlag, &receivedFrameFlag, sizeof(ReceivedFrameFlag));
    std::memcpy(pOutPacketData, &dataReadOutPacket, sizeof(dataReadOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(dataReadOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

/*
 * 次に送信する IR コマンドパケットを生成
 */
void IrsensorBase::CreateModeSetPacket(
    size_t* pOutPacketSize,
    char* pOutPacketData,
    size_t maxSize,
    IrProcessorType type,
    int modeOffset,
    IrTeraPluginParameter param,
    IrMcuVersion requiredVersion,
    uint8_t spiImagePacketCountMax,
    IrCommandType commandType,
    IrImageTransferProtocol protocol
    ) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorModeSetOutPacket modeSetOutPacket = {};
    modeSetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    modeSetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_McuWrite));
    modeSetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    if (commandType == IrCommandType::Extension && type != IrProcessorType::Ready)
    {
        modeSetOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::ModeSetEx);
    }
    else
    {
        modeSetOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::ModeSet);
    }
    if (type == IrProcessorType::TeraPlugin)
    {
        modeSetOutPacket.mode = static_cast<uint8_t>(IrProcessorType::TeraPlugin)
            + static_cast<uint8_t>(modeOffset);
        modeSetOutPacket.pluginSettings.Set<nn::xcd::IrsensorModeSetPluginSettingPack::ParameterEnable>(param.isParameterEnabled ? 1 : 0);
        modeSetOutPacket.pluginSettings.Set<nn::xcd::IrsensorModeSetPluginSettingPack::ParameterSize>(static_cast<uint8_t>(param.parameterSize));
        modeSetOutPacket.pluginSettings.Set<nn::xcd::IrsensorModeSetPluginSettingPack::Reserved>(0);
        std::memcpy(&modeSetOutPacket.pluginParameter, &param.parameter, param.parameterSize);
    }
    else
    {
        modeSetOutPacket.mode = static_cast<uint8_t>(type);
    }
    modeSetOutPacket.transferPacketCount = spiImagePacketCountMax;
    modeSetOutPacket.majorVersionHigher = (requiredVersion.major >> 8) & 0xFF;
    modeSetOutPacket.majorVersionLower = requiredVersion.major & 0xFF;
    modeSetOutPacket.minorVersionHigher = (requiredVersion.minor >> 8) & 0xFF;
    modeSetOutPacket.minorVersionLower = requiredVersion.minor & 0xFF;
    modeSetOutPacket.imageTranserProtocol = static_cast<uint8_t>(protocol);
    std::memcpy(pOutPacketData, &modeSetOutPacket, sizeof(modeSetOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(modeSetOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

/*
 * IR の ModeGet コマンドのパケットを生成
 */
void IrsensorBase::CreateModeGetPacket(size_t* pOutPacketSize, char* pOutPacketData, size_t maxSize) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorModeGetOutPacket modeGetOutPacket = {};
    modeGetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    modeGetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_Ir));
    modeGetOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    modeGetOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::ModeGet);
    std::memcpy(pOutPacketData, &modeGetOutPacket, sizeof(modeGetOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(modeGetOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

/*
 * IR の RegisterRead コマンドのパケットを生成
 */
void IrsensorBase::CreateRegisterReadPacket(
    size_t* pOutPacketSize,
    char* pOutPacketData,
    size_t maxSize,
    uint8_t setFlag,
    IrReadRegisterSetting setting) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorRegisterReadOutPacket registerReadOutPacket = {};
    registerReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    registerReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_Ir));
    registerReadOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    registerReadOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::RegisterRead);
    registerReadOutPacket.setFlag = setFlag;
    registerReadOutPacket.bank = setting.bankId;
    registerReadOutPacket.startAddr = setting.startAddress;
    registerReadOutPacket.endAddr = setting.startAddress + static_cast<uint8_t>(setting.size) - 1;
    std::memcpy(pOutPacketData, &registerReadOutPacket, sizeof(registerReadOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(registerReadOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

/*
 * IR の RegisterWrite コマンドのパケットを生成
 */
void IrsensorBase::CreateRegisterWritePacket(
    size_t* pOutPacketSize,
    char* pOutPacketData,
    size_t maxSize,
    IrWriteRegisterSetting setting) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorRegisterWriteOutPacket registerWriteOutPacket = {};
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_McuWrite));
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    registerWriteOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::RegisterWrite);
    registerWriteOutPacket.regCount = static_cast<uint8_t>(setting.registerCount);
    for (int i = 0; i < IrWriteRegisterCountMax; i++)
    {
        registerWriteOutPacket.regBlock[i].bankId = setting.registerBlock[i].bankId;
        registerWriteOutPacket.regBlock[i].address = setting.registerBlock[i].address;
        registerWriteOutPacket.regBlock[i].value = setting.registerBlock[i].data;
    }
    std::memcpy(pOutPacketData, &registerWriteOutPacket, sizeof(registerWriteOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(registerWriteOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

/*
 * IR の RegisterWriteEx コマンドのパケットを生成
 */
void IrsensorBase::CreateRegisterWriteExPacket(
    size_t* pOutPacketSize,
    char* pOutPacketData,
    size_t maxSize,
    IrWriteRegisterSettingEx setting) NN_NOEXCEPT
{
    NN_UNUSED(maxSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketSize);
    NN_SDK_ASSERT_NOT_NULL(pOutPacketData);

    // パケットの値をパッキング
    IrsensorRegisterWriteExOutPacket registerWriteOutPacket = {};
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Protocol>(nn::Bit8(detail::CommandProtocol_Ir));
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::BluetoothMode>(nn::Bit8(detail::BluetoothMode_McuWrite));
    registerWriteOutPacket.header.Set<nn::xcd::detail::PacketModePackForSend::Reserved>(0);
    registerWriteOutPacket.command = static_cast<uint8_t>(IrsensorPacketType::RegisterWriteEx);
    registerWriteOutPacket.config.Set<nn::xcd::IrsensorWriteExConfigPack::RegisterNum>(nn::Bit8(setting.registerCount));
    registerWriteOutPacket.config.Set<nn::xcd::IrsensorWriteExConfigPack::FastModeFlag>(nn::Bit8(setting.fastModeFlag));
    for (int i = 0; i < IrWriteRegisterExCountMax; i++)
    {
        registerWriteOutPacket.regBlock[i].address.Set<nn::xcd::IrsensorWriteExAddressPack::Address>(nn::Bit8(setting.registerBlock[i].address));
        registerWriteOutPacket.regBlock[i].address.Set<nn::xcd::IrsensorWriteExAddressPack::Bank>(nn::Bit8(setting.registerBlock[i].bankId));
        registerWriteOutPacket.regBlock[i].value = setting.registerBlock[i].data;
    }
    std::memcpy(pOutPacketData, &registerWriteOutPacket, sizeof(registerWriteOutPacket));

    // 末尾に CRC8 を付与
    // CRC は BT mode + protocol を除いたデータに対して計算
    size_t packetSize = sizeof(registerWriteOutPacket);
    pOutPacketData[packetSize - 1] = detail::CalcCrc8(pOutPacketData + 1, packetSize - 2);

    NN_SDK_ASSERT_LESS_EQUAL(packetSize, maxSize);
    *pOutPacketSize = packetSize;
}

}} // namespace nn::xcd
