﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Input.h>

#include "xcd_BluetoothTypes.h"
#include "xcd_FirmwareVersionTypes.h"


namespace nn { namespace xcd {

// 複数のボタンが押下されてからの経過時間を表す構造体
struct ButtonTriggerElapsedTime
{
    uint16_t time[6];
};

//!< コマンドの実行結果を待ち受けるためのクラスです
class ICommandListener
{
    NN_DISALLOW_COPY(ICommandListener);
    NN_DISALLOW_MOVE(ICommandListener);

public:
    ICommandListener() NN_NOEXCEPT { /* 何もしない */ };
    virtual ~ICommandListener() NN_NOEXCEPT { /* 何もしない */ };

/**
 * @brief      HidCommandに対してAckが受信された際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 * @param[in]  id                           Ackに対応するHidCommandのId(Command_XXXXX::Id)
 *
 */
    virtual void NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        NN_UNUSED(id);
        // 何もしない
    };

/**
 * @brief      SetDataFormat が完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 * @param[in]  dataFormat                   変更された DataFormat
 *
 */
    virtual void NotifySetDataFormat(Result result, PeriodicDataFormat dataFormat) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        NN_UNUSED(dataFormat);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して LinkKey を受領が完了し、ペアリング情報を更新するタイミングとなった際に呼ばれる関数です。
 *
 * @param[in]  deviceInfo                   deviceInfo
 *
 */
    virtual void NotifyPairingDatabaseUpdate(const BluetoothDeviceInfo& deviceInfo) NN_NOEXCEPT
    {
        NN_UNUSED(deviceInfo);
        // 何もしない
    };

/**
 * @brief      HidCommandに対してPairing完了が受信された際の通知を受け取るための関数です。
 *
 * @param[in]  Result                       ペアリングの実行結果
 *
 */
    virtual void NotifyPairingComplete(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対してDeviceInfoが受信された際の通知を受け取るための関数です
 *
 * @param[in]  pDeviceInfo                  受信されたDeviceInfoの内容
 * @param[in]  firmwareVersion              受信されたファームウェアバージョン
 *
 */
    virtual void NotifyDeviceInfo(const DeviceInfo& deviceInfo, FirmwareVersionUnit firmwareVersion, int serialFlashFormatVersion) NN_NOEXCEPT
    {
        NN_UNUSED(deviceInfo);
        NN_UNUSED(firmwareVersion);
        NN_UNUSED(serialFlashFormatVersion);
        /* 何もしない */
    };

/**
 * @brief      HidCommand に対してボタンの経過時間取得が完了した際に通知を受けるための関数です。
 *
 * @param[in]  value                 ボタンが押されてからの経過時間
 *
 */
    virtual void NotifyButtonTriggerElapsedTime(const ButtonTriggerElapsedTime& value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対してMcuReadが受信された際の通知を受け取るための関数です
 *
 * @param[in]  pBuffer                   受信されたMCUからのデータ
 * @param[in]  size                      MCUから受信されたデータのサイズ
 *
 */
    virtual void NotifyMcuRead(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(pBuffer);
        NN_UNUSED(size);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して 6軸センサーのユーザー CAL への書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateSixAxisSensorUserCal(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して 左アナログスティックのユーザー CAL への書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateLeftAnalogStickUserCal(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して 右アナログスティックのユーザー CAL への書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateRightAnalogStickUserCal(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して コントローラーの色情報書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateControllerColor(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して コントローラーのデザイン情報書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateDesignInfo(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して シリアルフラッシュのバージョン書き込みが完了した際の通知を受け取るための関数です
 *
 * @param[in]  result                       Resultの実行結果
 *
 */
    virtual void NotifyUpdateFormatVersion(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        // 何もしない
    };

/**
 * @brief      HidCommandに対して シリアルナンバー 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   センサーのキャリブレーション値
 *
 */
    virtual void NotifyIdentificationCode(const IdentificationCode& value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して CAL1 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   センサーのキャリブレーション値
 *
 */
    virtual void NotifyCal1(const SensorCalibrationValue& value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して CAL2 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   アナログスティックのキャリブレーション値
 *
 */
    virtual void NotifyCal2(const AnalogStickValidRange& leftValue,
                            const AnalogStickValidRange& rightValue,
                            ::nn::util::Color4u8Type& mainColor,
                            ::nn::util::Color4u8Type& subColor) NN_NOEXCEPT
    {
        NN_UNUSED(leftValue);
        NN_UNUSED(rightValue);
        NN_UNUSED(mainColor);
        NN_UNUSED(subColor);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して Design 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  color                   色情報
 * @param[in]  variation               デザインバリエーション
 *
 */
    virtual void NotifyDesign(const DeviceColor& color,
                              uint8_t variation) NN_NOEXCEPT
    {
        NN_UNUSED(color);
        NN_UNUSED(variation);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して MODEL1 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   アナログスティックのキャリブレーション値
 *
 */
    virtual void NotifyModel1(const SensorState& sensorHorizontalOffset,
                              const AnalogStickDeviceParameter& leftStickParam,
                              const uint16_t leftOriginPlay,
                              const uint16_t leftCircuitValidRatio,
                              const bool     leftIsXScalingRequired) NN_NOEXCEPT
    {
        NN_UNUSED(sensorHorizontalOffset);
        NN_UNUSED(leftStickParam);
        NN_UNUSED(leftOriginPlay);
        NN_UNUSED(leftCircuitValidRatio);
        NN_UNUSED(leftIsXScalingRequired);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して MODEL2 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   アナログスティックのキャリブレーション値
 *
 */
    virtual void NotifyModel2(const AnalogStickDeviceParameter& rightStickParam,
                              const uint16_t rightOriginPlay,
                              const uint16_t rightCircuitValidRatio,
                              const bool     rightIsXScalingRequired) NN_NOEXCEPT
    {
        NN_UNUSED(rightStickParam);
        NN_UNUSED(rightOriginPlay);
        NN_UNUSED(rightCircuitValidRatio);
        NN_UNUSED(rightIsXScalingRequired);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して USERCAL1 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   アナログスティックのキャリブレーション値
 *
 */
    virtual void NotifyUserCal1(const bool isLeftStickAvailable,
                                const AnalogStickValidRange& leftValue,
                                const bool isRightStickAvailable,
                                const AnalogStickValidRange& rightValue,
                                const bool isSensorAvailable) NN_NOEXCEPT
    {
        NN_UNUSED(isLeftStickAvailable);
        NN_UNUSED(leftValue);
        NN_UNUSED(isRightStickAvailable);
        NN_UNUSED(rightValue);
        NN_UNUSED(isSensorAvailable);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して USERCAL2 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pValue                   アナログスティックのキャリブレーション値
 *
 */
    virtual void NotifyUserCal2(const SensorCalibrationValue& value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        /* 何もしない */
    };

/**
 * @brief      HidCommandに対して任意の SerialFlash 領域の読み出しが完了した際にに通知を受けるための関数です
 *
 * @param[in]  pBuffer                  読み出したシリアルフラッシュのデータを格納するバッファ
 * @param[in]  size                     読み出したサイズ
 *
 */
    virtual void NotifySerialFlashRead(const uint8_t* pBuffer, int size) NN_NOEXCEPT
    {
        NN_UNUSED(pBuffer);
        NN_UNUSED(size);
        /* 何もしない */
    };

/**
* @brief      HidCommandに対して ExtDevInfo 読み出しが完了した際にに通知を受けるための関数です
*
* @param[in]  extDeviceType                   外部拡張デバイスの ID
*
*/
    virtual void NotifyExtDevInfo(uint8_t extDeviceType) NN_NOEXCEPT
    {
        NN_UNUSED(extDeviceType);
        /* 何もしない */
    };

/**
* @brief      HidCommandに対して ExtDevRead 読み出しが完了した際にに通知を受けるための関数です
*
* @param[in]  status                          データのステータス
* @param[in]  pData                           データへのポインタ
* @param[in]  dataSize                        データのサイズ
*
*/
    virtual void NotifyExtDevRead(uint8_t status, const uint8_t* pData, uint8_t size) NN_NOEXCEPT
    {
        NN_UNUSED(status);
        NN_UNUSED(pData);
        NN_UNUSED(size);
        /* 何もしない */
    };
};

}} // namespace nn::xcd
